/* fileindex.c -- Copyright 1989, 1990 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* A simple program to give information about one or more files about
 * which information is stored in the NX-Text database.
 *
 * $Id: fileindex.c,v 1.4 91/03/02 18:56:53 lee Rel1-10 $
 */

#include "globals.h" /* defines and declarations for database filenames */

#include <stdio.h>
#include <sys/types.h>
#include <malloc.h>
#include "emalloc.h"
#include "fileinfo.h"

static char *Revision = "@(#) lqtext 2.3 89/11/34";

/* The position of the \n in the 26-char string returned by ctime(3): */
#define DATENEWLINE 24

char *progname;
int AsciiTrace = 0;

/** System calls and library functions used in this file: **/

/** Unix System calls: **/
extern void exit();
/** System Library Functions: **/

/** external lqtext functions: **/
extern void cleanupdb(), SetDefaults();
int SaveFileInfo(), GetFilterType();
#ifndef efree
 extern void efree();
#endif
/** Functions defined within this file: **/
void AddInfo(), AllInfo(), Display(), PrintInfo();

int AllFiles = 0;
int ListMode = 0;
int AddFiles = 0;

int
main(argc, argv)
    int argc;
    char *argv[];
{
    extern int optind, getopt();
    /** extern char *optarg; (unused at the moment) **/
    int ch;
    int ErrorFlag = 0;

    progname = argv[0];

    SetDefaults(argc, argv);

    /* All programs take Zz:Vv */
    while ((ch = getopt(argc, argv, "Zz:VvAax")) != EOF) {
	switch (ch) {
	case 'z':
	case 'Z':
	    break; /* done by SetDefaults(); */
	case 'V':
	    fprintf(stderr, "%s version %s\n", progname, Revision);
	    break;
	case 'v':
	    AsciiTrace = 1;
	    break;
	case 'A':
	    AddFiles = 1;
	    break;
	case 'a':
	    AllFiles = 1;
	    break;
	case 'l':
	    ListMode = 1;
	    break;
	case 'x':
	    ErrorFlag = (-1);
	    break;
	case '?':
	    ErrorFlag = 1;
	    break;
	}
    }

    /* Normally put call to lrqError here to give a helpful message,
     * but not yet ready to ship the error handling package, sorry
     */
    if (ErrorFlag) {
	fprintf(stderr, "%s: usage: %s [options] [files]\n",progname,progname);
	fprintf(stderr, "%s: options are:\n", progname);
	fputs("\
	-c file -- treat the named file as a list of common words\n\
	-d dir	-- use the lq-text database in the directory \"dir\"\n\
	-l	-- list mode: no header output or lines drawn\n\
	-s	-- show the list of saved files\n\
	-t N	-- set trace level to N [default: 0]\n\
	-V	-- print version information\n\
	-v	-- be verbose (same as -t 1)\n\
	-x	-- print this explanation\n\
\n\
In addition, if no files are given, the following are understood:\n\
	-A	-- add the named files to the list of known files\n\
	-a	-- list information about all files\n", stderr);
	exit((ErrorFlag > 0) ? 1 : 0);
    }

    if (AllFiles && AddFiles) {
	fprintf(stderr, "%s: do not use both -a and -A options\n", progname);
	fprintf(stderr, "\tuse %s -x for further explanation.\n", progname);
	exit(1);
    }

    if (optind >= argc && !AllFiles && !AddFiles) {
	fprintf(stderr,
	"%s: You must either give the -a option or specify files to list.\n",
		progname);
	fprintf(stderr, "\tuse %s -x for further explanation.\n", progname);
	exit(1);
    }

    if (!AddFiles || !ListMode) {
	printf("%-7.7s | T | %-20.20s | %s\n",
		"FID", "Date Last indexed", "Current Location");
	puts(
"========|===|======================|=========================================="
	);
    }
    if (AllFiles) {
	AllInfo();
    } else {
	if (AddFiles) {
	    extern lqWriteAccess();

	    lqWriteAccess();
	}

	while (optind < argc) {
	    if (AddFiles) {
		AddInfo(argv[optind++]);
	    } else {
		PrintInfo(argv[optind++]); /* ugh */
	    }
	}
    }
    cleanupdb(); /* close dbm files */
    exit(0);
    /*NOTREACHED*/
    return 1; /* for lint and gcc... */
}

void
PrintInfo(Name)
    char *Name;
{
    extern t_FileInfo *GetFileInfo();
    long FID;
    extern long atol();
    extern t_FID Name2FID();

    t_FileInfo *FileInfo;

    if ((FID = Name2FID(Name)) == (t_FID) 0) {
	fprintf(stderr, "No FID information for filename: %s\n", Name);
        if ((FID = atol(Name)) == (t_FID) 0) {
	    return;
	}
    }

    /* get info from the list */
    if ((FileInfo = GetFileInfo(FID)) == (t_FileInfo *) 0) {
	fprintf(stderr, "No index information for: %s\n", Name);
	return;
    }
    Display(FileInfo);
}

void
Display(FileInfo)
    t_FileInfo *FileInfo;
{
    extern char *ctime();
    char *DateString;

    DateString = ctime(&(FileInfo->Date));
    DateString[DATENEWLINE] = '\0'; /* delete the trailing newline */

    if (ListMode) {
	printf("%lu %d %s %s\n",
	FileInfo->FID, FileInfo->FilterType, &DateString[4], FileInfo->Name);
    } else {
	printf("%7lu | %d | %-20.20s | %s\n",
	    FileInfo->FID, FileInfo->FilterType, &DateString[4], FileInfo->Name);
    }
}

/**
Mon Sep 25 23:58:53 BST 1989
FID     | T | Date Last indexed    | Current Location
========|===|======================|===========================================
      1 | 0 | Sep 25 20:31:26 1989 | /usr2/liam/Bible/NT/John/john01.kjv
      2 | 0 | Sep 25 20:31:28 1989 | /usr2/liam/Bible/NT/John/john02.kjv
      3 | 0 | Sep 25 20:31:30 1989 | /usr2/liam/Bible/NT/John/john03.kjv
**/

void
AllInfo()
{
    extern long GetMaxFID();
    extern t_FileInfo *GetFileInfo();

    t_FileInfo *FileInfo;
    long FID;
    long MaxFid = GetMaxFID();

    for (FID = 0L; FID <= MaxFid; FID++) {
	if ((FileInfo = GetFileInfo(FID)) != (t_FileInfo *) 0) {
	    Display(FileInfo);
	    efree(FileInfo); /* NOTDONE use destroyfileinfo() */
	}
    }
    printf("Max File Identifier is %lu\n", MaxFid);
}

void
AddInfo(FileName)
    char *FileName;
{
    extern time_t time();
    extern unsigned long GetNextFID();
    t_FileInfo FileInfo;

    FileInfo.Name = FileName;
    (void) time(&(FileInfo.Date));
    FileInfo.FID = GetNextFID();
    FileInfo.Stream = 0; /* force GetFilterType to use open()? */

    /* determine filter type */
    FileInfo.FilterType = GetFilterType(&FileInfo);

    printf("%d %s (type %d) %s\n",
	    FileInfo.FID,
	    FileInfo.Name,
	    FileInfo.FilterType,
	    SaveFileInfo(&FileInfo) == 0 ?
			    "saved successfully." :
			    "not saved."
    );
}

/*
 * $Log:	fileindex.c,v $
 * Revision 1.4  91/03/02  18:56:53  lee
 * Now asks for write access iff [sic] necessary
 * 
 * Revision 1.3  90/10/06  00:50:50  lee
 * Prepared for first beta release.
 * 
 * Revision 1.2  90/08/29  21:44:51  lee
 * Alpha release
 * 
 * Revision 1.1  90/08/09  19:17:11  lee
 * Initial revision
 * 
 * Revision 2.2  89/10/08  20:45:46  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:14:18  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.2  89/09/16  21:16:17  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/07  21:05:55  lee
 * Initial revision
 *
 */
