/* smalldb.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* Simple interface to start and end dbm.
 * You may also need to supply dbm_store() and dbm_fetch(), but these
 * should certainly be macros.
 *
 * $Id: smalldb.c,v 1.5 91/03/03 00:15:22 lee Rel1-10 $
 *
 * $Log:	smalldb.c,v $
 * Revision 1.5  91/03/03  00:15:22  lee
 * Improved an error message and fixed a permissions bug.
 * 
 * Revision 1.4  91/03/02  18:52:48  lee
 * Default access is now read only -- lqWriteAccess must be called otherwise.
 * 
 * Revision 1.3  90/10/06  00:12:20  lee
 * Prepared for first beta release.
 * 
 * Revision 1.2  90/09/20  17:53:26  lee
 * slight error reporting improvement.
 * 
 * Revision 1.1  90/08/09  19:16:56  lee
 * Initial revision
 * 
 * Revision 2.2  89/10/08  20:47:14  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:15:55  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.2  89/09/16  21:18:39  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/07  21:06:11  lee
 * Initial revision
 * 
 *
 */

#include "globals.h"

#include <stdio.h>

#include <fcntl.h>
#ifdef BSD
# include <sys/param.h>
# define PATH_MAX MAXPATHLEN /* untested, sorry */
#else /*!BSD*/
# include <limits.h> /* for PATH_MAX */
#endif
#include "smalldb.h"
#include "emalloc.h"

extern int strcmp();
extern char *strcpy();

/* The physical database for the list of words, and for the list
 * of files, uses ndbm.
 * The advantage of this is that it takes only two file system accesses
 * to retrieve any data item (honest!).
 * It's also reasonably fast at insertion.
 * One disadvantage is that it doesn't cope if too many words have the
 * same (32-bit) hash function, although publicly available replacements
 * such as the GNU project's gdbm fix this.
 *
 * Since starting the database is expensive (two opens and a malloc),
 * I have a cache of DBM pointers and keep them open.  Versions of the
 * dbm routines that don't support more than one database will have to
 * have a cache-size of one!
 * I am not sure what the impact of this would be on performance; for
 * adding a new file it shouldn't be too bad, as the file list is examined
 * only once for each file, during reading, and the word database is looked
 * at (at least once for each distinct word) only on writing.
 * For retrieval, however, the word database will be looked at for each
 * word in the query, and the file database for (potentially) each match
 * of each word, so the requests will be more interspersed.
 * Under no circumstances is it acceptable to dispense with the cache, as
 * otherwise you will be doing (literally) thousands of calls to
 * open() and close() per second!
 *
 */

#undef startdb

#ifndef CACHE
/* It's unusual to deal with lots of databases at once, so let's not
 * waste RAM...
 */
# define CACHE 3
#endif

static char NameCache[CACHE][PATH_MAX + 1]; /* + 1 for \0, I think */
static DBM *Cache[CACHE]; /* (set to zero by definition) */

static int MaxInCache = (-1);

/* FileFlags and Mode are passed to dbm_open */
static int FileFlags = O_RDONLY;
static int FileModes = 0;

void
lqWriteAccess()
{
    FileFlags = O_RDWR|O_CREAT;
    FileModes = 0664; /* owner and group write, others read only */
}

DBM *
startdb(FilePrefix)
    char *FilePrefix;
{
    extern int errno;
    register int i;

    for (i = 0; i <= MaxInCache; i++) {
	if (Cache[i] && STREQ(NameCache[i], FilePrefix)) {
	    return Cache[i];
	}
    }

    /* Find an empty slot */
    for (i = 0; i <= MaxInCache; i++) {
	if (Cache[i] == (DBM *) 0) break;
    }

    if (i > MaxInCache) {
	if (i >= CACHE) i = 0;
    }

    if (Cache[i]) dbm_close(Cache[i]);
    NameCache[i][0] = '\0';

    errno = 0;

    if ((Cache[i] = dbm_open(FilePrefix, FileFlags, FileModes)) == (DBM *)0) {
	int e = errno;
	(void) fprintf(stderr, "%s: dbm_open error %d: ", progname, errno);
	errno = e;
	perror(FilePrefix);
	exit(1);
    }
    (void) strcpy(NameCache[i], FilePrefix);
    if (i > MaxInCache) MaxInCache = i;

    return Cache[i];
}

#undef enddb

/*ARGSUSED*/
void
enddb(db)
    DBM *db;
{
    /* no-op */
}

void
cleanupdb()
{
    register int i;

    for (i = 0; i <= MaxInCache; i++) {
	if (Cache[i]) dbm_close(Cache[i]);
	Cache[i] = (DBM *) 0;
	NameCache[i][0] = '\0';
    }
}
