/* WordInfo.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* WordInfo.c -- handle the database of words for NX-Text.
 * 
 * NX-Text keeps a master list of all of the words that have ever been
 * seen.  Currently, this is in dbm format (sdbm or ndbm).
 * For each word, there's an associated WID (a unique number), an offset
 * into the master database (see pblock.c), and possibly thesaurus info.
 *
 * $Id: WordInfo.c,v 2.11 90/10/13 03:10:07 lee Rel1-10 $
 *
 * $Log:	WordInfo.c,v $
 * Revision 2.11  90/10/13  03:10:07  lee
 * Type error -- efree() needs a char *.
 * 
 * Revision 2.10  90/10/06  00:12:01  lee
 * Prepared for first beta release.
 * 
 * Revision 2.9  90/09/29  23:47:30  lee
 * Reduced the size of a buffer, and plugged yet another memory leak!
 * 
 * Revision 2.8  90/09/10  13:38:50  lee
 * deleted declaration of sleep()
 * 
 * Revision 2.7  90/08/29  21:46:48  lee
 * Alpha release.
 * 
 * Revision 2.6  90/08/12  17:33:38  lee
 * malloc changes; added SlayWordInfo() and MakeWordInfo().
 * 
 * Revision 2.5  90/08/09  19:16:35  lee
 * BSD lint and fixes...
 * 
 * Revision 2.4  90/03/22  14:23:19  lee
 * new calls to efree();
 * Offset now stored as a block number, not a byte offset
 * 
 * Revision 2.3  90/03/21  14:59:13  lee
 * Numerous changes.  WID2WordInfo() no longer calles GetWordPlaces().
 * 
 * Revision 2.2  89/10/08  20:45:05  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:13:56  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.4  89/09/17  23:01:53  lee
 * Various fixes; NumberInBlock now a short...
 * 
 * Revision 1.3  89/09/16  21:16:07  lee
 * First demonstratable version.
 * 
 * Revision 1.2  89/09/11  00:35:03  lee
 * Some speedups, but WID not working properly...
 * 
 * Revision 1.1  89/09/07  21:05:51  lee
 * Initial revision
 * 
 *
 */

#include "globals.h" /* defines and declarations for database filenames */

#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "fileinfo.h"
#include "smalldb.h"
#include "wordindex.h"
#include "wordinfo.h"
#include "numbers.h"

#include "emalloc.h"

#include "wordrules.h" /* max word length */

#include "pblock.h"

/** declarations: **/
/** Unix system calls that need to be declared: **/
extern int open(), close(); /* (these are not the stdio fopen and fclose) */
extern int creat();
extern void exit();
extern long lseek(); /* watch out for this on 16 bit (286, PDP11) systems! */
extern int read(), write();
extern int stat();
extern unsigned alarm(/* unsigned */);

/** Unix Library Calls that need to be declared: **/
#ifndef tolower /* e.g. on SunOS */
extern int tolower();
#endif
extern void perror();
/* extern int sleep(); -- this is unsigned on some systems */
extern int lockf();
/** lqtext Library calls that need to be declared: **/
extern void Deletepblock();

/** Functions within this file that need to be declared: **/
t_WordInfo *MakeWordInfo();
void SlayWordInfo();

/** **/

extern int AsciiTrace;
extern char *progname;

#define new(type) ( ((type) *) emalloc(sizeof(type)) )

/* Format when using ndbm: */

typedef struct {
    t_WID WID;
    unsigned long Offset; /* position in the database */
    unsigned long NumberOfWordPlaces;
    char Word[1]; /* Cheat here */
} t_WordIndexEntry;

/* Replacement fomat, intended to be faster and to use much less disk space.
 * Still use dbm for Word2WID, but not for the reverse mapping.
 * Also, cache the most recently-used WordInfo entry...
 * I have not measured how much of a win this is, but a lot of the code
 * calls Word2Wid() and then WID2WordInfo().
 */

static int Widfd = (-1);

t_WordInfo *
WID2WordInfo(WID)
    t_WID WID;
{
    extern t_WordPlace *GetWordPlaces(); /* pblock.c */

    char Buffer[WIDBLOCKSIZE + 5]; /* the +5 allows for overrun... */
    char *q = Buffer;
    t_WordInfo *WP;

    /* The above calculation is derived like this:
     *
     * The entry contains the total number of pairs (>= 1 byte),
     * the length of the string (>= 1 byte), and the string (>= 3 bytes)
     * (actually >= MinWordLength bytes, but setting this less than 3
     * would be a major disaster!)
     * Hence, there are WIDBLOCKSIZE - (2 + length) bytes left for pairs.
     * Now, each pair is at least 2 bytes, so halve the remainder.  I add
     * one coz WIDBLOCKSIZE - 3 is odd, so I would otherwise lose one!
     */

    if (Widfd < 0) {
	if ((Widfd = open(WidIndexFile, O_RDWR|O_CREAT, 0766)) < 0) {
	    fprintf(stderr, "Can't open WID file \"%s\"\n", WidIndexFile);
	    exit(1);
	}
    }

    if (lseek(Widfd, (long) (WID * WIDBLOCKSIZE), 0) < 0) {
	return (t_WordInfo *) 0;
    }

    if (read(Widfd, Buffer, WIDBLOCKSIZE) != WIDBLOCKSIZE) {
	return (t_WordInfo *) 0;
    }

    {
	unsigned short L;

	if ((L = sReadNumber(&q)) == 0) {
	    (void) fprintf(stderr,
		    "%s: Database corrupt, WID %lu has length zero\n",
		    progname, WID);
	    return (t_WordInfo *) 0;
	}
	WP = MakeWordInfo(WID, (int) L, q);
	q += L;
    }

    WP->Offset = (sReadNumber(&q) >> 1) * BLOCKSIZE;
    WP->NumberOfWordPlaces = sReadNumber(&q);

    /* Now, maybe read some WordPlace tuplets: */
#if 1
    if (q - Buffer < WIDBLOCKSIZE) {
	WP->WordPlaces = GetWordPlaces(
	    WP->WID,
	    q,
	    WIDBLOCKSIZE - (q - Buffer),
	    WP->Offset,
	    WP->NumberOfWordPlaces
	);
	WP->WordPlacesInHere = WP->NumberOfWordPlaces;
    } else {
	fprintf(stderr, "%s: Internal error, block too small for %ld (%s)\n",
		progname, WP->WID, WP->Word);
	exit(1);
    }

#else
    WP->WordPlaces = (t_WordPlace *) 0;
    if (q - Buffer < WIDBLOCKSIZE) {
	WP->DataBlock = emalloc(WIDBLOCKSIZE + 5);
	(void) memcpy(WP->DataBlock, Buffer, WIDBLOCKSIZE);
	WP->WordPlaceStart = &(WP->DataBlock[q - Buffer]);
    }
#endif

    /* done! */
    return WP;
}

static char PairBuffer[WIDBLOCKSIZE + 5]; /* the +5 allows for overrun... */

/* Make WordInfo Block Header... */
void
MkWIBH(WordInfo, pblock)
    t_WordInfo *WordInfo;
    t_pblock *pblock;
{
    char *q = PairBuffer;

#ifdef ASCIITRACE
    if (AsciiTrace > 15) {
	fprintf(stderr, "\tMake info block header for %s, Offset %lu==%lu\n",
	WordInfo->Word, pblock->ChainStart, WordInfo->Offset);
    }
#endif

    sWriteNumber(&q, WordInfo->Length);
    (void) strncpy(q, WordInfo->Word, WordInfo->Length);
    q += WordInfo->Length;
    if (pblock) sWriteNumber(&q, (pblock->ChainStart / BLOCKSIZE) << 1);
    else sWriteNumber(&q, 0L);
    sWriteNumber(&q, WordInfo->NumberOfWordPlaces);

    WordInfo->WordPlaceStart = q;
    WordInfo->DataBlock = PairBuffer;
}

/* Make WordInfo Block ... */
int
MkWIB(WordInfo, pblock)
    t_WordInfo *WordInfo;
    t_pblock *pblock;
{
    extern unsigned short PutWordPlaces();

    /* See how many pairs from the given pblock fit into WordInfo,
     * and leave them in PairBuffer...
     */

    if (AsciiTrace > 3) {
	fprintf(stderr, "Make info block for %s\n", WordInfo->Word);
    }

    MkWIBH(WordInfo, pblock);

    if (pblock == (t_pblock *) 0) {
	/* No WordPlaces to put in! */
	WordInfo->WordPlacesInHere = 0;
	return 0;
    }

    return WordInfo->WordPlacesInHere = PutWordPlaces(
		pblock->WordPlaces,
		WordInfo->WID,
		WordInfo->WordPlaceStart,
		WIDBLOCKSIZE - (WordInfo->WordPlaceStart - PairBuffer),
		pblock->ChainStart,
		pblock->NumberOfWordPlaces);
}

char *
String2SixBitString(String)
    unsigned char *String;
{
    static unsigned char Buffer[MaxWordLength + 1];
    register unsigned char *p;
    register unsigned char *Bufp = Buffer;
    unsigned short Val;
    int BitsLeft = 0;

    /* BUG: we lose word-processing accents, etc. and 8-bitness if
     * we do this.  Also, it slows things down very, very slightly.
     */

    /* Some ascii character equivalents:
     * '0' 48 060 0x30
     * 'A' 65 0101 0x41
     * '_' 95 0137 0x5f
     * 'a' 97 0141 0x61
     */
    for (p = String; *p; p++) {
	if (!isalnum(*p) && *p != '\'' && *p != '_') {
	    return (char *) 0;
	}
	if (isupper(*p)) *p = tolower(*p);
	/* Store as
	 * 0-9 --> 0-9 (easy!)
	 * a-z --> 10...35
	 * _/' --> 36/37
	 * hence, I need 6 bits per character.  This also leaves rather
	 * a lot of bits spare (38..64, 27 or so spaces).  As I fold case,
	 * and don't have controls, I don't know what to do there.  I
	 * could store digrams.  There are 38*38 = 1444 of these, but
	 * some of them don't happen.  Not worth the effort.
	 */
	if (isdigit(*p)) {
	    Val = (*p) - '0';
	} else if (isalpha(*p)) {
	    Val = (*p) - 'a' + ('9' - '0');
	} else if (*p == '\'') {
	    Val = ('9' - '0') + ('z' - 'a') + 1;
	} else if (*p == '_') {
	    Val = ('9' - '0') + ('z' - 'a') + 2;
	} else {
#define NEXTISEIGHT ('9' - '0') + ('z' - 'a') + 3
	    Val = NEXTISEIGHT;
	}
	/* Write the first half */
	if (!(BitsLeft & 07)) { /* i.e. it's 0 or 8 */
	    *Bufp = (Val << 2);
	    BitsLeft = 2;
	} else {
	    /* top BITSLEFT bits */
	    *Bufp++ |= (Val >> (6 - BitsLeft));
	    *Bufp = (unsigned) (Val << (2 + BitsLeft)); /* lose some bits */
	    if ((BitsLeft -= 6) < 0) BitsLeft += 8;
	}
    }
    if (BitsLeft) {
	Bufp++;
    }
    *Bufp = 0;
    return (char *) Buffer;
}

unsigned char *
SixBitString2String(SixBitString)
    char *SixBitString;
{
    static unsigned char Buffer[MaxWordLength + 2];
    register unsigned char *p = (unsigned char *) SixBitString;
    int BitsLeft = 0;
    unsigned char *Bufp = Buffer;

    while (*p) {
	if (!(BitsLeft & 07)) { /* i.e. it's 0 or 8 */
	    *Bufp++ = (*p) >> 2;
	    BitsLeft = 2;
	} else {
	    /* W R O N G */
	    /* bottom BITSLEFT bits */
	    *Bufp = ((*p) << (6 - BitsLeft));
	    /* Rest */
	    *Bufp = (unsigned) ((*p) << (2 + BitsLeft)); /* lose some bits */
	    if ((BitsLeft -= 6) < 0) BitsLeft += 8;
	}
    }
    return (unsigned char *) "notdone'fixme"; /* NOTDONE FIXME */
}

#ifdef TESTSIX
char *progname= "testsix";
main(argc, argv)
    int argc;
    char *argv[];
{
    extern char *gets();

    char Line[4096];
    int Encode = 1;

    if (argc != 3) {
	fprintf(stderr, "bad arg count; usage: %s -[de]\n", progname);
    }
    if (STREQ(argv[1], "-e")) Encode = 1;
    else if (STREQ(argv[1], "-d")) Decode = 1;
    else {
	fprintf(stderr, "usage: %s -[d|e]\n", progname);
	exit(1);
    }
    while (gets(Line) != (char *) 0) {
	char *Result;

	if (Encode) {
	    Result = String2SixBitString(Line);
	} else {
	    if (STREQ(Line, "(cannot be encoded)")) {
		Result = "(this line was not saved)";
	    } else {
		Result = SixBitString2String(line);
	    }
	}
	if (Result) {
	    printf("%s\n", Result);
	} else {
	    printf("(cannot be encoded)\n");
	}
    }
}

#endif /*TESTSIX*/


t_WID
Word2WID(Word, Length)
    char *Word;
    unsigned int Length;
{
    DBM *db;
    datum key, data;
    char *q;
    t_WID WID;
    char Buffer[8];
	/* enough for the binary representation of a number -- see numbers.c;
	 * this is _not_ sizeof(long).  It's probably 5, in fact, although
	 * for small numbers it's less.
	 */

    if (Length > MaxWordLength) {
	Length = MaxWordLength; /* NOTE: no trailing \0 required. */
    }

    /* contact database server */
    if ((db = startdb(WordIndex)) == (DBM *) 0) {
	fprintf(stderr, "dbmopen(%s) failed\n", WordIndex);
	exit(2);
    }

    key.dptr = Word;
    key.dsize = Length;

    data = dbm_fetch(db, key);

    enddb(db);

    if (data.dsize == 0) {
	return (t_WID) 0;
    }

    /* do this because ReadNumber will leave q pointing beyond Buffer: */
    (void) memcpy(Buffer, data.dptr, data.dsize);
    q = Buffer;
    WID = sReadNumber(&q);
    if (q - Buffer != data.dsize) {
	fprintf(stderr, "Word2Wid failed... got %lu\n");
	return (t_WID) 0;
    }
    return WID;
}
    
char *
WID2Word(WID)
    t_WID WID;
{
    t_WordInfo *W;
    char *Word;

    if (WID == (t_WID) 0) {
	return (char *) 0;
    }

    if ((W = WID2WordInfo(WID)) == (t_WordInfo *) 0) {
	return (char *) 0;
    }
    Word = W->Word;
    W->Word = (char *) 0;
    SlayWordInfo(W);
    return Word;
}

int
PutWordInfoIntoIndex(WordInfo, Offset)
    t_WordInfo *WordInfo;
    unsigned long Offset;
{
    DBM *db;
    char NumBuf[sizeof(t_WID) + 1];
    char *q = NumBuf;
    datum key, data;
    int RetVal;

    /** First, write the WID itself, so we can go from Word to WID */

    key.dptr = WordInfo->Word;
    key.dsize = WordInfo->Length;

    sWriteNumber(&q, WordInfo->WID);

    data.dptr = NumBuf;
    data.dsize = q - NumBuf;

    /* contact database server */
    if ((db = startdb(WordIndex)) == (DBM *) 0) {
	fprintf(stderr, "dbmopen(%s) failed\n", WordIndex);
	exit(2);
    }

    RetVal = dbm_store(db, key, data, DBM_REPLACE);

    enddb(db);

    /** Now, ensure that we have a physical block for WordInfo.  If
     ** we don't, there is something very wrong in pblock.c, our only
     ** possible caller.
     **/

    if (WordInfo->DataBlock == (char *) 0) {
	if (Offset) {
	    fprintf(stderr, "WARNING: WordInfo corrupt for \"%s\"\n",
			    WordInfo->Word);
	}
	(void) MkWIB(WordInfo, (t_pblock *) 0);
    }

    /** Now write the physical entry... */

    if (Widfd < 0) {
	if ((Widfd = open(WidIndexFile, O_RDWR|O_CREAT, 0766)) < 0) {
	    fprintf(stderr, "Can't open WID file \"%s\"\n", WidIndexFile);
	    exit(1);
	}
    }

    if (lseek(Widfd, (long) (WordInfo->WID * WIDBLOCKSIZE), 0) < 0) {
	perror("lseek");
	exit(1);
    }

    if (write(Widfd, WordInfo->DataBlock, WIDBLOCKSIZE) != WIDBLOCKSIZE) {
	perror("write");
	exit(1);
    }

    return RetVal;
}

int
WID_sig()
{
    return 0;
}

t_WID
GetMaxWID()
{
    extern int errno;
    extern long atol();

    int fd;
    char Buffer[20]; /* large enough to sprintf() a WID */
    struct stat StatBuf;
#if 0
    int FileKey; /* what one gets from a lock... */
    int NumberOfTriesLeft = 5;
#endif

    /* ensure that the file is there */
    if (stat(WidFile, &StatBuf) == -1) {
	return 0;
    }

    if ((fd = open(WidFile, O_RDWR, 0)) < 0) {
	fprintf(stderr, "Warning: Can't open WID file");
	return 0;
    }

#if 0
    errno = 0;

    /** Lock the file **/

    do {
	/* Set a timeout of 2 seconds */
	signal(SIGALRM, WID_sig);
	(void) alarm(3);
	if ((FileKey = lockf(fd, F_LOCK, 0L)) < 0) {
	    switch (errno) {
	    case EACCES: /*[sic]*/ /* another process has the lock */
		fprintf(stderr, "Please wait...\n");
		/* shouldn't happen */
		break;
	    case EDEADLK:
		fprintf(stderr, "Warning: can't lock \"%s\" -- EDEADLK\n", WidFile);
		FileKey = 1;
		break;
	    case EINTR:
		fprintf(stderr, "Please Wait... someone has the key...\n");
		sleep(1);
		break;
	    }
	}
	if (--NumberOfTriesLeft <= 0) {
	    fprintf(stderr, "Warning: can't lock ");
	    perror(WidFile);
	    (void) close(fd);
	    return 0;
	}
    } while (FileKey < 0);
    (void) alarm(0);

    if (stat(WidFile, &StatBuf) == -1) {
	fprintf(stderr, "It went away!\n");
	return 0;
    }
#endif

    /* Read the file */
    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "Can't read from \"%s\"\n", WidFile);
	exit(1);
    }

#if 0
    /** Unlock the file **/
    if (lockf(fd, F_ULOCK, 0L) < 0 && FileKey == 0) {
	fprintf(stderr, "Warning: might not have unlocked \"%s\"\n",
				WidFile);
    }
#endif
    (void) close(fd);

    Buffer[StatBuf.st_size] = '\0';

    return atol(Buffer);
}


t_WID LastNextWIDVal = (t_WID) 0;

#undef GetNextWID

t_WID GetNextWID(), GetMaxWID();

INLINE
t_WID
SpoofGetNextWID()
{
    static int SinceLastUpdate = 0;

    /* Call the real function sometimes, so that the database does
     * get updated in case of a crash or for other users.
     */
    if (++SinceLastUpdate > 500) {
	SinceLastUpdate = 0;
	return GetNextWID(1);
    }

    if (LastNextWIDVal == (t_WID) 0) {
	SinceLastUpdate = 0;
	LastNextWIDVal = GetMaxWID();
    }
    return ++LastNextWIDVal;
}

void
WriteCurrentMaxWID()
{
    (void) GetNextWID(1);
}

t_WID
GetNextWID(WriteCurrent)
    int WriteCurrent; /* simply write the current MaxWID if true */
{
    extern int errno;
    extern long atol();

    int fd;
    char Buffer[20];
    struct stat StatBuf;
#if 0
    int FileKey; /* what one gets from a lock... */
    int NumberOfTriesLeft = 5;
#endif
    t_WID Result;

    /** Alter the file, so other programs can see the new words...
     **/

    /* ensure that the file is there */
    if (stat(WidFile, &StatBuf) == -1) {
	fprintf(stderr, "Creating WID file \"%s\"\n", WidFile);
	if ((fd = creat(WidFile, 02666)) < 0) {
	    fprintf(stderr, "Can't create WID file \"%s\"\n", WidFile);
	    exit(1);
	}
	(void) close(fd);
	return GetNextWID(WriteCurrent);

	/*NOTREACHED*/
    }

    if ((fd = open(WidFile, O_RDWR, 0)) < 0) {
	fprintf(stderr, "Can't open WID file");
	perror(WidFile);
	exit(1);
    }

#if 0
    errno = 0;

    /** Lock the file **/

    do {
	/* Set a timeout of 2 seconds */
	signal(SIGALRM, WID_sig);
	(void) alarm(3);
	if ((FileKey = lockf(fd, F_LOCK, 0L)) < 0) {
	    switch (errno) {
	    case EACCES: /*[sic]*/ /* another process has the lock */
		fprintf(stderr, "Please wait...\n");
		/* shouldn't happen */
		break;
	    case EDEADLK:
		fprintf(stderr, "Warning: can't lock \"%s\" -- EDEADLK\n", WidFile);
		FileKey = 1;
		break;
	    case EINTR:
		fprintf(stderr, "Please Wait... someone has the key...\n");
		sleep(1);
		break;
	    }
	}
	if (--NumberOfTriesLeft <= 0) {
	    fprintf(stderr, "Warning: can't lock the file \"%s\"\n", WidFile);
	}
    } while (FileKey < 0);
    (void) alarm(0);

    if (stat(WidFile, &StatBuf) == -1) {
	fprintf(stderr, "It went away!\n");
	exit(1);
    }
#endif

    /* Read the file */

    if (read(fd, Buffer, (unsigned int) StatBuf.st_size) < 0) {
	fprintf(stderr, "Can't read from \"%s\"\n", WidFile);
	exit(1);
    }

    Buffer[StatBuf.st_size] = '\0';

    Result = atol(Buffer);

    /* if WriteCurrent is set, we should not increment anything */
    if (!WriteCurrent) {
	++LastNextWIDVal;
	++Result;
    }

    if (Result < LastNextWIDVal) {
	Result = LastNextWIDVal;
    }

    (void) sprintf(Buffer, "%lu\n", Result);

    /* Move to the start of the file and write the now value.
     * No need to truncate the file, because it didn't shrink!
     */
    (void) lseek(fd, 0, 0L);
    (void) write(fd, Buffer, (unsigned int) strlen(Buffer));

#if 0
    /** Unlock the file **/

    if (lockf(fd, F_ULOCK, 0L) < 0 && FileKey == 0) {
	fprintf(stderr, "Warning: might not have unlocked \"%s\"\n",
				WidFile);
    }
#endif
    (void) close(fd);

    return Result;
}

int
DeleteWord(Word)
    char *Word;
{
    extern t_pblock *Getpblock();

    t_WID WID;
    t_WordInfo *WordInfo;
    t_pblock *tmp;

    if ((WID = Word2WID(Word, strlen(Word))) == (t_WID) 0) {
	return -1; /* not there */
    }

    /* get info from the list */
    if ((WordInfo = WID2WordInfo(WID)) == (t_WordInfo *) 0) {
	return -1;
    }

    if ((tmp = Getpblock(WordInfo)) != (t_pblock *) NULL) {
	Deletepblock(tmp);
	(void) efree(tmp);
    }

    /* delete the offset from the database, but retain the WID: */
    WordInfo->Offset = 0L;
    WordInfo->NumberOfWordPlaces = 0L;
    WordInfo->WordPlacesInHere = 0;
    PutWordInfoIntoIndex(WordInfo, 0L);
    SlayWordInfo(WordInfo);

    return 0;
}

/* Routines to create and destroy WordInfo structures */
INLINE t_WordInfo *
MakeWordInfo(WID, Length, Word)
    t_WID WID;
    int Length;
    char *Word; /* the word, which might not be nul-terminated */
{
    register t_WordInfo *WP;
    WP = (t_WordInfo *) emalloc(sizeof(t_WordInfo));

    WP->WID = WID;
    WP->FID = (t_FID) 0;
    WP->Next = (t_WordInfo *) 0;
    WP->NumberOfWordPlaces = 0;
    WP->DataBlock = (char *) 0;
    WP->WordPlaceStart = (char *) 0;
    WP->WordPlaces = (t_WordPlace *) 0;
    WP->WordPlacesInHere = 0;
    WP->WordPlace.FID = 0; /* mark as invalid */
    WP->WordPlace.Flags = 0; /* this gets used anyway, so set it to zero! */

    WP->Word = emalloc(Length + 1);

    (void) strncpy(WP->Word, Word, Length);
    WP->Length = Length;
    WP->Word[Length] = '\0'; /* strncpy does not add a null */
    WP->Offset = 0;

    return WP;
}

void
SlayWordInfo(WP)
    t_WordInfo *WP;
{
    if (!WP) return;
    if (WP->Word) efree(WP->Word);
    if (WP->WordPlaces) efree((char *)WP-> WordPlaces);

    WP->Next = (t_WordInfo *) 0;
	/* The above line is to force a run-time error in the common
	 * (but wrong) case
	 * for (w = WordList; w; w = w->Next) SlayWordInfo(w);
	 */
    efree((char *) WP);
}
