(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;RTN-DATA.LSP

;;;The following functions define the data structures for the RTN parser
;;;and provide tools for redefining and modifying the network and lexicon.
;;;The sample lexicon provided is identical to the lexicon given in the
;;;beginning of Chapter Three.  The RTN is identical to Grammar 3.4.  The
;;;user has the option of using this lexicon and RTN or creating different
;;;ones.

;;;An RTN is defined as a list of nodes.  A node is a list whose first
;;;element is the name of the node (eg.  S2) and whose subsequent elements
;;;are the arcs that start at this node.  An arc is a list whose first
;;;element is the numeric arc label, whose second element is the arc
;;;function, and whose third element is the name of the node the arc
;;;terminates at.
									
;;;Arc functions are of five types: CAT-ARC, WORD-ARC, JUMP-ARC,
;;;PUSH-ARC AND POP-ARC.  The argument for CAT-ARC is a syntactic category,
;;;the argument for WORD-ARC is a word, JUMP-ARC and POP-ARC take no
;;;arguments, and the argument for PUSH-ARC is the name of the new network
;;;it will begin.  Some typical examples of arc functions are: (PUSH-ARC
;;;NP), (POP-ARC), and (CAT-ARC ADJ).

(DEFVAR *RTN* NIL)

(DEFVAR *SAMPLE-RTN* '((S (1 (PUSH-ARC NP) S1))
		       (S1 (1 (CAT-ARC VERB) S2))
		       (S2 (1 (POP-ARC) T)
			   (2 (PUSH-ARC NP) S2))
		       (NP (1 (CAT-ARC ART) NP1)
			   (2 (CAT-ARC NUM) NP1)
			   (3 (CAT-ARC PRON) NP2))
		       (NP1 (1 (CAT-ARC ADJ) NP1)
			    (2 (CAT-ARC NOUN) NP2))
		       (NP2 (1 (POP-ARC) T))))

(DEFUN NODE-NAME (node) (car node))

(DEFUN ARCS (node) (cdr node))

(DEFUN ARC-NUM (arc) (car arc))

(DEFUN ARC-FUNC (arc) (cadr arc))

(DEFUN ARC-ARG (arc) (cadr (arc-func arc)))

(DEFUN DESTINATION (arc) (caddr arc))


;;ADD-ARC adds a new node and arc to the current RTN, or adds a new arc
;;to a node already in the RTN.  It takes two arguments: (1) the name of
;;the node (which may or may not be in the current RTN) and (2) an arc.
;;This function can be used to incrementally build an RTN or modify one.

(DEFUN ADD-ARC (node-name arc)
  (let ((node (assoc node-name *rtn*)))
    (cond ((null node)
	   (setf *rtn* (cons (list node-name arc)
			   *rtn*)))
	  (t
	   (let ((new-node (append (list node-name)
				   (order-insert nil arc (arcs node)))))
	     (setf *rtn* (cons new-node
			     (remove node *rtn*))))))))
 

;;ORDER-INSERT is called by ADD-ARC (above) to insert an arc into the
;;proper place in the RTN.  It is an iterative function with three
;;arguments: (1) befores (2) new-arc (the arc being entered), and (3)
;;afters.  Befores and afters are the arcs whose position are before and
;;after new-arc in the proper arc-list of the RTN.

(DEFUN ORDER-INSERT (befores new-arc afters)
  (cond ((null afters)
	 (append (reverse befores)
		 (list new-arc)))
	((<= (arc-num new-arc)
	     (arc-num (car afters)))
	 (append
	   (append (reverse befores) (list new-arc))
	   afters))
	(t 
	 (order-insert (cons (car afters) befores)
		       new-arc
		       (cdr afters))))) 


;;DELETE-ARC deletes an arc from the current RTN.  It takes two
;;arguments: (1) the name of the node the arc starts at,, and (2) the arc
;;number.  This function is used to modify an RTN.

(DEFUN DELETE-ARC (node-name arc-num)
  (let ((old-node (assoc node-name *rtn*)))
    (let ((new-node (remove (assoc arc-num (arcs old-node))
			    old-node)))
      (cond ((null old-node)
	     (print "No such arc in RTN"))
	    ((equal (length new-node) 1)
	     (setf *rtn* (remove old-node *rtn*)))
	    (t
	     (setf *rtn* (cons new-node
			     (remove old-node *rtn*))))))))


;;MODIFY-RTN initializes the network to an appropriate value and calls
;;MODIFY-RTN-LOOP.

(DEFUN MODIFY-RTN ()
  (terpri)
  (princ-r "Will you modify the current network (M)," 2)
  (princ-r "or build one from scratch (S)?" 2)
  (terpri)
  (princ "  ==> (M or S) ")
  (let ((answer (read)))
    (cond ((equal answer 'S)
	   (setf *rtn* nil))
	  ((not (equal answer 'M))
	   (princ-r "* INCORRECT RESPONSE *" 3)
	   (princ-r "-- Assume Default: m --"))))

  (p-lines 2)
  (modify-rtn-loop))


;;ARC-INSTR prints out detailed instructions on modify the network.  In
;;particular it describes the commands to add an arc to and delete an
;;arc from the current network.  This function is called from
;;MODIFY-RTN-LOOP.  

 (DEFUN ARC-INSTR ()
  (princ-r "To add an arc, enter a list of:" 4)
  (princ-r "(1) ADD-ARC " 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the unaugmented arc." 5)
  (terpri)
  (princ-r "An unaugmented arc is a list of: " 2)
  (princ-r "(1) the arc number" 5)
  (princ-r "(2) the arc function " 5)
  (princ-r "(3) the node the arc terminates at" 5)
  (terpri)
  (princ-r "For example, to enter the arc:" 2)
  (terpri)
  (princ-r "(1 (PUSH-ARC NP) S2)" 5)
  (terpri)
  (princ-r "from node S, enter the list:" 2)
  (terpri)
  (princ-r "(ADD-ARC 'S '(1 (PUSH-ARC NP) S2))" 5)
  (p-lines 2)
  (princ-r "To delete an entire arc, enter a list of:" 4)
  (princ-r "(1) DELETE-ARC" 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the arc number." 5)
  (terpri)
  (princ-r "For example, to delete the arc from the example" 2)
  (princ-r "above, enter the list:" 2)
  (terpri)
  (princ-r "(DELETE-ARC 'S 1)" 5))


;;MODIFY-RTN-LOOP is a loop that repeats until the user enters DONE.  It
;;either adds an arc to the network, deletes an arc from the network, or
;;prints out ARC-INSTR (above), depending upon the input of the user.

(DEFUN MODIFY-RTN-LOOP ()
  (p-lines 1)
  (princ-r "Commands: (ADD-ARC node-name arc)" 2)
  (princ-r "(DELETE-ARC node-name arc-num)" 12)
  (princ-r "DONE" 12)
  (princ-r "HELP" 12)
  (p-lines 1)
  (princ-r "Current RTN:" 2)
  (pprint-list *rtn*)
  (princ "  ==> ")
  (let ((i (read)))
    (terpri)
    (cond ((equal i 'done)
	   *rtn*)
	  ((equal i 'help)
	   (arc-instr)
	   (p-lines 1)
	   (princ-i "To begin, press any letter and return:" 2)
	   (read)
	   (modify-rtn-loop))
	  ((or (not (listp i))
	       (not (equal (length i) 3)))
	   (princ-r "* INCORRECT NOTATION *" 2)
	   (modify-rtn-loop))
	  ((member (car i) '(add-arc delete-arc))
	   (eval i)
	   (modify-rtn-loop))
	  (t
	   (princ-r "* INCORRECT NOTATION *" 2)
	   (modify-rtn-loop)))))



;;;The lexicon is a list composed of words and sytactic entries for those
;;;words.  In general, a syntactic entry can contain any amount of
;;;syntactic information.  This parser, in particular, requires only the
;;;syntactic category of the word.  Therefore, in the lexicon, a word with
;;;n different entries is represented as a list whose first element is the
;;;word, and whose next n elements are lists each containing one of those
;;;categories.

(DEFUN WORD-ENTRY (word *lexicon*) (assoc word *lexicon*))

(DEFUN WORD-NAME (wrd-entry) (car wrd-entry))

(DEFUN ENTRIES-LIST (wrd-entry) (cdr wrd-entry))

(DEFUN SYNTACTIC-TYPE (entry) (car entry))

(DEFVAR *LEXICON* NIL)

(DEFVAR *SAMPLE-LEXICON* '((the (ART))
			   (a (ART))
			   (one (NUM) (PRON))
			   (wild (ADJ))
			   (green (ADJ) (NOUN))
			   (dogs (NOUN))
			   (man (NOUN) (VERB))
			   (saw (NOUN) (VERB)) 
			   (cried (VERB))
			   (broke (VERB))
			   (faded (VERB))) )


;;ADD-WORD enters a word into the lexicon named *lexicon*.  It takes as
;;its arguments (1) a word, and (2) its syntactic category.  

(DEFUN ADD-WORD (word syntactic-cat)
  (let ((wrd-entry (word-entry word *lexicon*)))
    (cond ((null wrd-entry)
	   (setf *lexicon* (cons (list word
				     (list syntactic-cat))
			       *lexicon*)))
	  (t
	   (let ((new-wrd-entry (append (list (word-name wrd-entry))
					(cons (list syntactic-cat)
					      (entries-list wrd-entry)))))
	     (setf *lexicon* (cons new-wrd-entry
				 (remove wrd-entry *lexicon*))))))))


;;DELETE-WORD deletes an entry from the lexicon.  It takes as its
;;arguments (1) a word, and (2) its syntactic category.

(DEFUN DELETE-WORD (word category)
  (let ((wrd-entry (word-entry word *lexicon*)))
    (let ((new-wrd-entry (remove
			 (assoc category (entries-list wrd-entry))
			 wrd-entry)))
      (cond ((null wrd-entry)
	     (print "No such entry in lexicon."))
	    ((equal (length new-wrd-entry) 1)                ;WORD ENTRY IS EMPTY
	     (setf *lexicon* (remove wrd-entry *lexicon*)))
	    (t
	     (setf *lexicon* (cons new-wrd-entry
				 (remove wrd-entry *lexicon*))))))))


;;LOOK-UP takes as its arguments: (1) the word it is to look up in the
;;lexicon, and (2) a syntactic category.  It returns the syntactic
;;information if the lexicon has an entry containing both the word and a
;;list of that particular syntactic category.  Otherwise, it returns nil.

(DEFUN LOOK-UP (word category)
  (assoc category
	 (entries-list (word-entry word *lexicon*))))	


;;MODIFY-LEXICON initialized the lexicon appropriately and calls
;;MODIFY-LEX-LOOP to modify it.

(DEFUN MODIFY-LEXICON ()
  (princ-r "Will you modify the current lexicon (M)," 2)
  (princ-r "or build one from scratch (S)?" 2) 
  (terpri)
  (princ "  ==> (M or S) ")
  (cond ((equal (read) 'S)
 	 (setf *lexicon* nil)))
  (modify-lex-loop))


;;BASIC-ENTRY-INSTR prints out detailed instructions on how to modify
;;the lexicon.  In particular, it describes the commands for adding an
;;entry to or deleting an entry from the lexicon.

(DEFUN BASIC-ENTRY-INSTR ()
  (p-lines 2)
  (princ-r "BUILD OR MODIFY LEXICON ONE ENTRY AT A TIME." 2)
  (terpri)
  (princ-r "To add an entry to the lexicon, first add the word and" 4)
  (princ-r "its syntactic category.  Then add any other syntactic " 2)
  (princ-r "information." 2)
  (terpri)
  (princ-r "To add a word and its syntactic category, enter a list of:" 4)
  (princ-r "(1) ADD-WORD " 5)
  (princ-r "(2) the word being entered" 5)
  (princ-r "(3) the syntactic category of the word." 5)
  (terpri)
  (princ-r "For example, to enter the noun: SAIL, enter the list:" 2)
  (terpri)
  (princ-r "(ADD-WORD 'SAIL 'NOUN)" 5)
  (terpri)
  (princ-r "Then, to enter the verb: SAIL, enter the list:" 2)
  (terpri)
  (princ-r "(ADD-WORD 'SAIL 'VERB)" 5)
  (terpri)
  (princ-r "To delete an entry from the lexicon, enter a list of:" 4)
  (princ-r "(1) DELETE-WORD " 5)
  (princ-r "(2) the word being deleted" 5)
  (princ-r "(3) the syntactic category of the word" 5)
  (terpri)
  (princ-r "For example, to delete the noun SAIL from the example" 2)
  (princ-r "above, enter the list:" 2)
  (terpri)
  (princ-r "(DELETE-WORD 'SAIL 'NOUN)" 5))


;;MODIFY-LEX-LOOP is a loop that repeats until the user enters DONE.  It
;;either enters a word into the lexicon, deletes a word from the
;;lexicon, or prints out BASIC-ENTRY-INSTR (above), depending upon the
;;input of the user.

(DEFUN MODIFY-LEX-LOOP ()
  (p-lines 1)
  (princ-r "Commands: (ADD-WORD word syntactic-category)" 2)
  (princ-r "(DELETE-WORD word syntactic-category)" 12)
  (princ-r "DONE" 12)
  (princ-r "HELP" 12)
  (p-lines 1)
  (princ-r "Current Lexicon:" 2)
  (pprint-list *lexicon*)
  (terpri)
  (princ "  ==> ")
  (let ((i (read)))
    (terpri)
    (cond ((equal i 'done)
	   *lexicon*)
	  ((equal i 'help)
	   (basic-entry-instr)
	   (p-lines 1)
	   (princ-i "To begin, press any letter and return:" 2)
	   (read)
	   (modify-lex-loop))
	  ((or (not (listp i))
	       (not (equal (length i) 3)))
	   (princ-r "* INCORRECT NOTATION *" 2)
	   (modify-lex-loop))
	  ((member (car i)  '(add-word delete-word))
	   (eval i)
	   (modify-lex-loop))
	  (t
	   (princ-r "* INCORRECT NOTATION *" 2)
	   (modify-lex-loop)))))








