(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;RTN.LSP

;;The following code is an implementation of a simple recursive
;;transition network.  It uses the data functions and structures defined
;;in RTN-DATA.LISP and the functions defined in UTILITIES.LISP.

;;BACKUPS is a global variable that stores backup states for the current
;;parse.  The parser chooses the last backup state on BACKUPS as its
;;current state by calling POP-BACKUPS.  POP-BACKUPS removes the last
;;backup state entered onto BACKUPS and returns this state.

(DEFVAR *BACKUPS* NIL)

(DEFUN POP-BACKUPS ()
  (let ((state (car *backups*)))
    (setf *backups* (cdr *backups*))
    state))


;;CAT-ARC, WORD-ARC, JUMP-ARC, PUSH-ARC, and POP-ARC are arc functions
;;which take as their arguments: (1) an arc (which is always the arc
;;from which the arc-function is called, (2) the sentence pointer, and
;;(3) the list of return points.  All three functions return a new parse
;;state.  A parse state is composed of a list of (1) the name of the
;;current node,(2) the sentence pointer, and (3) the list of return
;;points.

(DEFUN CAT-ARC (arc pointer returns)
  (cond ((look-up (aref *sentence* pointer)
		  (arc-arg arc))
	 (list (destination arc)
	       (1+ pointer)
	       returns))
	(t
	 nil)))

(DEFUN WORD-ARC (arc pointer returns)
  (cond ((equal (aref *sentence* pointer)
		(arc-arg arc))
	 (list (destination arc)
	       (1+ pointer)
	       returns))
	(t
	 nil)))

(DEFUN JUMP-ARC (arc pointer returns)
  (list (destination arc)
	pointer
	returns))

(DEFUN PUSH-ARC (arc pointer returns)
  (cond ((null (aref *sentence* pointer))
	 nil)
	(t
	 (list (arc-arg arc)
	       pointer
	       (cons (destination arc) returns)))))

(DEFUN POP-ARC (arc pointer returns)
  (cond (returns               ;RETURNS IS NOT EMPTY
	 (list (car returns)
	       pointer
	       (cdr returns)))
	((and (null returns)   ;RETURNS IS EMPTY, POINTER IS AT END OF SENTENCE
              (null (aref *sentence* pointer)))
	 'success)
	(t
	 nil)))

  
;;START-PARSE is the top-level function called by the user to start the
;;parse-loop.  It defines the network and lexicon and then calls
;;PARSE-LOOP.

(DEFUN START-PARSE()
  (p-lines 1)
  (princ-r "Will you use the sample RTN (S), the current RTN (C)" 2)
  (princ-r "or a modification (M) [DEFAULT]?" 2)
  (princ "  ==> (S C M) ")
  (let ((answer (read)))
    (cond ((equal answer 'm)
	   (modify-rtn))
	  ((equal answer 's)
	   (setf *rtn* *sample-rtn*))))
  (terpri)
  (princ-r "Will you use the sample lexicon (S), the current lexicon (C)" 2)
  (princ-r "or a modification (M)?" 2)
  (princ "  ==> (S C M) ")
  (let ((answer (read)))
    (cond ((equal answer 'm)
	   (modify-lexicon))
	  ((equal answer 's)
	   (setf *lexicon* *sample-lexicon*))))
  (parse-loop))


;;PARSE-LOOP sets the tracer to the user's requested level of detail and
;;queries the user for the sentence to be parsed.  It calls the PARSE
;;function on this sentence and then either repeats the loop and parses
;;another sentence, calls START-PARSE so the user can modify the RTN or
;;lexicon, or returns 'END, according to the users responses to the
;;queries.

(DEFUN PARSE-LOOP()
  (terpri)
  (princ-r "Enter your parse preference: P = single parse with Prompt for another [DEFAULT]" 2)
  (princ-r "                             A = all possible parses" 2)
  (princ "  ==> ")
  (let ((parse-flag (read)))
    (cond ((not (or (equal parse-flag 'P)
		    (equal parse-flag 'A)))
	   (setf parse-flag 'P)))
    (terpri)
    (princ-r "Enter your trace preference: 0 = No Trace," 2)
    (princ-r "                             1 = General Trace " 2)
    (princ-r "                             2 = Detailed Trace" 2)
    (princ "  ==> ")
    (let ((trace-flag (read)))
      (cond ((not (or (equal trace-flag 0)
		      (equal trace-flag 1)
		      (equal trace-flag 2)))
	     (terpri)
	     (princ-r "* INCORRECT RESPONSE *" 3)
	     (princ-r "-- Assume Default: 2 --" 2)
	     (setf trace-flag 2)))
      (terpri)
      (princ-r "Enter the sentence to be parsed as an ordered list of words." 2)
      (princ-r "- For the text example type: (ONE SAW THE MAN)" 2)
      (terpri)
      (princ "  ==> ")
      (let ((i (read)))
	(setf *sentence* (make-array (1+ (length i)) :initial-contents
				     (append i (list nil)))))
      (setf *backups* nil)
      (set-up trace-flag)
      (parse '(s 0 nil) parse-flag trace-flag))
    (terpri)
    (princ-r "Will you modify the RTN or lexicon? (Y or N)" 2)
    (princ "  ==> ")
    (cond ((equal (read) 'y)
	   (start-parse))
	  (t
	   (terpri)
	   (princ-r "Will you parse another sentence? (Y or N)" 2)
	   (princ "  ==> ")
	   (cond ((equal (read) 'n)
		  'end)
		 (t
		  (parse-loop)))))))


;;PARSE is the body of the RTN parser.  It takes as its arguments (1) a
;;state (consisting of a node name, the sentence pointer, and a list of
;;return states), and (2) a trace flag which determines the detail of the
;;trace for the current parse.  PARSE returns either 'SUCCESS!  or 'FAIL
;;to reflect the success or failure of the particular parse.  If the parse
;;is successful, PARSE either continues to print 'SUCCESS for each of the
;;other successful parses of the sentence (if parse-flag equals 'A) or
;;asks the user if another parse is desired (if parse-flag equals 'P).

(DEFUN PARSE (current-state parse-flag trace-flag)
  (let ((node-name (car current-state))
	(pointer (cadr current-state))
	(returns (caddr current-state)))

    ;GET TRACE DATA, UPDATE *BACKUPS*                         
    (let ((start-node (assoc node-name *rtn*))
	  (bu-temp nil)
	  (arcs-followed nil))
      
      (mapcar #'(lambda (arc)
		  (let ((state (apply (car (arc-func arc))
				      (list arc pointer returns))))
                        ;STATE IS RESULT OF APPLYING ARC FUNCTION
		    (cond (state    ;IF ARC CAN BE FOLLOWED SUCCESSFULLY,
			   (setf bu-temp   ;ADD RESULTING STATE TO TEMP-BU
				 (append (list state) bu-temp))
			   (cond ((equal trace-flag 2)
				  (setf arcs-followed       
					(append (list
						  (list (node-name start-node)
							(arc-num arc)))
				 ;;ADD ARC LABELS TO ARCS FOLLOWED
						arcs-followed))))))))
	      (arcs start-node))

      (print-data current-state
		  trace-flag
		  (reverse arcs-followed))
      (setf *backups* (append (reverse bu-temp) *backups*))))
  
  ;FOLLOW ARC TO NEXT NODE
  (next-node parse-flag trace-flag))


;;NEXT-NODE sets the current state to be the last state entered onto
;;BACKUPS.  If the parser discovers that the sentence is not legal, i.e.
;;current-state equals 'FAIL, NEXT-NODE returns 'FAIL.  If it has arrived
;;at a legal parse, i.e.  current-state equals 'SUCCESS, it returns
;;'SUCCESS and either continues to search for other possible parses (if
;;trace-flag equals 'A), or searches for another parse if the user
;;responds positively to a prompt.

(DEFUN NEXT-NODE (parse-flag trace-flag)
  (let ((current-state (pop-backups)))
    (cond ((null current-state)
	   (terpri)
	   (princ-r 'fail))                                 ;PARSE FAILS
	  ((equal current-state 'success)
	   (terpri)
	   (princ-r 'success!)
	   (cond ((equal parse-flag 'A)
		  (next-node parse-flag trace-flag))
		 (t
		  (princ-r "Would you like another parse for this sentence? (Y or N)" 2)
		  (princ "  ==> ")
		  (cond ((equal (read) 'y)
			 (next-node parse-flag trace-flag))
			(t
			 'success)))))                    ;PARSE SUCCEEDS
	  (t
	   (parse current-state parse-flag trace-flag)))))
           ;ARC IS FOLLOWED TO NEXT NODE,
           ;PARSE IS CALLED ON NEW STATE.


;;SET-UP sets up the screen for displaying parse information during the
;;parse.  Its only argument is the trace preference of the user.

(DEFUN SET-UP (trace-flag)
  (p-lines 5)
  (princ-i "SENTENCE:  0" 2)                ;PRINT SENTENCE WITH INDICES                
  (do ((a 0 (1+ a)))                      
      ((equal (aref *sentence* a) nil) t)
    (princ-i (aref *sentence* a) 2)
    (princ-i (1+ a) 2))
  (p-lines 1)

  (cond ((equal trace-flag 0)
	 't)
	((equal trace-flag 1)
	 (princ-i "CURRENT NODE" 2)
	 (princ-i "SENTENCE POINTER" 8)
	 (princ-r "RETURN POINTS" 4)
	 (p-lines 1))
	((equal trace-flag 2)
	 (princ-i "CURRENT STATE" 2)
	 (princ-i "ARCS FOLLOWED" 12)
	 (princ-r "BACKUP STATES" 12)
	 (p-lines 1))))


;;PRINT-DATA displays parse information on the screen according to the
;;trace mode chosen by the user.  It has three arguments: (1) the current
;;state, (2) the trace flag, and (3) a list of the node and arc numbers of
;;the arcs that were followed (non-null only if trace flag equals 2).

 (DEFUN PRINT-DATA (current-state trace-flag arcs-followed)
  (let ((node-name (car current-state))
	(pointer (cadr current-state))
	(returns (caddr current-state)))
    (cond ((equal trace-flag 0)
	   't)
	  ((equal trace-flag 1)
	   (terpri)
	   (princ-i node-name 7)
	   (princ-i pointer (- 22 (length (prin1-to-string node-name))))
	   (princ-r returns (- 18 (length (prin1-to-string pointer)))))
	  ((equal trace-flag 2)
	   (terpri)
	   (princ-i current-state 2)
	   (p-spaces (- 25 (length (prin1-to-string current-state))))
	   (cond ((null (caar arcs-followed))
		  (princ 'none)
		  (princ-r *backups* 21))
		 (t
		  (princ-arc (caar arcs-followed) (cadar arcs-followed))
		  (princ-r *backups*
			   (- 25 (+ (length (prin1-to-string (caar arcs-followed)))
				    (length (prin1-to-string (cadar arcs-followed)))
				    1)))))
	   (cond ((> (length arcs-followed) 1)
		  (princ-i "(&" 27)
		  (mapcar #'(lambda (arc)
			      (princ " ")
			      (princ-arc (car arc) (cadr arc)))
			  (cdr arcs-followed))
		  (princ-r " for b.u.)")))))))
	   
	   
    


