(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-


;;;                        LOGIC PARSER

;;; The logic parser (LOGIC-PARSER.LSP) uses the variables LEXICON and
;;;SYNTAX RULES defined in LOGIC-DATA.LSP or by the user directly.  Each
;;;syntax rule and lexicon entry is converted by the parser to an axium
;;;of the form: (G (SG1 ... SGn)).  The parser works as a backwards
;;;chaining theorem prover.  (In order to show G, it suffices to show
;;;each of SG1 ...  SGn.  It stops when it arrives at a unification of
;;;the initial statement and the axiums.  


;;The following statements define G as the goal of the rule and SG1 ... SGn
;;as the subgoals of the rule.  The predicate of a statement is the first
;;item of the list.  For example, the predicate of (noun ?p1 ?noun ?p2) is
;;noun.
					     
(DEFUN GOAL (rule) (car rule))
(DEFUN SUBGOALS (rule) (cdr rule))
(DEFUN PRED (statement) (car statement))


;;SUBSTITUT takes a statement and a list of bindings and replaces every
;;variable in the statement by the value it is bound to in the bindings  
;;list.

(DEFUN SUBSTITUT (statement bindings)
  (cond ((null statement) nil)
	((null bindings) statement)
	((atom statement)
	 (let ((sub (assoc statement bindings)))
	   (if (null sub)
	       statement
	       (cadr sub))))
	(t
	 (cons (substitut (car statement) bindings)
	       (substitut (cdr statement) bindings)))))


;;NEW-VARIABLES replaces every variable in item not already on the
;;bindings list by a new variable, and adds the list of the old and new
;;variable to bindings.  It returns a list of revised item and bindings.

(DEFUN NEW-VARIABLES (item bindings)
  (let ((item (substitut item bindings)))
    (cond ((null item) (list nil bindings))
	  ((and (pattern-var item)
		(assoc item (mapcar #'reverse bindings)))
	   (list item bindings))
	  ((pattern-var item)
	   (let ((m (gentemp "?T")))
	     (list m (cons (list item m) bindings))))
	  ((atom item) (list item bindings))
	  (t ;list
	   (let ((new-car (new-variables (car item) bindings)))
	     (let ((new-cdr (new-variables (cdr item) (cadr new-car))))
	       (list (cons (car new-car)
			   (car new-cdr))
		     (cadr new-cdr))))))))


;;SIMILAR-STATEMENT-LIST takes a statement and a set of rules and searches
;;the rules for the first rule (or statement) whose goal begins with the
;;same predicate as the statement.  If none are found it returns nil.
;;Otherwise, it calls MATCH-GOAL of statement and the subset of rules
;;beginning with the partially matched rule, where the rule has been
;;rewritten with new variables.

(DEFUN SIMILAR-STATEMENT-LIST (statement rules)
  (if (null rules)
      nil
      (if (equal (pred (goal (first  rules)))
		 (pred statement))
	  (match-goal statement (cons (car (new-variables (first rules) nil))
				      (rest rules)))
	  (similar-statement-list statement (rest rules)))))


;;MATCH-GOAL determines if the partially matched rule (the first rule of
;;rules) matches statement.  If it doesn't match, it calls
;;SIMILAR-STATEMENT-LIST of statement and rules, with the first rule
;;omitted.  If it does match, it returns a list of bindings and rules,
;;where the bindings are those returned by MATCH and the first rule of
;;rules is modified by substituting the variables of the first
;;rule by the variables they are bound to by the new variable bindings.

;;For example, '((NP 1 ?NAME 3) ((NAME 1 ?NAME 3))) could be the
;;first element of the list returned if the entered statement were '(NP
;;1 ?NP 3) and a rule in the list was ((NP ?P1 ?NAME ?P2) (NAME ?P1
;;?NAME ?P2)). The rest of the rule list would be identical to the rules
;;of the argument.

(DEFUN MATCH-GOAL (statement rules)
  (let ((bindings-list (match statement (goal (first rules)))))
    (cond ((null bindings-list)
	   (similar-statement-list statement (rest rules)))
	  ((null (car bindings-list))
	   (list '(()) rules))
	  (t
	   (list (car bindings-list)
		 (append (list
			   (cons (substitut statement       
					     (car bindings-list))
				 (substitut (subgoals (first rules)) 
					     (car bindings-list))))
			 (rest rules)))))))


;;;LEGAL-CONSTITUENT and LEGAL-PARTS form the body of the logic
;;;parser program.  Together they perform a backwards chaining routine.
;;;LEGAL-CONSTITUENT takes a statement such as '(S 1 ?S 3) and finds the
;;;first rule whose goal matches the statement.  It
;;;calls LEGAL-PARTS to determine whether or not the subgoals of that
;;;rule unify with the statement, the lexicon, and the other rules.
;;;LEGAL-PARTS in turn calls LEGAL-CONSTITUENT for each subgoal of the
;;;rule to determine whether or not it is itself a legal constituent for
;;;the current problem.  These alternating nested calls end when
;;;LEGAL-PARTS is called for a rule which is a statement and thus has no
;;;subgoals.  Examples of such a statement are (ISNOUN BOOK) and (WORD
;;;JOHN 1 2).

;;LEGAL-CONSTITUENT calls SIMILAR-STATEMENT to get rule-list and
;;bindings for the variables of the first rule of rule-list). If
;;rule-list is null it returns nil.  Otherwise, it calls LEGAL-PARTS for
;;the first rule of rule-list.  If LEGAL-PARTS returns nil,
;;LEGAL-CONSTITUENT calls itself with the argument rules equal to to
;;rule-list with the first rule omitted.  If LEGAL-PARTS returns a list
;;of bindings, LEGAL-CONSTITUENT returns a list of these bindings and
;;rule-list.

(DEFUN LEGAL-CONSTITUENT (statement rules global-rules)
  (if (null statement)
      '(())
      (let ((sim-statement (similar-statement-list statement rules)))
	(let ((bindings (car sim-statement))
	      (rule-list (cadr sim-statement)))
	  (if (null rule-list)
	      nil
	      (let ((final-bindings (legal-parts (car rule-list)
						 bindings
						 global-rules
						 global-rules)))
		(cond ((null final-bindings)
		       (legal-constituent statement
					  (cdr rule-list)
					  global-rules))
		      (t
		       (cons final-bindings rule-list)))))))))


;;LEGAL-PARTS is called on a single rule.  If the rule consists of a
;;single statement (which may or may not be followed by nil), bindings
;;is returned.  Otherwise, LEGAL-CONSTITUENT is called on the first
;;subgoal of the rule.  If it returns a null list of bindings,
;;LEGAL-PARTS returns nil.  If it returns '(()), i.e., if the subgoal is
;;null, LEGAL-PARTS returns bindings made up of the bindings.  Finally,
;;if LEGAL-CONSTITUENT returns an ordinary list of bindings and rules
;;(not of the two cases described above), then the result of calling
;;LEGAL-NEXT-PART on rule is returned. 

(DEFUN LEGAL-PARTS (rule bindings rule-list global-rules)
  (cond ((or (equal (length rule) 1)
	     (null (cadr rule)))  ; RULE IS OF LENGTH 2 AND LAST ELEMENT
				  ; IS NIL
	 bindings)
	(t
	 (let ((lc (legal-constituent (caar (subgoals rule))
				      rule-list
				      global-rules)))
	   (let ((recent-bindings (car lc))
		 (remaining-rules (cdr lc)))
	     (cond ((null recent-bindings)     ;FIRST SUBGOAL IS NOT
					       ;LEGAL CONSTITUENT.
		    nil)
		   ((null (car recent-bindings))
		    bindings)
		    ;(append bindings recent-bindings))
		   (t
		    (legal-next-part rule
				     bindings
				     recent-bindings
				     remaining-rules
				     global-rules))))))))


;;LEGAL-NEXT-PART is called by LEGAL-PARTS if the first subgoal of rule is
;;a legal constituent.  It deletes the first subgoal of rule, and calls
;;LEGAL-PARTS on the modified rule with bindings set to recent-bindings,
;;i.e., the bindings returned by the call to LEGAL-PARTS for the first
;;subgoal.  If this call to LEGAL-PARTS returns nil, then LEGAL-PARTS is
;;called again with the original rule and bindings, and the first rule
;;of rule-list omitted.  Thus, the program searches for another set of
;;bindings that make the first subgoal of rule a legal constituent, and
;;repeats the process from there.  Alternatively, if the call within
;;LEGAL-NEXT-PART to LEGAL-PARTS returns a list of bindings,
;;LEGAL-NEXT-PART returns a merging of this new list of bindings with
;;the argument called bindings. 

(DEFUN LEGAL-NEXT-PART (rule bindings recent-bindings remaining-rules global-rules)
  (let ((new-bindings
	  (legal-parts (list (substitut (car rule)
					recent-bindings)
			     (substitut (cdadr rule)
					recent-bindings))
		       recent-bindings
		       global-rules
		       global-rules)))
    (cond ((null new-bindings)
	   (legal-parts rule bindings (cdr
					remaining-rules) global-rules))
	  (t
	   (reduce-bindings (append new-bindings
				    bindings))))))


;;LEGAL-SENTENCE is the top-level command entered by the user to
;;determine if word-list, a sentence, is legal given a particular set of
;;grammar rules.  If it is not legal, it returns nil.  If it is legal,
;;it returns the parse of word-list (the list to which the variable ?s
;;is bound in the bindings returned by the first call to
;;LEGAL-CONSTITUENT). 

(DEFUN LEGAL-SENTENCE (word-list)
  (let ((global-rules (append syntax-rules
			      lexicon
			      (make-axiums word-list))))
    (terpri)
    (terpri)
    (pretty (cadr (assoc '?s
			 (car (legal-constituent
			       (list 's 1 '?s (1+ (length word-list)))
			       global-rules
			       global-rules))))
	    0)))
		  

;;PRETTY takes a list representation of a parse and prints it in a 
;;readable form.

(DEFUN PRETTY (item level &aux newlevel)
  (cond ((listp item)
	 (princ "(")
	 (setq newlevel (+ 2 (pretty (car item) level)))
	 (princ " ")
	 (pretty (cadr item) newlevel)
	 (and (cddr item)
	      (mapcar #'(lambda (x)
			  (p-spaces newlevel)
			  (pretty x newlevel))
		      (cddr item)))
	 (princ ")")
	 level)
	(t
	 (prin1 item)
	 (+ level (length (string item))))))

;;P-SPACES prints n spaces on a new line.

(DEFUN P-SPACES (n)
  (terpri)
  (do ((x n (1- x)))
      ((zerop x))
    (princ " ")))

;;ADJUST adds binding to b-list (a list of bindings) and unifies it with
;;the rest of b-list.  It replaces variables with constant values
;;wherever possible throughout the list of bindings.

(DEFUN ADJUST (binding b-list) 
  (cond ((not (pattern-var (car binding)))
	 b-list)
	(t
	 (let ((b (assoc (car binding) b-list)))
	   (cond ((equal b binding)
		  b-list)
		 ((null b)
		  (append (list (list (car binding)
				      (substitut (cadr binding) b-list)))
			  (substitut b-list (list binding))))
		 ((constantp (cdr binding))
		  (adjust (list (cadr b) (cadr binding))
			  (append (list binding)
				  (substitut (remove b b-list)
					      (list binding)))))
		 ((constant (cdr b))
		  (adjust (list (cadr binding) (cadr b))
			  b-list))
		 (t
		  (cond ((listp (cdr b))
			 (adjust (list (cdr binding) (cdr b)) b-list))
			(t
			 (adjust (list (cdr b) (cdr binding))
				 b-list)))))))))

;;CONSTANT determines whether or not an item is a constant.  If it
;;contains a variable, it is not.  A list not containing a variable is a
;;constant.

(DEFUN CONSTANT (item)
  (cond ((or (null item) (numberp item)) 't)
	((atom item)
	 (not (pattern-var item)))
	((not (constant (car item)))
	 nil)
	(t
	 (constant (cdr item)))))


;;REDUCE-BINDINGS takes a list of bindings that may contain repeats but
;;no contradicting values and eliminates repeats of the same variable,
;;and assigns to each variable its constant value if it has one.

(DEFUN REDUCE-BINDINGS (binding-list)
  (DEFUN SORT-ITER (new-list tail)
    (if (null tail)
	new-list
	(sort-iter (adjust (car tail) new-list)
		   (cdr tail))))
  (sort-iter nil binding-list))


;;MAKE-AXIUMS takes a list of words and returns axiums to become part of
;;the rule-list for parsing the sentence made up of those words.

(DEFUN MAKE-AXIUMS (word-list)
  (DEFUN AXIUMS-ITER (axiums n tail-list)
    (cond ((null tail-list)
	   (reverse axiums))
	  (t
	   (axiums-iter
	     (append (list (list (list 'word (car tail-list) n (1+ n))))
		     axiums)
	     (1+ n)
	     (cdr tail-list)))))
  (axiums-iter nil 1 word-list))


;;;**********************************************************************

;;;STANDARD UNIFICATION ALGORITHM


;;MATCH returns a list of bindings that unify pat1 and pat2.

(DEFUN MATCH (pat1 pat2)
  (match-with-bindings pat1 pat2 nil))


;;MATCH-WITH-BINDINGS calls VAR-MATCH if one pattern is a variable,
;;returns bindings if both patterns are equal atoms, and  returns nil if
;;they are unequal atoms.  If both patterns are lists, it calls itself
;;on the cars of each pattern and the cdrs of each pattern, using the
;;bindings resulting from the former call for the bindings argument in
;;the latter call. 

(DEFUN MATCH-WITH-BINDINGS (pat1 pat2 bindings)
  (cond ((pattern-var pat1)
	 (var-match pat1 pat2 bindings))
	((pattern-var pat2)
	 (var-match pat2 pat1 bindings))
	((atom pat1)
	 (if (equal pat1 pat2)
	     (list bindings)))
	((atom pat2) nil)    ;PATTERNS ARE UNEQUAL ATOMS
	(t                   ;BOTH PATTERNS ARE LISTS
	 (let ((car-result
		 (match-with-bindings (car pat1)
				      (car pat2)
				      bindings)))
	   (and car-result
		(match-with-bindings (cdr pat1)
				     (cdr pat2)
				     (car car-result)))))))

;;VAR-MATCH returns a list of bindings if variable and item are equal.
;;Otherwise, if variable is listed in bindings, it calls
;;MATCH-WITH-BINDINGS for the item the variable is bound to and item.
;;If it is not listed in bindings and variable is not CONTAINED-IN
;;item, the binding of variable and item is added to bindings and a list
;;of bindings is returned.

(DEFUN VAR-MATCH (variable item bindings)
  (if (equal variable item)
      (list bindings)
      (let ((var-binding (get-binding variable bindings)))
	(cond (var-binding
	       (match-with-bindings var-binding item bindings))
	      ((not (contained-in variable item bindings))
	       (list (add-binding variable item bindings)))))))


;;CONTAINED-IN returns true if item contains variable or any item which
;;is bound to variable in bindings.

(DEFUN CONTAINED-IN (variable item bindings)
  (cond ((atom item) nil)
	((null item) nil)
	((pattern-var item)
	 (or (equal variable item)
	     (contained-in variable
			   (get-binding item bindings)
			   bindings)))
	(t
	 (or (contained-in variable (car item) bindings)
	     (contained-in variable (cdr item) bindings)))))


;;ADD-BINDING adds the binding of variable and item to bindings.

(DEFUN ADD-BINDING (variable item bindings)
  (cons (list variable item) bindings))



;;PATTERN-VAR returns true if item is a variable (begins with ?) and false
;;otherwise.

(DEFUN PATTERN-VAR (item)
  (cond ((or (numberp item)
	     (listp item))
	 nil)
	(t
	 (equal (string (char (prin1-to-string item) 0))
		"?"))))


;;GET-BINDING returns the value the variable is bound to in bindings.

(DEFUN GET-BINDING (variable bindings)
  (cadr (assoc variable bindings)))






