(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;ELIZA.LSP (To be used in conjunction with ELIZA-DATA.LSP)


;;*RESP-RULE-LIST* is a list of pairs of words.  When the first member
;;of the pair is present in a sentence, it is replaced by the second.
;;The user might want to augment this list with contractions or other
;;words.  Note words with apostrophes may require special treatment.          

(DEFVAR *RESP-RULE-LIST* '((I YOU) (ME YOU) (YOU ME) (MY YOUR)
			   (YOUR MY) (AM ARE) (ARE AM)))

(DEFUN BEGIN-ELIZA ()
  (terpri)
  (print '(HELLO. WHAT IS YOUR NAME))
  (terpri)
  (terpri)
  (princ "==> ")
  (let ((name (read)))
    (terpri)
    (cond ((atom name)
	   (print '(IT IS IMPORTANT FOR US TO COME TO AN UNDERSTANDING))
	   (terpri)
	   (print '(I WILL WORK WITH YOU ONLY UNDER ONE CONDITION.))
	   (terpri)
	   (print '(THAT YOU SURROUND EACH OF YOUR RESPONSES WITH
			 A SET OF PARETHESIS))
	   (terpri)
	   (print (append '(IS THAT UNDERSTOOD) (list name '?)))
	   (terpri)
	   (terpri)
	   (princ "==> ")
	   (cond ((not (listp (read)))
		  (terpri)
		  (print '(BYE)))
		 (t
		  (terpri)
		  (print (append '(WHAT WOULD YOU LIKE TO TALK ABOUT
					TODAY)
				 (list name '?)))
		  (eliza-driver-loop))))
	  (t
	   (print (append '(WHAT WOULD YOU LIKE TO TALK ABOUT TODAY)
			  (append name (list '?))))
	   (eliza-driver-loop)))))

;;ELIZA-DRIVER-LOOP is the top level command loop of the program.  It
;;reads the sentence entered by the user, returns a response, and then
;;awaits another sentence by the user.  The loop ends when the user
;;enters (BYE).

(DEFUN ELIZA-DRIVER-LOOP ()
  (terpri)
  (terpri)
  (princ "==> ")
  (let ((sentence (read)))
    (cond ((equal sentence '(bye))
	   '(bye))
	  (t
	   (terpri)
	   (print (return-response sentence))
	   (eliza-driver-loop)))))

;;RETURN-RESPONSE returns a list of words that will be the response
;;to sentence, which is input by the user.

(DEFUN RETURN-RESPONSE (sentence)
  (let ((key-list (key-search sentence)))
    (if (null key-list)
	(rules sentence *resp-rule-list*)
	(get-response key-list sentence))))

;;GET-RESPONSE returns the response associated with a matched keyword on
;;key-list, or the list '(TELL ME MORE) if there is no match.  Key-list
;;is a list of keywords and their corresponding data as it is listed in
;;*keyword-list*.

(DEFUN GET-RESPONSE (key-list sentence)
  (let ((response (match-result (car key-list)
				      sentence)))
	  (cond ((and (null response)
		      (null (cdr key-list)))
		 '(TELL ME MORE))
		((null response)
		 (get-response (cdr key-list) sentence))
		(t
		 response))))

;;KEY-SEARCH returns a rank-ordered list of the keywords in a sentence.
;;It uses the local functions SEARCH-ITER and ORDER-INSERT.  SEARCH-ITER
;;tests each word of the sentence to see if it is a keyword.  If it is,
;;it calls ORDER-INSERT to place the word in the proper location of
;;key-list.  After all the words of sentence have been examined,
;;KEY-SEARCH returns key-list.

(DEFUN KEY-SEARCH (sentence)
   
    (DEFUN ORDER-INSERT (word key-list)
      (DEFUN INSERT-ITER (front-list word key-list)
	(cond ((null key-list)
	       (setf key-list (append front-list
				      (list word))))
	      ((> (rank (keyword? word))
		  (rank (keyword? (car key-list))))
	       (append front-list (list word) key-list))
	      (t
	       (insert-iter (append front-list
				    (list (car key-list)))
			    word
			    (cdr key-list)))))
      (insert-iter '() word key-list))

    (DEFUN SEARCH-ITER (sentence key-list)
      (cond ((null sentence)
	     key-list)
	    ((keyword? (car sentence))
	     (setf key-list (order-insert (car sentence)
					  key-list))
	     (search-iter (cdr sentence) key-list))
	    (t
	     (search-iter (cdr sentence) key-list))))

    (search-iter sentence nil))

;;MATCH-RESULT returns a response to sentence associated with keyword.
;;Where there are several possible response patterns associated with
;;keyword, it uses the first response pattern that matches sentence, and
;;returns the corresponding response.
 
(DEFUN MATCH-RESULT (keyword sentence)
  
  ;M-R-ITER returns a response formed by the first pattern of data that
  ;matches sentence.
  
  (DEFUN M-R-ITER (data sentence)
    (let ((result (match (car data) sentence)))
      (if (or (equal (length data) 1)
	      (not (null result)))
	  result
	  (m-r-iter (cdr data) sentence))))

  (m-r-iter (pattern-info (keyword? keyword)) sentence))

;;MATCH returns the result of trying to match sentence to each of the
;;pattern-response combinations listed in data.  It returns either the
;;first matched response or nil.  It does this by a single call to the
;;internally defined function 

(DEFUN MATCH (data sentence)
  
  (DEFUN M-ITER (pat resp sent symbol-table)
    (cond ((null pat)
	   (translate resp symbol-table))
	  ((variable (car pat))
	   (let ((first-part (match-first pat (cadr pat) sent)))
	     (setf symbol-table
		   (insert (car pat)
			   (rules first-part *resp-rule-list*)
			   symbol-table))
	     (m-iter (cdr pat)
		     resp
		     (trunc-iter first-part sent)
		     symbol-table)))
	  ((equal (car pat) (car sent))
	   (m-iter (cdr pat)
		   resp
		   (cdr sent)
		   symbol-table))))
  
  (m-iter (car data) (cadr data) sentence '(*table*)))

;;TRANSLATE replaces variables in a response pattern by what they match to
;;and returns the resulting response.  It uses the internally defined
;;function VALUE to return the value of word (its first argument)
;;according to symbol-table (its second argument) if word is a variable.

(DEFUN TRANSLATE (resp symbol-table)
  (DEFUN VALUE (word symbol-table)
    (if (variable word)
	(lookup word symbol-table)
	word))
  (if (null resp)
      nil
      (flatten
	(cons (value (car resp) symbol-table)
	      (translate (cdr resp) symbol-table)))))


;;MATCH-FIRST matches the first variable of pat to the word(s) of
;;sentence up to but not including word-after-var(iable). Word-after-var
;;will always be the word following the variable being matched in a
;;pattern.  MATCH-FIRST does this by a single call to the internally
;;defined function MATCH-ITER.

(DEFUN MATCH-FIRST (pat word-after-var sentence)
  (DEFUN MATCH-ITER (temp sent-tail pat word-after-var sentence)
    (cond ((null word-after-var)
	   sent-tail)
	  ((equal (append temp (member word-after-var sentence))
		  sentence)
	   temp)
	  (t
	   (match-iter (append temp (list (car sent-tail)))
		       (cdr sent-tail)
		       pat
		       word-after-var
		       sentence))))
  (match-iter nil sentence pat word-after-var sentence))
 
;;INSERT inserts var, a variable, and value, its value, into table.
;;Lookup returns the value of var, a variable, by looking it up in
;;table.

(DEFUN INSERT (var value table)
  (cons (car table)
	(cons (cons var value)
	      (cdr table))))

(DEFUN LOOKUP (var table)
  (cdr (assoc var (cdr table))))

;;RULES replaces certain words in SENTENCE by certain other words, according
;;to the RULE-LIST (i.e., *resp-rule-list*).

(DEFUN RULES (sentence rule-list)
  (cond ((null sentence)
	 nil)
	((atom sentence)
	 (let ((n (assoc sentence rule-list)))
	   (if (null n)
	       sentence
	       (cdr n))))
	(t
	 (flatten (cons (rules (car sentence) rule-list)
			(rules (cdr sentence) rule-list))))))

;;FLATTEN converts a nested list to a list of atoms by removing all
;;nested parethesis.

(DEFUN FLATTEN (lst)
  (cond ((null lst)
	 nil)
	((atom lst)
	 (list lst))
	(t
	 (append (flatten (car lst))
		 (flatten (cdr lst))))))

;;Each keyword and its relevant information is stored as an element on
;;*KEYWORD-LIST*.  An element consists of a list of (1) the keyword, (2)
;;its rank, and (3) a list of its pattern-responses.  A pattern response
;;is a list of a word pattern, often containing variables, and a
;;response.

(DEFVAR *KEYWORD-LIST* NIL)

;;KEYWORD? returns the entire entry of word in *KEYWORD-LIST* if word is
;;a keyword, and returns nil if it is not.

(DEFUN KEYWORD? (word)
  (assoc word *keyword-list*))

;;RANK returns the rank of the entry.

(DEFUN RANK (keyword-entry)
  (cadr keyword-entry))

;;PATTERN-INFO returns a list of the pattern-responses of an entry.

(DEFUN PATTERN-INFO (keyword-entry)
  (caddr keyword-entry))

;;INSERT-KEY adds keyword to *keyword-list*.  A keyword must be an
;;a list of the three members defined above.

(DEFUN INSERT-KEY (keyword rank pattern-info)
  (if (and (atom keyword)
	   (numberp rank)
	   (listp pattern-info))
      (setf *keyword-list* (cons
			     (list keyword rank pattern-info)
			     (remove (keyword? keyword) *keyword-list*)))
      (princ-r "ERROR: Incorrect notation for keyword." 2)))

;;TRUNC-ITER returns what remains of SENTENCE after WORDS are removed.

(DEFUN TRUNC-ITER (words sentence)
  (if (null words)
      sentence
      (trunc-iter (cdr words) (cdr sentence))))

;;VARIABLE returns true if word begins with a question mark and nil
;;otherwise.

(DEFUN VARIABLE (word)
  (equal (string
	   (char (prin1-to-string word) 0))
	 "?"))
      
