(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;ATN-INTERPRET.LSP


;;;The following two functions are used to interpret and apply the
;;;notation the user uses to write tests and actions for the arcs.

;;INTERPRET is the main function.  It takes as its arguments the
;;expression to interpret, a list of local registers, and a list of the
;;current registers. Depending on the format of the expression,
;;INTERPRET calls itself recursively on some part of the expression, it
;;calls INTERPRET-LOOP on the expression, it applies a function to the
;;result of ARG-INTERPRET of the function arguments, or it returns a
;;value directly.

(DEFUN INTERPRET (expr * current-registers)
  (cond ((equal expr '*)
	 *)
	((member expr '(T NONE))
	 expr)
	((atom expr)
	 (cadr (assoc expr (cdr current-registers))))
        ((and (equal (car expr) 'R)
	      (equal (length expr) 2))
	 (interpret (cadr expr) * current-registers))
	((equal (car expr) 'R)
	 (cadr (assoc (cadr expr)
		      (cdr (interpret (cons 'R (cddr expr))
				      *
				      current-registers)))))
	((equal (car expr) 'V)
	 (cadr expr))
	(t                                      ;ASSUME LISP EXPRESSION
	 (eval (cons (car expr)
		     (interpret-args (cdr expr)
				     *
				     current-registers))))))

;;INTERPRET-ARGS takes as its arguments an argument list, the local
;;registers, and the current registers of the parse.  It returns a list
;;of arguments each of which is the interpretation of a member arg-list.

(DEFUN INTERPRET-ARGS (arg-list * current-registers)
  (do ((new-args nil)
       (args arg-list (cdr args)))
      ((null args) (reverse new-args))
    (setf new-args
	  (cons (list 'quote
		      (interpret (car args)
				 *
				 current-registers))
		new-args))))


;;;In the ATN parser, an arc is successfully followed if its arc test is
;;;passed.  Included in following an arc is updating the current
;;;registers of the network containing the arc by carrying out arc
;;;actions.

;;An action can either be a direct register assignment, or it can be a
;;conditional register assignment.  If it is the former, REG-ASSIGN
;;returns true, REGISTER returns the register the will receive the new
;;value, and VALUE returns the new value.

(DEFUN ACTION-REG (action) (car action))
(DEFUN VALUE (action) (caddr action))
(DEFUN REG-ASSIGN (action)
  (equal (cadr action) '<))

;;DO-ACTIONS updates the current-registers for a network or returns a
;;set of preset registers for a push-arc by performing each of the
;;actions in the actions list.  It takes as its arguments a list of
;;actions to be performed, a list of local registers, a list of the
;;current registers of the parse, and a list of the registers to be
;;added to be changed.

(DEFUN DO-ACTIONS (actions * current-registers new-registers hold-list)
  (mapcar #'(lambda (action)
	      (cond ((reg-assign action)
		     (setf new-registers
			   (do-reg-assign action
					  *
					  current-registers
					  new-registers))
		     (setf current-registers
			   (do-reg-assign action
					  *
					  current-registers
					  current-registers)))
		    ((equal (car action) 'IF)
		     (let ((reg-and-holds
			     (do-if-action action
					   *
					   current-registers
					   new-registers
					   hold-list)))
		       (setf new-registers (car reg-and-holds))
		       (setf hold-list (cadr reg-and-holds))))
		    ((equal (car action) 'hold)
		     (setf hold-list
			   (do-hold action *
				    current-registers
				    new-registers
				    hold-list)))))
	  actions)
  (list new-registers hold-list))

;;DO-REG-ASSIGN does the interpreted action.

(DEFUN DO-REG-ASSIGN (action * current-registers new-registers)
  (add-register (cadr (action-reg action))
		(interpret (value action)
			   *
			   current-registers)
		new-registers))

;;DO-IF-ACTION does the interpreted action following the antecedant if the
;;interpretation of the antecedant is true.  Otherwise, it does the
;;alternative (ELSE) interpreted action if one exists.

(DEFUN DO-IF-ACTION (action * current-registers new-registers hold-list)
  (cond ((interpret (cadr action) * current-registers)
	 (do-actions (caddr action)
		     *
		     current-registers
		     new-registers
		     hold-list))
	(t
	 (do-actions (cadddr action)
		     *
		     current-registers
		     new-registers
		     hold-list))))

;;DO-HOLD does a hold action.  If hold is followed by an argument an
;;interpreted expression is held.  Otherwise, the current registers are
;;held.

(DEFUN DO-HOLD (action * current-registers new-registers hold-list)
  (cond ((null (cadr action))
	 (add-to-hold (car new-registers) new-registers hold-list))
	(t
	 (add-to-hold (car new-registers)
		      (interpret (cadr action)
				 *
				 current-registers)
		      hold-list))))
;;ADD-TO-HOLD adds a list of a network name and a set of registers to
;;the hold list.

(DEFUN ADD-TO-HOLD (network-name registers hold-list)
  (cons (list network-name
	      registers)
	hold-list))

;;SEARCH-HOLD-LIST checks each item on the hold list and returns it if
;;it is the constituent sought after.

(DEFUN SEARCH-HOLD-LIST (const lst)
  (cond ((null lst)
	 nil)
	((equal (caadar lst) const)
	 (car lst))
	(t
	 (search-hold-list const (cdr lst)))))


;;ADD-REGISTER is a function called by DO-ACTIONS to add new registers
;;to the current registers or change a value of a current register.  It
;;takes as its arguments, the register name, the value, and the list of
;;current registers.

(DEFUN ADD-REGISTER (reg val current-registers)
  (let ((r (assoc reg (cdr current-registers))))
    (cond (r (setf current-registers
		   (cons (car current-registers)
			 (cons (list reg val)
			       (remove r (cdr current-registers))))))
	  (t (setf current-registers
		   (cons (car current-registers)
			 (cons (list reg val)
			       (cdr current-registers)))))))
  current-registers)

;;AUX-AGREE returns true if verb order is legal. (See text.)

(DEFUN AUXAGREE (v aux-list)
  (let ((root-last (interpret '(r root)
			      nil
			      (car (last aux-list))))
	(form-v (interpret '(r form)
			   nil
			   v)))
    (cond ((null aux-list)
	   (intersection form-v '(pres past)))
	  ((and (equal root-last 'be)
		(intersection form-v '(ing)))
	   'progressive)
	  ((and (equal root-last 'be)
		(intersection form-v '(en)))
	   'passive)
	  ((and (equal root-last 'have)
		(intersection form-v '(en)))
	   'perfect)
	  ((intersection form-v '(inf))
	   'modal))))

(DEFUN AGR (feature1 feature2)
  (intersection feature1 feature2))

;;TRANS returns true if the verb is transitive.

(DEFUN TRANS (subcat-reg)
  (intersection subcat-reg '(obj)))

;;INTRANS returns true if the verb is intransitive.

(DEFUN INTRANS (subcat-reg)
  (not (intersection subcat-reg '(obj))))









