(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;ATN-DATA.LSP


;;;The following functions define the data structures for the ATN parser
;;;and provide tools for redefining and modifying the network and
;;;lexicon.

;;;The general structure of the augmented transition network (ATN) is
;;;similar to that of the recursive transition network (RTN).  Each arc,
;;;however, is a list composed of five members.  In addition to the
;;;numeric arc label, the arc function, and the name of the node at
;;;which the arc terminates, an arc contains an arc test and arc
;;;actions.

;;;In addition to the five types of arcs used in the RTN parser, this ATN
;;;parser is supplied with a virtual arc.  The corresponding arc
;;;function, VIR-ARC, takes a constituent type as its argument.

   		   
(DEFUN NODE-NAME (node) (car node))  

(DEFUN ARCS (node) (cdr node))

(DEFUN ARC-NUM (arc) (car arc))

(DEFUN ARC-FUNC (arc) (cadr arc))

(DEFUN ARC-ARG (arc) (cadr (arc-func arc)))

(DEFUN ARC-ARG2 (arc) (caddr (arc-func arc)))

(DEFUN DESTINATION (arc) (caddr arc))

(DEFUN AUG-INFO (arc) (cdddr arc))

(DEFUN TEST (aug-info) (car aug-info))

(DEFUN ACTIONS (aug-info) (cadr aug-info))


;;ADD-ARC adds a new node and arc to the current network, or adds a new
;;arc to a node already in the network.  It takes two arguments: (1) the
;;name of the node (which may or may not be in the current network) and
;;(2) an arc.  The newly created arc has the default value (T) for its
;;arc test, and the default value (NONE) for its arc actions.  This
;;function can be used to incrementally build a network or to modify one.

(DEFUN ADD-ARC (node-name arc)
  (let ((node (assoc node-name *atn*))
	(atn-arc (append arc (copy-alist (list '(t) '(none))))))
    (cond ((null node)
	   (setf *atn* (cons (list node-name atn-arc)
			   *atn*)))
	  (t
	   (let ((new-node (append (list node-name)
				   (order-insert nil atn-arc (arcs node)))))
	       (setf *atn* (cons new-node
			       (remove node *atn*)))))))
  (print-atn (list (assoc node-name *atn*))))


;;ORDER-INSERT is called by ADD-ARC (above) to insert an arc into the
;;proper place in the network.  It is an iterative function with three
;;arguments: (1) befores (2) new-arc (the arc being entered), and (3)
;;afters.  Befores and afters are the arcs whose position are before and
;;after new-arc in the proper arc-list of the network.

(DEFUN ORDER-INSERT (befores new-arc afters)
  (cond ((null afters)
	 (append (reverse befores)
		 (list new-arc)))
	((<= (arc-num new-arc)
	     (arc-num (car afters)))
	 (append
	   (append (reverse befores) (list new-arc))
	   afters))
	(t 
	 (order-insert (cons (car afters) befores)
		       new-arc
		       (cdr afters)))))


;;DELETE-ARC deletes an arc from the current network.  It takes two
;;arguments: (1) the name of the node the arc starts at, and (2) the arc
;;number.  This function is used to modify a network.

(DEFUN DELETE-ARC (node-name arc-num)
  (let ((old-node (assoc node-name *atn*)))
    (let ((new-node (remove (assoc arc-num (arcs old-node))
			    old-node)))
      (cond ((null old-node)
	     (print "No such arc in ATN"))
	    ((equal (length new-node) 1)
	     (setf *atn* (remove old-node *atn*)))
	    (t
	     (setf *atn* (cons new-node
			     (remove old-node *atn*)))))))
  (print-atn (list (assoc node-name *atn*))))


;;ADD-TEST and ADD-ACTION can be used to add arc tests and arc actions
;;to an augmented network. ADD-TEST replaces the current test by its
;;argument.  ADD-ACTION adds an action to the current list of arc
;;actions.
          
(DEFUN NEW-TEST (node-name arc-num tst)
  (let ((node  (assoc node-name *atn*)))
    (let ((arc (assoc arc-num (cdr node))))
      (rplaca (test (aug-info arc))
	      tst)
      (p-lines 1)
      (print-arcs (list arc) 0))))

(DEFUN ADD-ACTION (node-name arc-num action)
  (let ((node (assoc node-name *atn*)))
    (let ((arc (assoc arc-num (cdr node))))
      (let ((a (actions (aug-info arc))))
	(cond ((equal a '(none))
	       (rplaca a (list action)))
	      (t
	       (rplaca a (cons action
			       (car (copy-alist a)))))))
      (p-lines 1)
      (print-arcs (list arc) 0))))


;;DELETE-ACTION deletes an arc action from an augmented network.
;;Action-num (its third argument) is a number that represents the
;;actions placement within the list of arc actions.  For example, if
;;action-num equals two, then the second action of the action list will
;;be deleted.

(DEFUN DELETE-ACTION (node-name arc-num action-num)
  (let ((node (assoc node-name *atn*)))
    (let ((arc (assoc arc-num (cdr node))))
      (let ((action (nth (1- action-num)
			 (car (actions (aug-info arc))))))
		    
	(let ((new-actions (copy-list
			     (remove action
				     (car (actions (aug-info arc)))))))
	  (cond ((null new-actions)
		 (rplaca (actions (aug-info arc))
			 'none))
		(t
		 (rplaca (actions (aug-info arc))
			 new-actions)))
	  (p-lines 1)
	  (print-arcs (list arc) 0))))))


;;ARC-INSTR prints out detailed instructions on how to add an arc to
;;and delete an arc from the current network.    

(DEFUN ARC-INSTR ()
  (princ-r "ADDING AND DELETING ARCS" 2)
  (terpri)
  (princ-r "To add an arc, enter a list of:" 4)
  (terpri)
  (princ-r "(1) ADD-ARC " 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the unaugmented arc" 5)
  (terpri)
  (princ-r "where an unaugmented arc is a list of " 2)
  (terpri)
  (princ-r "(1) the arc number" 5)
  (princ-r "(2) the arc function " 5)
  (princ-r "(3) the node the arc terminates at." 5)
  (terpri)
  (princ-r "Since the arc test and actions are entered separately," 2)
  (princ-r "they are initially given the default values" 2)
  (princ-r "(T) and (NONE), respectively." 2)
  (terpri)
  (princ-r "For example, to enter the arc:" 2)
  (terpri)
  (princ-r "(1 (PUSH-ARC NP) S2 (T) (NONE)" 5)
  (terpri)
  (princ-r "from node S, enter the list:" 2)
  (terpri)
  (princ-r "(ADD-ARC 'S '(1 (PUSH-ARC NP) S2))." 5)
  (p-lines 2)
  (princ-r "To delete an entire arc, enter a list of:" 4)
  (terpri)
  (princ-r "(1) DELETE-ARC" 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the arc number." 5)
  (terpri)
  (princ-r "For example, to delete the arc from the example" 2)
  (princ-r "above, enter the list:" 2)
  (terpri)
  (princ-r "(DELETE-ARC 'S 1)" 5))


;;TEST-ACTION-INSTR prints out detailed instructions on how to modify
;;the tests and actions of an arc.  
    
(DEFUN TEST-ACTION-INSTR ()
  (princ-r "ADDING AND DELETING TESTS AND ACTIONS" 2)
  (terpri)
  (princ-r "To replace an arc test by a new test, enter a list of:" 4)
  (terpri)
  (princ-r "(1) NEW-TEST" 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the arc number" 5)
  (princ-r "(4) the new test (i.e. (EQUAL NUM PLURAL))" 5)
  (terpri)
  (princ-r "Similarily, to add a new action to the list of arc actions," 2)
  (princ-r "enter a list whose first element is ADD-ACTION and whose" 2)
  (princ-r "other elements are analogous to those above." 2)
  (terpri)
  (princ-r "Examples that modify" 2)
  (terpri)
  (princ-r "(1 (PUSH-ARC NP) S2 (T) (NONE)" 5)
  (terpri)
  (princ-r "are: " 2)
  (terpri)
  (princ-r "(NEW-TEST 'S 1 '(EQUAL NUM PLURAL))" 5)
  (princ-r "(ADD-ACTION 'S 1 '((R SUBJ) < *))." 5)
  (terpri)
  (princ-r "After executing these commands the modified arc is:" 2)
  (terpri)
  (princ-r "(1 (PUSH-ARC NP) S2 ((EQUAL NUM PLURAL)) ((((R SUBJ) < *))))" 5)
  (terpri)
  (princ-r "To delete an action from the list of arc actions, enter" 4)
  (princ-r "a list of:" 2)
  (terpri)
  (princ-r "(1) DELETE-ACTION" 5)
  (princ-r "(2) the name of the node the arc starts at" 5)
  (princ-r "(3) the arc number." 5)
  (princ-r "(4) the location of action to be deleted (1 is 1st, 2 is 2nd, etc.)" 5)
  (terpri)
  (princ-r "To delete the action of the arc above, enter the list:" 2)
  (terpri)
  (princ-r "(DELETE-ACTION 'S 1 1)" 5)
  (terpri))


;;PRINT-ATN-INSTR prints out detailed instructions on how to print
;;all or part of the network out.  The print commands to which it refers
;;are available from within MODIFY-ATN-LOOP as well as from the top
;;level.

(DEFUN PRINT-NETWORK-INSTR ()
  (princ-r "PRINTING THE CURRENT NETWORK" 2)
  (terpri)
  (princ-r "To display the current network, enter:" 4)
  (terpri)
  (princ-r "(PRINT &optional node-prefix)" 5)
  (terpri)
  (princ-r "If PRINT has no argument, the entire network is" 2)
  (princ-r "displayed on the screen.  Otherwise, only those" 2)
  (princ-r "nodes whose node-names begin with node-prefix" 2)
  (princ-r "are returned."2)
  (terpri)
  (princ-r "Typical print commands are:" 2)
  (terpri)
  (princ-r "(PRINT)" 5)
  (princ-r "(PRINT 'NP)" 5)
  (princ-r "(PRINT 'AUX2)" 5)
  (terpri))


;;ATN-INSTRUCTIONS prints out all three of the individual sets of
;;instructions listed above, and then lists the choice of commands
;;availabe to the user from within MODIFY-ATN-LOOP.

(DEFUN ATN-INSTRUCTIONS ()
  (terpri)
  (arc-instr)
  (p-lines 1)
  (princ-i "Type any character and return to continue:" 2)
  (read)
  (p-lines 1)
  (test-action-instr)
  (princ-i "Type any character and return to continue:" 2)
  (read)
  (p-lines 1)
  (print-network-instr)
  (princ-i "Type any character and return to continue:" 2)
  (read)
  (terpri)
  (print-atn-commands))


;;PRINT-ATN-COMMANDS lists all the commands available to the user from
;;within MODIFY-ATN-LOOP.  Then, it returns the user to the loop.

(DEFUN PRINT-ATN-COMMANDS ()
  (p-lines 1)
  (princ-r "Commands: (ADD-ARC node-name arc)" 2)
  (princ-r "(DELETE-ARC node-name arc-num)" 12)
