(print "****************************************************************************")
(print "Notice: Copyright 1987 by The Benjamin/Cummings Publishing Company, Inc.")
(print "        All rights reserved.  No part of this software may be reproduced,")
(print "        stored in a retrieval system, or transmitted, in any form or by")
(print "        any means, electronic, mechanical, photocopying, recording, or")
(print "        otherwise, without the prior written permission of Benjamin/Cummings")
(print "        Publishing Company.")
(print "****************************************************************************")

;;; -*- Mode:Common-Lisp; Base:10 -*-
;;;ATN.LSP


;;;The following code is an implementation of an augmented transition
;;;network.  It uses the data functions and structures defined in
;;;ATN-DATA.LISP, funtions related to register assignment and
;;;interpretation defined in INTERPRET.LISP, and the print functions and
;;;other general functions defined in UTILITIES.LISP.


;;BACKUPS is a global variable that stores backup states for the current
;;parse.  The parser chooses the last backup state on BACKUPS as its
;;current state by calling POP-BACKUPS.  POP-BACKUPS removes the last
;;backup state entered onto BACKUPS and returns this state.

(DEFVAR *BACKUPS* NIL)

(DEFUN POP-BACKUPS ()
  (let ((state (car *backups*)))
    (setf *backups* (cdr *backups*))
    state))

;;;CAT-ARC, PUSH-ARC, POP-ARC, WORD-ARC, JUMP-ARC and VIR-ARC are arc
;;;functions which take as their arguments: (1) an arc (which should always
;;;be the arc from which the arc-function was called, (2) the sentence
;;;pointer, (3) the list of return points, (4) the current registers, and
;;;(5) the list of held constituents.  All five functions return a new
;;;parse state.  A parse state is composed of a similar list of four items:
;;;(1) the name of the current node, (2) the sentence pointer, (3) the list
;;;of return points, (4) the current registers (current parse for the
;;;sentence) and (5) a list of the held constituents.  A return point
;;;consists of a list of the name of the node to return to, the test and
;;;actions to be performed before actually returning, and the current
;;;registers for the sentence at that node.


;;CAT-ARC adds all the states returned by CAT-REC to the backup list.
;;It returns the first state.

(DEFUN CAT-ARC (arc pointer returns current-registers hold-list)
  (let ((temps (cat-loop (look-up (aref *sentence* pointer)

				  (arc-arg arc))
			 arc
			 pointer
			 returns
			 current-registers
			 hold-list)))
    (cond ((> (length temps) 1)
	   (setf *backups* (append (cdr temps) *backups*))))
    (car temps)))


;;CAT-REC applies the arc tests to each lexicon entry of the current
;;word which is of the particular syntactic category. For each lexicon
;;entry for which the arc test evaluates to true, the arc actions are
;;performed and a new state is created.  CAT-REC returns a list of all
;;the new states.

(DEFUN CAT-LOOP (entries arc pointer returns current-registers hold-list)
  (let ((temps nil))
    (cond ((null entries)
	   nil)
	  (t
	   (mapcar #'(lambda (*)
		       (let ((reg-and-holds
			       (do-actions (car (actions (aug-info arc)))
					   *
					   current-registers
					   current-registers
					   hold-list)))
			 (cond ((null *)
				nil)
			       ((interpret (car (test (aug-info arc)))
					   *
					   current-registers)
				(setf temps
				      (append (list (list (destination arc)
							  (1+ pointer)
							  returns
							  (car reg-and-holds)
							  (cadr reg-and-holds)))
					      temps))))))
		   entries)))
    temps))


;;A virtual arc can be followed if the hold list contains a structure
;;that is of the same syntactic category as the argument of the
;;arc-funtion of the arc, and the arc test evaluates to true. 

(DEFUN VIR-ARC (arc pointer returns current-registers hold-list)
  (let ((held-const (search-hold-list (arc-arg arc) hold-list)))
    (cond ((null held-const)
	   nil)
	  ((interpret (car (test (aug-info arc)))
		      (cadr held-const)
		      current-registers)
	   (list (destination arc)
		 pointer
		 returns
		 (car (do-actions (car (actions (aug-info arc)))
				  (cadr held-const)
				  current-registers
				  current-registers
				  hold-list))
		 (remove held-const hold-list)))
	  (t
	   nil))))


;;A word arc can be followed if the root of the current word in the
;;sentence is identical to the argument of the arc-function of the arc,
;;and the arc test evaluates to true.  In the current implementation, if
;;several entries in the lexicon have the correct root, then the first
;;entry which results in a positive evaluation of the arc test becomes
;;the local registers from which a new state is derived. 

(DEFUN WORD-ARC (arc pointer returns current-registers hold-list)
  (word-rec (look-up (aref *sentence* pointer)
		    'all)
	    arc
	    pointer
	    returns
	    current-registers
	    hold-list))

(DEFUN WORD-REC (entries arc pointer returns current-registers hold-list)
  (let ((* (first entries)))
    (cond ((null entries)
	   nil)
	  ((and (equal (interpret '(r root) nil *)
		       (arc-arg arc))
		(interpret (car (test (aug-info arc)))
			   *
			   current-registers))
	   (let ((reg-and-holds
		   (do-actions (car (actions (aug-info arc)))
			       *
			       current-registers
			       current-registers
			       hold-list)))
	     (list (destination arc)
		   (1+ pointer)
		   returns
		   (car reg-and-holds)
		   (cadr reg-and-holds))))
	  (t
	   (word-rec (cdr entries) arc pointer returns current-registers hold-list)))))


;;To follow a push arc successfully, the current word of the sentence
;;cannot be nil (the end of the sentence).  PUSH-ARC
;;will cons the new return state to RETURNS which includes the node to
;;return to, arc tests and actions, and the current registers. It sets
;;current-registers to a list of the name of the new network being pushed. 
  
(DEFUN PUSH-ARC (arc pointer returns current-registers hold-list)
  (cond ((null (aref *sentence* pointer))
	 nil)
	(t
	 (list (arc-arg arc)       ;NEW-NODE
	       pointer
	       (cons (list (destination arc)
			   (aug-info arc)
			   current-registers)
		     returns)
	       (car (do-actions (arc-arg2 arc)
				'()
				current-registers
				(list (arc-arg arc))
				hold-list))
	       hold-list))))


;;A pop arc can be followed successfully if RETURNS is null and the
;;current word of the sentence is nil (the end of the sentence), or if
;;the current word is not nil, RETURNS is not null, and the test of the
;;next return state can be passed.  In the former case, a list of
;;'SUCCESS and the current registers are returned.  In the latter case,
;;a state is returned with current node equal to that of the first
;;return state in RETURNS, and with current registers equal to the
;;result of a call to DO-ACTIONS on the actions and registers of the
;;return state. 

(DEFUN POP-ARC (arc pointer returns current-registers hold-list)
  (let ((next-state (car returns)))
    (let ((aug-info (cadr next-state))
	  (higher-registers (caddr next-state)))
      
      (cond ((and next-state                     ;RETURNS IS NOT EMPTY
		  (null (assoc (car current-registers)
			       hold-list))
		  (interpret (car (test aug-info))
			     current-registers
			     higher-registers))
	     (let ((reg-and-holds
		     (do-actions (car (actions aug-info))
				 current-registers
				 higher-registers
				 higher-registers
				 hold-list)))
	       (list (car next-state)
		     pointer
		     (cdr returns)
		     (car reg-and-holds)
		     (cadr reg-and-holds))))
	    ((and (null returns)          ;RETURNS IS EMPTY AND
		                          ;POINTER IS AT END OF SENTENCE
		  (null (assoc (car current-registers)
			       hold-list))
		  (null (aref *sentence* pointer)))
	     (list 'success current-registers))
	    (t
	     nil)))))


;;A jump arc can always be followed.  It returns a state with a new
;;node: the destination of the arc of its argument.  The pointer,
;;returns, and current registers remain the same.

(DEFUN JUMP-ARC (arc pointer returns current-registers hold-list)
  (cond ((interpret (car (test (aug-info arc)))
		    nil
		    current-registers)
	 (list (destination arc)
	       pointer
	       returns
	       current-registers
	       hold-list))
	(t
	 nil)))

;;;The parse procedure of the ATN parser is similar to that of the RTN
;;;parser.  BEGIN-PARSER calls START-PARSE which enters PARSE-LOOP.
;;;Each loop through PARSE-LOOP parses a sentence by recursively calling
;;;PARSE after NEXT-NODE leads the parser to a new state.  

;;The global variables *ATN*, *LEXICON*, *SENTENCE*, AND *BACKUPS* are
;;initialized to NIL.  They are reset during START-PARSE and PARSE.

(DEFVAR *ATN* NIL)
(DEFVAR *LEXICON* NIL)
(DEFVAR *SENTENCE* NIL)
(DEFVAR *BACKUPS* NIL)

;;START-PARSE is the function called by the user to start the
;;parse-loop.  It gives the user the option of entering his or her own
;;network and lexicon, using the sample data defined in ATN-DATA.LISP,
;;or, after the first call to START-PARSE, using or modifying the
;;currently defined network and lexicon.  Then it calls PARSE-LOOP.

(DEFUN START-PARSE()
  (p-lines 1)
  (princ-r "Will you use the sample ATN (S), the current ATN (C) or a modification (M)?" 2)
  (princ "  ==> (S C M) ")
  (let ((answer (read)))
    (cond ((equal answer 'M)
	   (modify-atn))
	  ((equal answer 'S)
	   (setf *atn* *sample-atn*))))
  (terpri)
  (princ-r "Will you use the sample lexicon (S), the current lexicon (C) or a modification (M)?" 2)
  (princ "  ==> (S C M) ")
  (let ((answer (read)))
    (cond ((equal answer 'M)
	   (modify-lexicon))
	  ((equal answer 'S)
	   (setf *lexicon* *sample-lexicon*))))
  (terpri)
  (parse-loop))


;;PARSE-LOOP sets the parse options and queries the user for the
;;sentence to be parsed.  It calls PARSE on this sentence.  Then it
;;either repeats the loop and parses another sentence, calls START-PARSE
;;so the user can modify the RTN or lexicon, or returns 'END, according
;;to the users responses to its queries.

(DEFUN PARSE-LOOP()
  (let ((options (set-parse-options)))
    (let ((parse-flag (car options))
	  (trace-flag (cadr options)))
      (terpri)
      (princ-r "Enter the sentence to be parsed as an ordered list of words." 2)
      (princ-r "- For a text example type: (WHO IS THE BABY CARRYING)" 2)
      (terpri)
      (princ "  ==> ")
      (let ((i (read)))
	(setf *sentence* (make-array (1+ (length i)) :initial-contents
				     (append i (list nil)))))
      (setf *backups* nil)
      (set-up trace-flag)
      (parse '(s 0 nil (S) nil) parse-flag trace-flag)
      (terpri)
      (princ-r "Will you modify the network or lexicon? (Y or N)" 2)
      (princ "  ==> ")
      (cond ((equal (read) 'y)
	     (start-parse))
	    (t
	     (terpri)
	     (princ-r "Will you parse another sentence? (Y [DEFAULT] or N)" 2)
	     (princ "  ==> ")
	     (cond ((equal (read) 'n)
		    'end)
		   (t
		    (terpri)
		    (parse-loop))))))))


;;SET-PARSE-OPTIONS sets the parse-flag to the users requested mode and
;;the trace-flag to the user's requested level of detail.  

(DEFUN SET-PARSE-OPTIONS ()
  (princ-r "Enter your parse preference: P = single parse with Prompt for another [DEFAULT]" 2)
  (princ-r "                             A = all possible parses" 2)
  (princ "  ==> ")
  (let ((parse-flag (read)))
    (terpri)
    (cond ((not (or (equal parse-flag 'P)
		    (equal parse-flag 'A)))
	   (setf parse-flag 'P)))
    (princ-r "Enter your trace preference: 0 = No Trace" 2)
    (princ-r "                             1 = General Trace" 2)
    (princ-r "                             2 = Detailed Trace [DEFAULT]" 2)
    (princ "  ==> ")
    (let ((trace-flag (read)))
      (cond ((not (member trace-flag '(0 1 2)))
	     (setf trace-flag 2)))
      (list parse-flag trace-flag))))
      
  
;;PARSE is the body of the top down ATN parser.  It takes as its
;;arguments (1) a state (consisting of a node name, the sentence
;;pointer, a list of return states, a list of the current registers, and
;;a hold list), (2) a parse flag, and (3) a trace flag.  PARSE returns
;;either 'SUCCESS!  or 'FAIL to reflect the success or failure of the
;;particular parse, and prints out the final list of registers.

(DEFUN NDE (current-state) (car current-state))
(DEFUN PTR (current-state) (cadr current-state))
(DEFUN RET (current-state) (caddr current-state))
(DEFUN REG (current-state) (cadddr current-state))
(DEFUN H-L (current-state) (nth 4 current-state))

(DEFUN PARSE (current-state parse-flag trace-flag)
  (let ((node-name (nde current-state))
	(pointer (ptr current-state))
	(returns (ret current-state))
	(current-registers (reg current-state))
	(hold-list (h-l current-state)))

    (let ((start-node (assoc node-name *atn*))
	  (bu-temp nil)
	  (arcs-followed nil))
    
      (mapcar #'(lambda (arc)
		  (let ((state
			 (apply (car (arc-func arc))
				(list arc pointer returns current-registers hold-list))))
		    (cond (state       ;IF  ARC CAN BE FOLLOWED SUCCESSFULLY,
			   (setf bu-temp      ; ADD RESULTING STATE TO TEMP-BU
				 (append (list state) bu-temp))
			   (cond ((equal trace-flag 2)
				  (setf arcs-followed  ;ADD ARC-LABEL TO
					               ; ARCS-FOLLOWED (TRACE=2)
					(append (list
						  (list (node-name start-node)
							(arc-num arc)))
						arcs-followed))))))))
	      (arcs start-node))
      
      (print-data current-state
		  trace-flag
		  (reverse arcs-followed))
      
      (setf *backups* (append (reverse bu-temp) *backups*))
      (next-node parse-flag trace-flag))))    ;FOLLOW ARC TO NEXT NODE


;;NEXT-NODE sets the current state to the last backup state entered on
;;*BACKUPS*.  If this state is nil, it returns 'FAIL.  If this state is
;;a list whose car is 'SUCCESS, it prints out the final list of
;;registers and then continues to search for another parse or returns
;;success, depending upon the value of PARSE-FLAG.  Otherwise, NEXT-NODE
;;calls parse with the new current state. 
  
(DEFUN NEXT-NODE (parse-flag trace-flag)
  (let ((current-state (pop-backups)))
    (cond ((null current-state)
	   (princ "NO (FURTHER) PARSES"))                 ;PARSE FAILS
	  ((equal (car current-state) 'success)           ;PARSE SUCCEEDS
	   (p-lines 1)
	   (p-spaces 2)
	   (print-reg-list (cadr current-state) 2)
	   (p-lines 2)
	   (cond ((equal parse-flag 'A)
		  (next-node parse-flag trace-flag))
		 (t
		  (princ-r "Would you like another parse for this sentence? " 2)
		  (princ-r "(Y or N)" 2)
		  (princ "  ==> ")
		  (cond ((equal (read) 'y)
			 (next-node parse-flag trace-flag))
			(t
			 'success)))))
	  (t
	   (parse current-state parse-flag trace-flag)))))
           ;ARC IS FOLLOWED TO NEXT NODE,
           ;PARSE IS CALLED ON NEW STATE.


;;SET-UP sets up the screen for displaying parse information during the
;;parse.  Its only argument is the trace preference of the user.

(DEFUN SET-UP (trace-flag)
  (p-lines 5)
  (princ-i "SENTENCE:  0" 2)                ;PRINT SENTENCE WITH INDICES                
  (do ((a 0 (1+ a)))                      
      ((equal (aref *sentence* a) nil) t)
    (princ-i (aref *sentence* a) 2)
    (princ-i (1+ a) 2)) 
  (p-lines 1)

  (cond ((equal trace-flag 0)
	 't)
	((equal trace-flag 1)
	 (princ-i "NODE" 2)
	 (princ-i "POINTER" 8)
	 (princ-i "RETURNS" 5)
	 (princ-r "REGISTERS" 8)
	 (p-lines 1))
	((equal trace-flag 2)
	 (princ-i "STATE" 9)
	 (princ-i "REGISTERS" 25)
	 (princ-r "ARCS FOLLOWED" 30)
	 (p-lines 1))))

	   
;;PRINT-DATA displays parse information on the screen according to the
;;trace mode chosen by the user.  It has three arguments: (1) the
;;current state, (2) the trace flag, and (3) a list of the node and arc
;;numbers of the arcs that were followed (non-null only if trace flag
;;equals 2).

(DEFUN PRINT-DATA (current-state trace-flag arcs-followed)
  (let ((node-name (nde current-state))
	(pointer (ptr current-state))
	(returns (ret current-state))
	(registers (reg current-state)))
    
    (cond ((equal trace-flag 0)
	   't)
	  ((equal trace-flag 1)
	   (terpri)
	   (princ-i node-name 2)
	   (princ-i pointer (- 12 (length (prin1-to-string node-name))))
	   (princ-i (firsts returns) (- 12 (length (prin1-to-string pointer))))
	   (p-spaces (- 15 (length (prin1-to-string (firsts returns)))))
	   (print-reg-list registers 41)
	   (terpri))
	  ((equal trace-flag 2) 
	   (terpri)
	   (princ-i "CURRENT: " 2)
	   (print-state current-state) ;(NODENAME,POINTER,RETURN-NODES) REGISTERS
	   (p-spaces 75)
	   (cond ((null (caar arcs-followed))  ;ARCS-FOLLOWED 
		  (princ 'none))
		 (t
		  (princ-arc (caar arcs-followed)
			     (cadar arcs-followed))))
	   (cond ((> (length arcs-followed) 1)
		  (princ "(&")
		  (mapcar #'(lambda (arc)
			      (princ " ")
			      (princ-arc (car arc) (cadr arc)))
			  (cdr arcs-followed))
		  (princ-r " for b.u.)")))
	   (mapcar #'(lambda (bu-state)
		       (terpri)
		       (princ-i "BACK-UP: " 2)
		       (print-state bu-state))
		   *backups*)))))
	   

;;PRINT-STATE has one argument: state.  It prints out an abbreviated
;;list of the state, and then prints out its current registers.  It is
;;called by PRINT-DATA only when the trace-flag equals 2. 

(DEFUN PRINT-STATE (state)
  (princ (short-form state))
  (p-spaces (- 22 (length (prin1-to-string (short-form state)))))
  (print-reg-list (reg state) 33)
  (terpri))

;;SHORT-FORM has one argument: a state.  It returns an abbreviated list
;;of the state in which return states are represented only by their node
;;names, and the registers are omitted altogether.  This function is
;;used in PRINT-STATE (above) for printing out trace information when
;;the trace-flag equals 2.

(DEFUN SHORT-FORM (state)
  (list (nde state)
	(ptr state)
	(firsts (ret state))))






