
/**********************************************************************
 * $Id: Selection.c,v 1.10 92/11/30 11:32:56 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/


/* NOTE: THIS IS NOT A WIDGET!  Rather, this is an interface to a widget.
   It implements policy, and gives a (hopefully) easier-to-use interface
   than just directly making your own form. */

#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Cardinals.h>

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>	
#include <X11/Xaw/List.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Viewport.h>

#include "SelectionP.h"

/*
 * After we have set the list in the listWidget we set the
 * list to a magic value.  So that when a SetValues request is made
 * on the list we will notice it, and reset it.
 */
#define MAGIC_NUMBER		((String *) 3)
#define TOP_OF_VIEWPORT		((int)0)
#define BOTTOM_OF_VIEWPORT	((int)1)
#define MAX(x,y)	((x) > (y) ? (x) : (y))
#define MIN(x,y)	((x) < (y) ? (x) : (y))

static XtResource resources[] = {
  { XtNcallback,     XtCCallback,      XtRCallback,  sizeof(XtCallbackList),
      XtOffset(SelectionWidget, selection.callback),     XtRCallback, NULL},
  {XtNlistLabel,     XtCListLabel,     XtRString,    sizeof(String),
     XtOffset(SelectionWidget, selection.listLabel),     XtRString,   NULL},
  {XtNlist,          XtCList,          XtRPointer,   sizeof(String *),
     XtOffset(SelectionWidget, selection.list),          XtRString,   NULL},
  {XtNnumberStrings, XtCNumberStrings, XtRInt,       sizeof(int),
     XtOffset(SelectionWidget, selection.numberStrings), XtRString,   "0"},
  {XtNlistHeight,    XtCListHeight,    XtRDimension, sizeof(Dimension),
     XtOffset(SelectionWidget, selection.listHeight),    XtRString,   "200"},
  {XtNvalueLabel,    XtCValueLabel,    XtRString,    sizeof(String),
     XtOffset(SelectionWidget, selection.valueLabel),    XtRString,   NULL},
  {XtNvalue,         XtCValue,         XtRString,    sizeof(String),
     XtOffset(SelectionWidget, selection.value),         XtRString,   NULL},
};

static void	Initialize() ;
static void	ConstraintInitialize() ;
static Boolean	SetValues();
static void	ChangeManaged() ;
static void	CreateSelectionValueWidget();
static void	selectionCB() ;

typedef enum SelectType {
  SelectFirst, SelectLast, SelectNext, SelectPrevious, SelectCurrent 
} SelectType ;

static void	SelectItem	 ARGS((SelectionWidget, SelectType)) ;
static void 	SelectItemAction ARGS((Widget, 
				       XEvent *, String *, Cardinal *));
static void 	PassKeyAction	 ARGS((Widget, 
				       XEvent *, String *, Cardinal *));

static XtActionsRec actions[] =
{
  /* {name, procedure}, */
    {"SelectItem",	SelectItemAction},
    {"PassKey",		PassKeyAction},
};

static char translations[] =
  "Shift<Key>Up:	SelectItem(first)	\n\
   None<Key>Down:	SelectItem(next)	\n\
   None<Key>Up:		SelectItem(previous)	\n\
   Shift<Key>Down:	SelectItem(last)	\n\
   <Key>:		PassKey()		\n\
";


SelectionClassRec selectionClassRec = {
  {				/* core_class fields */
    /* superclass         */    (WidgetClass) &formClassRec,
    /* class_name         */    "Selection",
    /* widget_size        */    sizeof(SelectionRec),
    /* class_initialize   */    NULL,
    /* class_part init    */    NULL,
    /* class_inited       */    FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    XtInheritRealize,
    /* actions            */    actions,
    /* num_actions        */    XtNumber(actions),
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE,
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    NULL,
    /* resize             */    XtInheritResize,
    /* expose             */    XtInheritExpose,
    /* set_values         */    SetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    NULL,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    translations,
    /* query_geometry     */	XtInheritQueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
    },
  {				/* composite_class fields */
    /* geometry_manager   */   XtInheritGeometryManager,
    /* change_managed     */   ChangeManaged,
    /* insert_child       */   XtInheritInsertChild,
    /* delete_child       */   XtInheritDeleteChild,
    /* extension          */   NULL
    },
  {				/* constraint_class fields */
    /* subresourses       */   NULL,
    /* subresource_count  */   0,
    /* constraint_size    */   sizeof(SelectionConstraintsRec),
    /* initialize         */   ConstraintInitialize,
    /* destroy            */   NULL,
    /* set_values         */   NULL,
    /* extension          */   NULL
    },
  {				/* form_class fields */
    /* layout             */   XtInheritLayout
    },
  {				/* selection_class fields */
    /* extension          */   NULL
    }
};

WidgetClass selectionWidgetClass = (WidgetClass)&selectionClassRec ;

/*********************************************************************
 *	Name:		Initialize
 *	Description:	initialize procedure for the Selection widget
 *	Parameters:
 *	  Widget	request - the requested format
 *	  Widget	new	 - the accepted format
 *	  ArgList	args	 - the args passed at creation
 *	  Cardinal	*numArgs - the number of args
 *	Return Value:
 *	  static void	Initialize - NONE
 *********************************************************************/
static void	Initialize(request, new, args, numArgs)
  Widget	request ;
  Widget	new;
  ArgList	args ;
  Cardinal	*numArgs ;
{
  SelectionWidget sw = (SelectionWidget)new ;

  sw->selection.listLabelW
    = XtVaCreateManagedWidget("listLabel", labelWidgetClass, new,
			      XtNlabel,		sw->selection.listLabel,
			      XtNtop,		XtChainTop,
			      XtNbottom,	XtChainTop,
			      XtNleft,		XtChainLeft,
			      XtNright,		XtChainLeft,
			      XtNvertDistance, 	0,
			      XtNhorizDistance,	0,
			      NULL) ;

  sw->selection.viewportW 
    = XtVaCreateManagedWidget("viewport", viewportWidgetClass, new,
			      XtNheight,   sw->selection.listHeight,
			      XtNtop,      XtChainTop,
			      XtNbottom,   XtChainBottom,
			      XtNleft,     XtChainLeft,
			      XtNright,    XtChainRight,
			      XtNfromVert, sw->selection.listLabelW,
			      NULL) ;

  sw->selection.listW
    = XtVaCreateManagedWidget("list", listWidgetClass, sw->selection.viewportW,
			      XtNlist,		sw->selection.list,
			      XtNnumberStrings,	sw->selection.numberStrings,
			      NULL) ;

  if (sw->selection.numberStrings == 0)
    XtVaGetValues(sw->selection.listW,
		  XtNnumberStrings, &sw->selection.numberStrings, NULL) ;

  XtSelectionHighlight(sw, 0) ;
      
  sw->selection.list = MAGIC_NUMBER ;

  if (sw->selection.value) {
    sw->selection.value = XtNewString(sw->selection.value) ;
    CreateSelectionValueWidget(sw);
  } else {
    sw->selection.valueLabelW = NULL ;
    sw->selection.valueW      = NULL ;
  }

  XtAddCallback(sw->selection.listW, XtNcallback, selectionCB, (XtPointer)sw) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		ConstraintInitialize
 *	Description:	ensures that all the command widgets are at
 *			the bottom and below all of the std widgets
 *	Parameters:
 *	  Widget	request - the requested widget
 *	  Widget	new 	- the accepted one
 *	Return Value:
 *	  static void	ConstraintInitialize - NONE
 *********************************************************************/
static void	ConstraintInitialize(request, new)
  Widget	request ;
  Widget	new ;
{
  SelectionWidget 	sw = (SelectionWidget)new->core.parent ;
  SelectionConstraints 	constraint = 
    (SelectionConstraints)((Widget)new)->core.constraints ;

  /*  if not a button, then just use the defaults */
  if (!XtIsSubclass(new, commandWidgetClass))
    return ;

  constraint->form.top       = XtChainBottom ;
  constraint->form.bottom    = XtChainBottom ;
  constraint->form.left      = XtChainLeft ;
  constraint->form.right     = XtChainLeft ;
  if (sw->selection.valueW == NULL) 
    constraint->form.vert_base = sw->selection.viewportW;
  else
    constraint->form.vert_base = sw->selection.valueW;

  if (sw->composite.num_children > 1) {
    WidgetList	children = sw->composite.children;
    Widget	*childP;

    for (childP = children + sw->composite.num_children - 1 ;
	 childP >= children ; childP-- ) {
      if (   *childP == sw->selection.listLabelW
	  || *childP == sw->selection.viewportW
	  || *childP == sw->selection.valueLabelW 
	  || *childP == sw->selection.valueW)
	break;
      if (XtIsManaged(*childP) && XtIsSubclass(*childP, commandWidgetClass)) {
	constraint->form.horiz_base = *childP;
	break;
      }
    }
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		ChangeManaged
 *	Description:	makes sure that the viewport and value widget
 *			are the same width and at least as wide as all
 *			the command widgets
 *	Parameters:
 *	  Widget	genericWidget - the widget
 *	Return Value:
 *	  static void	ChangeManaged - NONE
 *********************************************************************/
#define MIN_WIDTH	100
#define SCROLLBAR_WIDTH	14
#define SUPER_CLASS_CHANGE_MANAGED(classRec)	\
  (((CompositeClassRec *)(classRec).core_class.superclass)->composite_class.change_managed)
/********************************************************************/
static void	ChangeManaged(genericWidget)
  Widget	genericWidget ;
{
  SelectionWidget 	widget = (SelectionWidget)genericWidget ;

  Widget	viewport = widget->selection.viewportW ;
  Widget	list     = widget->selection.listW ;
  Dimension	listWidth, buttonsWidth, minWidth ;

  if (viewport == NULL || !XtIsManaged(viewport)) {
    (*SUPER_CLASS_CHANGE_MANAGED(selectionClassRec)) ((Widget)widget);
    return ;
  }

  buttonsWidth = 0 ;
  if (widget->composite.num_children > 1) {
    WidgetList	children = widget->composite.children;
    Widget	*childP;

    for (childP = children + widget->composite.num_children - 1 ;
	 childP >= children ; childP-- ) {
      if (XtIsManaged(*childP) && XtIsSubclass(*childP, commandWidgetClass)) {
	buttonsWidth
	  += (*childP)->core.width
	    + ((FormConstraints)(*childP)->core.constraints)->form.dx ;
      }
    }
  }

  if (list && XtIsManaged(list)) {
    listWidth = list->core.width ;
    if (list->core.height > viewport->core.height)
      listWidth += SCROLLBAR_WIDTH ;
  } else {
    listWidth = 0 ;
  }

  minWidth = listWidth > buttonsWidth ? listWidth : buttonsWidth ;
  if (minWidth < MIN_WIDTH)
    minWidth = MIN_WIDTH ;

  XtVaSetValues(viewport,
		XtNwidth, minWidth, NULL) ;
  if (widget->selection.valueW != NULL)
    XtVaSetValues(widget->selection.valueW, XtNwidth, minWidth, NULL) ;

  (*SUPER_CLASS_CHANGE_MANAGED(selectionClassRec)) ((Widget)widget);
}
/********************************************************************/


/*********************************************************************
 *	Name:		SetValues
 *	Description:	set values proc
 *	Parameters:
 *	  Widget	current - 
 *	  Widget	request - 
 *	  Widget	new - 
 *	  ArgList 	in_args - 
 *	  Cardinal 	*numArgs - 
 *	Return Value:
 *	  static Boolean	SetValues - FALSE (never need redisplay)
 *********************************************************************/
static Boolean	SetValues(current, request, new, in_args, numArgs)
  Widget	current ;
  Widget	request ;
  Widget	new;
  ArgList 	in_args;
  Cardinal 	*numArgs;
{
  SelectionWidget w = (SelectionWidget)new;
  SelectionWidget old = (SelectionWidget)current;

  if (w->selection.listLabel != old->selection.listLabel)
    XtVaSetValues(w->selection.listLabelW,
		  XtNlabel, w->selection.listLabel, NULL) ;

  if (w->selection.listHeight != old->selection.listHeight)
    XtVaSetValues(w->selection.viewportW,
		  XtNheight, w->selection.listHeight, NULL) ;

  if (w->selection.list != old->selection.list
      || w->selection.numberStrings != old->selection.numberStrings ) {
    Widget	 listWidget = w->selection.listW ;
    String	*list	    = w->selection.list ;
      
    XtUnmanageChild(listWidget) ;

    if (list != MAGIC_NUMBER)
      XawListChange(listWidget, list, 0, 0, True) ;
    else
      XtVaSetValues(listWidget,
		    XtNnumberStrings, w->selection.numberStrings, NULL) ;

    XtVaGetValues(listWidget,
		  XtNnumberStrings, &w->selection.numberStrings, NULL) ;

    XtManageChild(listWidget) ;
    if (XtIsRealized((Widget)w))
      XtSelectionHighlight(w, 0) ;

    if (w->selection.valueW)
      XtSelectionSetValueString((Widget)w, "") ;

    w->selection.list = MAGIC_NUMBER ;
  }

  if (w->selection.valueLabelW
      && w->selection.valueLabel != old->selection.valueLabel)
      XtVaSetValues(w->selection.valueLabelW,
		    XtNlabel, w->selection.valueLabel, NULL) ;

  if (w->selection.value != old->selection.value) {
    if (w->selection.value == NULL) {
      /* only get here if it wasn't NULL before. */
      XtDestroyWidget(old->selection.valueLabelW);
      XtDestroyWidget(old->selection.valueW);
      w->selection.valueLabelW = NULL ;
      w->selection.valueW      = NULL ;
    } else if (old->selection.value == NULL) { 
      /* create a new value widget. */
      w->core.width  = old->core.width;
      w->core.height = old->core.height;
      w->selection.value = XtNewString(w->selection.value) ;
      CreateSelectionValueWidget(w);
    } else {			
      /* Widget ok, just change string. */
      XtSelectionSetValueString((Widget)w, w->selection.value) ;
    }
  }
  return False;
}
/********************************************************************/


/*********************************************************************
 *	Name:		CreateSelectionValueWidget
 *	Description:	Creates the selection widget's value widget.
 *			Must be called only when w->selection.value
 *			is non-nil.
 *	Parameters:
 *	  Widget	w - the selection widget.
 *	Return Value:
 *	  static void	CreateSelectionValueWidget - NONE
 *********************************************************************/
static void	CreateSelectionValueWidget(sw)
  SelectionWidget	sw ;
{
  sw->selection.valueLabelW 
    = XtVaCreateManagedWidget("valueLabel", labelWidgetClass, (Widget)sw,
			      XtNlabel,	 	sw->selection.valueLabel,
			      XtNtop,		XtChainBottom,
			      XtNbottom,	XtChainBottom,
			      XtNleft,	 	XtChainLeft,
			      XtNright,	 	XtChainLeft,
			      XtNfromVert,	sw->selection.viewportW,
			      NULL) ;

  sw->selection.valueW 
    = XtVaCreateWidget("value", asciiTextWidgetClass, (Widget)sw,
		       XtNtype,			XawAsciiString,
		       XtNeditType,		XawtextEdit,
		       XtNstring,		sw->selection.value,
		       XtNresizable,		True,
		       XtNleft,			XtChainLeft,
		       XtNright,		XtChainRight,
		       XtNtop,      		XtChainBottom,
		       XtNbottom,   		XtChainBottom,
		       XtNfromVert,		sw->selection.valueLabelW,
		       NULL) ;

  /* if the value widget is being added after buttons,
   * then the buttons need new layout constraints.
   */
  if (sw->composite.num_children > 1) {
    WidgetList	children = sw->composite.children;
    Widget	*childP;

    for (childP = children + sw->composite.num_children - 1;
	 childP >= children; childP-- ) {
      if (*childP == sw->selection.valueLabelW 
	  || *childP == sw->selection.listLabelW
	  || *childP == sw->selection.valueW
	  || *childP == sw->selection.viewportW)
	continue;
      if (XtIsManaged(*childP) && XtIsSubclass(*childP, commandWidgetClass)) {
	((SelectionConstraints)(*childP)->core.constraints)->form.vert_base 
	    = sw->selection.valueW;
      }
    }
  }
  XtManageChild(sw->selection.valueW);
}
/********************************************************************/


/*********************************************************************
 *	Name:		selectionCB
 *	Description:	
 *	Parameters:
 *	  Widget	w - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	selectionCB - 
 *********************************************************************/
static void	selectionCB(w, clientData, callData)
  Widget	w ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  XtCallCallbacks((Widget)clientData, XtNcallback, callData) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		actions
 *	Description:	Actions for moving around the list
 *	Parameters:
 *	  Widget	w      - the widget. MUST BE A SELECTION WIDGET
 *	  XEvent	*event - the event UNUSED
 *	  String	*params    - UNUSED
 *	  Cardinal	*numParams - UNUSED
 *	Return Value:
 *	  static void 	actions - NONE
 *********************************************************************/
static Boolean		itemVisible(sw)
  SelectionWidget	sw ;
{
  XawListReturnStruct	*rs ;
  XFontStruct		*font ;
  Position		itemTop, itemBottom, y ;
  Dimension		space, height, itemHeight ;

  rs = XtSelectionGetSelected(sw) ;
  if (rs == NULL || rs->list_index == XAW_LIST_NONE)
    return TRUE ;

  XtVaGetValues(sw->selection.listW,	
		XtNfont,	&font, 
		XtNrowSpacing,	&space, 
		XtNy,		&y, 
		NULL) ;
  XtVaGetValues(sw->selection.viewportW,	
		XtNheight,	&height, 
		NULL) ;

  itemHeight = font->max_bounds.descent + font->max_bounds.ascent + space ;

  itemTop    = y + rs->list_index*itemHeight  ;
  itemBottom = itemTop + itemHeight ;
  
  if (itemBottom > height || itemTop < 0)
    return FALSE ;
  else
    return TRUE ;
}
/********************************************************************/
static void		makeItemVisible(sw, where)
  SelectionWidget	sw ;
  int			where ;
{
  XawListReturnStruct	*rs ;
  XFontStruct		*font ;
  Position		itemTop, itemBottom, y ;
  Dimension		space, listHeight, viewportHeight, itemHeight ;

  if (!XtIsRealized((Widget)sw->selection.viewportW) )
    return ;

  rs = XtSelectionGetSelected(sw) ;
  if (rs == NULL || rs->list_index == XAW_LIST_NONE)
    return ;

  XtVaGetValues(sw->selection.listW,	
		XtNfont,	&font, 
		XtNrowSpacing,	&space, 
		XtNy,		&y, 
		XtNheight,	&listHeight,
		NULL) ;

  XtVaGetValues(sw->selection.viewportW,	
		XtNheight,	&viewportHeight, 
		NULL) ;

  itemHeight = font->max_bounds.descent + font->max_bounds.ascent + space ;

  itemTop    = rs->list_index*itemHeight  ;
  itemBottom = itemTop + itemHeight ;

#if XtSpecificationRelease > 4
  if (where == TOP_OF_VIEWPORT)
    XawViewportSetCoordinates(sw->selection.viewportW, 
			      0, MIN(itemTop,
				     MAX(0, listHeight - viewportHeight))) ;
  else
    XawViewportSetCoordinates(sw->selection.viewportW, 
			      0, MAX(0, itemBottom - viewportHeight)) ;
#endif
}
/********************************************************************/
static void 	PassKeyAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  SelectionWidget	sw = (SelectionWidget)w ;
  Widget		value ;
  XawListReturnStruct	*rs ;

  if (!XtIsSubclass(w, selectionWidgetClass))
    return ;

  if (event->type != KeyPress && event->type != KeyRelease)
    return ;

  value = sw->selection.valueW ;
  if (value == NULL || !XtIsManaged(value))
    return ;
  
  event->xkey.window = XtWindow(value) ;
  XSendEvent(XtDisplay(value), event->xkey.window, 
	     True, KeyPressMask|KeyReleaseMask, event) ;
}
/********************************************************************/
static void		selectItem(sw, which)
  SelectionWidget	sw ;
  SelectType		which ;
{
  int			index ;
  XawListReturnStruct	*rs = XtSelectionGetSelected((Widget)sw) ;

  switch(which) {
  case SelectFirst:
    index = 0 ;
    break ;

  case SelectLast:
    index = sw->selection.numberStrings - 1 ;
    break ;

  case SelectNext:
    if (rs == NULL || rs->list_index == XAW_LIST_NONE 
	|| rs->list_index == sw->selection.numberStrings - 1)
      index = 0 ;
    else 
      index = rs->list_index + 1 ;
    break ;

  case SelectPrevious:
    if (rs == NULL || rs->list_index == XAW_LIST_NONE || rs->list_index == 0)
      index = sw->selection.numberStrings - 1 ;
    else 
      index = rs->list_index - 1 ;
    break ;

  case SelectCurrent:
    if (rs == NULL || rs->list_index == XAW_LIST_NONE)
      index = 0 ;
    else 
      index = rs->list_index ;
    break ;

  default:
    index = 0 ;
    break ;
  }

  XtSelectionHighlight((Widget)sw, index) ;

#if XtSpecificationRelease >= 5
  if (!itemVisible(sw))
    makeItemVisible(sw, BOTTOM_OF_VIEWPORT) ;
#endif

  rs = XtSelectionGetSelected((Widget)sw) ;
  if (rs->list_index != XAW_LIST_NONE)
    XtCallCallbacks((Widget)sw, XtNcallback, rs) ;
}
/********************************************************************/
static void 	SelectItemAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  SelectionWidget	sw = (SelectionWidget)w ;

  if (*numParams > 1 || !XtIsSubclass(w, selectionWidgetClass))
    return ;

  if (*numParams == 0 || strcmp(params[0], "first") == 0) {
    selectItem(sw, SelectFirst) ;
  } else if (strcmp(params[0], "last") == 0) {
    selectItem(sw, SelectLast) ;
  } else if (strcmp(params[0], "next") == 0) {
    selectItem(sw, SelectNext) ;
  } else if (strcmp(params[0], "previous") == 0) {
    selectItem(sw, SelectPrevious) ;
  } else if (strcmp(params[0], "current") == 0) {
    selectItem(sw, SelectCurrent) ;
  } else {
    return ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtSelectionAddButton
 *	Description:	adds a button to the selection widget
 *	Parameters:
 *	  Widget	 selection - 
 *	  String	 name - 
 *	  XtCallbackProc function - 
 *	  XtPointer	 param - 
 *	Return Value:
 *	  Widget		 XtSelectionAddButton - 
 *********************************************************************/
Widget		 XtSelectionAddButton(widget, name, function, param)
  Widget	 widget ;
  String	 name ;
  XtCallbackProc function ;
  XtPointer	 param ;
{
  Widget button;

  if (!XtIsSubclass(widget, selectionWidgetClass))
    return NULL ;

  /* Correct Constraints are all set in ConstraintInitialize(). */
  button = XtVaCreateManagedWidget(name, commandWidgetClass, widget, NULL) ;

  if (function != NULL)		/* don't add NULL callback func. */
    XtAddCallback(button, XtNcallback, function, param);

  return button ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtSelectionGetValueString
 *	Description:	
 *	Parameters:
 *	  Widget w - 
 *	Return Value:
 *	  char	 *XtSelectionGetValueString - 
 *********************************************************************/
char	 *XtSelectionGetValueString(w)
  Widget w ;
{
  String		newValue ;
  SelectionWidget	sw = (SelectionWidget)w ;

  if (!XtIsSubclass(w, selectionWidgetClass))
    return NULL ;

  XtVaGetValues(sw->selection.valueW, XtNstring, &newValue, NULL) ;

  if (sw->selection.value)
    XtFree(sw->selection.value) ;

  sw->selection.value = newValue ? XtNewString(newValue) : newValue ;

  return sw->selection.value ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtSelectionSetValueString
 *	Description:	
 *	Parameters:
 *	  Widget genericWidget - 
 *	  String newValue - 
 *	Return Value:
 *	  void     XtSelectionSetValueString - 
 *********************************************************************/
void     XtSelectionSetValueString(genericWidget, newValue)
  Widget genericWidget ;
  String newValue ;
{
  SelectionWidget widget = (SelectionWidget)genericWidget ;

  if (!XtIsSubclass(genericWidget, selectionWidgetClass))
    return ;

  if (widget->selection.value)
    XtFree(widget->selection.value) ;

  widget->selection.value = newValue ? XtNewString(newValue) : newValue ;

  if (widget->selection.valueW)
    XtVaSetValues(widget->selection.valueW,
		  XtNstring, widget->selection.value, NULL) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtSelectionGetSelected
 *	Description:	
 *	Parameters:
 *	  Widget		genericWidget - 
 *	Return Value:
 *	  XawListReturnStruct	*XtSelectionGetSelected - 
 *********************************************************************/
XawListReturnStruct	*XtSelectionGetSelected(genericWidget)
  Widget		genericWidget ;
{
  SelectionWidget	widget = (SelectionWidget)genericWidget ;

  if (XtIsSubclass(genericWidget, selectionWidgetClass)
      && widget->selection.listW)
    return XawListShowCurrent(widget->selection.listW) ;
  else
    return NULL ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtSelectionHighlight
 *	Description:	highlights a given item in the selection list
 *	Parameters:
 *	  Widget	genericWidget - the selection widget
 *	  int		item - the item index
 *	Return Value:
 *	  void		XtSelectionHighlight - NONE
 *********************************************************************/
void		XtSelectionHighlight(genericWidget, item)
  Widget	genericWidget ;
  int		item ;
{
  SelectionWidget	widget = (SelectionWidget)genericWidget ;

  if (!XtIsSubclass(genericWidget, selectionWidgetClass))
    return ;

  if (widget->selection.listW && item < widget->selection.numberStrings)
    XawListHighlight(widget->selection.listW, item) ;

#if XtSpecificationRelease >= 5
  if (!itemVisible(widget))
    makeItemVisible(widget, TOP_OF_VIEWPORT) ;
#endif
}
/********************************************************************/
