
/**********************************************************************
 * $Id: Plot.c,v 1.2 92/11/30 11:32:49 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <varargs.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Cardinals.h>

#include <X11/Xaw/Label.h>
#include <X11/Xaw/Viewport.h>

#include "PlotP.h"

/* just a number so that I can tell if the user tried to change 
   the resources */
#define MAGIC_NUMBER		(float)(1.2345e-7)
#define MAGIC_NUMBER_STR	"1.2345e-7"

static XtResource resources[] = {
  {XtNlabel, XtCLabel, XtRString, sizeof(String),
     XtOffset(PlotWidget, plot.label), XtRString, NULL},
  {XtNformat, XtCFormat, XtRString, sizeof(String),
     XtOffset(PlotWidget, plot.format), XtRString, "% 8g"},

  { XtNcanvasXOffset, XtCCanvasOffset, XtRFloatPointer, sizeof(float *),
      XtOffset(PlotWidget, plot.canvasXOffset), XtRString, MAGIC_NUMBER_STR},
  { XtNcanvasYOffset, XtCCanvasOffset, XtRFloatPointer, sizeof(float *),
      XtOffset(PlotWidget, plot.canvasYOffset), XtRString, MAGIC_NUMBER_STR},
  { XtNcanvasWidth, XtCCanvasDimension, XtRFloatPointer, sizeof(float *),
      XtOffset(PlotWidget, plot.canvasWidth), XtRString,   MAGIC_NUMBER_STR},
  { XtNcanvasHeight, XtCCanvasDimension, XtRFloatPointer, sizeof(float *),
      XtOffset(PlotWidget, plot.canvasHeight), XtRString,  MAGIC_NUMBER_STR},
};

static void	ClassInitialize() ;
static void	Initialize() ;
static void	ChangeManaged() ;
static Boolean	SetValues();
static void	resetScaleLabels() ;
static int	labelPrintf() ;


PlotClassRec plotClassRec = {
  {				/* core_class fields */
    /* superclass         */    (WidgetClass) &formClassRec,
    /* class_name         */    "Plot",
    /* widget_size        */    sizeof(PlotRec),
    /* class_initialize   */    ClassInitialize,
    /* class_part init    */    NULL,
    /* class_inited       */    FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    XtInheritRealize,
    /* actions            */    NULL,
    /* num_actions        */    0,
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE,
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    NULL,
    /* resize             */    XtInheritResize,
    /* expose             */    XtInheritExpose,
    /* set_values         */    SetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    NULL,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    NULL,
    /* query_geometry     */	XtInheritQueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
    },
  {				/* composite_class fields */
    /* geometry_manager   */   XtInheritGeometryManager,
    /* change_managed     */   ChangeManaged,
    /* insert_child       */   XtInheritInsertChild,
    /* delete_child       */   XtInheritDeleteChild,
    /* extension          */   NULL
    },
  {				/* constraint_class fields */
    /* subresourses       */   NULL,
    /* subresource_count  */   0,
    /* constraint_size    */   sizeof(PlotConstraintsRec),
    /* initialize         */   NULL,
    /* destroy            */   NULL,
    /* set_values         */   NULL,
    /* extension          */   NULL
    },
  {				/* form_class fields */
    /* layout             */   XtInheritLayout
    },
  {				/* plot_class fields */
    /* extension	  */   NULL
    }
};

WidgetClass plotWidgetClass = (WidgetClass)&plotClassRec ;
  
/*********************************************************************
 *	Name:		CvtStringToFloatPtr
 *	Description:	converts a string to a float value and sets
 *			a pointer to the value
 *	Parameters:
 *	  XrmValue	*args - 
 *	  Cardinal	*numArgs - 
 *	  XrmValuePtr	from, to - 
 *	Return Value:
 *	  static Boolean	CvtStringToFloatPtr - TRUE on success,
 *				FALSE on falure.
 *********************************************************************/
static Boolean	CvtStringToFloatPtr(display, args, numArgs, from, to, data)
  Display	*display ;
  XrmValuePtr	args ;
  Cardinal	*numArgs ;
  XrmValuePtr	from, to ;
  XtPointer	data ;
{
  Boolean	badConvert = FALSE ;
  String	string, ptr ;
  float		floatValue ;
  static float	*floatPtr ;

  if (*numArgs != 0) {
    badConvert = TRUE ;
    return !badConvert ;
  }
  
  string = from->addr ;

  floatPtr   = NULL ;
  floatValue = strtod(string, &ptr) ;
  if (ptr == string)
    badConvert = TRUE ;

  if (badConvert) {
    XtStringConversionWarning(string, XtRFloatPointer) ;
  } else {
    floatPtr  = XtNew(float) ;
    *floatPtr = floatValue ;
    if (NULL == to->addr)
      to->addr = (caddr_t) &floatPtr ;
    else if (to->size < sizeof(float *))
      badConvert = TRUE ;
    else
      *(float **)to->addr = floatPtr;
    
    to->size = sizeof(float *) ;
  }
  return !badConvert ;
}
/********************************************************************/

/********************************************************************/
static void	ClassInitialize () {
  XawInitializeWidgetSet() ;
  XtSetTypeConverter(XtRString, XtRFloatPointer, CvtStringToFloatPtr, 
		     NULL, 0, XtCacheAll, NULL) ;
}
/********************************************************************/

/********************************************************************/
static float	*dupFloatPtr(ptr, newPtr)
  const	float	*ptr ;
  float		*newPtr ;
{
  if (newPtr == NULL)
    newPtr = (float *)XtMalloc(sizeof(float)) ;

  if (ptr)
    *newPtr = *ptr ;
  else
    *newPtr = 0.0 ;
  return newPtr ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		Initialize
 *	Description:	
 *	Parameters:
 *	  Widget	requestWidget - 
 *	  Widget	newWidget - 
 *	  ArgList	args - 
 *	  Cardinal	*numArgs - 
 *	Return Value:
 *	  static void	Initialize - 
 *********************************************************************/
/* ARGSUSED */
static void	Initialize(requestWidget, newWidget, in_args, numArgs)
  Widget	requestWidget ;
  Widget	newWidget ;
  ArgList	in_args ;
  Cardinal	*numArgs ;
{
  PlotWidget 	new     = (PlotWidget)newWidget ;
  PlotWidget 	request = (PlotWidget)requestWidget ;

  XtCheckSubclass(requestWidget, plotWidgetClass, 
		  "Initializer for plotWidgetClass") ;

  /* make private copies of everything */
  if (NULL != request->plot.label)
    new->plot.label = (char *)strdup(request->plot.label) ;

  new->plot.canvasXOffset = dupFloatPtr(request->plot.canvasXOffset, NULL) ;
  new->plot.canvasYOffset = dupFloatPtr(request->plot.canvasYOffset, NULL) ;
  new->plot.canvasWidth   = dupFloatPtr(request->plot.canvasWidth,   NULL) ;
  new->plot.canvasHeight  = dupFloatPtr(request->plot.canvasHeight,  NULL) ;

  /* now start making the widgets */
  new->plot.yScaleWidget[1] 
    = XtVaCreateManagedWidget("yMax", labelWidgetClass, newWidget,
			      XtNtop,		XtChainTop,
			      XtNbottom,	XtChainTop,
			      XtNleft,		XtChainLeft,
			      XtNright,		XtChainLeft,
			      NULL) ;
  
  new->plot.yScaleWidget[0]
    = XtVaCreateManagedWidget("yMin", labelWidgetClass, newWidget,
			      XtNtop,		XtChainBottom,
			      XtNbottom,	XtChainBottom,
			      XtNleft,		XtChainLeft,
			      XtNright,		XtChainLeft,
			      NULL) ;
  new->plot.canvasWidget
    = XtVaCreateManagedWidget("canvas", canvasWidgetClass, newWidget, 
			    XtNtop,	 	XtChainTop,
			    XtNbottom,	 	XtChainBottom,
			    XtNleft,	 	XtChainLeft,
			    XtNright,	 	XtChainRight,
			    XtNfromHoriz,	new->plot.yScaleWidget[1],
			    NULL) ;

  if (*new->plot.canvasXOffset != MAGIC_NUMBER)
    XtVaSetValues(new->plot.canvasWidget,
		  XtNcanvasXOffset, new->plot.canvasXOffset, NULL) ;
  if (*new->plot.canvasYOffset != MAGIC_NUMBER)
    XtVaSetValues(new->plot.canvasWidget,
		  XtNcanvasYOffset, new->plot.canvasYOffset, NULL) ;
  if (*new->plot.canvasWidth != MAGIC_NUMBER)
    XtVaSetValues(new->plot.canvasWidget,
		  XtNcanvasWidth, new->plot.canvasWidth, NULL) ;
  if (*new->plot.canvasHeight != MAGIC_NUMBER)
    XtVaSetValues(new->plot.canvasWidget,
		  XtNcanvasHeight, new->plot.canvasHeight, NULL) ;

  new->plot.xScaleWidget[0]
    = XtVaCreateManagedWidget("xMin", labelWidgetClass, newWidget,
			      XtNtop,		XtChainBottom,
			      XtNbottom,	XtChainBottom,
			      XtNleft,		XtChainLeft,
			      XtNright,		XtChainLeft,
			      XtNfromVert,	new->plot.canvasWidget,
			      XtNfromHoriz, 	new->plot.yScaleWidget[1],
			      NULL) ;
  new->plot.labelWidget
    = XtVaCreateWidget("label", labelWidgetClass, newWidget,
		       XtNtop,		XtChainBottom,
		       XtNbottom,	XtChainBottom,
		       XtNfromVert,	new->plot.canvasWidget,
		       NULL) ;
  if (NULL != new->plot.label) {
    XtVaSetValues(new->plot.labelWidget,
		  XtNlabel, new->plot.label,
		  NULL) ;
    XtManageChild(new->plot.labelWidget) ;
  }

  new->plot.xScaleWidget[1]
    = XtVaCreateManagedWidget("xMax", labelWidgetClass, newWidget,
			      XtNtop,		XtChainBottom,
			      XtNbottom,	XtChainBottom,
			      XtNleft,		XtChainRight,
			      XtNright,		XtChainRight,
			      XtNfromVert,	new->plot.canvasWidget,
			      NULL) ;

  resetScaleLabels(new) ;

  if (NULL != new->plot.label)
    labelPrintf(new->plot.labelWidget, "%s", new->plot.label) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		SetValues
 *	Description:	
 *	Parameters:
 *	  Widget	currentWidget, requestWidget, newWidget - 
 *	  ArgList 	in_args - 
 *	  Cardinal 	*numArgs - 
 *	Return Value:
 *	  static Boolean	SetValues - 
 *********************************************************************/
static Boolean	SetValues(currentWidget, requestWidget, newWidget,
			  in_args, numArgs)
  Widget	currentWidget, requestWidget, newWidget;
  ArgList 	in_args;
  Cardinal 	*numArgs;
{
  PlotWidget	old     = (PlotWidget)currentWidget ;
  PlotWidget	request = (PlotWidget)requestWidget ;
  PlotWidget	new     = (PlotWidget)newWidget ;
  Arg		args[10] ;
  Cardinal	n ;

  XtCheckSubclass(requestWidget, plotWidgetClass, 
		  "Set values routine for plotWidgetClass") ;

  n = 0 ;
  if (request->plot.canvasXOffset != old->plot.canvasXOffset) {
    new->plot.canvasXOffset = old->plot.canvasXOffset ;
    dupFloatPtr(request->plot.canvasXOffset, new->plot.canvasXOffset) ;
    XtSetArg(args[n], XtNcanvasXOffset, new->plot.canvasXOffset), ++n ;
  }

  if (request->plot.canvasYOffset != old->plot.canvasYOffset) {
    new->plot.canvasYOffset = old->plot.canvasYOffset ;
    dupFloatPtr(request->plot.canvasYOffset, new->plot.canvasYOffset) ;
    XtSetArg(args[n], XtNcanvasYOffset, new->plot.canvasYOffset), ++n ;
  }

  if (request->plot.canvasWidth != old->plot.canvasWidth) {
    new->plot.canvasWidth = old->plot.canvasWidth ;
    dupFloatPtr(request->plot.canvasWidth, new->plot.canvasWidth) ;
    XtSetArg(args[n], XtNcanvasWidth, new->plot.canvasWidth), ++n ;
  }

  if (request->plot.canvasHeight != old->plot.canvasHeight) {
    new->plot.canvasHeight = old->plot.canvasHeight ;
    dupFloatPtr(request->plot.canvasHeight, new->plot.canvasHeight) ;
    XtSetArg(args[n], XtNcanvasHeight, new->plot.canvasHeight), ++n ;
  }

  if (n > 0) {
    XtSetValues(new->plot.canvasWidget, args, n) ;
    resetScaleLabels(new) ;
  }

  if (request->plot.label != old->plot.label) {
    new->plot.label = (char *)strdup(request->plot.label) ;
    XtFree(old->plot.label) ;
    labelPrintf(new->plot.labelWidget, "%s", new->plot.label) ;
  }
  return FALSE ;
}
/********************************************************************/

static void	ChangeManaged(genericWidget)
  Widget	genericWidget ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  Widget	canvas, label, yMin, xMax ;
  Dimension	canvasWidth, canvasHeight ;
  PlotConstraints	constraint ;

  XtCheckSubclass(genericWidget, plotWidgetClass, 
		  "Set values routine for plotWidgetClass") ;

  canvas = widget->plot.canvasWidget ;
  if (NULL != canvas && XtIsManaged(canvas)) {
    canvasWidth  = canvas->core.width ;
    canvasHeight = canvas->core.height ;
  } else {
    canvasWidth  = 0 ;
    canvasHeight = 0 ;
  }
  yMin = widget->plot.yScaleWidget[0] ;
  if (NULL != yMin && XtIsManaged(yMin)) {
    constraint = (PlotConstraints)yMin->core.constraints ;
    constraint->form.dy = canvasHeight - yMin->core.height ;
  }
  xMax = widget->plot.xScaleWidget[1] ;
  if (NULL != xMax && XtIsManaged(xMax)) {
    constraint = (PlotConstraints)xMax->core.constraints ;
    constraint->form.horiz_base = yMin ;
    constraint->form.dx         = canvasWidth - xMax->core.width ;
  }

  label = widget->plot.labelWidget ;
  if (NULL != label && XtIsManaged(label)) {
    constraint = (PlotConstraints)label->core.constraints ;
    constraint->form.horiz_base = yMin ;
    constraint->form.dx         = (canvasWidth - label->core.width)/2 ;
  }

  (*((CompositeClassRec *)plotClassRec.core_class.superclass)->composite_class.change_managed)
    (genericWidget) ;
}

/***********************************************************************
 *	Name:		labelPrintf
 *	Description:	same as printf, but writes to the XtNlabel
 *			resource of a widget. It doesn't update
 *			the field if it the string is unchanged.
 *	Parameters:	va_alist - this uses varargs, so be careful
 *				that the format string is correct.
 *	Return Value:	whatever vsprintf returns.
 ***********************************************************************/
static int	labelPrintf(va_alist)
  va_dcl 
{
  va_list 	argList ;
  Widget	label ;
  Arg		args[32] ;
  Cardinal	n ;
  char		*format, *oldString, newString[BUFSIZ] ;
  int		returnValue ;

  va_start(argList) ;
  label  = va_arg(argList, Widget) ;
  format = va_arg(argList, char *) ;

  XtCheckSubclass(label, labelWidgetClass, "labelPrintf") ;

  n = 0 ;
  XtSetArg(args[n], XtNlabel, &oldString),	n++ ;
  XtGetValues(label, args, n) ;

  returnValue = vsprintf(newString, format, argList) ;

  if (strcmp(newString, oldString)) {
    n = 0 ;
    XtSetArg(args[n], XtNlabel, newString),	n++ ;
    XtSetValues(label, args, n) ;
  }

  va_end(argList) ;
  return returnValue ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		resetScaleLabels
 *	Description:	resets the labels with the proper values from
 *			the plot canvas widget child
 *	Parameters:
 *	  PlotWidget	widget - the plot widget
 *	Return Value:
 *	  static void	resetScaleLabels - NONE
 *********************************************************************/
static void	resetScaleLabels(widget)
  PlotWidget	widget ;
{
  float		*xPtr, *yPtr, *widthPtr, *heightPtr ;

  XtVaGetValues(widget->plot.canvasWidget,
		XtNcanvasXOffset, 	&xPtr,
		XtNcanvasYOffset, 	&yPtr,
		XtNcanvasWidth, 	&widthPtr,
		XtNcanvasHeight, 	&heightPtr,
		NULL) ;

  *widget->plot.canvasXOffset = *xPtr ;
  labelPrintf(widget->plot.xScaleWidget[0], widget->plot.format, *xPtr) ;

  *widget->plot.canvasYOffset = *yPtr ;
  labelPrintf(widget->plot.yScaleWidget[0], widget->plot.format, *yPtr) ;

  *widget->plot.canvasWidth = *widthPtr ;
  labelPrintf(widget->plot.xScaleWidget[1], widget->plot.format, 
	      *xPtr + *widthPtr) ;

  *widget->plot.canvasHeight = *heightPtr ;
  labelPrintf(widget->plot.yScaleWidget[1], widget->plot.format, 
	      *yPtr + *heightPtr);
}
/********************************************************************/


/********************************************************************/
void	XtPlotAddPoints(genericWidget, points, numPoints, redraw)
  Widget	genericWidget ;
  CanvasPoint	*points ;
  int		numPoints ;
  Boolean	redraw ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  XtCanvasAddPoints(widget->plot.canvasWidget, points, numPoints, redraw) ;
  resetScaleLabels(widget) ;
}
/********************************************************************/
void	XtPlotAddCircles(genericWidget, circles, numCircles, redraw)
  Widget	genericWidget ;
  CanvasCircle	*circles ;
  int		numCircles ;
  Boolean	redraw ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  XtCanvasAddCircles(widget->plot.canvasWidget, circles, numCircles, redraw);
  resetScaleLabels(widget) ;
}
/********************************************************************/
void	XtPlotAddSegments(genericWidget, segments, numSegments, redraw)
  Widget	genericWidget ;
  CanvasSegment	*segments ;
  int		numSegments ;
  Boolean	redraw ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  XtCanvasAddSegments(widget->plot.canvasWidget, segments, numSegments,redraw);
  resetScaleLabels(widget) ;
}
/********************************************************************/
void	XtPlotRedraw(genericWidget)
  Widget	genericWidget ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  XtCanvasRedraw(widget->plot.canvasWidget) ;
  resetScaleLabels(widget) ;
}
/********************************************************************/
void	XtPlotClear(genericWidget, redraw)
  Widget	genericWidget ;
  Boolean	redraw ;
{
  PlotWidget	widget = (PlotWidget)genericWidget ;
  XtCanvasClear(widget->plot.canvasWidget, redraw) ;
  resetScaleLabels(widget) ;
}
/********************************************************************/
