
/**********************************************************************
 * $Id: Network.c,v 1.4 92/12/02 13:07:24 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <values.h>
#include <nan.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>
#include "NetworkP.h"

#include "lines.h"
#include "lines2.h"
#include "shadow.h"
#include "weave.h"
#include "solid.h"

/* %%% should be NULL, according to the spec */
#define UnspecifiedPixmap (Pixmap)2
#define MAX(x,y)	((x) > (y) ? (x) : (y))
#define MIN(x,y)	((x) < (y) ? (x) : (y))

#define offset(field)	XtOffset(NetworkWidget, network.field)
static XtResource resources[] = {
  { XtNunitBackgroundStyle, XtCBackgroundStyle, XtRStyle, sizeof(XtStyle),
      offset(unitBackgroundStyle), XtRStyle, XtEshadow},
  { XtNunitBackground, XtCBackground,   XtRPixel, sizeof(Pixel),
      offset(unitBackground), XtRString, XtDefaultBackground},
  { XtNnegativeColor, XtCNegativeColor, XtRPixel, sizeof(Pixel),
      offset(negativeColor), XtRString, XtDefaultForeground },
  { XtNpositiveColor, XtCPositiveColor, XtRPixel, sizeof(Pixel),
      offset(positiveColor), XtRString, XtDefaultBackground },

  {XtNunitHeight,    XtCUnitDimension, XtRDimension, sizeof(Dimension),
     offset(unitHeight),    XtRString,   "20"},
  {XtNunitWidth,     XtCUnitDimension, XtRDimension, sizeof(Dimension),
     offset(unitWidth),     XtRString,   "20"},
  {XtNunitBorderWidth, XtCUnitDimension, XtRDimension, sizeof(Dimension),
     offset(unitBorderWidth), XtRString,   "1"},
  {XtNunitSep,       XtCUnitSep,       XtRDimension, sizeof(Dimension),
     offset(unitSep),       XtRString,   "3"},
  {XtNrowColumnMode, XtCRowColumnMode, XtRBoolean,   sizeof(Boolean),
     offset(rowColumnMode), XtRString,   "True"},

  {XtNnumUnits,      XtCNumUnits,      XtRInt,       sizeof(int),
     offset(numUnits),      XtRString,   "0"},
  {XtNunitArray,     XtCUnitArray,     XtRPointer,   sizeof(UnitDisplayRec *),
     offset(unitArray),     XtRPointer,  NULL},
  {XtNmaxValue,      XtCMaxValue,      XtRFloatPointer,   sizeof(float *),
     offset(maxValuePtr),     XtRString,  "1.0"},
  {XtNfixedMax,	     XtCFixedMax,      XtRBoolean,   sizeof(Boolean),
     offset(fixedMax),	      XtRString,   "True"},

  {XtNincrement,     XtCIncrement,     XtRInt,   sizeof(int),
     offset(increment),	      XtRString,   "10"},

  { XtNqueryCallback, XtCCallback,      XtRCallback,  sizeof(XtCallbackList),
      offset(queryCallback),  XtRCallback, NULL},
  { XtNmotionCallback,XtCCallback,     XtRCallback,  sizeof(XtCallbackList),
      offset(motionCallback), XtRCallback, NULL},
  { XtNchangeValueCallback,XtCCallback,XtRCallback,  sizeof(XtCallbackList),
      offset(changeValueCallback), XtRCallback, NULL},
};
#undef offset


/* Private Functions and procedures */
static void	ClassInitialize () ;
static void	Initialize () ;
static Boolean	SetValues ();
static void	Destroy    () ;
static void	Redisplay  () ;
static void	QueryAction();
static void	SelectAction();
static void	DecrementValueAction();
static void	IncrementValueAction();
static void	DragAction();
static void	DropAction();
static void	setGCs() ;
static GC	getGC () ;
static int	setUnitDimensions() ;
static void	setBackgroundPixmap() ;
static void	Realize() ;

static XtActionsRec actions[] =
{
  /* {name, procedure}, */
    {"query",		QueryAction},
    {"incrementValue",	IncrementValueAction},
    {"decrementValue",	DecrementValueAction},
    {"select",		SelectAction},
    {"drag",		DragAction},
    {"drop",		DropAction},
};

static char translations[] =
  "<Btn1Up>:		query()	\n\
   <Btn2Down>:		select()\n\
   <Btn2Motion>:	drag()	\n\
   <Btn2Up>:		drop()	\n\
   None<Btn3Up>:	incrementValue()	\n\
   Shift<Btn3Up>:	decrementValue()	\n\
";


NetworkClassRec networkClassRec = {
  {				/* core_class fields */
    /* superclass         */    (WidgetClass) &widgetClassRec,
    /* class_name         */    "Network",
    /* widget_size        */    sizeof(NetworkRec),
    /* class_initialize   */    ClassInitialize,
    /* class_part init    */    NULL,
    /* class_inited       */    FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    Realize,
    /* actions            */    actions,
    /* num_actions        */    XtNumber(actions),
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE,
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    Destroy,
    /* resize             */    XtInheritResize,
    /* expose             */    Redisplay,
    /* set_values         */    SetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    NULL,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    translations,
    /* query_geometry     */	XtInheritQueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
    },
  {				/* network_class fields */
    /* empty              */   0
    }
};

WidgetClass networkWidgetClass = (WidgetClass)&networkClassRec ;

static Boolean	CvtStringToStyle(display, args, numArgs, from, to, data)
  Display	*display ;
  XrmValuePtr	args ;
  Cardinal	*numArgs ;
  XrmValuePtr	from, to ;
  XtPointer	data ;
{
  Boolean	badConvert = FALSE ;
  String	string ;
  static XtStyle style ;

  if (*numArgs != 0) {
    badConvert = TRUE ;
    return !badConvert ;
  }
  
  string = from->addr ;

  if (strcmp(string, "plain") == 0)
    style = XtPlain ;
  else if (strcmp(string, "shadow") == 0)
    style = XtShadow ;
  else if (strcmp(string, "lines") == 0)
    style = XtLines ;
  else if (strcmp(string, "lines2") == 0)
    style = XtLines2 ;
  else if (strcmp(string, "solid") == 0)
    style = XtSolid ;
  else if (strcmp(string, "weave") == 0)
    style = XtWeave ;
  else
    badConvert = TRUE ;
  
  if (badConvert) {
    XtStringConversionWarning(string, XtRStyle) ;
  } else {
    if (to->addr == NULL)
      to->addr = (XtPointer) &style ;
    else if (to->size < sizeof(XtStyle))
      badConvert = TRUE ;
    else
      *(XtStyle *)to->addr = style;
    
    to->size = sizeof(XtStyle) ;
  }
  return !badConvert ;
}
  
/*********************************************************************
 *	Name:		CvtStringToFloatPtr
 *	Description:	converts a string to a float value and sets
 *			a pointer to the value
 *	Parameters:
 *	  XrmValue	*args - 
 *	  Cardinal	*numArgs - 
 *	  XrmValuePtr	from, to - 
 *	Return Value:
 *	  static Boolean	CvtStringToFloatPtr - TRUE on success,
 *				FALSE on falure.
 *********************************************************************/
static Boolean	CvtStringToFloatPtr(display, args, numArgs, from, to, data)
  Display	*display ;
  XrmValuePtr	args ;
  Cardinal	*numArgs ;
  XrmValuePtr	from, to ;
  XtPointer	data ;
{
  Boolean	badConvert = FALSE ;
  String	string, ptr ;
  float		floatValue ;
  static float	*floatPtr ;

  if (*numArgs != 0) {
    badConvert = TRUE ;
    return !badConvert ;
  }
  
  string = from->addr ;

  floatPtr   = NULL ;
  floatValue = strtod(string, &ptr) ;
  if (ptr == string)
    badConvert = TRUE ;

  if (badConvert) {
    XtStringConversionWarning(string, XtRFloatPointer) ;
  } else {
    floatPtr  = XtNew(float) ;
    *floatPtr = floatValue ;
    if (NULL == to->addr)
      to->addr = (XtPointer) &floatPtr ;
    else if (to->size < sizeof(float *))
      badConvert = TRUE ;
    else
      *(float **)to->addr = floatPtr;
    
    to->size = sizeof(float *) ;
  }
  return !badConvert ;
}
/********************************************************************/

static void	ClassInitialize () {
  XawInitializeWidgetSet() ;
  XtSetTypeConverter(XtRString, XtRStyle, CvtStringToStyle, 
		     NULL, 0, XtCacheAll, NULL) ;
  XtSetTypeConverter(XtRString, XtRFloatPointer, CvtStringToFloatPtr, 
		     NULL, 0, XtCacheAll, NULL) ;
}

/***********************************************************************
 *	Name:		Initialize
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	Initialize(request, new, args, numArgs)
  Widget	request ;
  Widget	new ;
  ArgList	args ;
  Cardinal	numArgs ;
{
  NetworkPart	*newNet = &((NetworkWidget)new)->network ;
  NetworkPart	*reqNet = &((NetworkWidget)request)->network ;

  int		idx ;
  Dimension	width, height ;

  if (reqNet->unitArray != NULL) {
    newNet->unitArray = (UnitDisplayRec *)XtCalloc(newNet->numUnits, 
						   sizeof(UnitDisplayRec)) ;
    memcpy(newNet->unitArray,
	   reqNet->unitArray, newNet->numUnits*sizeof(UnitDisplayRec)) ;
    newNet->unitArrayP = (UnitDisplayRecP *)XtCalloc(newNet->numUnits, 
						     sizeof(UnitDisplayRecP)) ;
  } else {
    newNet->unitArrayP = NULL ;
  }

  if (newNet->fixedMax) {
    if (newNet->maxValuePtr == NULL)
      newNet->maxValue = 0.0 ;
    else
      newNet->maxValue = *newNet->maxValuePtr ;
  } else {
    newNet->maxValue = 0.0 ;
    for (idx = 0 ; idx < newNet->numUnits ; ++idx)
      if (!isNaN(newNet->unitArray[idx].value))
	newNet->maxValue = MAX(newNet->maxValue, 
			       fabs(newNet->unitArray[idx].value)) ;
  }
  newNet->maxValuePtr = &newNet->maxValue ;

  width = height = 1 ;
  for (idx = 0 ; idx < newNet->numUnits ; ++idx) {
    UnitDisplayRecP	*unitRecP = &newNet->unitArrayP[idx] ;
    setUnitDimensions(new, idx) ;
    width
      = MAX(width, unitRecP->back.x + unitRecP->back.width + newNet->unitSep) ;
    height
      = MAX(height, unitRecP->back.y + unitRecP->back.height+newNet->unitSep) ;
  }
  new->core.width  = MAX(new->core.width,  width) ;
  new->core.height = MAX(new->core.height, height) ;

  newNet->borderGC   = NULL ;
  newNet->backGC     = NULL ;
  newNet->positiveGC = NULL ;
  newNet->negativeGC = NULL ;

  newNet->unitBackgroundPixmap 	 = UnspecifiedPixmap ;
  newNet->changeBackgroundPixmap = TRUE ;
  newNet->exposedRegion		 = NULL ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		SetValues
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static Boolean	SetValues(current, request, new, in_args, numArgs)
  Widget	current, request, new;
  ArgList 	in_args;
  Cardinal 	*numArgs;
{
  NetworkWidget newW = (NetworkWidget)new;
  NetworkWidget oldW = (NetworkWidget)current;
  NetworkWidget reqW = (NetworkWidget)request;
  int		idx ;
  Boolean	redraw = FALSE ;
  Boolean	netChanged = FALSE ;

  /* The whole network is changed, lots of stuff to do */
  if (oldW->network.unitArray != reqW->network.unitArray) {
    netChanged = TRUE ;
    redraw     = TRUE ;
    if (oldW->network.unitArray != NULL) {
      XtFree((char *)oldW->network.unitArray) ;
      XtFree((char *)oldW->network.unitArrayP) ;
    }
    if (reqW->network.unitArray != NULL) {
      newW->network.unitArray 
	= (UnitDisplayRec *)XtCalloc(newW->network.numUnits, 
				     sizeof(UnitDisplayRec)) ;
      memcpy(newW->network.unitArray,
	     reqW->network.unitArray, 
	     newW->network.numUnits*sizeof(UnitDisplayRec)) ;
      newW->network.unitArrayP 
	= (UnitDisplayRecP *)XtCalloc(newW->network.numUnits, 
				      sizeof(UnitDisplayRecP)) ;
    } else {
      newW->network.unitArrayP = NULL ;
    }
  }

  if (newW->network.fixedMax) {
    if (newW->network.maxValuePtr == NULL)
      newW->network.maxValue = 0.0 ;
    else
      newW->network.maxValue = *newW->network.maxValuePtr ;
  } else if (netChanged) {
    newW->network.maxValue = 0.0 ;
    for (idx = 0 ; idx < newW->network.numUnits ; ++idx)
      if (!isNaN(newW->network.unitArray[idx].value))
	newW->network.maxValue 
	  = MAX(newW->network.maxValue,
		fabs(newW->network.unitArray[idx].value)) ;
  }
  newW->network.maxValuePtr = &newW->network.maxValue ;

  /* redraw if the maximum value has changed */
  if (newW->network.maxValue != oldW->network.maxValue)
    redraw = TRUE ;

  /* If dimensions or network have changed, we must recalc display */
  if (netChanged
      || newW->network.maxValue != oldW->network.maxValue
      || newW->network.unitWidth != oldW->network.unitWidth
      || newW->network.unitSep != oldW->network.unitSep
      || newW->network.unitArray != oldW->network.unitArray) {
    Dimension	width, height ;
    redraw = TRUE ;
    width = height = 1 ;
    for (idx = 0 ; idx < newW->network.numUnits ; ++idx) {
      UnitDisplayRecP	*unitRecP = &newW->network.unitArrayP[idx] ;
      setUnitDimensions(new, idx) ;
      width = MAX(width, 
		  unitRecP->back.x + unitRecP->back.width 
		  + newW->network.unitSep);
      height = MAX(height, 
		   unitRecP->back.y + unitRecP->back.height 
		   + newW->network.unitSep);
    }
    newW->core.width  = MAX(new->core.width,  width) ;
    newW->core.height = MAX(new->core.height, height) ;
  }

  /* check unit background pixmap styles */
  if (newW->network.unitBackgroundStyle != oldW->network.unitBackgroundStyle) {
    redraw = TRUE ;
    newW->network.changeBackgroundPixmap = TRUE ;
  }

  /* check the colours and line widths */
  if (   newW->network.unitBackground	!= oldW->network.unitBackground
      || newW->network.negativeColor	!= oldW->network.negativeColor
      || newW->network.positiveColor	!= oldW->network.positiveColor
      || newW->network.unitBorderWidth	!= oldW->network.unitBorderWidth
      || newW->network.rowColumnMode	!= oldW->network.rowColumnMode) {
    redraw = TRUE ;
  }

  /* All of these mean we have to redraw */
  if (redraw)
    return True ;
  else 
    return False ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		Destroy
 *	Description:	destructor for the Network. It deallocates
 *			all memory allocated by initialize.
 *	Parameters:	
 *		Widget	widget - the widget to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	Destroy(widget)
  Widget	widget ;
{
  NetworkWidget	oldW = (NetworkWidget)widget ;

  if (oldW->network.unitArray != NULL)
    XtFree((char *)(char *)oldW->network.unitArray) ;
  if (oldW->network.unitArrayP != NULL)
    XtFree((char *)(char *)oldW->network.unitArrayP) ;

  if (oldW->network.borderGC != NULL)
    XtReleaseGC((Widget)oldW, oldW->network.borderGC) ;
  if (oldW->network.backGC != NULL)
    XtReleaseGC((Widget)oldW, oldW->network.backGC) ;
  if (oldW->network.positiveGC != NULL)
    XtReleaseGC((Widget)oldW, oldW->network.positiveGC) ;
  if (oldW->network.negativeGC != NULL)
    XtReleaseGC((Widget)oldW, oldW->network.negativeGC) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		QueryAction
 *	Description:	action procedure for any button press, it
 *			just calls any query callbacks on the callback list
 *	Parameters:	
 *		Widget		widget - the widget the query went to
 *		XEvent		*event - the query event (button release)
 *		String		*argv  - argument vector (unused)
 *		Cardinal	argc   - number of args (unused)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
#define PointInRect(rect, x, y)	((   (x) > (rect).x \
				  && (y) > (rect).y \
				  && (x) < (rect).x + (rect).width \
				  && (y) < (rect).y + (rect).height) ? \
				 TRUE : FALSE)
/**********************************************************************/
static int	findUnitIdx(widget, x, y)
  NetworkWidget	widget ;
  Position	x ;
  Position	y ;
{
  int		  idx ;
  int		  numUnits = widget->network.numUnits ;
  UnitDisplayRecP *array   = widget->network.unitArrayP ;

  for (idx = 0 ; idx < numUnits ; ++idx) {
    XRectangle	*rectangle = &array[idx].back ;
    if (PointInRect(*rectangle, x, y) == TRUE) 
      break ;
  }
  if (idx >= numUnits)
    return -1 ;
  else
    return idx ;
}
/**********************************************************************/
static void	QueryAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget	netW = (NetworkWidget)widget ;
  int		idx ;

  idx = findUnitIdx(netW, event->xbutton.x, event->xbutton.y) ;

  if (idx >= 0) {
    NetworkReturnStruct	returnStruct ;
    returnStruct.x      = event->xbutton.x ;
    returnStruct.y      = event->xbutton.y ;
    returnStruct.index  = idx ;
    returnStruct.unitRec= netW->network.unitArray[idx] ;

    XtCallCallbacks(widget, XtNqueryCallback, (XtPointer)&returnStruct) ;
  }
}
/**********************************************************************/
static void	IncrementValueAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget		netW = (NetworkWidget)widget ;
  NetworkReturnStruct	returnStruct ;
  int			idx ;
  UnitDisplayRec	unit ;

  idx = findUnitIdx(netW, event->xbutton.x, event->xbutton.y) ;

  if (idx < 0)
    return ;
  
  memcpy(&unit, &netW->network.unitArray[idx], sizeof(unit)) ;
  if (!isNaN(unit.value)) {
    if (netW->network.maxValue == 0.0)
      unit.value += 1.0*netW->network.increment/100 ;
    else 
      unit.value += netW->network.maxValue*netW->network.increment/100 ;
  }  
  XtNetworkSetUnit(widget, idx, &unit) ;

  returnStruct.x      = event->xbutton.x ;
  returnStruct.y      = event->xbutton.y ;
  returnStruct.index  = idx ;
  memcpy(&returnStruct.unitRec, &unit, sizeof(unit)) ;

  XtCallCallbacks(widget, XtNchangeValueCallback, (XtPointer)&returnStruct) ;
}
/**********************************************************************/
static void	DecrementValueAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget		netW = (NetworkWidget)widget ;
  NetworkReturnStruct	returnStruct ;
  int			idx ;
  UnitDisplayRec	unit ;

  idx = findUnitIdx(netW, event->xbutton.x, event->xbutton.y) ;

  if (idx < 0)
    return ;
  
  memcpy(&unit, &netW->network.unitArray[idx], sizeof(unit)) ;
  if (!isNaN(unit.value)) {
    if (netW->network.maxValue == 0.0)
      unit.value -= 1.0*netW->network.increment/100 ;
    else 
      unit.value -= netW->network.maxValue*netW->network.increment/100 ;
  }
  XtNetworkSetUnit(widget, idx, &unit) ;

  returnStruct.x      = event->xbutton.x ;
  returnStruct.y      = event->xbutton.y ;
  returnStruct.index  = idx ;
  memcpy(&returnStruct.unitRec, &unit, sizeof(unit)) ;

  XtCallCallbacks(widget, XtNchangeValueCallback, (XtPointer)&returnStruct) ;
}
/**********************************************************************/
static void	SelectAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget	netW = (NetworkWidget)widget ;
  XRectangle	*rectangle ;
  GC		gc ;
  XGCValues	values;
  XtGCMask	mask ;
  int		idx ;

  /* reset the motion gc */
  if (netW->network.motionGC != NULL)
    XtReleaseGC((Widget)netW, netW->network.motionGC) ;

  values.foreground = netW->core.border_pixel ;
  values.background = netW->core.background_pixel ;
  values.function   = GXxor ;
  values.cap_style  = CapNotLast ;

  mask = GCForeground | GCBackground | GCFunction | GCCapStyle ;
  netW->network.motionGC = gc = XtGetGC((Widget)widget, mask, &values);

  /* find the unit we are in */
  idx = findUnitIdx(netW, event->xbutton.x, event->xbutton.y) ;

  /* no unit, use last one moved */
  if (idx < 0) {
    rectangle = &netW->network.lastRect ;
    rectangle->x = rectangle->y = 0 ;
    rectangle->width = rectangle->height = 0 ;
    return ;
  } else {
    rectangle = &netW->network.unitArrayP[idx].back ;
  }

  /* draw the rectangle and set the lasRect */
  netW->network.motionIdx = idx ;
  XDrawRectangle(XtDisplay(widget), XtWindow(widget), gc,
		 rectangle->x, rectangle->y, 
		 rectangle->width, rectangle->height) ;
  memcpy(&netW->network.lastRect, rectangle, sizeof(XRectangle)) ;
}
/**********************************************************************/
static void	DragAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget	netW  = (NetworkWidget)widget ;
  XRectangle	*rect = &netW->network.lastRect ;

  XDrawRectangle(XtDisplay(widget), XtWindow(widget), netW->network.motionGC,
		 rect->x, rect->y, rect->width, rect->height) ;
  rect->x = event->xbutton.x - netW->network.unitHeight/2 ;
  rect->y = event->xbutton.y - netW->network.unitWidth/2 ;
  XDrawRectangle(XtDisplay(widget), XtWindow(widget), netW->network.motionGC,
		 rect->x, rect->y, rect->width, rect->height) ;
}
/**********************************************************************/
static void	DropAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	argc ;
{
  NetworkWidget		netW = (NetworkWidget)widget ;
  NetworkReturnStruct	returnStruct ;
  UnitDisplayRec 	unit, *unitPtr ;
  XRectangle	*rectangle ;
  int		idx, row, column ;

  rectangle = &netW->network.lastRect ;
  XDrawRectangle(XtDisplay(widget), XtWindow(widget), netW->network.motionGC,
		 rectangle->x, rectangle->y, 
		 rectangle->width, rectangle->height) ;

  if (event->xbutton.y > widget->core.height || event->xbutton.y < 0
      || event->xbutton.x > widget->core.width  || event->xbutton.x < 0) {
    row    = -1 ;
    column = -1 ;
  } else {
    row    = event->xbutton.y/(netW->network.unitHeight+netW->network.unitSep);
    column = event->xbutton.x/(netW->network.unitWidth+netW->network.unitSep) ;
    for (idx = 0 ; idx < netW->network.numUnits ; ++idx) {
      unitPtr = &netW->network.unitArray[idx] ;
      if (unitPtr->row == row && unitPtr->column == column)
	break ;
    }

    if (idx == netW->network.motionIdx)
      return ;

    if (idx < netW->network.numUnits) {
      XBell(XtDisplay(widget), (int)0) ;
      return ;
    }
  }
  idx = netW->network.motionIdx ;
  memcpy(&unit, &netW->network.unitArray[idx], sizeof(unit)) ;
  unit.row    = row ;
  unit.column = column ;
  XtNetworkSetUnit(widget, idx, &unit) ;

  returnStruct.x     = event->xbutton.x ;
  returnStruct.y     = event->xbutton.y ;
  returnStruct.index = idx ;
  memcpy(&returnStruct.unitRec, &netW->network.unitArray[idx], sizeof(unit)) ;

  XtCallCallbacks(widget, XtNmotionCallback, (XtPointer)&returnStruct) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		XtNetworkSetUnit
 *	Description:	sets the values in the UnitDisplayRec struct for a
 *			unit in a network.
 *	Parameters:	
 *		Widget	w	- the network widget
 *		int	unitIdx	- the index of the unit to update
 *		UnitDisplayRec	*unit - the unit rec with the values to set
 *	Return Value:	
 ***********************************************************************/
#define UNIT_MOVED(w, unit, idx)					\
  (((NetworkWidget)w)->network.unitArray[idx].column != unit->column	\
   || ((NetworkWidget)w)->network.unitArray[idx].row != unit->row)

#define UNIT_CHANGED(w, unit, idx)					\
  (!((isNaN(((NetworkWidget)w)->network.unitArray[idx].value)		\
      && isNaN(unit->value)) 						\
     || ((int)(((NetworkWidget)w)->network.unitHeight*			\
	       ((NetworkWidget)w)->network.unitWidth*			\
	       ((NetworkWidget)w)->network.unitArray[idx].value		\
	       /((NetworkWidget)w)->network.maxValue)			\
	 ==								\
	 (int)(((NetworkWidget)w)->network.unitHeight*			\
	       ((NetworkWidget)w)->network.unitWidth*			\
	       unit->value/((NetworkWidget)w)->network.maxValue))))

#define MAX_VALUE_CHANGED(w, unit)					\
  (!((NetworkWidget)w)->network.fixedMax				\
   && !isNaN(unit->value)						\
   && fabs(unit->value) > ((NetworkWidget)w)->network.maxValue)
/**********************************************************************/
static Boolean		resetUnit(w, unitIdx, unit, region)
  Widget		w;
  int			unitIdx ;
  UnitDisplayRec	*unit ;
  Region		region ;
{
  NetworkWidget	net = (NetworkWidget)w ;
  XRectangle	rect ;
  Boolean	redisplay = FALSE ;

  /* If the maximum value is changed we have to reset all the 
   * unit dimensions, and clear the whole window */
  if (MAX_VALUE_CHANGED(w, unit)) {
    int		idx ;
    net->network.maxValue = fabs(unit->value) ;
    memcpy(&net->network.unitArray[unitIdx], unit, sizeof(UnitDisplayRec)) ;
    for (idx = 0 ; idx < net->network.numUnits ; ++idx)
      setUnitDimensions(net, idx) ;
    if (XtIsRealized(w)) {
      redisplay = TRUE ;
      rect.x = 0 ;
      rect.y = 0 ;
      rect.width  = net->core.width ;
      rect.height = net->core.height ;
      XUnionRectWithRegion(&rect, region, region) ;
    }
  } else if (UNIT_MOVED(w, unit, unitIdx) || UNIT_CHANGED(w, unit, unitIdx)) {
    /* if the unit is moved, or the value is changed */
    if (XtIsRealized(w)) {
      Dimension	borderWidth ;
      redisplay    = TRUE ;
      rect         = net->network.unitArrayP[unitIdx].back ;
      borderWidth  = net->network.unitBorderWidth ;
      rect.x      -= borderWidth ;
      rect.y      -= borderWidth ;
      rect.width  += 2*borderWidth ;
      rect.height += 2*borderWidth ;
      XUnionRectWithRegion(&rect, region, region) ;
      if (UNIT_MOVED(w, unit, unitIdx))
	XClearArea(XtDisplay(w), XtWindow(w), 
		   rect.x, rect.y, rect.width, rect.height, False) ;

      memcpy(&net->network.unitArray[unitIdx], unit, sizeof(UnitDisplayRec)) ;
      setUnitDimensions(w, unitIdx) ;

      rect         = net->network.unitArrayP[unitIdx].back ;
      borderWidth  = net->network.unitBorderWidth ;
      rect.x      -= borderWidth ;
      rect.y      -= borderWidth ;
      rect.width  += 2*borderWidth ;
      rect.height += 2*borderWidth ;

      XUnionRectWithRegion(&rect, region, region) ;
    } else {
      memcpy(&net->network.unitArray[unitIdx], unit, sizeof(UnitDisplayRec)) ;
      setUnitDimensions(w, unitIdx) ;
    }

  } else {
    memcpy(&net->network.unitArray[unitIdx], unit, sizeof(UnitDisplayRec)) ;
  }
  return redisplay ;
}
/**********************************************************************/
void		XtNetworkSetUnit(w, unitIdx, unit)
  Widget		w;
  int			unitIdx ;
  UnitDisplayRec	*unit ;
{
  Region	region ;

  if (unitIdx >= ((NetworkWidget)w)->network.numUnits)
    return ;

  region = XCreateRegion() ;

  if (TRUE == resetUnit(w, unitIdx, unit, region))
    Redisplay(w, NULL, region) ;

  XDestroyRegion(region) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		XtNetworkSetUnits
 *	Description:	sets the values in the UnitDisplayRec struct for
 *			ALL units in a network.
 *	Parameters:	
 *		Widget	w	- the network widget
 *		int	unitIdx	- the index of the unit to update
 *		UnitDisplayRec	*unit - the array of unit recs
 *	Return Value:	NONE
 ***********************************************************************/
void		XtNetworkSetUnits(w, unit)
  Widget		w;
  UnitDisplayRec	*unit ;
{
  Region	region ;
  Boolean	redraw = FALSE ;
  int		idx, numUnits ;

  region   = XCreateRegion() ;
  numUnits = ((NetworkWidget)w)->network.numUnits ;
  for (idx = 0 ; idx < numUnits ; ++idx) {
    if (TRUE == resetUnit(w, idx, unit+idx, region))
      redraw = TRUE ;
  }
  if (redraw)
    Redisplay(w, NULL, region) ;

  XDestroyRegion(region) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		XtNetworkGetUnit
 *	Description:	returns the UnitStruct for a requested unit index
 *	Parameters:	
 *		Widget	w	- the network widget
 *		int	idx 	- the index of the unit to get
 *	Return Value:	
 *		UnitDisplayRec	*XtNetworkGetUnit 
 *				- a pointer to the UnitDisplayRec
 *			NOTE: USER IS RESPONSIBLE FOR FREEING IT
 ***********************************************************************/
UnitDisplayRec	*XtNetworkGetUnit(w, idx)
  Widget	w;
  int		idx ;
{
  NetworkWidget		netW = (NetworkWidget)w ;
  UnitDisplayRec	*unitPtr ;

  if (idx >= netW->network.numUnits)
    return NULL ;

  unitPtr = (UnitDisplayRec *)XtMalloc(sizeof(UnitDisplayRec)) ;
  memcpy((char *)unitPtr, 
	 (char *)(&netW->network.unitArray[idx]), sizeof(UnitDisplayRec)) ;
  return unitPtr ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setUnitDimensions
 *	Description:	calculates the dimensions of all the rectangles
 *			and segments used to display an individual unit	
 *	Parameters:	
 *		Widget	widget - the network widget
 *		int	idx    - the index of the unit in the widget arrays
 *	Return Value:	
 *		True
 ***********************************************************************/
static int	setUnitDimensions(widget, idx) 
  Widget	widget ;
  int		idx ;
{
  NetworkPart	*network = &((NetworkWidget)widget)->network ;
  int		area ;
  XRectangle	*rectPtr ;
  XSegment	*segmentPtr ;
  double	ratio ;

  UnitDisplayRec	*unitRec  = &network->unitArray[idx] ;
  UnitDisplayRecP	*unitRecP = &network->unitArrayP[idx] ;

  /* Set the dimensions of the background */
  rectPtr    = &unitRecP->back ;
  if (network->rowColumnMode == True) {
    if (unitRec->column < 0 && unitRec->row < 0) {
      rectPtr->x = -2*(network->unitWidth  + network->unitSep) ;
      rectPtr->y = -2*(network->unitHeight + network->unitSep) ;
    } else {
      rectPtr->x = unitRec->column * (network->unitWidth + network->unitSep)
	+ network->unitSep ;
      rectPtr->y = unitRec->row    * (network->unitHeight + network->unitSep)
	+ network->unitSep ;
    }
  } else {
    rectPtr->x = unitRec->column ;
    rectPtr->y = unitRec->row ;
  }
  rectPtr->width  = network->unitWidth ;
  rectPtr->height = network->unitHeight ;

  /* calculate the dimensions of the foreground rectangle */
  rectPtr = &unitRecP->valueRectangle ;
  if (network->maxValue == 0.0 || isNaN(unitRec->value)) {
    area = 0 ;
    rectPtr->width  = 0 ;
    rectPtr->height = 0 ;
  } else {
    ratio = fabs(unitRec->value)/(network->maxValue) ;
    ratio = MIN(ratio, 1.0) ;
    area  = ratio * (network->unitHeight)*(network->unitWidth) ;

    rectPtr->width = sqrt(ratio)*(network->unitWidth) ;
    if (rectPtr->width == 0)
      rectPtr->height = 0 ;
    else
      rectPtr->height = area/rectPtr->width ;
  }
  rectPtr->x = unitRecP->back.x + (network->unitWidth  - rectPtr->width )/2 ;
  rectPtr->y = unitRecP->back.y + (network->unitHeight - rectPtr->height)/2 ;

  /* calculate the dimensions of the line for the remaining value */
  segmentPtr = &unitRecP->valueSegment ;
  segmentPtr->x1 = rectPtr->x + rectPtr->width ;
  segmentPtr->y1 = rectPtr->y + rectPtr->height - 1 ;
  segmentPtr->x2 = segmentPtr->x1 ;
  segmentPtr->y2 = segmentPtr->y1 - (area - rectPtr->width*rectPtr->height) ;

  return True ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		Redisplay
 *	Description:	the expose event handler for the unit widget.
 *	Parameters:	
 *		Widget	widget - the unit widget
 *		XEvent	*event - the expose event  UNUSED
 *		Region	region - the expose region
 *	Return Value:	
 *		NONE
 ***********************************************************************/
#define UnitExposed(region, rect, borderWidth) \
  ((XRectInRegion((region), 			\
		 (rect).x - (borderWidth),	\
		 (rect).y - (borderWidth),	\
		 (rect).width + 2*(borderWidth),	\
		 (rect).height+ 2*(borderWidth)) == RectangleOut) \
   || ((rect).width + 2*(borderWidth) <= 0 \
       && (rect).height+ 2*(borderWidth) <= 0) ? False : True)

static void	Redisplay(widget, event, region)
  Widget	widget ;
  XEvent	*event ;
  Region	region ;
{
  NetworkWidget	netW = (NetworkWidget)widget ;
  XRectangle	*rectangle ;
  XSegment	*segment ;
  GC		gc ;
  int		idx, numExposed, numPositive, numNegative, numNaN ;

  static XRectangle	*backRects   = NULL ;
  static XRectangle	*borderRects = NULL ;
  static XRectangle	*posRects    = NULL ;
  static XRectangle	*negRects    = NULL ;
  static XSegment	*posSegs     = NULL ;
  static XSegment	*negSegs     = NULL ;
  static XSegment	*nanSegs     = NULL ;
  static int		maxExposed ;

  if (netW->network.exposedRegion == NULL) 
    netW->network.exposedRegion = XCreateRegion() ;

  XUnionRegion(region, 
	       netW->network.exposedRegion, netW->network.exposedRegion) ;

  /* If we have more exposes coming, return */
  if (event != NULL && ((XExposeEvent *)event)->count != 0)
    return ;
  
  if (netW->network.changeBackgroundPixmap) {
    setBackgroundPixmap(netW) ;
    netW->network.changeBackgroundPixmap = FALSE ;
  }
  setGCs(widget) ;

  /* make sure all the arrays are large enough to hold everything */
  if (maxExposed < netW->network.numUnits) {
    maxExposed = netW->network.numUnits ;
    if (backRects != NULL)
      XtFree((char *)backRects) ;
    backRects = (XRectangle *)XtMalloc(maxExposed * sizeof(XRectangle)) ;
    if (borderRects != NULL)
      XtFree((char *)borderRects) ;
    borderRects = (XRectangle *)XtMalloc(maxExposed * sizeof(XRectangle)) ;
    if (posRects != NULL)
      XtFree((char *)posRects) ;
    posRects = (XRectangle *)XtMalloc(maxExposed * sizeof(XRectangle)) ;
    if (negRects != NULL)
      XtFree((char *)negRects) ;
    negRects = (XRectangle *)XtMalloc(maxExposed * sizeof(XRectangle)) ;
    if (posSegs != NULL)
      XtFree((char *)posSegs) ;
    posSegs = (XSegment *)XtMalloc(maxExposed * sizeof(XSegment)) ;
    if (negSegs != NULL)
      XtFree((char *)negSegs) ;
    negSegs = (XSegment *)XtMalloc(maxExposed * sizeof(XSegment)) ;
    if (nanSegs != NULL)
      XtFree((char *)nanSegs) ;
    nanSegs = (XSegment *)XtMalloc(maxExposed * 2*sizeof(XSegment)) ;
  }

  /* find all exposed units and store the rectangles and segments 
   * in the proper arrays */
  numExposed  = 0 ;
  numPositive = 0 ;
  numNegative = 0 ;
  numNaN      = 0 ;
  for (idx = 0 ; idx < netW->network.numUnits ; ++idx) {
    UnitDisplayRecP	*unitRec    = &netW->network.unitArrayP[idx] ;
    Dimension		borderWidth = netW->network.unitBorderWidth ;

    if (UnitExposed(netW->network.exposedRegion,
		    unitRec->back, borderWidth) == False)
      continue ;

    backRects[numExposed] = unitRec->back ;

    if (netW->network.unitBorderWidth > 0) {
      borderRects[numExposed].x = backRects[numExposed].x - borderWidth ;
      borderRects[numExposed].y = backRects[numExposed].y - borderWidth ;
      borderRects[numExposed].width 
	= backRects[numExposed].width  + borderWidth ;
      borderRects[numExposed].height 
	= backRects[numExposed].height + borderWidth ;
    }
    rectangle = &unitRec->valueRectangle ;
    segment   = &unitRec->valueSegment ;
    if (isNaN(netW->network.unitArray[idx].value)) {
      nanSegs[numNaN].x1 = borderRects[numExposed].x ;
      nanSegs[numNaN].x2 = nanSegs[numNaN].x1 + borderRects[numExposed].width ;
      nanSegs[numNaN].y1 = borderRects[numExposed].y ;
      nanSegs[numNaN].y2 = nanSegs[numNaN].y1 + borderRects[numExposed].height;
      ++numNaN ;
      nanSegs[numNaN].x1 = borderRects[numExposed].x ;
      nanSegs[numNaN].x2 = nanSegs[numNaN].x1 + borderRects[numExposed].width ;
      nanSegs[numNaN].y2 = borderRects[numExposed].y ;
      nanSegs[numNaN].y1 = nanSegs[numNaN].y2 + borderRects[numExposed].height;
      ++numNaN ;
    } else if (netW->network.unitArray[idx].value < 0.0) {
      negRects[numNegative] = unitRec->valueRectangle ;
      negSegs[numNegative]  = unitRec->valueSegment ;
      ++numNegative ;
    } else {
      posRects[numPositive] = unitRec->valueRectangle ;
      posSegs[numPositive]  = unitRec->valueSegment ;
      ++numPositive ;
    }
    ++numExposed ;
  }

  /* Now draw everything in one fell swoop */
  if (numExposed > 0) {
    gc = netW->network.backGC ;
    XFillRectangles(XtDisplay(widget), XtWindow(widget), gc,
		    backRects, numExposed) ;
    if (numNaN > 0) {
      gc = getGC((Widget)widget, netW->core.border_pixel,
		 netW->core.background_pixel, netW->core.border_pixmap, 2) ;
      XDrawSegments(XtDisplay(widget), XtWindow(widget), gc,
		    nanSegs, numNaN) ;
      XtReleaseGC((Widget)widget, gc) ;
    }
    if (numNegative > 0) {
      gc = netW->network.negativeGC ;
      XFillRectangles(XtDisplay(widget), XtWindow(widget), gc,
		      negRects, numNegative) ;
      XDrawSegments(XtDisplay(widget), XtWindow(widget), gc,
		    negSegs, numNegative) ;
    }
    if (numPositive > 0) {
      gc = netW->network.positiveGC ;
      XFillRectangles(XtDisplay(widget), XtWindow(widget), gc,
		      posRects, numPositive) ;
      XDrawSegments(XtDisplay(widget), XtWindow(widget), gc,
		    posSegs, numPositive) ;
    }
    if (netW->network.unitBorderWidth > 0) {
      gc = netW->network.borderGC ;
      XDrawRectangles(XtDisplay(widget), XtWindow(widget), gc,
		      borderRects, numExposed) ;
    }
  }

  XDestroyRegion(netW->network.exposedRegion) ;
  netW->network.exposedRegion = NULL ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setBackgroundPixmap
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	setBackgroundPixmap(w)
  NetworkWidget	w ;
{
  if (w->network.unitBackgroundPixmap != UnspecifiedPixmap)
    XFreePixmap(XtDisplay(w), w->network.unitBackgroundPixmap) ;

  if (w->network.unitBackgroundStyle == XtLines)
    w->network.unitBackgroundPixmap = 
      XCreatePixmapFromBitmapData(XtDisplay(w), XtWindow(w), 
				  lines_bits, lines_width, lines_height,
				  w->network.negativeColor,
				  w->network.positiveColor,
				  DefaultDepthOfScreen(XtScreen(w))) ;
  else if (w->network.unitBackgroundStyle == XtShadow)
    w->network.unitBackgroundPixmap = 
      XCreatePixmapFromBitmapData(XtDisplay(w), XtWindow(w), 
				  shadow_bits, shadow_width, shadow_height,
				  w->network.negativeColor,
				  w->network.positiveColor,
				  DefaultDepthOfScreen(XtScreen(w))) ;
  else if (w->network.unitBackgroundStyle == XtLines2)
    w->network.unitBackgroundPixmap = 
      XCreatePixmapFromBitmapData(XtDisplay(w), XtWindow(w), 
				  lines2_bits, lines2_width, lines2_height,
				  w->network.negativeColor,
				  w->network.positiveColor,
				  DefaultDepthOfScreen(XtScreen(w))) ;
  else if (w->network.unitBackgroundStyle == XtSolid)
    w->network.unitBackgroundPixmap = 
      XCreatePixmapFromBitmapData(XtDisplay(w), XtWindow(w), 
				  solid_bits, solid_width, solid_height,
				  w->network.negativeColor,
				  w->network.positiveColor,
				  DefaultDepthOfScreen(XtScreen(w))) ;
  else if (w->network.unitBackgroundStyle == XtWeave)
    w->network.unitBackgroundPixmap = 
      XCreatePixmapFromBitmapData(XtDisplay(w), XtWindow(w), 
				  weave_bits, weave_width, weave_height,
				  w->network.negativeColor,
				  w->network.positiveColor,
				  DefaultDepthOfScreen(XtScreen(w))) ;
  else
    w->network.unitBackgroundPixmap = UnspecifiedPixmap ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setGCs
 *	Description:	sets the GC's for the network to the proper values
 *	Parameters:	
 *		Widget	widget - the network widget
 *	Return Value:	NONE
 ***********************************************************************/
static void	setGCs(widget) 
  Widget	widget ;
{
  NetworkWidget	netW = (NetworkWidget)widget ;

  if (netW->network.borderGC != NULL)
    XtReleaseGC((Widget)netW, netW->network.borderGC) ;
  netW->network.borderGC = getGC((Widget)widget, 
				 netW->core.border_pixel,
				 netW->core.background_pixel,
				 netW->core.border_pixmap,
				 netW->network.unitBorderWidth) ;
  if (netW->network.backGC != NULL)
    XtReleaseGC((Widget)netW, netW->network.backGC) ;
  netW->network.backGC = getGC((Widget)widget, 
			       netW->network.unitBackground,
			       netW->network.unitBackground,
			       netW->network.unitBackgroundPixmap,
			       1) ;
  if (netW->network.positiveGC != NULL)
    XtReleaseGC((Widget)netW, netW->network.positiveGC) ;
  netW->network.positiveGC = getGC((Widget)widget, 
				   netW->network.positiveColor,
				   netW->network.unitBackground,
				   UnspecifiedPixmap,
				   1) ;
  if (netW->network.negativeGC != NULL)
    XtReleaseGC((Widget)netW, netW->network.negativeGC) ;
  netW->network.negativeGC = getGC((Widget)widget, 
				   netW->network.negativeColor,
				   netW->network.unitBackground,
				   UnspecifiedPixmap,
				   1) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getGC
 *	Description:	returns a copy type GC with foreground etc. set
 *			to the proper colour etc.
 *	Parameters:	
 *		Widget	widget - the widget to make the GC for
 *	Return Value:	
 *		GC	getGC - the copy GC
 ***********************************************************************/
static GC	getGC(widget, fg, bg, tile, width)
  Widget	widget ;
  Pixel		fg ;
  Pixel		bg ;
  Pixmap	tile ;
  Dimension	width ;
{
  XGCValues	values;
  XtGCMask	mask ;
  GC		gc ;

  values.foreground	= fg ;
  values.background	= bg ;
  values.function 	= GXcopy ;
  values.cap_style	= CapNotLast ;

  mask = GCForeground | GCBackground | GCFunction |GCCapStyle ;

  if (width > 1) {
    values.line_width	= width ;
    mask |= GCLineWidth ;
  }

  if (tile != UnspecifiedPixmap) {
    values.tile		= tile ;
    mask |= GCTile ;
  }

  gc = XtGetGC((Widget)widget, mask, &values);

  if (tile != UnspecifiedPixmap)
    XSetFillStyle(XtDisplay(widget), gc, FillTiled) ;

  return gc ;
}
/**********************************************************************/

static void	Realize(w, valueMask, attributes)
  Widget	 w;
  Mask		 *valueMask;
  XSetWindowAttributes *attributes;
{
  (*networkWidgetClass->core_class.superclass->core_class.realize)
    (w, valueMask, attributes);

  /* check if we can do backing-store */
  if (NotUseful != XDoesBackingStore(XtScreen(w))) {
    unsigned long	 	valueMask ;
    XSetWindowAttributes	attributes ;
    valueMask = CWBackingStore ;
    attributes.backing_store = WhenMapped ;
    XChangeWindowAttributes(XtDisplay(w), XtWindow(w), valueMask, &attributes);
  }
}
