
/**********************************************************************
 * $Id: Graph.c,v 1.4 92/11/30 11:32:39 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <values.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "GraphP.h"

/* Some standard defines  */
#define WF(widget,  field)	(((Widget)widget)->core.field)
#define GWF(widget, field)	(((GraphWidget)widget)->graph.field)
#define MAX(x, y)		((x) > (y) ? (x) : (y))
#define MIN(x, y)		((x) < (y) ? (x) : (y))
#define ABS(x)			((x) < (0.0) ? (-(x)) : (x))

/* Resources for the Graph Widget */
static XtResource resources[] = {
#define offset(field) XtOffset(GraphWidget, graph.field)
  /* {name, class, type, size, offset, default_type, default_addr}, */
  { XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
      XtOffset(GraphWidget, core.height), XtRString, "100" },
  { XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
      XtOffset(GraphWidget, core.width), XtRString, "200" },
  { XtNfixedMin, XtCFixedScale, XtRBoolean, sizeof(Boolean),
      offset(fixedMin), XtRBoolean, FALSE },
  { XtNfixedMax, XtCFixedScale, XtRBoolean, sizeof(Boolean),
      offset(fixedMax), XtRBoolean, FALSE },
  { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
      offset(foregroundPixel), XtRString, XtDefaultForeground },
  { XtNinternalBorder, XtCWidth, XtRDimension,  sizeof(Dimension),
      offset(internalBorder), XtRImmediate, (XtPointer)4},
  { XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct*),
      offset(font), XtRString, "XtDefaultFont" },
  { XtNnumColumns, XtCNumColumns, XtRInt, sizeof(int),
      offset(numColumns), XtRString, "50" },
  { XtNjumpScroll, XtCJumpScroll, XtRInt, sizeof(int),
      offset(jumpScroll), XtRString, "25" },
  { XtNlabel, XtCLabel, XtRString, sizeof(String),
      offset(label), XtRString, NULL },
  { XtNmaxFormat, XtCFormat, XtRString, sizeof(String),
      offset(maxFormat), XtRString, "% f "},
  { XtNmaxValue, XtCMaxValue, XtRPointer, sizeof(float*),
      offset(maxValue), XtRString, NULL },
  { XtNminFormat, XtCFormat, XtRString, sizeof(String),
      offset(minFormat), XtRString, "% f "},
  { XtNminValue, XtCMinValue, XtRPointer, sizeof(float*),
      offset(minValue), XtRString, NULL },
#undef offset
};

/* Private Functions and procedures */
static void	initialize() ;
static Boolean	setValues ();
static void	destroy   () ;
static void	redisplay () ;
static void	graphAction  ();
static void	setGCs  () ;
static void	setLabelWidth() ;
static Boolean	rescaleGraph () ;
static void	shiftGraph   () ;
static void	drawGraphSegments() ;
static void	setRectDimensions() ;

  /* Translations */
static XtActionsRec actions[] =
{
  /* {name, procedure}, */
    {"graph",	graphAction},
};
static char translations[] =
"<Key>:		graph()	\n\
";

GraphClassRec graphClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Graph",
    /* widget_size		*/	sizeof(GraphRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	destroy,
    /* resize			*/	XtInheritResize,
    /* expose			*/	redisplay,
    /* set_values		*/	setValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* graph fields */
    /* empty			*/	0
  }
};

WidgetClass graphWidgetClass = (WidgetClass)&graphClassRec;

/***********************************************************************
 *	Name:		Initialize
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	initialize(request, new, args, numArgs)
  Widget	request;
  Widget	new;
  ArgList	args;
  Cardinal	numArgs;
{
  /* Make sure that we our own private copy of the label */
  if (GWF(request, label) != NULL)
    GWF(new, label) = XtNewString(GWF(request, label)) ;

  if (GWF(request, maxFormat) != NULL)
    GWF(new, maxFormat) = XtNewString(GWF(request, maxFormat)) ;

  if (GWF(request, minFormat) != NULL)
    GWF(new, minFormat) = XtNewString(GWF(request, minFormat)) ;

  /* Ditto for the minimum value */
  GWF(new, minValue) = (float *)XtMalloc(sizeof(float)) ;
  if (GWF(request, minValue) != NULL)
    *GWF(new, minValue) = *GWF(request, minValue) ;
  else
    *GWF(new, minValue) = 0.0 ;

  /* and the maximum values */
  GWF(new, maxValue) = (float *)XtMalloc(sizeof(float)) ;
  if (GWF(request, maxValue) != NULL)
    *GWF(new, maxValue) = *GWF(request, maxValue) ;
  else
    *GWF(new, maxValue) = 0.0 ;

  /* Now set the internal fields */
  GWF(new, values) = (float *)XtCalloc(GWF(new, numColumns), sizeof(float)) ;

  setGCs(new) ;
  setLabelWidth(new) ;
  setRectDimensions(new) ;

  GWF(new, currentColumn) = 0 ;
  GWF(new, stepSize)
    = ((float)GWF(new, graphWidth))/(GWF(new, numColumns) - 1) ;
}
/***********************************************************************/


/***********************************************************************
 *	Name:		setValues
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static Boolean	setValues(current, request, new, in_args, numArgs)
  Widget	current, request, new;
  ArgList 	in_args;
  Cardinal 	*numArgs;
{
  Boolean	redraw = FALSE ;

  /* Make sure that we have our own private copies of the labels */
  if (GWF(request, label) != GWF(current, label)) {
    if (GWF(request, label))
      GWF(new, label) = XtNewString(GWF(request, label)) ;
    
    if (GWF(current, label))
      XtFree((char *)GWF(current, label)) ;

    setLabelWidth(new) ;
    redraw = TRUE ;
  }
  if (GWF(request, minFormat) != GWF(current, minFormat)) {
    if (GWF(request, minFormat))
      GWF(new, minFormat) = XtNewString(GWF(request, minFormat)) ;
    
    if (GWF(current, minFormat))
      XtFree((char *)GWF(current, minFormat)) ;

    setLabelWidth(new) ;
    redraw = TRUE ;
  }
  if (GWF(request, maxFormat) != GWF(current, maxFormat)) {
    if (GWF(request, maxFormat))
      GWF(new, maxFormat) = XtNewString(GWF(request, maxFormat)) ;
    
    if (GWF(current, maxFormat))
      XtFree((char *)GWF(current, maxFormat)) ;

    setLabelWidth(new) ;
    redraw = TRUE ;
  }

  /* Ditto for the minimum values */
  if (GWF(request, minValue) != GWF(current, minValue)) {
    GWF(new, minValue) = (float *)XtMalloc(sizeof(float)) ;
    if (GWF(request, minValue) != NULL)
      *GWF(new, minValue) = *GWF(request, minValue) ;
    else
      *GWF(new, minValue) = 0.0 ;
    if (GWF(current, minValue))
      XtFree((char *)GWF(current, minValue)) ;
    redraw = TRUE ;
  }

  /* and the maximum values */
  if (GWF(request, maxValue) != GWF(current, maxValue)) {
    GWF(new, maxValue) = (float *)XtMalloc(sizeof(float)) ;
    if (GWF(request, maxValue) != NULL)
      *GWF(new, maxValue) = *GWF(request, maxValue) ;
    else
      *GWF(new, maxValue) = 0.0 ;
    if (GWF(current, maxValue))
      XtFree((char *)GWF(current, maxValue)) ;
    redraw = TRUE ;
  }

  if (GWF(request, jumpScroll) > GWF(request, numColumns))
    GWF(new, jumpScroll) = GWF(request, numColumns) ;

  if (GWF(request, jumpScroll) <= 0)
    GWF(new, jumpScroll) = 1 ;

  /* Now set the internal fields */
  if (GWF(request, numColumns) != GWF(current, numColumns)) {
    GWF(new, values) = (float *)XtCalloc(GWF(new, numColumns), sizeof(float)) ;
    GWF(new, stepSize)
      = ((float)GWF(new, graphWidth))/(GWF(new, numColumns) - 1) ;
    GWF(new, currentColumn) = MIN(GWF(request, currentColumn), 
				  GWF(request, numColumns)) ;
    memcpy(GWF(new, values), 
	   GWF(current, values), GWF(new, currentColumn)*sizeof(float)) ;

    if (GWF(current, values))
      XtFree((char *)GWF(current, values)) ;
    redraw = TRUE ;
  }

  return redraw ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setGCs
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	setGCs(widget)
  Widget	widget ;
{
  XGCValues	values;
  XtGCMask	mask ;

  values.foreground	= GWF(widget, foregroundPixel) ;
  values.background	= WF(widget, background_pixel) ;
  values.font		= GWF(widget, font)->fid ;
  values.function 	= GXcopy ;
  
  mask = GCForeground | GCBackground | GCFont | GCFunction ;

  GWF(widget, gc)      = XtGetGC(widget, mask, &values);
  values.clip_mask	= None ;
  mask |= GCClipMask ;
  GWF(widget, graphGC) = XtGetGC(widget, mask, &values);
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setLabelWidth
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	setLabelWidth(widget)
  Widget 	widget;
{
  XFontStruct	*fs = GWF(widget, font) ;
  String	newLine, label ;
  char		limit[128] ;
  int		lineWidth, labelWidth ;

  sprintf(limit, GWF(widget, maxFormat), -1.1) ;
  labelWidth = XTextWidth(fs, limit, (int) strlen(limit)) ;

  sprintf(limit, GWF(widget, minFormat), -1.1) ;
  GWF(widget, labelWidth) = XTextWidth(fs, limit, (int) strlen(limit)) ;
  GWF(widget, labelWidth) = MAX(GWF(widget, labelWidth), labelWidth) ;

  label = GWF(widget, label) ;

  if (label == NULL) {
    labelWidth = 0 ;
  } else if ((newLine = strchr(label, '\n')) == NULL) {
    labelWidth = XTextWidth(fs, label, (int) strlen(label)) ;
  } else {
    labelWidth = 0 ;
    for ( ; newLine != NULL ; newLine = strchr(label, '\n')) {
      lineWidth  = XTextWidth(fs, label, (int)(newLine - label));
      labelWidth = MAX(labelWidth, lineWidth) ;
      label = newLine + 1;
    }
    if (*label) {
      lineWidth  = XTextWidth(fs, label, strlen(label));
      labelWidth = MAX(labelWidth, lineWidth) ;
    }
  }
  GWF(widget, labelWidth) = MAX(GWF(widget, labelWidth), labelWidth) ;
}
/**********************************************************************/

static void	setRectDimensions(widget)
  Widget	widget ;
{
  GWF(widget, graphWidth)  
    = WF(widget, width) 
      - GWF(widget, labelWidth) 
	- 2 * GWF(widget, internalBorder) ;
  GWF(widget, graphHeight) 
    = (WF(widget, height) - 2 * GWF(widget, internalBorder)) ;
  GWF(widget, stepSize)	
    = ((double)GWF(widget, graphWidth))/(GWF(widget, numColumns) - 1) ;
}

/***********************************************************************
 *	Name:		Destroy
 *	Description:	destructor for the Graph. It deallocates
 *			all memory allocated by initialize.
 *	Parameters:	
 *		Widget	widget - the widget to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	destroy(widget)
  Widget	widget ;
{
  if (GWF(widget, label) != NULL)
    XtFree((char *)GWF(widget, label)) ;

  XtFree((char *)GWF(widget, minValue)) ;
  XtFree((char *)GWF(widget, maxValue)) ;

  if (GWF(widget, gc) != NULL)
    XtReleaseGC(widget, GWF(widget, gc)) ;
  if (GWF(widget, graphGC) != NULL)
    XtReleaseGC(widget, GWF(widget, graphGC)) ;
}
/***********************************************************************/


/***********************************************************************
 *	Name:		graphAction
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void graphAction(widget, event, argv, argc)
  Widget	widget ;
  XEvent	*event ;
  String	*argv ;
  Cardinal	*argc ;
{
#ifdef DEBUG
  fprintf(stderr, "graphAction!!\n") ;
#endif
}
/**********************************************************************/


/***********************************************************************
 *	Name:		drawBoundStrings
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
#define DRAWSTRING(widget, x, y, str)    \
  XDrawString(XtDisplay(widget), XtWindow(widget), GWF(widget, gc), \
	      x, y, str, strlen(str))
#define TEXTLENGTH(widget, str)	\
  XTextWidth(GWF(widget, font), str, strlen(str))

/**********************************************************************/
static void	drawBoundStrings(widget)
  Widget	widget ;
{
  char	numString[32] ;
  int	yMin, yMax, xMax ;
  int	x, y ;

  /* Note that since vertical pixel values increase as you move down
     the screen, yMax < yMin. */
  yMax = GWF(widget, internalBorder) ;
  yMin = yMax + GWF(widget, graphHeight) ;
  xMax = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;

  sprintf(numString, GWF(widget, minFormat), *GWF(widget, minValue)) ;
  x = xMax - TEXTLENGTH(widget, numString) ;
  y = yMin - GWF(widget, font)->max_bounds.descent ;
  DRAWSTRING(widget, x, y, numString) ;

  sprintf(numString,GWF(widget, maxFormat), *GWF(widget, maxValue)) ;
  x = xMax - TEXTLENGTH(widget, numString) ;
  y = yMax + GWF(widget, font)->max_bounds.ascent ;
  DRAWSTRING(widget, x, y, numString) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		drawUpperBound
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
#define DRAWLINE(widget, x1, y1, x2, y2) \
  XDrawLine(XtDisplay(widget), XtWindow(widget), GWF(widget, gc), \
	    (x1), (y1), (x2), (y2))
/**********************************************************************/
static void	drawUpperBound(widget)
  Widget	widget ;
{
  int x1, x2;
  int y ;
  
  x1 = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;
  x2 = x1 + GWF(widget, graphWidth) ;
  y  = GWF(widget, internalBorder) ;
  DRAWLINE(widget, x1, y, x2, y);
}
/**********************************************************************/
static void	drawLowerBound(widget)
  Widget	widget ;
{
  int x1, x2;
  int y ;
  
  x1 = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;
  x2 = x1 + GWF(widget, graphWidth) ;
  y  = GWF(widget, internalBorder) + GWF(widget, graphHeight) ;
  DRAWLINE(widget, x1, y, x2, y);
}
/**********************************************************************/


/***********************************************************************
 *	Name:		drawGraphLabel
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	drawGraphLabel(widget) 
  Widget	widget ;
{
  char	*label, *newLine ;
  int	numLines, x, y, fontHeight ;

  label = GWF(widget, label) ;
  if (label == NULL)
    return ;

  for (numLines = 1, newLine = strchr(label, '\n') ;
       newLine != NULL ; 
       ++numLines, newLine = strchr(newLine + 1, '\n'))
    ;

  fontHeight 
    = GWF(widget, font)->max_bounds.ascent
      + GWF(widget, font)->max_bounds.descent ;

  x = GWF(widget, internalBorder) ;
  y = GWF(widget, internalBorder) + GWF(widget, graphHeight) 
    + GWF(widget, font)->max_bounds.ascent 
      - (GWF(widget, graphHeight) + numLines*fontHeight)/2 ;
  
  for (newLine = strchr(label, '\n') ;
       newLine != NULL ;
       newLine = strchr(label, '\n')) {
    XDrawString(XtDisplay(widget), XtWindow(widget), GWF(widget, gc),
		x, y, label, (int)(newLine - label)) ;
    y += fontHeight ;
    label = newLine + 1 ;
  }
  if (*label)
    XDrawString(XtDisplay(widget), XtWindow(widget), GWF(widget, gc),
		x, y, label, strlen(label)) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		redisplay
 *	Description:	The expose event handler for the Graph widget
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	redisplay(widget, event, region)
  Widget	widget ;
  XEvent	*event ;
  Region	region ;
{
  int	xoffset ;

  setRectDimensions(widget) ;

  if (XRectInRegion(region, 
		    GWF(widget, internalBorder), GWF(widget, internalBorder),
		    GWF(widget, labelWidth),     GWF(widget, graphHeight)) 
      != RectangleOut) {
    drawBoundStrings(widget) ;	/* draw all the bound labels */
    drawGraphLabel(widget) ;	/* draw all the graph labels */
  }

  /* Now actually draw the graphs */
  xoffset = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;
  if (XRectInRegion(region,
		    xoffset, GWF(widget, internalBorder),
		    GWF(widget, graphWidth), 1) != RectangleOut)
    drawUpperBound(widget) ;

  if (GWF(widget, currentColumn) > 0
      && XRectInRegion(region,
		       xoffset, GWF(widget, internalBorder) + 1,
		       GWF(widget, graphWidth),
		       GWF(widget, graphHeight) - 2) != RectangleOut)
    drawGraphSegments(widget, 0, GWF(widget, currentColumn) - 1) ;

  if (XRectInRegion(region,
		    xoffset, 
		    GWF(widget, internalBorder) + GWF(widget, graphHeight) - 1,
		    GWF(widget, graphWidth), 1) != RectangleOut) {
    drawLowerBound(widget) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		graphAddValues
 *	Description:	This routine adds a value that is given to it 
 *			to the graph.
 *	Parameters:	
 *		double	newValue - the value to add to the graph
 *	Return Value:	NONE
 ***********************************************************************/
void graphAddValue(widget, newValue)
  Widget	widget ;
  double	newValue ;
{
  Boolean	rescale = FALSE ;
  int		currentColumn ;

  currentColumn = GWF(widget, currentColumn) ;
  GWF(widget, values)[currentColumn] = newValue ;

  if (newValue > *GWF(widget, maxValue) || newValue < *GWF(widget, minValue))
    rescale = TRUE ;

  if (XtIsRealized(widget) == TRUE) {
    if (currentColumn == 0)
      drawGraphSegments(widget, 0, currentColumn) ;
    else
      drawGraphSegments(widget, currentColumn - 1, currentColumn) ;
  }

  /* update the current value.  This needs to be done even when the graph
     is shifted, as all shift_graph does is move the pointers. */
  ++GWF(widget, currentColumn) ;

  /* Check to see whether the graph needs to be shifted. */
  while (GWF(widget, currentColumn) >= GWF(widget, numColumns)) {
    shiftGraph(widget) ;
    rescale = TRUE ;
  }

  if (rescale == TRUE)
    rescaleGraph(widget) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		graphReset
 *	Description:	Resets the graph completely (i.e. clears all values)
 *	Parameters:	
 *		Widget	widget - the graph widget to reset
 *	Return Value:	NONE
 ***********************************************************************/
void		graphReset(widget)
  Widget	widget ;
{
  if (XtIsSubclass(widget, graphWidgetClass)) {
    GWF(widget, currentColumn) = 0 ;
    rescaleGraph(widget) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		drawGraphSegments
 *	Description:	draws the graph between two (not necessarily
 *			adjacent columns
 *	Parameters:	
 *		Widget	widget - the graph widget
 *		int	firstColumn - the first column to draw from
 *		int	lastColumn - the last column to draw to
 *	Return Value:	
 ***********************************************************************/
static void	drawGraphSegments(widget, firstColumn, lastColumn)
  Widget	widget ;
  int		firstColumn ;
  int		lastColumn ;
{
  int		idx, numPoints ;
  Dimension	xoffset, yoffset, height ;
  float		minValue, span ;
  float		*values, stepSize ;
  XRectangle	rectangle[1] ;

  static XPoint	*points ;
  static int	maxPoints ;

  numPoints = lastColumn - firstColumn + 1 ;
  if (numPoints > maxPoints) {
    maxPoints = numPoints ;
    points    = (XPoint *)XtRealloc((char *)points, maxPoints*sizeof(XPoint)) ;
  }

  values   = GWF(widget, values) ;
  minValue = *GWF(widget, minValue) ;
  span     = *GWF(widget, maxValue) - minValue ;
  stepSize = GWF(widget, stepSize) ;
  height   = GWF(widget, graphHeight) ;
  yoffset  = GWF(widget, internalBorder) + height ;
  xoffset  = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;
  
  for (idx = 0 ; idx < numPoints ; ++idx) {
    points[idx].x = xoffset + (firstColumn + idx)*stepSize ;
    points[idx].y = yoffset - height*(values[firstColumn+idx] - minValue)/span;
  }

  rectangle[0].x = GWF(widget, internalBorder) + GWF(widget, labelWidth) ;
  rectangle[0].y = GWF(widget, internalBorder)  ;
  rectangle[0].width  = GWF(widget, graphWidth) ;
  rectangle[0].height = GWF(widget, graphHeight) ;

  XSetClipRectangles(XtDisplay(widget), GWF(widget, graphGC),
		     0, 0, rectangle, 1, YXBanded) ;
  if (numPoints == 1) {
    XDrawPoint(XtDisplay(widget), XtWindow(widget), GWF(widget, graphGC), 
	       points[0].x, points[0].y) ;
  } else {
    XDrawLines(XtDisplay(widget), XtWindow(widget), GWF(widget, graphGC),
	       points, numPoints, CoordModeOrigin) ;
  }
  XSetClipMask(XtDisplay(widget), GWF(widget, graphGC), None) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		graphShift
 *	Description:	external call to shift a graph widget
 *	Parameters:
 *	  Widget	widget - the graph to shift
 *	Return Value:
 *	  void graphShift - NONE
 *********************************************************************/
void		graphShift(widget)
  Widget	widget ;
{
  if (XtIsSubclass(widget, graphWidgetClass)) {
    shiftGraph(widget) ;
  }
}
/********************************************************************/


/***********************************************************************
 *	Name:		shiftGraph
 *	Description:	shifts the values plotted in the graph by
 *			jumpScroll columns. It is used when the window
 *			is full.
 *	Parameters:	
 *		Widget	widget - the graph Widget
 *	Return Value:	NONE
 ***********************************************************************/
static void	shiftGraph(widget)
  Widget	widget ;
{
  int	newColumn ;
  int	jumpScroll = GWF(widget, jumpScroll) ;
  int	lastColumn = MAX(0, GWF(widget, currentColumn) - jumpScroll) ; 
  float	*values    = GWF(widget, values) ;
  
  for (newColumn = 0  ; newColumn < lastColumn ; ++newColumn)
    values[newColumn] = values[newColumn + jumpScroll] ;

  GWF(widget, currentColumn) = lastColumn ;

  if (XtIsRealized(widget) == True) {
    XClearArea(XtDisplay(widget), XtWindow(widget), 
	       GWF(widget, internalBorder) + GWF(widget, labelWidth),
	       GWF(widget, internalBorder) + 1,
	       GWF(widget, graphWidth),
	       GWF(widget, graphHeight) - 1, TRUE) ;
  }
}
/**********************************************************************/


/*********************************************************************
 *	Name:		graphRescale
 *	Description:	external call to rescale a graph widget
 *	Parameters:
 *	  Widget	widget - the graph to rescale
 *	Return Value:
 *	  void graphRescale - NONE
 *********************************************************************/
void		graphRescale(widget)
  Widget	widget ;
{
  if (XtIsSubclass(widget, graphWidgetClass))
    rescaleGraph(widget) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		rescaleGraph
 *	Description:	Recalculates the maximum and minimum values
 *			for all the graphs
 *	Parameters:	
 *		Widget	widget - the graph widget
 *	Return Value:	True if we did rescale, False if not
 ***********************************************************************/
static Boolean	rescaleGraph(widget)
  Widget	widget ;
{
  int		columnIdx ;
  Boolean	changed = FALSE ;

  float		*values = GWF(widget, values) ;
  float		maxValue, minValue, newSpan, oldSpan ;

  /* if we have both min and max fixed, we can't rescale */
  if (GWF(widget, fixedMin) == TRUE && GWF(widget, fixedMax) == TRUE)
    return FALSE ;

  minValue =  MAXFLOAT ;
  maxValue = -MAXFLOAT ;
  for (columnIdx = 0  ; columnIdx < GWF(widget, currentColumn) ; ++columnIdx) {
    float	value = values[columnIdx] ;
    minValue = MIN(minValue, value) ;
    maxValue = MAX(maxValue, value) ;
  }
  /* if we didn't process any values, set limits to something reasonable */
  if (columnIdx == 0) {
    minValue = 0.0 ;
    maxValue = 0.0 ;
  }

  newSpan = maxValue - minValue ;
  oldSpan = *GWF(widget, maxValue) - *GWF(widget, minValue) ;
  if (GWF(widget, fixedMin) != TRUE &&
      (*GWF(widget, minValue) > minValue || 2.0*newSpan < oldSpan)) {
    *GWF(widget, minValue) = minValue - newSpan/2.0 ;
    changed = TRUE ;
  }
  if (GWF(widget, fixedMax) != TRUE &&
      (*GWF(widget, maxValue) < maxValue || 2.0*newSpan < oldSpan)) {
    *GWF(widget, maxValue) = maxValue + newSpan/2.0 ;
    changed = TRUE ;
  }

  if (changed && XtIsRealized(widget) == True)
    XClearArea(XtDisplay(widget), XtWindow(widget), 
	       0, GWF(widget, internalBorder) + 1,
	       WF(widget, width), GWF(widget, graphHeight) - 1, TRUE) ;

  return changed ;
}
/**********************************************************************/
