
/**********************************************************************
 * $Id: Canvas.c,v 1.3 92/11/30 11:32:35 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <values.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/XawInit.h>

#include "CanvasP.h"

#ifndef MIN
#define MIN(x, y)	((x) < (y) ? (x) : (y))
#endif
#ifndef MIN3
#define MIN3(x, y, z)	((x) < MIN(y, z) ? (x) : MIN(y, z))
#endif

#ifndef MAX
#define MAX(x, y)	((x) > (y) ? (x) : (y))
#endif
#ifndef MAX3
#define MAX3(x, y, z)	((x) > MAX(y, z) ? (x) : MAX(y, z))
#endif

#define offset(field)	XtOffset(CanvasWidget, canvas.field)
static XtResource resources[] = {
  { XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
      XtOffset(CanvasWidget, core.width), XtRString, "300"},
  { XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
      XtOffset(CanvasWidget, core.height), XtRString, "300"},

  { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
      offset(foregroundColor), XtRString, XtDefaultForeground},
  { XtNlineWidth, XtCLineWidth, XtRDimension, sizeof(Dimension),
      offset(lineWidth), XtRString, "1"},
  { XtNpointSize, XtCPointSize, XtRDimension, sizeof(Dimension),
      offset(pointSize), XtRString, "4"},

  { XtNautoScale, XtCAutoScale, XtRBoolean, sizeof(Boolean),
      offset(autoScale), XtRString, "FALSE"},
  { XtNcanvasXOffset, XtCCanvasOffset, XtRFloatPointer, sizeof(float *),
      offset(canvasXOffset), XtRString, "0.0"},
  { XtNcanvasYOffset, XtCCanvasOffset, XtRFloatPointer, sizeof(float *),
      offset(canvasYOffset), XtRString, "0.0"},
  { XtNcanvasWidth, XtCCanvasDimension, XtRFloatPointer, sizeof(float *),
      offset(canvasWidth), XtRString, "1.0"},
  { XtNcanvasHeight, XtCCanvasDimension, XtRFloatPointer, sizeof(float *),
      offset(canvasHeight), XtRString, "1.0"},
};
#undef offset


/* Methods for the Canvas Widget class */
static void	ClassInitialize() ;
static void	Initialize() ;
static Boolean	SetValues();
static void	Destroy() ;
static void	Redisplay() ;

/* Private functions */
static GC	setGC() ;
static void	drawPoints() ;
static void	drawSegments() ;
static void	drawCircles() ;
static Boolean	rescalePoints() ;
static Boolean	rescaleSegments() ;
static Boolean	rescaleCircles() ;

CanvasClassRec canvasClassRec = {
  {				/* core_class fields */
    /* superclass         */    (WidgetClass) &widgetClassRec,
    /* class_name         */    "Canvas",
    /* widget_size        */    sizeof(CanvasRec),
    /* class_initialize   */    ClassInitialize,
    /* class_part init    */    NULL,
    /* class_inited       */    FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    XtInheritRealize,
    /* actions            */    NULL,
    /* num_actions        */    0,
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE,
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    Destroy,
    /* resize             */    XtInheritResize,
    /* expose             */    Redisplay,
    /* set_values         */    SetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    NULL,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    NULL,
    /* query_geometry     */	XtInheritQueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
    },
  {				/* canvas_class fields */
    /* drawPointsProc	  */	drawPoints,
    /* drawSegmentsProc	  */	drawSegments,
    /* drawCirclesProc	  */	drawCircles,
    /* rescalePointsProc  */	rescalePoints,
    /* rescaleSegmentsProc*/	rescaleSegments,
    /* rescaleCirclesProc */	rescaleCircles,
    /* extension          */   	NULL
    }
};

WidgetClass canvasWidgetClass = (WidgetClass)&canvasClassRec ;

/*********************************************************************
 *	Name:		CvtStringToFloatPtr
 *	Description:	converts a string to a float value and sets
 *			a pointer to the value
 *	Parameters:
 *	  XrmValue	*args - 
 *	  Cardinal	*numArgs - 
 *	  XrmValuePtr	from, to - 
 *	Return Value:
 *	  static Boolean	CvtStringToFloatPtr - TRUE on success,
 *				FALSE on falure.
 *********************************************************************/
static Boolean	CvtStringToFloatPtr(display, args, numArgs, from, to, data)
  Display	*display ;
  XrmValuePtr	args ;
  Cardinal	*numArgs ;
  XrmValuePtr	from, to ;
  XtPointer	data ;
{
  Boolean	badConvert = FALSE ;
  String	string, ptr ;
  float		floatValue ;
  static float	*floatPtr ;

  if (*numArgs != 0) {
    badConvert = TRUE ;
    return !badConvert ;
  }
  
  string = from->addr ;

  floatPtr   = NULL ;
  floatValue = strtod(string, &ptr) ;
  if (ptr == string)
    badConvert = TRUE ;

  if (badConvert) {
    XtStringConversionWarning(string, XtRFloatPointer) ;
  } else {
    floatPtr  = XtNew(float) ;
    *floatPtr = floatValue ;
    if (NULL == to->addr)
      to->addr = (XtPointer) &floatPtr ;
    else if (to->size < sizeof(float *))
      badConvert = TRUE ;
    else
      *(float **)to->addr = floatPtr;
    
    to->size = sizeof(float *) ;
  }
  return !badConvert ;
}
/********************************************************************/

static void	ClassInitialize () {
  XawInitializeWidgetSet() ;
  XtSetTypeConverter(XtRString, XtRFloatPointer, CvtStringToFloatPtr, 
		     NULL, 0, XtCacheAll, NULL) ;
}

/*********************************************************************
 *	Name:		Initialize
 *	Description:	
 *	Parameters:
 *	  Widget	requestWidget - 
 *	  Widget	newWidget - 
 *	  ArgList	args - 
 *	  Cardinal	numArgs - 
 *	Return Value:
 *	  static void	Initialize - 
 *********************************************************************/
static void	Initialize(requestWidget, newWidget, args, numArgs)
  Widget	requestWidget ;
  Widget	newWidget ;
  ArgList	args ;
  Cardinal	numArgs ;
{
  CanvasWidget	request = (CanvasWidget)requestWidget ;
  CanvasWidget	new     = (CanvasWidget)newWidget ;

  XtCheckSubclass(requestWidget, canvasWidgetClass, 
		  "Initializer for canvasWidgetClass") ;

  /* set all the scaling variables and make private copies */
  new->canvas.scaleChanged = TRUE ;
  new->canvas.resetMinMax  = TRUE ;

  new->canvas.canvasXOffset = (float *)XtMalloc(sizeof(float)) ;
  if (NULL != request->canvas.canvasXOffset)
    *new->canvas.canvasXOffset = *request->canvas.canvasXOffset ;
  else
    *new->canvas.canvasXOffset = 0.0 ;

  new->canvas.canvasYOffset = (float *)XtMalloc(sizeof(float)) ;
  if (NULL != request->canvas.canvasYOffset)
    *new->canvas.canvasYOffset = *request->canvas.canvasYOffset ;
  else
    *new->canvas.canvasYOffset = 0.0 ;

  new->canvas.canvasWidth = (float *)XtMalloc(sizeof(float)) ;
  if (NULL != request->canvas.canvasWidth)
      *new->canvas.canvasWidth = *request->canvas.canvasWidth ;
  else
    *new->canvas.canvasWidth = 0.0 ;

  new->canvas.canvasHeight = (float *)XtMalloc(sizeof(float)) ;
  if (NULL != request->canvas.canvasHeight)
    *new->canvas.canvasHeight = *request->canvas.canvasHeight ;
  else
    *new->canvas.canvasHeight = 0.0 ;

  /* initialize the colours */
  new->canvas.changeGC     = TRUE ;
  new->canvas.foregroundGC = NULL ;

  /* initialize arrays of figures */
  new->canvas.points   = NULL ;
  new->canvas.segments = NULL ;
  new->canvas.circles  = NULL ;

  new->canvas.numPoints	  = 0 ;
  new->canvas.numSegments = 0 ;
  new->canvas.numCircles  = 0 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		SetValues
 *	Description:	
 *	Parameters:
 *	  Widget	currentWidget, requestWidget, newWidget - 
 *	  ArgList 	in_args - 
 *	  Cardinal 	*numArgs - 
 *	Return Value:
 *	  static Boolean	SetValues - 
 *********************************************************************/
static Boolean	SetValues(currentWidget, requestWidget, newWidget,
			  in_args, numArgs)
  Widget	currentWidget, requestWidget, newWidget;
  ArgList 	in_args;
  Cardinal 	*numArgs;
{
  CanvasWidget	old     = (CanvasWidget)currentWidget ;
  CanvasWidget	request = (CanvasWidget)requestWidget ;
  CanvasWidget	new     = (CanvasWidget)newWidget ;

  XtCheckSubclass(requestWidget, canvasWidgetClass, 
		  "Set values routine for canvasWidgetClass") ;

  if (request->canvas.autoScale == FALSE) {
    if (request->canvas.canvasXOffset != old->canvas.canvasXOffset) {
      new->canvas.scaleChanged  = TRUE ;
      new->canvas.canvasXOffset = old->canvas.canvasXOffset ;
      if (NULL != request->canvas.canvasXOffset)
	*new->canvas.canvasXOffset = *request->canvas.canvasXOffset ;
      else
	*new->canvas.canvasXOffset = 0.0 ;
    }

    if (request->canvas.canvasYOffset != old->canvas.canvasYOffset) {
      new->canvas.scaleChanged  = TRUE ;
      new->canvas.canvasYOffset = old->canvas.canvasYOffset ;
      if (NULL != request->canvas.canvasYOffset)
	*new->canvas.canvasYOffset = *request->canvas.canvasYOffset ;
      else
	*new->canvas.canvasYOffset = 0.0 ;
    }

    if (request->canvas.canvasWidth != old->canvas.canvasWidth) {
      new->canvas.scaleChanged = TRUE ;
      new->canvas.canvasWidth = old->canvas.canvasWidth ;
      if (NULL != request->canvas.canvasWidth)
	*new->canvas.canvasWidth = *request->canvas.canvasWidth ;
      else
	*new->canvas.canvasWidth = 0.0 ;
    }

    if (request->canvas.canvasHeight != old->canvas.canvasHeight) {
      new->canvas.scaleChanged = TRUE ;
      new->canvas.canvasHeight = old->canvas.canvasHeight ;
      if (NULL != request->canvas.canvasHeight)
	*new->canvas.canvasHeight = *request->canvas.canvasHeight ;
      else
	*new->canvas.canvasHeight = 0.0 ;
    }
  }
  /* initialize the colours */
  if (request->canvas.foregroundColor != old->canvas.foregroundColor)
    new->canvas.changeGC = TRUE ;
  if (request->canvas.lineWidth != old->canvas.lineWidth)
    new->canvas.changeGC = TRUE ;
  if (request->canvas.pointSize != old->canvas.pointSize)
    new->canvas.changeGC = TRUE ;

  if (new->canvas.changeGC || new->canvas.scaleChanged)
    return TRUE ;
  else
    return FALSE ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		Destroy
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	Return Value:
 *	  static void	Destroy - 
 *********************************************************************/
static void	Destroy(widget)
  Widget	widget ;
{
  CanvasWidget	old = (CanvasWidget)widget ;

  XtCheckSubclass(widget, canvasWidgetClass, 
		  "Destroy routine for canvasWidgetClass") ;

  /* free the resources */
  if (NULL != old->canvas.canvasXOffset)
    XtFree((char *)old->canvas.canvasXOffset) ;
  if (NULL != old->canvas.canvasYOffset)
    XtFree((char *)old->canvas.canvasYOffset) ;
  if (NULL != old->canvas.canvasWidth)
    XtFree((char *)old->canvas.canvasWidth) ;
  if (NULL != old->canvas.canvasHeight)
    XtFree((char *)old->canvas.canvasHeight) ;

  /* free the GC */
  if (NULL != old->canvas.foregroundGC)
    XtReleaseGC((Widget)old, old->canvas.foregroundGC) ;

  /* free private arrays of points etc. */
  if (NULL != old->canvas.points)
    XtFree((char *)old->canvas.points) ;
  if (NULL != old->canvas.segments)
    XtFree((char *)old->canvas.segments) ;
  if (NULL != old->canvas.circles)
    XtFree((char *)old->canvas.circles) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setGC
 *	Description:	
 *	Parameters:
 *	  CanvasWidget	widget - 
 *	Return Value:
 *	  static void	setGC - 
 *********************************************************************/
static GC	setGC(widget)
  CanvasWidget	widget ;
{
  if (widget->canvas.changeGC) {
    XGCValues	values;
    XtGCMask	mask ;
    if (NULL != widget->canvas.foregroundGC)
      XtReleaseGC((Widget)widget, widget->canvas.foregroundGC) ;

    values.foreground = widget->canvas.foregroundColor ;
    values.line_width = widget->canvas.lineWidth ;
    values.function   = GXcopy ;
    values.cap_style  = CapNotLast ;
    mask = GCForeground | GCLineWidth | GCFunction | GCCapStyle ;

    widget->canvas.foregroundGC = XtGetGC((Widget)widget, mask, &values);
    widget->canvas.changeGC     = FALSE ;
  }
  return widget->canvas.foregroundGC ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		rescalePoints
 *	Description:	
 *	Parameters:
 *	  CanvasWidget	widget - 
 *	  CanvasPoints	*points - 
 *	  int		numPoints - 
 *	Return Value:
 *	  static Boolean	rescalePoints - 
 *********************************************************************/
static Boolean	rescalePoints(widget, points, numPoints)
  CanvasWidget	widget ;
  CanvasPoint	*points ;
  int		numPoints ;
{
  float		minX, minY, maxX, maxY ;
  Boolean	scaleChanged = FALSE ;
  int		idx ;

  if (widget->canvas.resetMinMax == TRUE) {
    minX = minY =  MAXFLOAT ;
    maxX = maxY = -MAXFLOAT ;
    *widget->canvas.canvasXOffset = minX ;
    *widget->canvas.canvasYOffset = minY ;
    *widget->canvas.canvasWidth   = 0.0 ;
    *widget->canvas.canvasHeight  = 0.0 ;
    widget->canvas.resetMinMax = FALSE ;
  } else {
    minX = *widget->canvas.canvasXOffset ;
    minY = *widget->canvas.canvasYOffset ;
    maxX = *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth ;
    maxY = *widget->canvas.canvasYOffset + *widget->canvas.canvasHeight ;
  }

  for (idx = 0 ; idx < numPoints ; ++idx) {
    CanvasPoint	*ptr = &points[idx] ;
    minX = MIN(minX, ptr->x) ;
    minY = MIN(minY, ptr->y) ;
    maxX = MAX(maxX, ptr->x) ;
    maxY = MAX(maxY, ptr->y) ;
  }

  if (minX != MAXFLOAT && minX != *widget->canvas.canvasXOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasXOffset = minX ;
  }

  if (minY != MAXFLOAT && minY != *widget->canvas.canvasYOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasYOffset = minY ;
  }

  if (maxX != -MAXFLOAT
      && maxX != *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasWidth = maxX-*widget->canvas.canvasXOffset ;
  }

  if (maxY != -MAXFLOAT
      && maxY != *widget->canvas.canvasYOffset+*widget->canvas.canvasHeight) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasHeight = maxY-*widget->canvas.canvasYOffset ;
  }

  return scaleChanged ;
}
/*********************************************************************
 *	Name:		drawPoints
 *	Description:	
 *	Parameters:
 *	  CanvasWidget	widget - 
 *	  CanvasPoints	*points - 
 *	  int		numPoints - 
 *	Return Value:
 *	  static void	drawPoints - 
 *********************************************************************/
static void	drawPoints(widget, points, numPoints)
  CanvasWidget	widget ;
  CanvasPoint	*points ;
  int		numPoints ;
{
  static XRectangle	*xRectangles = NULL ;
  static int		arraySize    = 0 ;

  int		idx, boxSize ;
  float		xScale, yScale ;
  float		xOffset, yOffset ;

  if (arraySize < numPoints) {
    arraySize = numPoints ;
    if (NULL != xRectangles)
      XtFree((char *)xRectangles) ;
    xRectangles = (XRectangle *)XtCalloc(arraySize, sizeof(XRectangle)) ;
  }
  xScale = widget->core.width/(*widget->canvas.canvasWidth) ;
  yScale = widget->core.height/(*widget->canvas.canvasHeight) ;
  xOffset = -xScale*(*widget->canvas.canvasXOffset) ;  
  yOffset = widget->core.height + yScale*(*widget->canvas.canvasYOffset) ;

  boxSize = widget->canvas.pointSize ;
  for (idx = 0 ; idx < numPoints ; ++idx) {
    xRectangles[idx].x = xOffset + xScale*points[idx].x - boxSize/2 ;
    xRectangles[idx].y = yOffset - yScale*points[idx].y - boxSize/2;
    xRectangles[idx].width  = boxSize ;
    xRectangles[idx].height = boxSize ;
  }
  XFillRectangles(XtDisplay(widget), XtWindow(widget), setGC(widget),
		  xRectangles, numPoints) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		rescaleSegments
 *	Description:	
 *	Parameters:
 *	  CanvasWidget		widget - 
 *	  CanvasSegments	*segments - 
 *	  int			numSegments - 
 *	Return Value:
 *	  static Boolean	rescaleSegments - 
 *********************************************************************/
static Boolean	rescaleSegments(widget, segments, numSegments)
  CanvasWidget	widget ;
  CanvasSegment	*segments ;
  int		numSegments ;
{
  float		minX =  MAXFLOAT, minY =  MAXFLOAT ;
  float		maxX = -MAXFLOAT, maxY = -MAXFLOAT ;
  Boolean	scaleChanged = FALSE ;
  int		idx ;

  if (widget->canvas.resetMinMax == TRUE) {
    widget->canvas.resetMinMax = FALSE ;
  } else {
    minX = *widget->canvas.canvasXOffset ;
    minY = *widget->canvas.canvasYOffset ;
    maxX = *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth ;
    maxY = *widget->canvas.canvasYOffset + *widget->canvas.canvasHeight ;
  }

  for (idx = 0 ; idx < numSegments ; ++idx) {
    CanvasSegment	*ptr = &segments[idx] ;
    minX = MIN3(minX, ptr->x1, ptr->x2) ;
    minY = MIN3(minY, ptr->y1, ptr->y2) ;
    maxX = MAX3(maxX, ptr->x1, ptr->x2) ;
    maxY = MAX3(maxY, ptr->y1, ptr->y2) ;
  }

  if (minX != MAXFLOAT && minX < *widget->canvas.canvasXOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasXOffset = minX ;
  }

  if (minY != MAXFLOAT && minY != *widget->canvas.canvasYOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasYOffset = minY ;
  }

  if (maxX != -MAXFLOAT
      && maxX != *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasWidth = maxX-*widget->canvas.canvasXOffset ;
  }

  if (maxY != -MAXFLOAT
      && maxY != *widget->canvas.canvasYOffset+*widget->canvas.canvasHeight) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasHeight = maxY-*widget->canvas.canvasYOffset ;
  }

  return scaleChanged ;
}
/*********************************************************************
 *	Name:		drawSegments
 *	Description:	
 *	Parameters:
 *	  CanvasWidget		widget - 
 *	  CanvasSegments	*segments - 
 *	  int			numSegments - 
 *	Return Value:
 *	  static void	drawSegments - 
 *********************************************************************/
static void	drawSegments(widget, segments, numSegments)
  CanvasWidget	widget ;
  CanvasSegment	*segments ;
  int		numSegments ;
{
  static XSegment	*xSegments  = NULL ;
  static int		arraySize = 0 ;

  int		idx ;
  float		xScale, yScale ;
  float		xOffset, yOffset ;

  if (arraySize < numSegments) {
    arraySize = numSegments ;
    if (NULL != xSegments)
      XtFree((char *)xSegments) ;
    xSegments = (XSegment *)XtCalloc(arraySize, sizeof(XSegment)) ;
  }
  xScale = widget->core.width/(*widget->canvas.canvasWidth) ;
  yScale = widget->core.height/(*widget->canvas.canvasHeight) ;
  xOffset = -xScale*(*widget->canvas.canvasXOffset) ;  
  yOffset = widget->core.height + yScale*(*widget->canvas.canvasYOffset) ;

  for (idx = 0 ; idx < numSegments ; ++idx) {
    xSegments[idx].x1 = xOffset + xScale*segments[idx].x1 ;
    xSegments[idx].y1 = yOffset - yScale*segments[idx].y1 ;
    xSegments[idx].x2 = xOffset + xScale*segments[idx].x2 ;
    xSegments[idx].y2 = yOffset - yScale*segments[idx].y2 ;
  }
  XDrawSegments(XtDisplay(widget), XtWindow(widget), setGC(widget),
		xSegments, numSegments) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		rescaleCircles
 *	Description:	
 *	Parameters:
 *	  CanvasWidget	widget - 
 *	  CanvasCircles	*circles - 
 *	  int		numCircles - 
 *	Return Value:
 *	  static Boolean	rescaleCircles - 
 *********************************************************************/
static Boolean	rescaleCircles(widget, circles, numCircles)
  CanvasWidget	widget ;
  CanvasCircle	*circles ;
  int		numCircles ;
{
  float		minX =  MAXFLOAT, minY =  MAXFLOAT ;
  float		maxX = -MAXFLOAT, maxY = -MAXFLOAT ;
  Boolean	scaleChanged = FALSE ;
  int		idx ;

  if (widget->canvas.resetMinMax == TRUE) {
    widget->canvas.resetMinMax = FALSE ;
  } else {
    minX = *widget->canvas.canvasXOffset ;
    minY = *widget->canvas.canvasYOffset ;
    maxX = *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth ;
    maxY = *widget->canvas.canvasYOffset + *widget->canvas.canvasHeight ;
  }

  for (idx = 0 ; idx < numCircles ; ++idx) {
    CanvasCircle	*ptr = &circles[idx] ;
    minX = MIN(minX, ptr->x - ptr->radius) ;
    minY = MIN(minY, ptr->y - ptr->radius) ;
    maxX = MAX(maxX, ptr->x + ptr->radius) ;
    maxY = MAX(maxY, ptr->y + ptr->radius) ;
  }

  if (minX != MAXFLOAT && minX < *widget->canvas.canvasXOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasXOffset = minX ;
  }

  if (minY != MAXFLOAT && minY != *widget->canvas.canvasYOffset) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasYOffset = minY ;
  }

  if (maxX != -MAXFLOAT
      && maxX != *widget->canvas.canvasXOffset + *widget->canvas.canvasWidth) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasWidth = maxX-*widget->canvas.canvasXOffset ;
  }

  if (maxY != -MAXFLOAT
      && maxY != *widget->canvas.canvasYOffset+*widget->canvas.canvasHeight) {
    scaleChanged = TRUE ;
    *widget->canvas.canvasHeight = maxY-*widget->canvas.canvasYOffset ;
  }

  return scaleChanged ;
}
/*********************************************************************
 *	Name:		drawCircles
 *	Description:	
 *	Parameters:
 *	  CanvasWidget	widget - 
 *	  CanvasCircles	*circles - 
 *	  int		numCircles - 
 *	Return Value:
 *	  static void	drawCircles - 
 *********************************************************************/
static void	drawCircles(widget, circles, numCircles)
  CanvasWidget	widget ;
  CanvasCircle	*circles ;
  int		numCircles ;
{
  static XArc	*xArcs  = NULL ;
  static int	arraySize = 0 ;

  int		idx ;
  float		xScale, yScale ;
  float		xOffset, yOffset ;

  if (arraySize < numCircles) {
    arraySize = numCircles ;
    if (NULL != xArcs)
      XtFree((char *)xArcs) ;
    xArcs = (XArc *)XtCalloc(arraySize, sizeof(XArc)) ;
  }
  xScale = widget->core.width/(*widget->canvas.canvasWidth) ;
  yScale = widget->core.height/(*widget->canvas.canvasHeight) ;
  xOffset = -xScale*(*widget->canvas.canvasXOffset) ;  
  yOffset = widget->core.height + yScale*(*widget->canvas.canvasYOffset) ;

  for (idx = 0 ; idx < numCircles ; ++idx) {
    xArcs[idx].x = xOffset + xScale*(circles[idx].x - circles[idx].radius) ;
    xArcs[idx].y = yOffset - yScale*(circles[idx].y + circles[idx].radius) ;
    xArcs[idx].width  = 2*xScale*circles[idx].radius ;
    xArcs[idx].height = 2*yScale*circles[idx].radius ;
    xArcs[idx].angle1 = 0 ;
    xArcs[idx].angle2 = 23040 ; /* 360 * 64 (don't ask me why) */
  }
  XDrawArcs(XtDisplay(widget), XtWindow(widget), setGC(widget),
	    xArcs, numCircles) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		Redisplay
 *	Description:	the expose event handler for the unit widget.
 *	Parameters:	
 *		Widget	widget - the unit widget
 *		XEvent	*event - the expose event  UNUSED
 *		Region	region - the expose region
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	Redisplay(genericWidget, event, region)
  Widget	genericWidget ;
  XEvent	*event ;
  Region	region ;
{
  CanvasWidget	 widget = (CanvasWidget)genericWidget ;
  CanvasDrawProc draw ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "Expose handler for canvasWidgetClass") ;

  /* If we have more exposes coming, return */
  if (NULL != event && ((XExposeEvent *)event)->count != 0)
    return ;

  if (widget->canvas.numPoints > 0) {
    draw = ((CanvasWidgetClass)XtClass(widget))->canvas_class.drawPointsProc ;
    if (NULL != draw)
      draw(widget, widget->canvas.points, widget->canvas.numPoints) ;
  }
  if (widget->canvas.numSegments > 0) {
    draw = ((CanvasWidgetClass)XtClass(widget))->canvas_class.drawSegmentsProc;
    if (NULL != draw)
      draw(widget, widget->canvas.segments, widget->canvas.numSegments) ;
  }

  if (widget->canvas.numCircles > 0) {
    draw = ((CanvasWidgetClass)XtClass(widget))->canvas_class.drawCirclesProc ;
    if (NULL != draw)
      draw(widget, widget->canvas.circles, widget->canvas.numCircles) ;
  }

  widget->canvas.scaleChanged = FALSE ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		XtCanvasAddPoints
 *	Description:	
 *	Parameters:
 *	  Widget	genericWidget - 
 *	  CanvasPoint	*points - 
 *	  int		numPoints - 
 *	  Boolean	redraw - 
 *	Return Value:
 *	  void	XtCanvasAddPoints - 
 *********************************************************************/
void	XtCanvasAddPoints(genericWidget, points, numPoints, redraw)
  Widget	genericWidget ;
  CanvasPoint	*points ;
  int		numPoints ;
  Boolean	redraw ;
{
  CanvasWidget	 widget = (CanvasWidget)genericWidget ;
  CanvasDrawProc	draw ;
  CanvasRescaleProc	rescale ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "XtCanvasAddPoints procedure for canvasWidgetClass") ;

  if (numPoints <= 0)
    return ;

  /* add the points to the widget */
  widget->canvas.numPoints += numPoints ;
  widget->canvas.points 
    = (CanvasPoint *) XtRealloc((char *)widget->canvas.points,
				widget->canvas.numPoints*sizeof(CanvasPoint)) ;

  memcpy(widget->canvas.points + widget->canvas.numPoints - numPoints, points, 
	 numPoints*sizeof(CanvasPoint));

  if (TRUE == widget->canvas.autoScale) {
    rescale =
      ((CanvasWidgetClass)XtClass(widget))->canvas_class.rescalePointsProc ;
    if (NULL != rescale)
      widget->canvas.scaleChanged = rescale(widget, points, numPoints) ;
  }

  
  if (TRUE == redraw && XtIsRealized((Widget)widget)) {
    if (widget->canvas.scaleChanged)
      XtCanvasRedraw(genericWidget) ;
    else {
      draw = ((CanvasWidgetClass)XtClass(widget))->canvas_class.drawPointsProc;
      if (NULL != draw)
	draw(widget, points, numPoints) ;
    }
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtCanvasAddSegments
 *	Description:	
 *	Parameters:
 *	  Widget	genericWidget - 
 *	  CanvasSegment	*segments - 
 *	  int		numSegments - 
 *	  Boolean	redraw - 
 *	Return Value:
 *	  void	XtCanvasAddSegments - 
 *********************************************************************/
void	XtCanvasAddSegments(genericWidget, segments, numSegments, redraw)
  Widget	genericWidget ;
  CanvasSegment	*segments ;
  int		numSegments ;
  Boolean	redraw ;
{
  CanvasWidget	 widget = (CanvasWidget)genericWidget ;
  CanvasDrawProc	draw ;
  CanvasRescaleProc	rescale ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "XtCanvasAddSegments procedure for canvasWidgetClass") ;

  if (numSegments <= 0)
    return ;

  /* add the segments to the widget */
  widget->canvas.numSegments += numSegments ;
  widget->canvas.segments 
    = (CanvasSegment *) XtRealloc((char *)widget->canvas.segments,
				  widget->canvas.numSegments *sizeof(CanvasSegment)) ;

  memcpy(widget->canvas.segments + widget->canvas.numSegments - numSegments, 
	 segments, numSegments*sizeof(CanvasSegment));

  if (TRUE == widget->canvas.autoScale) {
    rescale =
      ((CanvasWidgetClass)XtClass(widget))->canvas_class.rescaleSegmentsProc ;
    if (NULL != rescale)
      widget->canvas.scaleChanged = rescale(widget, segments, numSegments) ;
  }

  if (TRUE == redraw && XtIsRealized((Widget)widget)) {
    if (widget->canvas.scaleChanged)
      XtCanvasRedraw(genericWidget) ;
    else {
      draw =
	((CanvasWidgetClass)XtClass(widget))->canvas_class.drawSegmentsProc;
      if (NULL != draw)
	draw(widget, segments, numSegments) ;
    }
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtCanvasAddCircles
 *	Description:	
 *	Parameters:
 *	  Widget	genericWidget - 
 *	  CanvasCircle	*circles - 
 *	  int		numCircles - 
 *	  Boolean	redraw - 
 *	Return Value:
 *	  void	XtCanvasAddCircles - 
 *********************************************************************/
void	XtCanvasAddCircles(genericWidget, circles, numCircles, redraw)
  Widget	genericWidget ;
  CanvasCircle	*circles ;
  int		numCircles ;
  Boolean	redraw ;
{
  CanvasWidget	widget = (CanvasWidget)genericWidget ;
  CanvasDrawProc	draw ;
  CanvasRescaleProc	rescale ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "XtCanvasAddCircles procedure for canvasWidgetClass") ;

  if (numCircles <= 0)
    return ;

  /* add the circles to the widget */
  widget->canvas.numCircles += numCircles ;
  widget->canvas.circles 
    = (CanvasCircle *) XtRealloc((char *)widget->canvas.circles,
			      widget->canvas.numCircles*sizeof(CanvasCircle)) ;

  memcpy(widget->canvas.circles + widget->canvas.numCircles - numCircles, 
	 circles, numCircles*sizeof(CanvasCircle)) ;
  
  if (TRUE == widget->canvas.autoScale) {
    rescale =
      ((CanvasWidgetClass)XtClass(widget))->canvas_class.rescaleCirclesProc ;
    if (NULL != rescale)
      widget->canvas.scaleChanged = rescale(widget, circles, numCircles) ;
  }

  if (TRUE == redraw && XtIsRealized((Widget)widget)) {
    if (widget->canvas.scaleChanged)
      XtCanvasRedraw(genericWidget) ;
    else {
      draw =
	((CanvasWidgetClass)XtClass(widget))->canvas_class.drawCirclesProc ;
      if (NULL != draw)
	draw(widget, circles, numCircles) ;
    }
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtCanvasRedraw
 *	Description:	
 *	Parameters:
 *	  Widget	genericWidget - 
 *	Return Value:
 *	  void		XtCanvasRedraw - 
 *********************************************************************/
void		XtCanvasRedraw(genericWidget)
  Widget	genericWidget ;
{
  CanvasWidget	widget = (CanvasWidget)genericWidget ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "XtCanvasReset procedure for canvasWidgetClass") ;

  if (XtIsRealized((Widget)widget))
    XClearArea(XtDisplay(widget), XtWindow(widget), 0, 0, 0, 0, True) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		XtCanvasClear
 *	Description:	
 *	Parameters:
 *	  Widget	genericWidget - 
 *	  Boolean	redraw - 
 *	Return Value:
 *	  void		XtCanvasClear - 
 *********************************************************************/
void		XtCanvasClear(genericWidget, redraw)
  Widget	genericWidget ;
  Boolean	redraw ;
{
  CanvasWidget	widget = (CanvasWidget)genericWidget ;

  XtCheckSubclass(genericWidget, canvasWidgetClass, 
		  "XtCanvasReset procedure for canvasWidgetClass") ;

  widget->canvas.numCircles = 0 ;
  if (widget->canvas.circles) {
    XtFree((char *)widget->canvas.circles) ;
    widget->canvas.circles = NULL ;
  }
  widget->canvas.numPoints = 0 ;
  if (widget->canvas.points) {
    XtFree((char *)widget->canvas.points) ;
    widget->canvas.points = NULL ;
  }
  widget->canvas.numSegments = 0 ;
  if (widget->canvas.segments) {
    XtFree((char *)widget->canvas.segments) ;
    widget->canvas.segments = NULL ;
  }

  widget->canvas.resetMinMax = TRUE ;
  if (redraw && XtIsRealized((Widget)widget))
    XClearArea(XtDisplay(widget), XtWindow(widget), 0, 0, 0, 0, True) ;
}
/********************************************************************/
