
/**********************************************************************
 * $Id: unit.c,v 1.3 92/11/30 11:32:06 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "unit.h"
#include "traverse.h"

/***********************************************************************
 *	Private functions
 ***********************************************************************/
static void	initialize ARGS((Unit	unit)) ;
static void	finalize   ARGS((Unit	unit)) ;

static void	checkUnitName  ARGS((Unit	unit, void	*data)) ;
static void	checkUnitRegex ARGS((Unit	unit, void	*data)) ;
/**********************************************************************/

/***********************************************************************
 *	Private variables
 ***********************************************************************/
static UnitProc	createUnitHook  = NULL ;
static UnitProc	destroyUnitHook = NULL ;
/**********************************************************************/


/***********************************************************************
 *	Name:		setCreateUnitHook  (setDestroyUnitHook)
 *	Description:	sets the user hook to be called after (before)
 *			a unit is created (destroyed).
 *	Parameters:	
 *		UnitProc	hook - the procedure to be called:
 *				void hook (Unit	unit)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	setCreateUnitHook(hook)
  UnitProc	hook ;
{
  createUnitHook  = hook ;
}
/**********************************************************************/
void	setDestroyUnitHook(hook)
  UnitProc	hook ;
{
  destroyUnitHook = hook ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		createUnit 
 *	Description:	creates a unit and initializes all of its data
 *	Parameters:	
 *		char	*name - the name for the unit
 *		Group	group   - the group to insert the uni in
 *	Return Value:	
 *		Unit	createUnit - the new unit
 ***********************************************************************/
Unit	createUnit (name, group)
  char	*name ;
  Group	group ;
{
  Unit	unit ;

  unit = (Unit)malloc(sizeof(UnitRec)) ;

  if (unit == NULL) {
    simError(SIM_ERR_MALLOC, "createUnit \"%s\"", name) ;
    return unit ;
  }

  unit->name  = strdup(name) ;
  unit->group = group ;
  if (group != NULL)
    unit->net = group->net ;

  initialize (unit) ;

  if (createUnitHook != NULL)
    (*createUnitHook)(unit) ;

  if (group != NULL)
    groupAddUnit(group, unit) ;

  return unit ;
}
/***********************************************************************
 *	Name:		initialize
 *	Description:	init proc for the Unit class of objects
 *		Unit	unit - the unit object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	initialize (unit) 
  Unit		unit ;
{
  if (unit->group && unit->group->type & BIAS)
    unit->output = 1.0 ;
  else
    unit->output  = 0.0 ;

  unit->totalInput  = 0.0 ;
  unit->target  = 0.0 ;
  unit->outputDeriv = 0.0 ;
  unit->inputDeriv  = 0.0 ;

  if (unit->net != NULL)
    unit->timeSlices = unit->net->timeSlices ;

  if (unit->net->type & RECURRENT) {
    unit->outputHistory   = (Real *)calloc(unit->timeSlices, sizeof(Real)) ;
    unit->clampingHistory = (Real *)calloc(unit->timeSlices, sizeof(Real)) ;
    unit->targetHistory   = (Real *)calloc(unit->timeSlices, sizeof(Real)) ;
    if (unit->group && unit->group->type & BIAS) {
      int	idx ;
      for (idx = 0 ; idx < unit->timeSlices ; ++idx)
	unit->outputHistory[idx] = 1.0 ;
    }
  } else {
    unit->outputHistory   = NULL ;
    unit->clampingHistory = NULL ;
    unit->targetHistory   = NULL ;
  }

  unit->weightRow    = 0 ;
  unit->weightColumn = 0 ;
  unit->activityRow    = 0 ;
  unit->activityColumn = 0 ;

  unit->incomingLink = NULL ;
  unit->outgoingLink = NULL ;
  unit->numIncoming  = 0 ;
  unit->numOutgoing  = 0 ;
  unit->incomingArraySize  = 0 ;
  unit->outgoingArraySize  = 0 ;

  unit->extension = NULL ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		destroyUnit 
 *	Description:	destroys a unit
 *	Parameters:	
 *		Unit	unit - the unit to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	destroyUnit (unit) 
  Unit		unit ;
{
  if (unit->group != NULL) 
    groupDeleteUnit(unit->group, unit) ;

  if (destroyUnitHook != NULL)
    (*destroyUnitHook)(unit) ;

  finalize(unit) ;

  if (unit->name != NULL)
    free(unit->name) ;
  free(unit) ;
}
/***********************************************************************
 *	Name:		finalize
 *	Description:	finalize proc for the unit class of objects
 *			NOTE: this also destroys all links attached to it.
 *		Unit	unit - the unit object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	finalize (unit) 
  Unit		unit ;
{
  if (unit->outputHistory != NULL)
    free(unit->outputHistory) ;
  if (unit->clampingHistory != NULL)
    free(unit->clampingHistory) ;
  if (unit->targetHistory != NULL)
    free(unit->targetHistory) ;

  while (unit->numIncoming > 0)
    destroyLink(unit->incomingLink[unit->numIncoming-1]) ;

  while (unit->numOutgoing > 0)
    destroyLink(unit->outgoingLink[unit->numOutgoing-1]) ;

  if (unit->incomingLink != NULL)
    free(unit->incomingLink) ;
  if (unit->outgoingLink != NULL)
    free (unit->outgoingLink) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		unitAddLink 
 *	Description:	adds either an incoming or outgoing link
 *			to the unit, depending on mask.
 *	Parameters:	
 *		Unit	unit - the unit to add the link to
 *		Mask	mask - a mask identifying the type of link
 *				(eg. INCOMING, OUTGOING)
 *		Unit	link - the link to add in
 *	Return Value:	
 *		NONE
 ***********************************************************************/
#ifndef GRANULARITY
#define GRANULARITY 64
#endif
int	unitAddLink (unit, mask, link)
  Unit	unit ;
  Mask	mask ;
  Link	link ;
{
  Link	**arrayPtr ;
  int	*numPtr ;
  int	*sizePtr ;

  if (mask & INCOMING) {
    arrayPtr = &unit->incomingLink ;
    numPtr   = &unit->numIncoming ;
    sizePtr  = &unit->incomingArraySize ;
  } else {
    arrayPtr = &unit->outgoingLink ;
    numPtr   = &unit->numOutgoing ;
    sizePtr  = &unit->outgoingArraySize ;
  }

  if (*numPtr >= *sizePtr) {
    if (*sizePtr == 0) {
      /* set to sum default initial size */
      *sizePtr  = *numPtr + GRANULARITY ;
      *arrayPtr = (Link *)malloc ((*sizePtr)*sizeof(Link)) ;
    } else {
      /* double the size of the array */
      *sizePtr  = 2 * (*numPtr) ;
      *arrayPtr = (Link *)realloc (*arrayPtr, (*sizePtr)*sizeof(Link)) ;
    }
    if (*arrayPtr == NULL) {
      simError(SIM_ERR_MALLOC, "unitAddLink unit: \"%s\", link: \"%s\"", 
	       unit->name, link->name) ;
      return 0 ;
    }
  }

  (*arrayPtr)[*numPtr] = link ;
  ++(*numPtr) ;
  return 1 ;
}
/**********************************************************************/  


/***********************************************************************
 *	Name:		unitDeleteLink 
 *	Description:	deletes either an incoming or outgoing link
 *			from the unit, depending on mask.
 *	Parameters:	
 *		Unit	unit - the unit to delete from
 *		Mask	mask - a mask identifying the type of link
 *				(eg. INCOMING, OUTGOING)
 *		Unit	link - the link to delete
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	unitDeleteLink (unit, mask, link)
  Unit	unit ;
  Mask	mask ;
  Link	link ;
{
  Link	*array ;
  int	*numPtr, idx ;

  if (mask & INCOMING) {
    array  = unit->incomingLink ;
    numPtr = &unit->numIncoming ;
  } else {
    array  = unit->outgoingLink ;
    numPtr = &unit->numOutgoing ;
  }

  for (idx = 0 ; idx < *numPtr && array[idx] != link ; ++idx)
    ;

  if (array != NULL && array[idx] == link) {
    for ( ; idx < *numPtr ; ++idx)
      array[idx] = array[idx + 1] ;
    --(*numPtr) ;
  }
}
/**********************************************************************/  


/***********************************************************************
 *	Name:		unitFromName 
 *	Description:	finds the unit with the given name
 *	Parameters:	
 *		Net	net  - the net to search through
 *		String	name - the name of the required unit
 *	Return Value:	
 *		Unit	unitFromName - the FIRST unit with name "name",
 *				NULL if it does not exist.
 ***********************************************************************/
static Unit	thisUnit ;
/**********************************************************************/
Unit	unitFromName (net, name)
  Net		net ;
  String	name ;
{
  thisUnit = NULL ;
  netForAllUnitsBack (net, ALL, checkUnitName, (void*)name) ;
  return thisUnit ;
}
/**********************************************************************/
static void	checkUnitName (unit, data)
  Unit		unit ;
  void		*data ;
{
  if (strcmp(unit->name, (char *)data) == 0)
    thisUnit = unit ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		unitsFromRegex 
 *	Description:	finds the units whose names match the given
 *			regular expression
 *	Parameters:	
 *		Net	net   - the network to search through
 *		String	expression - the regular expression to match
 *	Return Value:	
 *		Unit	*unitsFromRegex - an array of units whose names
 *				match the regular expression. The final
 *				element in the array is NULL.
 *				The user is responsible for freeing it.
 *				NULL is returned if the regex is bad.
 ***********************************************************************/
static Unit	*matchedUnit ;
static int	numMatches ;
/**********************************************************************/
Unit	*unitsFromRegex (net, expression)
  Net		net ;
  String	expression ;
{
  char	buffer[BUFSIZ], *compiled ;

  if (strpbrk(expression, "$^") != NULL)
    return NULL ;

  sprintf(buffer, "^%s$", expression) ;
  compiled = regcmp(buffer, NULL) ;
  if (compiled == NULL)
    return NULL ;

  numMatches  = 0 ;
  matchedUnit = (Unit *)malloc(sizeof(Unit)) ;
  if (matchedUnit == NULL) {
    simError(SIM_ERR_MALLOC, "unitsFromRegex") ;
    return matchedUnit ;
  }
  matchedUnit[0] = NULL ;
  netForAllUnits(net, ALL, checkUnitRegex, (void*)compiled) ;

  free(compiled) ;
  return matchedUnit ;
}
/**********************************************************************/
static void	checkUnitRegex (unit, data)
  Unit		unit ;
  void		*data ;
{
  if (matchedUnit == NULL)
    return ;

  if (regex((char *)data, unit->name) != NULL) {
    matchedUnit[numMatches] = unit ;
    ++numMatches ;
    matchedUnit = (Unit *)realloc(matchedUnit, 
				    (numMatches + 1)*sizeof(Unit)) ;
    if (matchedUnit == NULL) {
      simError(SIM_ERR_MALLOC, "unitsFromRegex") ;
      return ;
    }
    matchedUnit[numMatches] = NULL ;
  }
}
/**********************************************************************/

