
/**********************************************************************
 * $Id: traverse.c,v 1.5 93/03/29 14:12:12 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "traverse.h"

#define TRAV_FORWARD	(Mask)0
#define TRAV_BACKWARD	(Mask)1
#define TRAV_RANDOM	(Mask)2

typedef void	(*TraverseNetProc) ARGS(()) ;
static void	traverseNet ARGS((Net	net,       Mask	mask, 
				  TraverseNetProc, void	*data, 
				  Mask	traverseType, 
				  int	unitsOrGroups,
				  int	negate)) ;

static void	traverseGroup ARGS((Group group, 
				    UnitDataProc, void *data, 
				    Mask   traverseType)) ;

/***********************************************************************
 *	Name:		unitForAll(Other)Links 
 *	Description:	for all links (NOT) in a mask set of a unit this will 
 *			call a proc passing in the link Object and data.
 *	Parameters:	
 *		Unit		unit - the unit to process.
 *		Mask		mask - the mask set
 *		LinkDataProc	proc - the procedure to execute for all links
 *		void		*data- client data
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void		unitForAllLinks (unit, mask, proc, data)
  Unit		unit ;
  Mask		mask ;
  LinkDataProc	proc ;
  void		*data ;
{
  int	idx, num ;
  Link	*link ;
  Mask	otherMask = mask & ~(INCOMING | OUTGOING) ;
  
  /* go through incoming if there is any flag we don't recognize */
  if ((mask & INCOMING) || otherMask) {
    num  = unit->numIncoming ;
    link = unit->incomingLink ;
    for (idx = 0 ; idx < num ; ++idx, ++link)
      if ((mask & INCOMING) || (otherMask & (*link)->type))
	(*proc) (*link, data) ;
  }
  if (mask & OUTGOING || otherMask) {
    num  = unit->numOutgoing ;
    link = unit->outgoingLink ;
    for (idx = 0 ; idx < num ; ++idx, ++link)
      if ((mask & OUTGOING) || (otherMask & (*link)->type))
	(*proc) (*link, data) ;
  }
}
/**********************************************************************/
void		unitForAllOtherLinks (unit, mask, proc, data)
  Unit		unit ;
  Mask		mask ;
  LinkDataProc	proc ;
  void		*data ;
{
  int	idx, num ;
  Link	*link ;
  Mask	otherMask = mask & ~(INCOMING | OUTGOING) ;
  
  /* go through incoming if INCOMING is not in the mask */
  if (!(mask & INCOMING)) {
    num  = unit->numIncoming ;
    link = unit->incomingLink ;
    for (idx = 0 ; idx < num ; ++idx, ++link)
      if (!(otherMask & (*link)->type))
	(*proc) (*link, data) ;
  }
  if (!(mask & OUTGOING)) {
    num  = unit->numOutgoing ;
    link = unit->outgoingLink ;
    for (idx = 0 ; idx < num ; ++idx, ++link)
      if (!(otherMask & (*link)->type))
	(*proc) (*link, data) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupForAllUnits / groupForAllUnitsBack
 *	Description:	for all units in a mask set of a group this will 
 *			call a proc passing in the unit Object and data.
 *			For Back, it traverses in opposite order.
 *	Parameters:	
 *		Group		group - the group to process.
 *		UnitDataProc	proc  - the procedure to execute for all units
 *		void		*data  - client data
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void		groupForAllUnits (group, proc, data)
  Group		group ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseGroup(group, proc, data, TRAV_FORWARD) ;
}
/**********************************************************************/
void		groupForAllUnitsBack (group, proc, data)
  Group		group ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseGroup(group, proc, data, TRAV_BACKWARD) ;
}
/**********************************************************************/
void		groupForAllUnitsRandom (group, proc, data)
  Group		group ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseGroup(group, proc, data, TRAV_RANDOM) ;
}
/**********************************************************************/
static void	traverseGroup(group, proc, data, traverseType)
  Group		group ;
  UnitDataProc	proc ;
  void		*data ;
  Mask		traverseType ;
{
  Unit		*unitPtr, *finalPtr ;

  switch(traverseType) {
  case TRAV_FORWARD:
    unitPtr  = group->unit ;
    finalPtr = unitPtr + group->numUnits ;
    while (unitPtr != finalPtr)
      (*proc) (*(unitPtr++), data) ;
    break ;

  case TRAV_BACKWARD:
    finalPtr = group->unit ;
    unitPtr  = finalPtr + group->numUnits ;
    while ((unitPtr--) != finalPtr)
      (*proc) (*unitPtr, data) ;
    break ;

  case TRAV_RANDOM:
    {
      int	idx ;
      int	numUnits = group->numUnits ;
      Unit	*unit    = group->unit ;
      Unit	*randomArray = NULL ;

      if (numUnits) {
	randomArray = (Unit *)calloc(numUnits, sizeof(Unit)) ;
	if (randomArray == NULL) {
	  simError(SIM_ERR_MALLOC, "traverseGroup: \"%s\"", group->name) ;
	  return ;
	}
      } 
  
      for (idx = 0 ; idx < numUnits ; ) {
	int	next = (int)(simRand() * numUnits) % numUnits ;
	if (randomArray[next] == NULL)
	  randomArray[next] = unit[idx++] ;
      }

      for (idx = 0 ; idx < numUnits ; ++idx)
	(*proc) (randomArray[idx], data) ;

      free ((void *)randomArray) ;
    } 
  default:
    break ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		netForAll(Other)Groups/Units/Back/Random 
 *	Description:	for all groups (units) in a mask set of a net 
 *			these will call a proc passing in the Object 
 *			and the user supplied data.
 *	Parameters:	
 *		Net		net  - the net to process.
 *		Mask		mask - the mask set for the groups.
 *		GroupDataProc	proc - the procedure to execute for all groups
 *		UnitDataProc	proc - the procedure to execute for all Units
 *		void		*data - any data you want to pass to the proc
 *	Return Value:	
 *		NONE
 ***********************************************************************/
#define GROUP_IN_SET(group, mask) (((group)->type & mask) ? TRUE : FALSE)
#define TRAV_GROUPS		(Mask)0
#define TRAV_UNITS		(Mask)1
/**********************************************************************/
void	netForAllGroups (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_FORWARD, TRAV_GROUPS, TRUE) ;
} 
/**********************************************************************/
void	netForAllGroupsBack (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_BACKWARD, TRAV_GROUPS, TRUE) ;
}
/**********************************************************************/
void	netForAllGroupsRandom (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_RANDOM, TRAV_GROUPS, TRUE) ;
}
/**********************************************************************/
void	netForAllOtherGroups (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_FORWARD, TRAV_GROUPS, FALSE) ;
} 
/**********************************************************************/
void	netForAllOtherGroupsBack (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_BACKWARD, TRAV_GROUPS, FALSE) ;
}
/**********************************************************************/
void	netForAllOtherGroupsRandom (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  GroupDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_RANDOM, TRAV_GROUPS, FALSE) ;
}
/**********************************************************************/
void	netForAllUnits (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_FORWARD, TRAV_UNITS, TRUE) ;
}
/**********************************************************************/
void	netForAllUnitsBack (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_BACKWARD, TRAV_UNITS, TRUE) ;
}
/**********************************************************************/
void	netForAllUnitsRandom (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_RANDOM, TRAV_UNITS, TRUE) ;
}
/**********************************************************************/
void	netForAllOtherUnits (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_FORWARD, TRAV_UNITS, FALSE) ;
}
/**********************************************************************/
void	netForAllOtherUnitsBack (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_BACKWARD, TRAV_UNITS, FALSE) ;
}
/**********************************************************************/
void	netForAllOtherUnitsRandom (net, mask, proc, data)
  Net		net ;
  Mask		mask ;
  UnitDataProc	proc ;
  void		*data ;
{
  traverseNet(net, mask, (TraverseNetProc)proc, data, TRAV_RANDOM, TRAV_UNITS, FALSE) ;
}
/**********************************************************************/
static void	sumUnits(group, data)
  Group		group ;
  void		*data ;
{
  *(int *)data += group->numUnits ;
}
/**********************************************************************/
static int	currentIdx ;
static void	collectUnits(unit, data)
  Unit		unit ;
  void		*data ;
{
  ((Unit *)data)[currentIdx++] = unit ;
}
/**********************************************************************/
static void	traverseNet(net, mask, proc, data, 
			    traverseType, unitsOrGroups, inSet)
  Net		net ;
  Mask		mask ;
  TraverseNetProc proc ;
  void		*data ;
  Mask		traverseType ;
  int		unitsOrGroups ;
  int		inSet ;
{
  Group		*groupPtr, *finalPtr ;

  switch(traverseType) {
  case TRAV_FORWARD:
    groupPtr = net->group ;
    finalPtr = groupPtr + net->numGroups ;
    while (groupPtr != finalPtr) {
      if (GROUP_IN_SET(*groupPtr, mask) == inSet) {
	if (unitsOrGroups == TRAV_UNITS)
	  traverseGroup(*groupPtr, (UnitDataProc)proc, data, TRAV_FORWARD) ;
	else
	  ((GroupDataProc)proc)(*groupPtr, data) ;
      }
      ++groupPtr ;
    }
    break ;

  case TRAV_BACKWARD:
    finalPtr = net->group ;
    groupPtr = finalPtr + net->numGroups ;
    while (groupPtr-- != finalPtr) {
      if (GROUP_IN_SET(*groupPtr, mask) == inSet) {
	if (unitsOrGroups == TRAV_UNITS)
	  traverseGroup(*groupPtr, (UnitDataProc)proc, data, TRAV_FORWARD) ;
	else
	  ((GroupDataProc)proc)(*groupPtr, data) ;
      }
    }
    break ;

  case TRAV_RANDOM:
    {
      Group	*group    = net->group ;
      int	numGroups = net->numGroups ;
      int	idx ;

      if (unitsOrGroups == TRAV_GROUPS) {
	Group	*groupArray = (Group *)calloc(numGroups, sizeof(Group)) ;
	if (groupArray == NULL) {
	  simError(SIM_ERR_MALLOC, "traverseNet: \"%s\"", net->name) ;
	  return ;
	}
  
	for (idx = 0 ; idx < numGroups ; ) {
	  int	next = (int)(simRand() * numGroups) % numGroups ;
	  if (groupArray[next] == NULL)
	    groupArray[next] = group[idx++] ;
	}

	for (idx = 0 ; idx < numGroups ; ++idx)
	  if (GROUP_IN_SET(*group, mask) == inSet)
	    ((GroupDataProc)proc)(groupArray[idx], data) ;

	free ((void *)groupArray) ;
      } else {
	Unit	*unitArray, *randomUnitArray ;
	int	numUnits = 0 ;
	traverseNet(net, mask, (TraverseNetProc)sumUnits, &numUnits, 
		    TRAV_FORWARD, TRAV_GROUPS, inSet) ;
	if (numUnits == 0)
	  return ;

	unitArray = (Unit *)calloc(numUnits, sizeof(Unit)) ;
	if (unitArray == NULL) {
	  simError(SIM_ERR_MALLOC, "traverseNet: \"%s\"", net->name) ;
	  return ;
	}
	randomUnitArray = (Unit *)calloc(numUnits, sizeof(Unit)) ;
	if (randomUnitArray == NULL) {
	  simError(SIM_ERR_MALLOC, "traverseNet: \"%s\"", net->name) ;
	  return ;
	}
  
	currentIdx = 0 ;
	traverseNet(net, mask, (TraverseNetProc)collectUnits, unitArray,
		    TRAV_FORWARD, TRAV_UNITS, inSet) ;

	for (idx = 0 ; idx < numUnits ; ) {
	  int	next = (int)(simRand() * numUnits) % numUnits ;
	  if (randomUnitArray[next] == NULL)
	    randomUnitArray[next] = unitArray[idx++] ;
	}

	for (idx = 0 ; idx < numUnits ; ++idx)
	  ((UnitDataProc)proc)(randomUnitArray[idx], data) ;

	free ((void *)unitArray) ;
	free ((void *)randomUnitArray) ;
	break ;
      }
      break ;
    }
  default:
    break ;
  }
}
/**********************************************************************/
