
/**********************************************************************
 * $Id: simUtils.c,v 1.3 93/02/09 16:44:51 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <varargs.h>

#include <xerion/simulator.h>
#define _SOURCE
#include "simUtils.h"
#undef _SOURCE

int	simErrno ;

char	*simErrString[] = {
  "Unknown simulator error",
  "Malloc returned a NULL value",
  "Bad array size"
  } ;
  
/***********************************************************************
 *	Name:		simError
 *	Description:	produces a message on standard error output,
 *			describing a simulator error and sets the integer
 *			value simerrno.
 *	Parameters:	
 *		int	simerrno - the error number to set
 *		char	*format - a format string (printf style) and
 *				associated args. simErr prints this
 *				string, a colon and a blank then a standard
 *				error message.
 *	Return Value:	NONE
 ***********************************************************************/
void	simError(va_alist)
  va_dcl 
{
  va_list 	args ;
  char		*format ;

  va_start(args) ;
  simErrno = va_arg(args, int) ;
  format   = va_arg(args, char *) ;

  if (format != NULL) {
    vfprintf(stderr, format, args) ;
    fprintf(stderr, ": ") ;
  }
  va_end(ap);

  if (simErrno < 0 || simErrno > MAX_SIMERRNO)
    simErrno = 0 ;
  fprintf(stderr, "%s\n", simErrString[simErrno]) ;
}
/**********************************************************************/



/* ************************************************* */
/* Here is a good random number generator
   and the auxiliary routines to print the execution time */

#define SHUFFLE 256		/* size of random array */
#define MULT 25173
#define MOD ((long int) 65536)
#define INCR 13849
#define FMOD ((double) 65536.0)

static int initialized = FALSE ;
static long int	randSeed = 696969;
static double	random_array[SHUFFLE];	/* random variables */

/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	double ISimpleRandReal(void) - returns a random number between 0 and 1
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables:
 *	double randSeed 	    - The rng seed
 * Returns: None
 * Calls: None
 * Description:
 *	This routine returns the random number generator between 0 and 1
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
static double ISimpleRandReal()
{
  randSeed = (MULT * randSeed + INCR) % MOD;
  return (double) randSeed / FMOD;
}
/***********************************************************************/

/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	initialize_rng() - to initialize the random number generator
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables: None
 * Returns: None
 * Calls:
 *	ISimpleRandReal()
 *	IRandReal()
 * Description:
 *	This routine initializes the random number generator
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
static void initialize_rng()
{
  int n;
  double x;
  
  for (n = 0; n < SHUFFLE; ++n)
    random_array[n] = ISimpleRandReal();
  for (n = 0; n < 1000; ++n)	/* warm up random generator */
    x = IRandReal();
}


/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	double IRandReal(void) - Shuffles random numbers in random_array[]
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables: None
 * Returns: None
 * Calls:
 *	ISimpleRandReal()
 * Description:
 *	This routine initializes the random number generator
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
double simRand()
{
  double rranf;
  int kranf;

  if (!initialized) {
    initialized = TRUE ;
    initialize_rng() ;
  }

  kranf = (int) (ISimpleRandReal() * SHUFFLE) % SHUFFLE;
  rranf = random_array[kranf];
  random_array[kranf] = ISimpleRandReal();
  
  return rranf;
}
/***********************************************************************/
void	simRandSeed(seed)
  int	seed ;
{
  if (!initialized) {
    initialized = TRUE ;
    initialize_rng() ;
  }

  randSeed = seed ;
  initialize_rng() ;
}
/***********************************************************************/
