
/**********************************************************************
 * $Id: sigmoid.c,v 1.5 93/03/29 14:12:10 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>

#include "sigmoid.h"
#include "sigmoidP.h"


/********************************************************************/
#define isTransformed(s) (!(   (s)->xOffset == 0.0 && (s)->yOffset == 0.0\
			    && (s)->scale   == 1.0 && (s)->squash  == 1.0))
/********************************************************************/
static Real	setXOffset	ARGS((Sigmoid, double offset)) ;
static Real	setYOffset	ARGS((Sigmoid, double offset)) ;
static Real	setScale	ARGS((Sigmoid, double scale)) ;
static Real	setSquash	ARGS((Sigmoid, double squash)) ;

static Real	getSigmoid	ARGS((Sigmoid, double y)) ;
static Real	getDeriv	ARGS((Sigmoid, double y)) ;

static void	destroy		ARGS((Sigmoid)) ;
/********************************************************************/

/*********************************************************************
 *	Name:		createSigmoid/destroySigmoid
 *	Description:	creates (destroys) a sigmoid distribution object
 *	Parameters:
 *	  double	xOffset - the x offset of the center of the sigmoid
 *	  double	yOffset - the y offset of the center of the sigmoid
 *	  double	squash  - the horizontal scale (squash)
 *	  double	scale   - the vertical scale (stretch)
 *	Return Value:
 *	  Sigmoid	createSigmoid  - the new sigmoid
 *	  void		destroySigmoid - NONE
 *********************************************************************/
Sigmoid	createSigmoid(xOffset, yOffset, squash, scale)
  double	xOffset ;
  double	yOffset ;
  double	squash ;
  double	scale ;
{
  Sigmoid	this ;
  
  this = (Sigmoid)malloc(sizeof(SigmoidRec)) ;
  if (this == NULL)
    return NULL ;

  this->setXOffset	= setXOffset ;
  this->setYOffset	= setYOffset ;
  this->setSquash	= setSquash ;
  this->setScale	= setScale ;

  this->getSigmoid	= getSigmoid ;
  this->getDeriv	= getDeriv ;

  this->destroy	= destroy ;

  MSsetXOffset(this, xOffset) ;
  MSsetYOffset(this, yOffset) ;
  MSsetSquash (this, squash) ;
  MSsetScale  (this, scale) ;

  return this ;
}
/********************************************************************/
static void	destroy(this)
  Sigmoid	this ;
{
  if (this)
    free((void *)this) ;
}
/********************************************************************/


/********************************************************************/
static Real	setXOffset(this, offset)
  Sigmoid	this ;
  double	offset ;
{
  this->xOffset = offset ;
  this->transform = isTransformed(this) ;

  return MSxOffset(this) ;
}
/********************************************************************/
static Real	setYOffset(this, offset)
  Sigmoid	this ;
  double	offset ;
{
  this->yOffset = offset ;
  this->transform = isTransformed(this) ;

  return MSyOffset(this) ;
}
/********************************************************************/
static Real	setScale(this, scale)
  Sigmoid	this ;
  double	scale ;
{
  if (scale == 0.0)
    scale = 1.0 ;

  this->scale   = scale ;
  this->transform = isTransformed(this) ;

  return MSscale(this) ;
}
/********************************************************************/
static Real	setSquash(this, squash)
  Sigmoid	this ;
  double	squash ;
{
  if (squash == 0.0)
    squash = 1.0 ;

  this->squash  = squash ;
  this->transform = isTransformed(this) ;

  return MSsquash(this) ;
}
/********************************************************************/


/**********************************************************************
 *	Name:         getSigmoid
 *	Description:  fast version of a sigmoid function, uses table
 *			lookup
 *	Parameters:
 *	  Sigmoid	this - the sigmoid to use
 *	  double	x - the value for which to calculate 
 *				the sigmoid
 *	Return Value: 
 *	  static Real	getSigmoid - the sigmoid value.
 **********************************************************************/
static Real	getSigmoid(this, x)
  Sigmoid	this ;
  double	x;
{
  Boolean	negative ;
  Real		y ;

  if (this->transform)
    x = (x - this->xOffset) * this->squash ;

  negative = x < 0.0 ;
  if (negative)
    x = -x ;

  if (x > xMax) {
    int 	tsize = sizeof(table) / sizeof(table[0]) ;

    y = xMaxGrad*(x - xMax) / xIncrement + table[tsize-1];
    if (y > 1) 
      y = 1 ;

  } else {
    Real		quotient, *entry;

    quotient = x/xIncrement;
    entry    = table + (int)quotient;

    y = *entry + (quotient - (int)quotient)*(*(entry+1) - *entry);
  }

  if (negative)
    y = 1.0 - y ;

  if (this->transform)
    y = y * this->scale + this->yOffset ;

  return y ;
}
/********************************************************************/
static Real	getDeriv(this, x)
  Sigmoid	this ;
  double	x;
{
  Real	y = MSsigmoid(this, x) ;

  return y * (1.0 - y) ;
}
/********************************************************************/
