
/**********************************************************************
 * $Id: qArray.c,v 1.4 92/11/30 11:31:56 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <xerion/useful.h>
#include "qArray.h"

typedef unsigned int	QAStorage ;

#define NUM_BITS(type)		(8*sizeof(type))
#define LEFTMOST_BIT(type)	((type)1 << (NUM_BITS(type) - 1))
#define ALL_BITS(type)		(~(type)0)
#define MAX_NUM(bits)					\
  ((unsigned int)(bits >= NUM_BITS(unsigned int) ?	\
		  ALL_BITS(unsigned int) : (((unsigned int)1 << bits) - 1)))

#ifndef MAX
#define MAX(x,y)	((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x,y)	((x) < (y) ? (x) : (y))
#endif

#define BIT(storage, offset)					\
  (offset ? ((storage) & (LEFTMOST_BIT(QAStorage) >> (offset)))	\
            : ((storage) & LEFTMOST_BIT(QAStorage)))
#define SHIFTED_BIT(storage,offset,shift)			\
  (shift ? (shift < 0 ? (BIT(storage, offset) << -shift) 	\
	                : (BIT(storage, offset) >> shift)) 	\
           : BIT(storage, offset))


static QAStorage	getBits	ARGS((QAStorage *, int, int)) ;
static QAStorage	setBits	ARGS((QAStorage *, int, int, 
				      unsigned int)) ;

static float	store		ARGS((QArray, int idx, double value)) ;
static float	retrieve	ARGS((QArray, int idx)) ;
static void	destroy		ARGS((QArray)) ;

/*********************************************************************
 *	Name:		createQArray
 *	Description:	creates a Quantized array object
 *	Parameters:
 *	  unsigned int	elements - the number of elements it will hold
 *	  unsigned int	bits - the number of bits to use to represent
 *				each element
 *	  double	min - the minimum value representable
 *	  double	max - the maximum value representable
 *	Return Value:
 *	  QArray	createQArray - the object
 *********************************************************************/
QArray	createQArray(elements, bits, min,  max)
  unsigned int	elements ;
  unsigned int	bits ;
  double	min ;
  double	max ;
{
  QArray	this = (QArray)malloc(sizeof(QArrayRec)) ;

  this->bits = bits ;
  this->min  = min ;
  this->max  = MAX(min, max) ;
  this->elements = elements ;

  if (this->bits <= 0 || this->bits > NUM_BITS(QAStorage)) {
    this->bits = 0 ;
    this->array.real = (float *)calloc(this->elements, sizeof(float)) ;
  } else {
    int	numBytes = 1 + (this->bits * this->elements / NUM_BITS(char)) ;
    this->array.storage = (char *)calloc(numBytes, sizeof(char)) ;
  }

  this->store	 = store ;
  this->retrieve = retrieve ;
  this->destroy	 = destroy ;

  return this ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroy
 *	Description:	destroys a Quantized array
 *	Parameters:
 *	  QArray	this - the object
 *	Return Value:
 *	  static void	destroy - NONE
 *********************************************************************/
static void	destroy(this)
  QArray	this ;
{
  if (this == NULL)
    return ;

  if (this->array.storage)
    free((void *)this->array.storage) ;

  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		store
 *	Description:	the store method for a QArray
 *	Parameters:
 *	  QArray	this - the object
 *	  int		idx - the index for where to store it
 *	  double	value - the value to store
 *	Return Value:
 *	  static float	store - the quantized value
 *********************************************************************/
static float	store(this, idx, value)
  QArray	this ;
  int		idx ;
  double	value ;
{
  QAStorage	qValue ;

  if (this == NULL)
    return 0.0 ;

  if (idx < 0 || idx >= this->elements)
    return 0.0 ;

  if (this->bits) {
    if (value > this->max)
      value = this->max ;
    else if (value < this->min)
      value = this->min ;
    
    if (this->max > this->min)
      value = (value - this->min)/(this->max - this->min) ;
    else
      value = 0.0 ;

    qValue = rint(value * MAX_NUM(this->bits)) ;
    setBits((QAStorage *)this->array.storage, this->bits, idx, qValue) ;
  } else {
    this->array.real[idx] = value ;
  }
  
  return MQAretrieve(this, idx) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		retrieve
 *	Description:	the retrieve method for a QArray
 *	Parameters:
 *	  QArray	this - the object
 *	  int		idx - the index of the element to retrieve
 *	Return Value:
 *	  static float	retrieve - the quantized value
 *********************************************************************/
static float	retrieve(this, idx)
  QArray	this ;
  int		idx ;
{
  QAStorage 	qValue ;

  if (this == NULL)
    return 0.0 ;

  if (idx < 0 || idx >= this->elements)
    return 0.0 ;

  if (this->bits) {
    qValue = getBits((QAStorage *)this->array.storage, this->bits, idx) ;
    return this->min + (this->max - this->min) * qValue / MAX_NUM(this->bits) ;
  } else {
    return this->array.real[idx] ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		getBits
 *	Description:	gets the bits representing a quantized value
 *			from a bit array.
 *	Parameters:
 *	  QAStorage	*array - the array
 *	  int		elementSize - the number of bits per element
 *	  int		index - the index of the desired element
 *	Return Value:
 *	  static QAStorage	getBits - the bits representing the value
 *********************************************************************/
static QAStorage	getBits(array, elementSize, index)
  QAStorage	*array ;
  int		elementSize ;
  int		index ;
{
  int		byteOffset = elementSize * index / NUM_BITS(QAStorage) ;
  int		bitOffset  = elementSize * index % NUM_BITS(QAStorage) ;
  QAStorage	value	   = 0 ;
  int		idx, minBit, maxBit, shift ;

  if (elementSize > NUM_BITS(QAStorage))
    return (QAStorage)0 ;
  
  minBit = bitOffset ;
  maxBit = MIN(bitOffset + elementSize, NUM_BITS(QAStorage)) ;
  shift  = NUM_BITS(QAStorage) - minBit - elementSize ;
  for (idx = minBit ; idx < maxBit ; ++idx)
    value |= SHIFTED_BIT(array[byteOffset], idx, shift) ;

  minBit = 0 ;
  maxBit = MAX(bitOffset + elementSize - NUM_BITS(QAStorage), 0) ;
  shift  = NUM_BITS(QAStorage) - maxBit ;
  for (idx = minBit ; idx < maxBit ; ++idx)
    value |= SHIFTED_BIT(array[byteOffset + 1], idx, shift) ;

  return value ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setBits
 *	Description:	sets the bits in a bit array
 *	Parameters:
 *	  QAStorage	*array - the array
 *	  int		elementSize - the number of bits per element
 *	  int		index - the index of the element to set
 *	  unsigned int	bitsIn - the bits to store
 *	Return Value:
 *	  static QAStorage	setBits - the bits (stored)
 *********************************************************************/
static QAStorage	setBits(array, elementSize, index, bitsIn)
  QAStorage	*array ;
  int		elementSize ;
  int		index ;
  unsigned int	bitsIn ;
{
  int		byteOffset = elementSize * index / NUM_BITS(QAStorage) ;
  int		bitOffset  = elementSize * index % NUM_BITS(QAStorage) ;
  QAStorage	value	   = (QAStorage)bitsIn ;
  int		idx, minBit, maxBit, shift ;

  if (elementSize > NUM_BITS(QAStorage))
    return (QAStorage)0 ;
  
  minBit = NUM_BITS(QAStorage) - elementSize ;
  maxBit = bitOffset + elementSize <= NUM_BITS(QAStorage)
    ? NUM_BITS(QAStorage) : minBit + NUM_BITS(QAStorage) - bitOffset ;
  shift  = bitOffset - minBit ;
  for (idx = minBit ; idx < maxBit ; ++idx) {
    array[byteOffset] &= ~SHIFTED_BIT(ALL_BITS(QAStorage), idx, shift) ;
    array[byteOffset] |= SHIFTED_BIT(value, idx, shift) ;
  }

  minBit = maxBit ;
  maxBit = NUM_BITS(QAStorage) ;
  shift  = -minBit ;
  for (idx = minBit ; idx < maxBit ; ++idx) {
    array[byteOffset + 1] &= ~SHIFTED_BIT(ALL_BITS(QAStorage), idx, shift) ;
    array[byteOffset + 1] |=  SHIFTED_BIT(value, idx, shift) ;
  }
  return value ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		copyQArray
 *	Description:	procedure for copying one QArray to another
 *	Parameters:
 *	  const	QArray	array1 - the array to copy from
 *	  QArray	array2 - the array to copy to
 *	Return Value:
 *	  QArray	copyQArray - the array copied to
 *********************************************************************/
QArray		copyQArray(array1, array2)
  const	QArray	array1 ;
  QArray	array2 ;
{
  int		idx, elements ; ;
  
  if (MQAelements(array1) != MQAelements(array2))
    return NULL ;

  elements = MQAelements(array2) ;
  for (idx = 0 ; idx < elements ; ++idx) 
    MQAstore(array2, idx, MQAretrieve(array1, idx)) ;

  return array2 ;
}
/********************************************************************/
