
/**********************************************************************
 * $Id: mask.c,v 1.4 92/11/30 13:10:48 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "mask.h"

static int 	compare         ARGS((const void	*p1, const void	*p2)) ;
static int 	compareAnyClass ARGS((const void	*p1, const void	*p2)) ;
static int 	compareMask     ARGS((const void	*p1, const void	*p2)) ;
static int 	compareMaskAnyClass ARGS((const void *p1, const void *p2)) ;

static void	initMasks ARGS((void)) ;

typedef struct MaskRec {
  String	class ;
  String	name ;
  Mask		mask ;
} MaskRec ;

static MaskRec		*record ;
static unsigned int	numRecords = 0 ;
static Boolean		initialized = FALSE ;


/***********************************************************************
 *	Name:		registerClassMask
 *	Description:	registers an integer mask and a corresponding
 *			string to find it by. YOU SHOULD NOT REGISTER 0.
 *	Parameters:	
 *		Mask		newMask - the mask value to register
 *		const String	name - the name to associate with the mask
 *		const String	class - the class to register the mask with
 *					(used to narrow down the masks)
 *	Return Value:	
 *		Mask		registerClassMask - the value of the mask, 0
 *					if the name is already registered
 ***********************************************************************/
Mask		registerClassMask (newMask, class, name)
  Mask		newMask ;
  const String	class ;
  const String	name ;
{
  if (findClassMask(class, name) != 0)
    return 0 ;
  
  if (record == NULL) {
    record = (MaskRec *)malloc((numRecords+1)*sizeof(MaskRec)) ;
  } else {
    record = (MaskRec *)realloc(record, (numRecords+1)*sizeof(MaskRec)) ;
  }
  if (record == NULL) {
    simError(SIM_ERR_MALLOC, "registerMask \"%s\"", name) ;
    return 0 ;
  }

  record[numRecords].class = strdup(class) ;
  record[numRecords].name  = strdup(name) ;
  record[numRecords].mask  = newMask ;
  numRecords++ ;

  qsort((void *)record, numRecords, sizeof(MaskRec), compare) ;

  return findClassMask(class, name) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		findClassMask
 *	Description:	given a string, this returns the mask associated
 *			with it (as registered with register mask)
 *	Parameters:	
 *		const String	class - the class of the mask to find
 *		const String	name - the name of the mask to find
 *	Return Value:	
 *		Mask		findClassMask - the associated mask, 0 if
 *					no match is found
 ***********************************************************************/
Mask		findClassMask(class, name)
  const	String	class ;
  const	String	name ;
{
  MaskRec     	*thisRecord ;
  MaskRec	find ;
  Mask		mask ;

  if (!initialized)
    initMasks() ;

  if (record == NULL)
    return 0 ;

  find.class = strdup(class) ;
  find.name  = strdup(name) ;
  find.mask  = 0 ;

  if (strcmp(class, UNKNOWN_CLASS)) {
    thisRecord = (MaskRec *)bsearch((void *)&find, (void *)record, 
				    numRecords, sizeof(MaskRec), compare) ;
    if (thisRecord == NULL) {
      free(find.class) ;
      find.class = strdup(UNKNOWN_CLASS) ;
      thisRecord = (MaskRec *)bsearch((void *)&find, (void *)record, 
				      numRecords, sizeof(MaskRec), compare) ;
    }
  } else {
    thisRecord = (MaskRec *)lfind((void *)&find, (void *)record, 
				  &numRecords, sizeof(MaskRec), 
				  compareAnyClass) ;
  }
  free(find.class) ;
  free(find.name) ;

  if (thisRecord != NULL)
    mask = thisRecord->mask ;
  else
    mask = strtol(name, NULL, 0) ;

  return mask ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		classMaskName
 *	Description:	returns the name of a mask
 *	Parameters:
 *	  const String	class - the class of the mask (UNKNOWN_CLASS
 *				if you don't care.
 *	  Mask		mask -  the mask
 *	Return Value:
 *	  String	classMaskName - the name (good only till next
 *				invocation).
 *********************************************************************/
String	classMaskName(class, mask)
  const String	class ;
  Mask		mask ;
{
  MaskRec     	*thisRecord ;
  MaskRec	find ;
  static char	string[32] ;
  String	name ;

  if (!initialized)
    initMasks() ;

  if (record == NULL)
    return NULL ;

  find.class = strdup(class) ;
  find.mask  = mask ;

  if (strcmp(class, UNKNOWN_CLASS)) {
    thisRecord = (MaskRec *)lfind((void *)&find, (void *)record, 
				  &numRecords, sizeof(MaskRec), 
				  compareMask) ;
    if (thisRecord == NULL) {
      free(find.class) ;
      find.class = strdup(UNKNOWN_CLASS) ;
      thisRecord = (MaskRec *)lfind((void *)&find, (void *)record, 
				    &numRecords, sizeof(MaskRec), 
				    compareMask) ;
    }
  } else {
    thisRecord = (MaskRec *)lfind((void *)&find, (void *)record, 
				  &numRecords, sizeof(MaskRec), 
				  compareMaskAnyClass) ;
  }

  free(find.class) ;

  if (thisRecord != NULL)
    name = thisRecord->name ;
  else {
    sprintf(string, "%0x", mask) ;
    name = string ;
  }

  return name ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		registerMask
 *	Description:	registers an integer mask and a corresponding
 *			string to find it by. YOU SHOULD NOT REGISTER 0.
 *	Parameters:	
 *		int		newMask - the mask value to register
 *		const String	name - the name to associate with the mask
 *	Return Value:	
 *		Mask		registerMask - the value of the mask, 0
 *					if the name is already registered
 ***********************************************************************/
Mask		registerMask (newMask, name)
  Mask		newMask ;
  const String	name ;
{
  return registerClassMask(newMask, UNKNOWN_CLASS, name) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		findMask
 *	Description:	given a string, this returns the mask associated
 *			with it (as registered with register mask)
 *	Parameters:	
 *		const String	name - the name of the mask to find
 *	Return Value:	
 *		Mask		findMask - the associated mask, 0 if
 *					no match is found
 ***********************************************************************/
Mask		findMask(name)
  const String	name ;
{
  return findClassMask(UNKNOWN_CLASS, name) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		maskName
 *	Description:	returns the name of a mask
 *	Parameters:
 *	  Mask		mask -  the mask
 *	Return Value:
 *	  String	classMaskName - the name (good only till next
 *				invocation).
 *********************************************************************/
String	maskName(mask)
  Mask		mask ;
{
  return classMaskName(UNKNOWN_CLASS, mask) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		compare
 *	Description:	compares the names of two record structs
 *	Parameters:	pointers to the two structs
 *	Return Value:	-1, 0, 1
 ***********************************************************************/
static int compare(p1, p2)
  const void	*p1 ;
  const void	*p2 ;
{
  return (strcmp(((MaskRec *)p1)->class, ((MaskRec *)p2)->class) ?
	  strcmp(((MaskRec *)p1)->class, ((MaskRec *)p2)->class)
	  : strcmp(((MaskRec *)p1)->name, ((MaskRec *)p2)->name)) ;
}
/**********************************************************************/
static int compareAnyClass(p1, p2)
  const void	*p1 ;
  const void	*p2 ;
{
  return strcmp(((MaskRec *)p1)->name, ((MaskRec *)p2)->name) ;
}
/**********************************************************************/
static int compareMask(p1, p2)
  const void	*p1 ;
  const void	*p2 ;
{
  return (strcmp(((MaskRec *)p1)->class, ((MaskRec *)p2)->class) ?
	  strcmp(((MaskRec *)p1)->class, ((MaskRec *)p2)->class)
	  : ((MaskRec *)p1)->mask - ((MaskRec *)p2)->mask) ;
}
/**********************************************************************/
static int compareMaskAnyClass(p1, p2)
  const void	*p1 ;
  const void	*p2 ;
{
  return ((MaskRec *)p1)->mask - ((MaskRec *)p2)->mask ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		initMasks
 *	Description:	registers all of the standard masks from mask.h
 *	Parameters:	NONE
 *	Return Value:	NONE
 ***********************************************************************/
static void	initMasks ARGS((void)) {

  initialized = TRUE ;

  registerMask (ALL,		"ALL") ;
  registerMask (UNKNOWN,	"UNKNOWN") ;

  registerClassMask(RECURRENT,	NET_CLASS,	"RECURRENT") ;

  registerClassMask(INCOMING,	LINK_CLASS,	"INCOMING") ;
  registerClassMask(OUTGOING,	LINK_CLASS,	"OUTGOING") ;
  registerClassMask(FROZEN,	LINK_CLASS,	"FROZEN") ;
  registerClassMask(LOG_TRANSFORM, LINK_CLASS,	"LOG_TRANSFORM") ;

  registerClassMask(INPUT,	GROUP_CLASS, "INPUT") ;
  registerClassMask(BIAS,	GROUP_CLASS, "BIAS") ;
  registerClassMask(HIDDEN,	GROUP_CLASS, "HIDDEN") ;
  registerClassMask(OUTPUT,	GROUP_CLASS, "OUTPUT") ;

  registerClassMask(TRAINING,   EXAMPLESET_CLASS, "TRAINING") ;
  registerClassMask(TESTING,    EXAMPLESET_CLASS, "TESTING") ;
  registerClassMask(VALIDATION, EXAMPLESET_CLASS, "VALIDATION") ;
}
/**********************************************************************/
 
