
/**********************************************************************
 * $Id: link.c,v 1.7 92/11/30 13:11:39 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "link.h"

/***********************************************************************
 *	Private functions
 ***********************************************************************/
static void	initialize ARGS((Link	link)) ;
static void	finalize   ARGS((Link	link)) ;

static Link	getLinkMemory ARGS((char *name)) ;
static void	freeLinkMemory ARGS((Link link)) ;
/**********************************************************************/

/***********************************************************************
 *	Private variables
 ***********************************************************************/
static LinkProc	createLinkHook  = NULL ;
static LinkProc	destroyLinkHook = NULL ;
/**********************************************************************/


/***********************************************************************
 *	Name:		setCreateLinkHook  (setDestroyLinkHook)
 *	Description:	sets the user hook to be called after (before)
 *			a link is created (destroyed).
 *	Parameters:	
 *		LinkProc	hook - the procedure to be called:
 *				void hook (Link	link)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	setCreateLinkHook(hook)
  LinkProc	hook ;
{
  createLinkHook  = hook ;
}
/**********************************************************************/
void	setDestroyLinkHook(hook)
  LinkProc	hook ;
{
  destroyLinkHook = hook ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		createLink 
 *	Description:	creates a link and initializes all of its data
 *	Parameters:	
 *		char	*name - the name for the link
 *	Return Value:	
 *		Link	createLink - the new link
 ***********************************************************************/
Link	createLink (name, preUnit, postUnit, mask)
  String	name ;
  Unit		preUnit ;
  Unit		postUnit ;
  Mask		mask ;
{
  Link	link ;

  link = getLinkMemory(name) ;
  if (link == NULL) {
    simError(SIM_ERR_MALLOC, "createLink \"%s\"", name) ;
    return link ;
  }

  if (!(mask & ALL))
    mask = UNKNOWN ;
  link->type = mask ;

  if (preUnit && postUnit && preUnit->net == postUnit->net) {
    link->preUnit  = preUnit ;
    link->postUnit = postUnit ;
  } else {
    link->preUnit  = NULL ;
    link->postUnit = NULL ;
  }

  initialize (link) ;

  if (createLinkHook != NULL)
    (*createLinkHook) (link) ;

  return link ;
}
/***********************************************************************
 *	Name:		initialize
 *	Description:	init proc for the Link class of objects
 *		Link	link - the link object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	initialize (link) 
  Link		link ;
{
  link->weight      = 0.0 ;
  link->deltaWeight = 0.0 ;
  link->deriv       = 0.0 ;

  link->variableIdx = -1.0 ;
  link->scaleFactor = 1.0 ;

  link->weightRow[0]	= -1 ;
  link->weightRow[1]	= -1 ;
  link->weightColumn[0]	= -1 ;
  link->weightColumn[1]	= -1 ;

  link->extension = NULL ;

  if (link->preUnit && link->postUnit) {
    unitAddLink (link->preUnit,  OUTGOING, link) ;
    unitAddLink (link->postUnit, INCOMING, link) ;

    if (link->preUnit->net != NULL)
      netAddLink(link->preUnit->net, link) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		destroyLink 
 *	Description:	destroys a link
 *	Parameters:	
 *		Link	link - the link to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	destroyLink (link) 
  Link		link ;
{
  if (destroyLinkHook != NULL)
    (*destroyLinkHook)(link) ;

  finalize(link) ;

  freeLinkMemory(link) ;
}
/***********************************************************************
 *	Name:		finalize
 *	Description:	finalize proc for the link class of objects.
 *			It removes the link from the units it is connected
 *			to.
 *		Link	link - the link object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	finalize (link) 
  Link		link ;
{
  if (link->preUnit  != NULL)
    unitDeleteLink(link->preUnit,  OUTGOING, link) ;
  if (link->postUnit != NULL)
    unitDeleteLink(link->postUnit, INCOMING, link) ;

  if (link->preUnit != NULL && link->preUnit->net != NULL)
    netDeleteLink(link->preUnit->net, link) ;

  return ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		linkFromName 
 *	Description:	finds the link with the given name
 *	Parameters:	
 *		Net	net  - the net to search through
 *		String	name - the name of the required link
 *	Return Value:	
 *		Link	linkFromName - the FIRST link with name "name",
 *				NULL if it does not exist.
 ***********************************************************************/
Link	linkFromName (net, name)
  Net		net ;
  String	name ;
{
  Link	*linkArray = net->links ;
  int	numLinks   = net->numLinks ;
  Link	link       = NULL ;
  int	idx ;

  for (idx = 0 ; idx < numLinks ; ++idx) {
    if (strcmp(linkArray[idx]->name, name) == 0) {
      link = linkArray[idx] ;
      break ;
    }
  }
  return link ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		linksFromRegex 
 *	Description:	finds the links whose names match the given
 *			regular expression
 *	Parameters:	
 *		Net	net   - the network to search through
 *		String	expression - the regular expression to match
 *	Return Value:	
 *		Link	*linksFromRegex - an array of links whose names
 *				match the regular expression. The final
 *				element in the array is NULL.
 *				The user is responsible for freeing it.
 *				NULL is returned if the regex is bad.
 ***********************************************************************/
Link	*linksFromRegex (net, expression)
  Net		net ;
  String	expression ;
{
  Link	*linkArray = net->links ;
  int	numLinks   = net->numLinks ;
  char	buffer[BUFSIZ], *compiled ;
  Link	*matchedLink ;
  int	numMatches ;
  int	idx ;

  if (strpbrk(expression, "$^") != NULL)
    return NULL ;

  sprintf(buffer, "^%s$", expression) ;
  compiled = regcmp(buffer, NULL) ;
  if (compiled == NULL)
    return NULL ;

  numMatches  = 0 ;
  matchedLink = (Link *)malloc(sizeof(Link)) ;
  if (matchedLink == NULL) {
    simError(SIM_ERR_MALLOC, "linksFromRegex") ;
    return matchedLink ;
  }
  matchedLink[0] = NULL ;

  for (idx = 0 ; idx < numLinks ; ++idx) {
    if (regex(compiled, linkArray[idx]->name) != NULL) {
      matchedLink[numMatches] = linkArray[idx] ;
      ++numMatches ;
      matchedLink = (Link *)realloc(matchedLink, 
				    (numMatches + 1)*sizeof(Link)) ;
      if (matchedLink == NULL) {
	simError(SIM_ERR_MALLOC, "linksFromRegex") ;
	break ;
      }
      matchedLink[numMatches] = NULL ;
    }
  }

  free(compiled) ;
  return matchedLink ;
}
/**********************************************************************/


/*********************************************************************
 *	Description:
 *	These data structures are used by getLinkMemory and
 *	freeLinkMemory to handle the management of the memory for
 *	links.
 *		
 *********************************************************************/
#ifndef MAX_LINK_NAME
#define MAX_LINK_NAME	32
#endif

#ifndef GRANULARITY
#define GRANULARITY 512
#endif

typedef struct LinkMemory {
  LinkRec	link ;
  char		name[MAX_LINK_NAME] ;
  int		tableIdx ;
  Boolean	free ;
} LinkMemory ;

static LinkMemory	**linkMemoryTable ;
static Boolean		*tableHasFree ;
static int		numMemoryTables ;
/**********************************************************************/


/*********************************************************************
 *	Name:		getLinkMemory
 *	Description:	Utility to manage memroy for links. This is 
 *			necessary because the number of links precludes
 *			calling malloc for each of them (don't believe
 *			me? Try replacing these with malloc and free
 *			and try to build a network with 100000 
 *			connections.)
 *	Parameters:
 *	  char		*name - the name to give the link 
 *	Return Value:
 *	  static Link	getLinkMemory - a pointer to the memory
 *			for the link. DO NOT FREE IT. RELEASE IT
 *			WITH freeLinkMemory.
 *********************************************************************/
static Link	getLinkMemory(name)
  char		*name ;
{
  Link	link ;
  int	tableIdx, linkIdx, idx ;

  for (tableIdx = 0 ; tableIdx < numMemoryTables ; ++tableIdx) {
    if (tableHasFree[tableIdx]) {
      for (linkIdx = 0 ; linkIdx < GRANULARITY ; ++linkIdx) {
	if (linkMemoryTable[tableIdx][linkIdx].free)
	  break ;
      } 
      if (linkIdx == GRANULARITY)
	tableHasFree[tableIdx] = FALSE ;
      else
	break ;
    }
  }

  if (tableIdx == numMemoryTables) {
    ++numMemoryTables ;
    if (numMemoryTables == 1) {
      linkMemoryTable
	= (LinkMemory **)malloc(numMemoryTables*sizeof(LinkMemory *)) ;
      tableHasFree = (Boolean *)malloc(numMemoryTables*sizeof(Boolean)) ;
    } else {
      linkMemoryTable
	= (LinkMemory **)realloc(linkMemoryTable,
				 numMemoryTables*sizeof(LinkMemory *)) ;
      tableHasFree = (Boolean *)realloc(tableHasFree,
					numMemoryTables*sizeof(Boolean)) ;
    }
    
    linkMemoryTable[tableIdx] 
      = (LinkMemory *)malloc(GRANULARITY*sizeof(LinkMemory));
    tableHasFree[tableIdx] = TRUE ;

    for (idx = 0 ; idx < GRANULARITY ; ++idx) {
      linkMemoryTable[tableIdx][idx].tableIdx = tableIdx ;
      linkMemoryTable[tableIdx][idx].free     = TRUE ;
    }
    linkIdx = 0 ;
  }

  link = &linkMemoryTable[tableIdx][linkIdx].link ;
  if (name == NULL) {
    link->name = NULL ;
  } else {
    link->name = linkMemoryTable[tableIdx][linkIdx].name ;
    strncpy(link->name, name, MAX_LINK_NAME) ;
    link->name[MAX_LINK_NAME-1] = '\0' ;
  }

  linkMemoryTable[tableIdx][linkIdx].free = FALSE ;

  return link ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		freeLinkMemory
 *	Description:	releases memory used for a link and allows it
 *			to be used later
 *	Parameters:
 *	  Link		link - the link to free. MUST HAVE BEEN OBTAINED
 *			WITH getLinkMemory
 *	Return Value:
 *	  static void	freeLinkMemory - 
 *********************************************************************/
static void	freeLinkMemory(link)
  Link		link ;
{
  LinkMemory	*memory = (LinkMemory *)link ;

  memory->free = TRUE ;
  tableHasFree[memory->tableIdx] = TRUE ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		linkSetWeight
 *	Description:	method for setting a link's weight. It makes
 *			sure that all weights/vectors are synchronized
 *	Parameters:
 *	  Link		link - the link to set
 *	  Real		weight - the desired weight
 *	Return Value:
 *	  Real		linkSetWeight - the weight it is set to
 *********************************************************************/
Real		linkSetWeight(link, weight)
  Link		link ;
  double	weight ;
{
  Net	net ;

  if (link->preUnit)
    net = link->preUnit->net ;
  else if (link->postUnit)
    net = link->postUnit->net ;
  else
    net = NULL ;
  
  if (net) {
    switch (link->type & LOG_TRANSFORM) {
    case LOG_TRANSFORM:
      if (weight/link->scaleFactor > 0)
	net->variables[link->variableIdx] = log(weight/link->scaleFactor) ;
      else
	net->variables[link->variableIdx] = 0 ;
      break ;
    default:
      net->variables[link->variableIdx] = weight/link->scaleFactor ;
      break ;
    }
    syncValues(net, WeightsFromVector) ;
  } else {
    link->weight = weight ;
  }

  return link->weight ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		createVariable 
 *	Description:	creates a link and initializes all of its data
 *	Parameters:	
 *		char	*name - the name for the link
 *	Return Value:	
 *		Link	createLink - the new link
 ***********************************************************************/
Variable	createVariable (name, net, mask)
  String	name ;
  Net		net ;
  Mask		mask ;
{
  Variable	variable = createLink(name, NULL, NULL, mask) ;

  if (net && variable)
    netAddLink(net, variable) ;

  return variable ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		destroyVariable 
 *	Description:	destroys a variable
 *	Parameters:	
 *		Variable	variable - the variable to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	destroyVariable(variable, net) 
  Variable	variable ;
  Net		net ;
{
  if (net)
    netDeleteLink(net, variable) ;

  destroyLink(variable) ;
}
/**********************************************************************/
