
/**********************************************************************
 * $Id: group.c,v 1.5 92/11/30 11:31:45 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "group.h"
#include "traverse.h"

/***********************************************************************
 *	Private procedures
 ***********************************************************************/
static void	initialize ARGS((Group	group)) ;
static void	finalize   ARGS((Group	group)) ;

static void	groupActivityUpdate ARGS((Group	group)) ;
static void	groupGradientUpdate ARGS((Group	group)) ;

static void	checkGroupName  ARGS((Group	group, void	*data)) ;
static void	checkGroupRegex ARGS((Group	group, void	*data)) ;
/**********************************************************************/


/***********************************************************************
 *	Private variables
 ***********************************************************************/
static GroupProc	createGroupHook = NULL ;
static GroupProc	destroyGroupHook = NULL ;
/**********************************************************************/


/***********************************************************************
 *	Name:		setCreateGroupHook  (setDestroyGroupHook)
 *	Description:	sets the user hook to be called after (before)
 *			a group is created (destroyed).
 *	Parameters:	
 *		GroupProc	hook - the procedure to be called:
 *				void hook (Group	group)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void		setCreateGroupHook(hook)
  GroupProc	hook ;
{
  createGroupHook = hook ;
}
/**********************************************************************/
void		setDestroyGroupHook(hook)
  GroupProc	hook ;
{
  destroyGroupHook = hook ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		createGroup 
 *	Description:	creates a group and initializes all of its data
 *			as well as inserting it into a network
 *	Parameters:	
 *		char	*name - the name for the group
 *		Mask	mask  - a mask identifying the type of group
 *				eg. INPUT, HIDDEN
 *		Net	net   - the net to insert the group in
 *	Return Value:	
 *		Group	createGroup - the new group
 ***********************************************************************/
Group	createGroup (name, mask, net)
  String	name ;
  Mask		mask ;
  Net		net ;
{
  Group	group ;

  group = (Group)malloc(sizeof(GroupRec)) ;

  if (group == NULL) {
    simError(SIM_ERR_MALLOC, "createGroup \"%s\"", name) ;
    return group ;
  }

  group->name = strdup(name) ;
  if (!(mask & ALL))
    mask = UNKNOWN ;
  group->type = mask ;
  group->net  = net ;

  initialize(group) ;

  if (createGroupHook != NULL)
    (*createGroupHook)(group) ;

  if (net != NULL)
    netAddGroup(net, group) ;

  return group ;
}
/***********************************************************************
 *	Name:		initialize
 *	Description:	init proc for the Group class of objects
 *		Group	group - the group object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	initialize (group) 
  Group		group ;
{
  group->groupActivityUpdateProc = groupActivityUpdate ;
  group->groupGradientUpdateProc = groupGradientUpdate ;
  group->groupOtherUpdateProc    = NULL ;

  group->unitActivityUpdateProc = NULL ;
  group->unitGradientUpdateProc = NULL ;
  group->unitOtherUpdateProc    = NULL ;

  group->unit          = NULL ;
  group->numUnits      = 0 ;
  group->unitArraySize = 0 ;

  McostModel(group) = NULL ;

  group->extension     = NULL ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		destroyGroup 
 *	Description:	destroys a group
 *	Parameters:	
 *		Group	group - the group to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	destroyGroup (group)
  Group	group ;
{
  if (group->net != NULL)
    netDeleteGroup(group->net, group) ;

  if (destroyGroupHook != NULL)
    (*destroyGroupHook)(group) ;

  finalize(group) ;

  free(group->name) ;
  free(group) ;
}
/***********************************************************************
 *	Name:		finalize
 *	Description:	finalize proc for the group class of objects
 *		Group	group - the group object ;
 *		void	*data - client data, unused
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	finalize (group)
  Group		group ;
{
  int	idx ;

  for (idx = group->numUnits - 1 ; idx >= 0 ; --idx)
    destroyUnit(group->unit[idx]) ;

  if (group->unit != NULL)
    free (group->unit) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupAddUnit
 *	Description:	class procedure that adds a unit to a group.
 *	Parameters:	
 *		Group	group - the group to add the units to
 *		Unit	unit  - the unit to add
 *	Return Value:	
 *		NONE
 ***********************************************************************/
#ifndef GRANULARITY
#define GRANULARITY 32
#endif
int	groupAddUnit (group, unit)
  Group	group ;
  Unit	unit ;
{

  if (group->numUnits >= group->unitArraySize) {
    Unit	**arrayPtr = &group->unit ;
    int		*numPtr    = &group->unitArraySize ;
  
    if (*numPtr == 0) {
      *numPtr   = group->numUnits + GRANULARITY ;
      *arrayPtr = (Unit *)malloc ((*numPtr)*sizeof(Unit)) ;
    } else {
      *numPtr   = group->numUnits + GRANULARITY ;
      *arrayPtr = (Unit *)realloc (*arrayPtr, (*numPtr)*sizeof(Unit)) ;
    }
    if (*arrayPtr == NULL) {
      simError(SIM_ERR_MALLOC, "groupAddUnit group: \"%s\", unit: \"%s\"", 
	       group->name, unit->name) ;
      return 0 ;
    }
  }

  group->unit[group->numUnits] = unit ;
  ++(group->numUnits) ;
  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupDeleteUnit
 *	Description:	deletes a unit from a group
 *	Parameters:	
 *		Group	group - the group to delete the unit from
 *		Unit	unit  - the unit to delete
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	groupDeleteUnit (group, unit)
  Group	group ;
  Unit	unit ;
{
  int	idx ;

  for (idx = 0 ; idx < group->numUnits && group->unit[idx] != unit ; ++idx)
    ;

  if (group->unit != NULL && group->unit[idx] == unit) {
    for ( ; idx < group->numUnits ; ++idx)
      group->unit[idx] = group->unit[idx + 1] ;
    --(group->numUnits) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupActivityUpdate
 *	Description:	calls the unitActivityUpdate for all units in the
 *			group. Returns if the proc is NULL
 *		Group	group - the group object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	groupActivityUpdate (group)
  Group		group ;
{
  if (group->unitActivityUpdateProc != NULL) 
    groupForAllUnits(group, 
		     (UnitDataProc)(group->unitActivityUpdateProc), NULL) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupGradientUpdate
 *	Description:	calls the unitGradientUpdate for all units in the
 *			group. Returns if the proc is NULL
 *		Group	group - the group object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	groupGradientUpdate (group)
  Group		group ;
{
  if (group->unitGradientUpdateProc != NULL) 
    groupForAllUnitsBack(group, 
			 (UnitDataProc)(group->unitGradientUpdateProc), NULL) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupFromName 
 *	Description:	finds the group with the given name
 *	Parameters:	
 *		Net	net  - the network to search through
 *		String	name - the name of the required group
 *	Return Value:	
 *		Group	groupFromName - the FIRST group with name "name",
 *				NULL if it does not exist.
 ***********************************************************************/
static Group	thisGroup ;
/**********************************************************************/
Group		groupFromName (net, name)
  Net		net ;
  String	name ;
{
  thisGroup = NULL ;
  netForAllGroupsBack(net, ALL, checkGroupName, (void*)name) ;
  return thisGroup ;
}
/**********************************************************************/
static void	checkGroupName (group, data)
  Group		group ;
  void		*data ;
{
  if (strcmp(group->name, (char *)data) == 0)
    thisGroup = group ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		groupsFromRegex 
 *	Description:	finds the groups whose names match the given
 *			regular expression
 *	Parameters:	
 *		Net	net   - the network to search through
 *		String	expression - the regular expression to match
 *	Return Value:	
 *		Group	*groupsFromRegex - an array of groups whose names
 *				match the regular expression. The final
 *				element in the array is NULL.
 *				The user is responsible for freeing it.
 *				NULL is returned if the regex is bad.
 ***********************************************************************/
static Group	*matchedGroup ;
static int	numMatches ;
/**********************************************************************/
Group		*groupsFromRegex (net, expression)
  Net		net ;
  String	expression ;
{
  char	buffer[BUFSIZ], *compiled ;

  if (strpbrk(expression, "$^") != NULL)
    return NULL ;

  sprintf(buffer, "^%s$", expression) ;
  compiled = regcmp(buffer, NULL) ;
  if (compiled == NULL)
    return NULL ;

  numMatches   = 0 ;
  matchedGroup = (Group *)malloc(sizeof(Group)) ;
  if (matchedGroup == NULL) {
    simError(SIM_ERR_MALLOC, "groupsFromRegex") ;
    return matchedGroup ;
  }
  matchedGroup[0] = NULL ;
  netForAllGroupsBack(net, ALL, checkGroupRegex, (void*)compiled) ;

  free(compiled) ;
  return matchedGroup ;
}
/**********************************************************************/
static void	checkGroupRegex (group, data)
  Group		group ;
  void		*data ;
{
  if (regex((char *)data, group->name) != NULL) {
    matchedGroup[numMatches] = group ;
    ++numMatches ;
    matchedGroup = (Group *)realloc(matchedGroup, 
				    (numMatches + 1)*sizeof(Group)) ;
    if (matchedGroup == NULL) {
      simError(SIM_ERR_MALLOC, "groupsFromRegex") ;
    } else {
      matchedGroup[numMatches] = NULL ;
    }
  }
}
/**********************************************************************/
