/**********************************************************************
 * $Id: gaussian.c,v 1.5 93/02/09 16:44:50 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <xerion/useful.h>

#include "gaussian.h"
#include "sigmoid.h"

#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif
#ifndef M_SQR
#define M_SQR(x)	((x)*(x))
#endif

/********************************************************************/
static Real	setMean	        ARGS((Gaussian, double mean)) ;
static Real	setVariance     ARGS((Gaussian, double variance)) ;
static Real	setProportion	ARGS((Gaussian, double proportion)) ;
static Real	gaussianPdf	ARGS((Gaussian, double value)) ;
static Real	gaussianProb	ARGS((Gaussian, double x, double delta)) ;
static Real	gaussianSample	ARGS((Gaussian)) ;
static Real	responsibility  ARGS((Gaussian, double y)) ;
static void	destroyGaussian ARGS((Gaussian)) ;
/********************************************************************/
static	Real	gaussianConstant = 0.0 ;
/********************************************************************/

/*********************************************************************
 *	Name:		createGaussian/destroyGaussian
 *	Description:	creates (destroys) a gaussian distribution object
 *	Parameters:
 *	  Mixture	mixture	 - the mixture to add the gaussian to
 *	  Real		mean	 - the mean of the distribution
 *	  Real		variance - the variance
 *	  Gaussian	gaussian - the gaussian to destroy
 *	Return Value:
 *	  Gaussian	createGaussian  - the new gaussian
 *	  void		destroyGaussian - NONE
 *********************************************************************/
Gaussian	createGaussian(mixture, mean, variance)
  Mixture	mixture ;
  double	mean ;
  double	variance ;
{
  Gaussian	this ;
  
  this = (Gaussian)malloc(sizeof(GaussianRec)) ;
  if (this == NULL)
    return NULL ;

  this->mixture	 = mixture ;

  this->setMean		  = setMean ;
  this->setVariance	  = setVariance ;
  this->setProportion     = setProportion ;
  this->getDensity 	  = gaussianPdf ;
  this->getProbability    = gaussianProb ;
  this->getSample	  = gaussianSample ;
  this->getResponsibility = responsibility ;

  this->destroy		  = destroyGaussian ;

  if (gaussianConstant == 0.0)
    gaussianConstant = 1.0/sqrt(2.0*M_PI) ;

  MGsetMean(this, mean) ;
  MGsetVariance(this, variance) ;
  MGsetProportion(this, 0.0) ;

  return this ;
}
/********************************************************************/
static void	destroyGaussian(this)
  Gaussian	this ;
{
  free((void *)this) ;
}
/********************************************************************/


/********************************************************************/
static Real	setMean(this, mean)
  Gaussian	this ;
  double	mean ;
{
  this->mean    = mean ;
  this->changed = TRUE ;
  if (this->mixture)
    this->mixture->changed = TRUE ;
  return MGmean(this) ;
}
/********************************************************************/
static Real	setVariance(this, variance)
  Gaussian	this ;
  double	variance ;
{
  this->variance     = variance ;
  this->stdDeviation = sqrt(variance) ;
  this->changed	     = TRUE ;
  if (this->mixture)
    this->mixture->changed = TRUE ;
  return MGvariance(this) ;
}
/********************************************************************/
static Real	setProportion(this, proportion)
  Gaussian	this ;
  double	proportion ;
{
  this->proportion = proportion ;
  if (this->mixture)
    this->mixture->changed = TRUE ;
  return MGproportion(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gaussianPdf
 *	Description:	calculates the density density for a gaussian
 *			distribution for a given value
 *	Parameters:
 *	  Gaussian	this - the distribution
 *	  double	y	 - the value to calculate it for
 *	Return Value:
 *	  static Real	gaussianPdf - the density
 *********************************************************************/
static Real	gaussianPdf(this, y)
  Gaussian	this ;
  double	y ;
{
  if (this->changed || y != this->y) {
    Real	error 	 = y - MGmean(this) ;
    Real	exponent = -M_SQR(error)/(2.0*MGvariance(this)) ;
    
    this->y	  = y ;
    this->changed = FALSE ;
    this->density = gaussianConstant*exp(exponent)/MGstdDeviation(this) ;
  }
  return this->density ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gaussianProb
 *	Description: 	Uses a sigmoid to approximate the integral of
 *			a gaussian between (x - delta) and (x + delta)
 *			ONLY VALID FOR SMALL DELTA.
 *			
 *	Parameters:
 *	  Gaussian	this - the distribution
 *	  double	x	 - the center of the sample
 *	  double	delta	 - half the *small* range for integration.
 *	Return Value:
 *	  static Real	gaussianProb - the probability
 *********************************************************************/
/* A magic factor to make a sigmoid approximate an error function */
#define SIGMOID_K	(1.88)
/********************************************************************/
static Real	gaussianProb(this, x, delta)
  Gaussian	this ;
  double	x ;
  double	delta ;
{
  static Sigmoid sigmoid = NULL ;

  if (sigmoid == NULL)
    sigmoid = createSigmoid(0.0, 0.0, SIGMOID_K, 1.0) ;

  x -= MGmean(this) ;

  return (MSsigmoid(sigmoid, MGstdDeviation(this)*(x + delta))
	  - MSsigmoid(sigmoid, MGstdDeviation(this)*(x - delta))) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gaussianSample
 *	Description:	generates a sample from a gaussian distribution
 *	Parameters:
 *	  Gaussian	this - the distribution
 *	Return Value:
 *	  static Real	gaussianSample - the sample
 *********************************************************************/
#ifndef MAXINT
#define MAXINT	(~(int)(1 << sizeof(int)*8 - 1))
#endif
/********************************************************************/
static Real	gaussianSample(this)
  Gaussian	this ;
{
  static Boolean	iset = FALSE ;
  static Real		gset ;
  Real			sample ;

  if  (iset) {
    sample = MGmean(this) + gset*MGstdDeviation(this) ;
    iset   = FALSE ;
  } else {
    Real	fac, r, v1, v2;
    do {
      v1 = -1.0 + 2.0*simRand() ;
      v2 = -1.0 + 2.0*simRand() ;
      r  = v1*v1 + v2*v2;
    } while (r >= 1.0);

    fac = sqrt(-2.0*log(r)/r);

    sample = MGmean(this) + v2*fac*MGstdDeviation(this) ;
    gset   = v1*fac ;
    iset   = TRUE ;
  }

  return sample ;
}
/********************************************************************/


/********************************************************************/
static Real	responsibility(this, y)
  Gaussian	this ;
  double	y ;
{
  if (this->mixture)
    return (MGproportion(this) * MGdensity(this, y)
	    / MMdensity(this->mixture, y)) ;
  else
    return 1.0 ;
}
/********************************************************************/




/********************************************************************/
static Real	mixturePdf	ARGS((Mixture, double y)) ;
static Real	mixtureProb	ARGS((Mixture, double x, double delta)) ;
static Real	complexity	ARGS((Mixture, double y)) ;
static Real	derivatives	ARGS((Mixture, double y,
				      Real *dCdM, Real *dCdS, Real *dCdP)) ;
static void	destroyMixture	ARGS((Mixture)) ;
/********************************************************************/


/*********************************************************************
 *	Name:		createMixture/destroyMixture
 *	Description:	creates (destroys) a mixture distribution object
 *	Parameters:
 *	  int		numGaussians - the number of gaussians in the
 *			  mixture
 *	  double	min - the minimum value for the range the 
 *			  gaussians will be spread out on.
 *	  double	max - the maximum value.
 *	Return Value:
 *	  Mixture	createMixture  - the new mixture
 *	  void		destroyMixture - NONE
 *********************************************************************/
Mixture	createMixture(numGaussians, min, max)
  int		numGaussians ;
  double	min ;
  double	max ;
{
  Mixture	this ;
  Real		interval, mean ;
  int		idx ;
  
  this = (Mixture)malloc(sizeof(MixtureRec)) ;
  if (this == NULL)
    return NULL ;

  this->getDensity	= mixturePdf ;
  this->getProbability	= mixtureProb ;
  this->getSample	= NULL ;
  this->complexity	= complexity ;
  this->derivatives	= derivatives ;
  this->destroy		= destroyMixture ;

  this->numGaussians = 0 ;
  if (numGaussians == 0) {
    this->gaussian = NULL ;
  } else {
    this->gaussian = (Gaussian *)calloc(numGaussians,sizeof(Gaussian));
    
    interval = (max - min)/numGaussians ;
    mean     = min + interval/2.0 ;
    for (idx = 0 ; idx < numGaussians ; ++idx) {
      this->gaussian[idx] = createGaussian(this, mean, sqrt(interval/2.0)) ;
      ++this->numGaussians ;
      MGsetProportion(MMgaussian(this, idx), 1.0/numGaussians) ;
      mean += interval ;
    }
  }
  this->changed = TRUE ;

  return this ;
}
/********************************************************************/
static void	destroyMixture(this)
  Mixture	this ;
{
  int		idx ;

  for (idx = 0 ; idx < this->numGaussians ; ++idx)
    MGdestroy(MMgaussian(this, idx)) ;

  if (this->gaussian)
    free((void *)this->gaussian) ;

  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		mixturePdf
 *	Description:	returns the pdf of the mixture of gaussians
 *			at a particular value.
 *	Parameters:
 *	  Mixture	mixture - the mixture
 *	  double	y       - the value to get the pdf for
 *	Return Value:
 *	  static Real	mixturePdf - the density
 *********************************************************************/
static Real	mixturePdf(this, y)
  Mixture	this ;
  double	y ;
{
  if (this->changed || y != this->y) {
    Real	density	     = 0.0 ;
    int		numGaussians = MMnumGaussians(this) ;
    int		idx ;

    for (idx = 0 ; idx < numGaussians ; ++idx) {
      Gaussian	gaussian = MMgaussian(this, idx) ;
      density += MGproportion(gaussian)*MGdensity(gaussian, y) ;
    }
    this->y	  = y ;
    this->density = density ;
    this->changed = FALSE ;
  }
  return this->density ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		mixtureProb
 *	Description:	returns the prob of the mixture of gaussians
 *			at a particular value.
 *	Parameters:
 *	  Mixture	mixture - the mixture
 *	  double	x - the center of the sample
 *	  double	delta	 - half the *small* range for integration.
 *	Return Value:
 *	  static Real	mixtureProb - the density
 *********************************************************************/
static Real	mixtureProb(this, x, delta)
  Mixture	this ;
  double	x ;
  double	delta ;
{
  int	numGaussians = MMnumGaussians(this) ;
  Real	probability  = 0.0 ;
  int	idx ;

  for (idx = 0 ; idx < numGaussians ; ++idx) {
    Gaussian	gaussian = MMgaussian(this, idx) ;
    probability += MGproportion(gaussian)*MGprobability(gaussian, x, delta) ;
  }

  return probability ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		complexity
 *	Description:	returns the complexity measure for a given 
 *			value (the log density)
 *	Parameters:
 *	  Mixture	this - the mixture of guassians
 *	  double	y 	- the value to use.
 *	Return Value:
 *	  static Real	complexity - the complexity
 *********************************************************************/
static Real	complexity(this, y)
  Mixture	this ;
  double	y ;
{
  return -log(MMdensity(this, y)) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		derivatives
 *	Description:	calculates the partial derivatives of the
 *			complexity function of the mixture of gaussians
 *	Parameters:
 *	  Mixture	this	- the mixture
 *	  Real		y 	- the value to calculate the partials at
 *	  Real		*dCdM	- partials wrt the gaussian means
 *	  Real		*dCdS	- partials wrt the stdDeviations
 *	  Real		*dCdP	- partials wrt the proportions
 *		NOTE: all arrays must already exist and have at least
 *			MnumGaussian(mixture) elements.
 *	Return Value:
 *	  static Real	derivatives - the partial wrt y.
 *********************************************************************/
static Real	derivatives(this, y, dCdM, dCdS, dCdP)
  Mixture	this ;
  double	y ;
  Real		*dCdM ;
  Real		*dCdS ;
  Real		*dCdP ;
{
  int	numGaussians = MMnumGaussians(this) ;
  Real	dCdy ;
  int	idx ;

  /* initialize the derivatives that we have to sum up */
  dCdy = 0.0 ;
  for (idx = 0 ; idx < numGaussians ; ++idx) {
    dCdM[idx] = 0.0 ;
    dCdP[idx] = 0.0 ;
    dCdS[idx] = 0.0 ;
  }

  for (idx = 0 ; idx < numGaussians ; ++idx) {
    Gaussian	gaussian       = MMgaussian(this, idx) ;
    Real	responsibility = MGresponsibility(gaussian, y) ;
    Real	variance       = MGvariance(gaussian) ;
    Real	error	       = MGmean(gaussian) - y ;

    dCdy      -= responsibility*error/variance ;
    dCdM[idx] += responsibility*error/variance ;
    dCdP[idx] -= responsibility/MGproportion(gaussian) ;
    dCdS[idx] -= (responsibility*(M_SQR(error) - variance)
		  /(variance*MGstdDeviation(gaussian))) ;
  }

  return dCdy ;
}
/********************************************************************/
