
/**********************************************************************
 * $Id: example.c,v 1.12 93/03/29 14:12:08 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/simulator.h>
#include "simUtils.h"
#include "example.h"

#ifndef MAX
#define MAX(x,y)	((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x,y)	((x) < (y) ? (x) : (y))
#endif

#define DUPLICATE_INPUT_VECTORS(e,i,j)	\
    (((e)->inputQVector && ((e)->inputQVector[i] == (e)->inputQVector[j]) \
      || (e)->inputVector && ((e)->inputVector[i] == (e)->inputVector[j])) \
     ? TRUE : FALSE)

#define DUPLICATE_TARGET_VECTORS(e,i,j)	\
    (((e)->inputQVector && ((e)->inputQVector[i] == (e)->inputQVector[j]) \
      || (e)->inputVector && ((e)->inputVector[i] == (e)->inputVector[j])) \
     ? TRUE : FALSE)

/***********************************************************************
 *	Private procedures
 ***********************************************************************/
static void	initialize ARGS((ExampleSet	exampleSet)) ;
static void	finalize   ARGS((ExampleSet	exampleSet)) ;

static int	reset      ARGS((ExampleSet	exampleSet)) ;
static int	getNext    ARGS((ExampleSet	exampleSet)) ;
static int	getNumber  ARGS((ExampleSet	exampleSet, int	number)) ;
static void	moveExample	ARGS((ExampleSet, int index, int toIndex)) ;
static void	deleteExample	ARGS((ExampleSet, int index)) ;
static int	addExampleList	ARGS((ExampleSet exampleSet, int listSize, 
				      Real **input, int numInputs, 
				      Real **target, int numTargets)) ;
static int	getExampleList	ARGS((ExampleSet, int index, int *listSize, 
				      Real ***input, int *numInputs, 
				      Real ***target, int *numTargets)) ;

static void	setInput   ARGS((Unit	unit,  void	*data)) ;
static void	setTarget  ARGS((Unit	unit,  void	*data)) ;
static void	sumUnits   ARGS((Group	group, void	*data)) ;
/**********************************************************************/
static QArray	*quantizeVectors ARGS((Real **, int n1, int n2, int bits)) ;
static void	freeOldVector	 ARGS((Real **, int n)) ;
static void	freeOldQVector	 ARGS((QArray *, int n)) ;

/***********************************************************************
 *	Private variables
 ***********************************************************************/
static ExampleSetProc	createExampleSetHook  = NULL ;
static ExampleSetProc	destroyExampleSetHook = NULL ;
static ExampleProc	createExampleHook     = NULL ;
static ExampleProc	destroyExampleHook    = NULL ;
/**********************************************************************/
Real 			buildTime ;
Real 			getTime ;
/**********************************************************************/


/***********************************************************************
 *	Name:		setCreateExampleSetHook
 *	Description:	sets the user hook to be called whenever an exampleSet
 *			is created.
 *	Parameters:	
 *		ExampleSetProc	hook - the procedure to be called:
 *				void hook  (ExampleSet	exampleSet)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	setCreateExampleSetHook(hook)
  ExampleSetProc	hook ;
{
  createExampleSetHook = hook ;
}
/**********************************************************************/
void	setDestroyExampleSetHook(hook)
  ExampleSetProc	hook ;
{
  destroyExampleSetHook = hook ;
}
/**********************************************************************/
void	setCreateExampleHook(hook)
   ExampleProc	hook ;
{
  createExampleHook = hook ;
}
/**********************************************************************/
void	setDestroyExampleHook(hook)
  ExampleProc	hook ;
{
  destroyExampleHook = hook ;
}
 /**********************************************************************/


/***********************************************************************
 *	Name:		createExampleSet 
 *	Description:	creates an exampleSet and initializes all of its data
 *			as well as inserting it into a network
 *	Parameters:	
 *		char	*name - the name for the exampleSet
 *		Net	net   - the net to insert the exampleSet in
 *	Return Value:	
 *		ExampleSet	createExampleSet - the new exampleSet
 ***********************************************************************/
ExampleSet	createExampleSet (name, mask, net)
  String	name ;
  Mask		mask ;
  Net		net ;
{
  ExampleSet	exampleSet ;

  exampleSet = (ExampleSet)malloc (sizeof(ExampleSetRec)) ;

  if (exampleSet == NULL) {
    simError(SIM_ERR_MALLOC, "createExampleSet \"%s\"", name) ;
    return exampleSet ;
  }

  exampleSet->name = strdup (name) ;
  exampleSet->type = mask ;
  exampleSet->net  = net ;

  initialize (exampleSet) ;

  if (createExampleSetHook != NULL)
    (*createExampleSetHook) (exampleSet) ;

  return exampleSet ;
}
/***********************************************************************
 *	Name:		initialize
 *	Description:	init proc for the ExampleSet class of objects
 *		ExampleSet	exampleSet - the exampleSet object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	initialize (exampleSet) 
  ExampleSet	exampleSet ;
{
  exampleSet->resetAction     = reset ;
  exampleSet->getNextAction   = getNext ;
  exampleSet->getNumberAction = getNumber ;
  exampleSet->addExampleList  = addExampleList ;
  exampleSet->getExampleList  = getExampleList ;
  exampleSet->moveExample     = moveExample ;
  exampleSet->deleteExample   = deleteExample ;
 
  exampleSet->example          = NULL ;
  exampleSet->numExamples      = 0 ;
  exampleSet->currentIdx       = -1 ;
  exampleSet->exampleArraySize = 0 ;

  exampleSet->clampMask  = INPUT ;
  exampleSet->targetMask = OUTPUT ;

  exampleSet->permute   = FALSE ;
  exampleSet->inputBits = 0 ;
  exampleSet->targetBits = 0 ;

  exampleSet->scale = 1.0 ;
  exampleSet->shift = 0.0 ;
  
  exampleSet->extension = NULL ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		destroyExampleSet 
 *	Description:	destroys a exampleSet
 *	Parameters:	
 *		ExampleSet	exampleSet - the exampleSet to destroy
 *	Return Value:	
 *		NONE
 ***********************************************************************/
void	destroyExampleSet (exampleSet) 
  ExampleSet	exampleSet ;
{
  if (destroyExampleSetHook != NULL)
    (*destroyExampleSetHook) (exampleSet) ;

  finalize (exampleSet) ;

  free (exampleSet->name) ;
  free (exampleSet) ;
}
/***********************************************************************
 *	Name:		finalize
 *	Description:	finalize proc for the exampleSet class of objects
 *		ExampleSet	exampleSet - the exampleSet object ;
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	destroyExample(example)
  Example	example ;
{
  if (example == NULL)
    return ;

  if (destroyExampleHook)
    (*destroyExampleHook)(example);

  if (example->name)
    free(example->name) ;

  if (example->inputVector)
    freeOldVector(example->inputVector, example->numVectors) ;
  if (example->targetVector)
    freeOldVector(example->targetVector, example->numVectors) ;

  if (example->inputQVector)
    freeOldQVector(example->inputQVector, example->numVectors) ;
  if (example->targetQVector)
    freeOldQVector(example->targetQVector, example->numVectors) ;

}
/**********************************************************************/
static void	finalize (exampleSet) 
  ExampleSet	exampleSet ;
{
  if (exampleSet->example != NULL) {
    Example	example = exampleSet->example ;
    int		idx ;

    /* for all the examples */
    for (idx = 0 ; idx < exampleSet->numExamples ; ++idx)
      destroyExample(&example[idx]) ;
    free (exampleSet->example) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		exampleSetAddExample 
 *	Description:	adds an example to an example set.
 *	Parameters:	
 *		ExampleSet	exampleSet - the example set to add to
 *		Real		*input - the array of inputs to add
 *		int		numInputs - the number of input values
 *		Real		*target - the array of targets to add
 *		int		numTargets - the number of target values
 *	Return Value:	
 *		NONE
 ***********************************************************************/
int	exampleSetAddExample (exampleSet, input, numInputs, target, numTargets)
  ExampleSet	exampleSet ;
  Real		*input ;
  int		numInputs ;
  Real		*target ;
  int		numTargets ;
{
  Real		**inputVector, **targetVector ;

  inputVector	 = (Real **)malloc(sizeof(Real *)) ;
  inputVector[0] = input ;

  targetVector    = (Real **)malloc(sizeof(Real *)) ;
  targetVector[0] = target ;

  return exampleSet->addExampleList(exampleSet, 1, 
				    inputVector, numInputs, 
				    targetVector, numTargets) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		exampleSetAddExampleList
 *	Description:	
 *	Parameters:
 *	  ExampleSet	exampleSet - 
 *	  int		listSize - 
 *	  Real		**input - 
 *	  int		numInputs - 
 *	  Real		**target - 
 *	  int		numTargets - 
 *	Return Value:
 *	  int	  exampleSetAddExampleList - the index of the example
 *********************************************************************/
int	  exampleSetAddExampleList(exampleSet, listSize,  input,  
				   numInputs, target, numTargets)
  ExampleSet	exampleSet ;
  int		listSize ;
  Real		**input ;
  int		numInputs ;
  Real		**target ;
  int		numTargets ;
{
  return exampleSet->addExampleList(exampleSet, listSize, 
				    input, numInputs, 
				    target, numTargets) ;
}
/********************************************************************/
int	  exampleSetGetExampleList(exampleSet, index, listSize,  input,  
				   numInputs, target, numTargets)
  ExampleSet	exampleSet ;
  int		index ;
  int		*listSize ;
  Real		***input ;
  int		*numInputs ;
  Real		***target ;
  int		*numTargets ;
{
  return exampleSet->getExampleList(exampleSet, index, listSize, 
				    input, numInputs, 
				    target, numTargets) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addExampleList
 *	Description:	Method for adding examples to the set. NOTE:
 *			the example arrays are used in place. They must
 *			be allocated using malloc, and remain in existence
 *			as long as the example set does. (it will free them
 *			when it is done).
 *	Parameters:
 *	  ExampleSet	exampleSet - the example set
 *	  int		listSize - the size of the list (1 for normal
 *				   nets, larger for recurrent nets
 *	  Real		**input  - the array (dimension listSize) of
 *				   input vectors.
 *	  int		numInputs - the dimension of each input vector
 *	  Real		**target - the array (dimension listSize) of
 *				   target vectors.
 *	  int		numTargets - the dimension of each target vector
 *	Return Value:
 *	  static int	addExampleList - 1
 *********************************************************************/
#ifndef GRANULARITY
#define GRANULARITY 64
#endif
static int	addExampleList(exampleSet, listSize,  input,  
			       numInputs, target, numTargets)
  ExampleSet	exampleSet ;
  int		listSize ;
  Real		**input ;
  int		numInputs ;
  Real		**target ;
  int		numTargets ;
{
  Example	example ;

  /* Make sure that the arrays are big enough */
  if (exampleSet->numExamples >= exampleSet->exampleArraySize) {
    if (exampleSet->exampleArraySize == 0) {
      /* set array size to some default size */
      exampleSet->exampleArraySize = exampleSet->numExamples + GRANULARITY ;
      exampleSet->example 
	= (ExampleRec *)malloc ((exampleSet->exampleArraySize)
				*sizeof(ExampleRec)) ;
    } else {
      /* double the size of the array */
      exampleSet->exampleArraySize = 2.0*exampleSet->numExamples ;
      exampleSet->example 
	= (ExampleRec *)realloc (exampleSet->example,
				 (exampleSet->exampleArraySize)
				 *sizeof(ExampleRec)) ;
    }
    if (exampleSet->example == NULL) {
      simError(SIM_ERR_MALLOC, "addExampleList \"%s\"",
	       exampleSet->name) ;
      return -1 ;
    }
  }

  example = &exampleSet->example[exampleSet->numExamples] ;

  if (exampleSet->inputBits) {
    example->inputQVector 
      = quantizeVectors(input, listSize, numInputs, exampleSet->inputBits) ;
    freeOldVector(input, listSize) ;
    example->inputVector = NULL ;
  } else {
    example->inputQVector = NULL ;
    example->inputVector  = input ;
  }
  example->numInputs = numInputs ;

  if (exampleSet->targetBits) {
    example->targetQVector
      = quantizeVectors(target, listSize, numTargets, exampleSet->targetBits) ;
    freeOldVector(target, listSize) ;
    example->targetVector = NULL ;
  } else {
    example->targetQVector = NULL ;
    example->targetVector  = target ;
  }

  example->numTargets = numTargets ;
  example->numVectors = listSize ;

  example->name      = NULL ;
  example->extension = NULL ;

  if (createExampleHook)
    (*createExampleHook) (example);

  ++(exampleSet->numExamples) ;

  return example - exampleSet->example ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		getExampleList
 *	Description:	returns a copy of a requested example
 *	Parameters:
 *	  ExampleSet	exampleSet - the example set to get it from
 *	  int		exampleIdx - the index of the requested example
 *	  int		*listSize -  the size of the lists (return)
 *	  Real		***input - the vector of inputs (return)
 *	  int		*numInputs - the number of inputs (return)
 *	  Real		***target - the vector of targets (return)
 *	  int		*numTargets - the number of targets (return)	
 *	Return Value:
 *	  static int	getExampleList - 0 on failure. NOTE: the example
 *			is allocated when requested, it is the users
 *			responsibility to free it when done.
 *********************************************************************/
static int	getExampleList(exampleSet, exampleIdx,
			       listSize,  input,  
			       numInputs, target, numTargets)
  ExampleSet	exampleSet ;
  int		exampleIdx ;
  int		*listSize ;
  Real		***input ;
  int		*numInputs ;
  Real		***target ;
  int		*numTargets ;
{
  int		unitIdx, idx, nextIdx ;
  Example	example ;
  
  if (exampleIdx < 0 || exampleIdx >= exampleSet->numExamples)
    return 0 ;
  example = &exampleSet->example[exampleIdx] ;

  *listSize   = example->numVectors ;
  *numInputs  = example->numInputs ;
  *numTargets = example->numTargets ;

  *input  = (Real **)calloc(*listSize, sizeof(Real *)) ;
  for (idx = 0 ; idx < *listSize ; ++idx) {
    if ((*input)[idx])
      continue ;

    (*input)[idx] = (Real *)calloc(*numInputs, sizeof(Real)) ;
    if (example->inputQVector)
      for (unitIdx = 0 ; unitIdx < *numInputs ; ++unitIdx)
	(*input)[idx][unitIdx] 
	  = MQAretrieve(example->inputQVector[idx], unitIdx) ;
    else
      memcpy((*input)[idx], 
	     example->inputVector[idx], *numInputs*sizeof(Real)) ;

    /* if the vector was shared, set all other pointers to NULL */
    for (nextIdx = idx + 1 ; nextIdx < *listSize ; ++nextIdx)
      if (DUPLICATE_INPUT_VECTORS(example, idx, nextIdx))
	(*input)[nextIdx] = (*input)[idx] ;
  }

  *target = (Real **)calloc(*listSize, sizeof(Real *)) ;
  for (idx = 0 ; idx < *listSize ; ++idx) {
    if ((*target)[idx])
      continue ;

    (*target)[idx] = (Real *)calloc(*numTargets, sizeof(Real)) ;
    if (example->targetQVector)
      for (unitIdx = 0 ; unitIdx < *numTargets ; ++unitIdx)
	(*target)[idx][unitIdx]
	  = MQAretrieve(example->targetQVector[idx], unitIdx) ;
    else
      memcpy((*target)[idx],
	     example->targetVector[idx], *numTargets*sizeof(Real)) ;

    /* if the vector was shared, set all other pointers to NULL */
    for (nextIdx = idx + 1 ; nextIdx < *listSize ; ++nextIdx)
      if (DUPLICATE_TARGET_VECTORS(example, idx, nextIdx))
	(*target)[nextIdx] = (*target)[idx] ;
  }
  return 1;
}
/********************************************************************/


/*********************************************************************
 *	Name:		deleteExample
 *	Description:	deletes an individual example from an example
 *			set
 *	Parameters:
 *	  ExampleSet	this - the example set
 *	  int		index - the index of the example to delete
 *	Return Value:
 *	  void	  deleteExample - NONE
 *********************************************************************/
void	  exampleSetDeleteExample(exampleSet, index)
  ExampleSet	exampleSet ;
  int		index ;
{
  if (exampleSet->deleteExample)
    exampleSet->deleteExample(exampleSet, index) ;
}
/********************************************************************/
static void	  deleteExample(this, index)
  ExampleSet	this ;
  int		index ;
{
  int	idx ;

  if (index < 0 || index >= this->numExamples)
    return ;

  destroyExample(&this->example[index]) ;
  for (idx = index + 1 ; idx < this->numExamples ; ++idx)
    this->example[idx - 1] = this->example[idx] ;

  --this->numExamples ;
  if (this->currentIdx >= this->numExamples)
    this->currentIdx = this->numExamples - 1 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		moveExample
 *	Description:	moves an example around inside an example set
 *	Parameters:
 *	  ExampleSet	this - the example set
 *	  int		index - the original index of the example
 *	  int		toIndex - the desired index
 *	Return Value:
 *	  static void	  moveExample - NONE
 *********************************************************************/
static void	  moveExample(this, index, toIndex)
  ExampleSet	this ;
  int		index ;
  int		toIndex ;
{
  int		idx ;
  ExampleRec	record ;

  if (index < 0 || index >= this->numExamples
      || toIndex < 0 || toIndex >= this->numExamples
      || index == toIndex)
    return ;

  record = this->example[index] ;

  if (index < toIndex) {
    for (idx = index + 1 ; idx <= toIndex ; ++idx)
      this->example[idx - 1] = this->example[idx] ;
  } else {
    for (idx = index ; idx > toIndex ; --idx)
      this->example[idx] = this->example[idx - 1] ;
  }
  this->example[toIndex] = record ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		exampleSetCopyExample
 *	Description:	copies an example from one place in and example 
 *			set to another place in another (or the same)
 *			example set
 *	Parameters:
 *	  ExampleSet	fromSet - the original example set
 *	  int		fromIdx - the original index
 *	  ExampleSet	toSet - the new set
 *	  int		toIdx - the new index
 *	Return Value:
 *	  void	exampleSetCopyExample - NONE
 *********************************************************************/
void	exampleSetCopyExample(fromSet, fromIdx, toSet, toIdx)
  ExampleSet	fromSet ;
  int		fromIdx ;
  ExampleSet	toSet ;
  int		toIdx ;
{
  int		numVectors, numInputs, numTargets ;
  Real		**input, **target ;

  if (fromSet == NULL || toSet == NULL)	
    return ;

  if (exampleSetGetExampleList(fromSet, fromIdx, &numVectors,
			       &input, &numInputs, &target, &numTargets) == 0)
    return ;
  
  exampleSetAddExampleList(toSet, numVectors,
			   input, numInputs, target, numTargets) ;
  if (fromSet->example[fromIdx].name)
    toSet->example[toSet->numExamples - 1].name
      = strdup(fromSet->example[fromIdx].name) ;

  if (toSet->moveExample)
    toSet->moveExample(toSet, toSet->numExamples - 1, toIdx) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		exampleSetMoveExample
 *	Description:	moves an example from one place in and example 
 *			set to another place in another (or the same)
 *			example set
 *	Parameters:
 *	  ExampleSet	fromSet - the original example set
 *	  int		fromIdx - the original index
 *	  ExampleSet	toSet - the new set
 *	  int		toIdx - the new index
 *	Return Value:
 *	  void	exampleSetMoveExample - NONE
 *********************************************************************/
void	exampleSetMoveExample(fromSet, fromIdx, toSet, toIdx)
  ExampleSet	fromSet ;
  int		fromIdx ;
  ExampleSet	toSet ;
  int		toIdx ;
{
  int		numVectors, numInputs, numTargets ;
  Real		**input, **target ;
  char		*name ;

  if (fromSet == NULL || toSet == NULL)	
    return ;

  if (fromSet != toSet) {

    if (exampleSetGetExampleList(fromSet, fromIdx, &numVectors,
				 &input, &numInputs, &target, &numTargets)
	== 0)
      return ;

    if (fromSet->example[fromIdx].name)
      name = strdup(fromSet->example[fromIdx].name) ;
    else
      name = NULL ;

    deleteExample(fromSet, fromIdx) ;

    exampleSetAddExampleList(toSet, numVectors,
			     input, numInputs, target, numTargets) ;
    toSet->example[toSet->numExamples - 1].name = name ;

    fromIdx = toSet->numExamples - 1 ;
  }

  if (toSet->moveExample)
    toSet->moveExample(toSet, fromIdx, toIdx) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		reset
 *	Description:	resets the example list in an example set
 *	Parameters:	
 *		ExampleSet	exampleSet - the exampleSet
 *	Return Value:	
 *		int		reset - the index of the current example
 ***********************************************************************/
static int	reset (exampleSet) 
  ExampleSet	exampleSet ;
{
  exampleSet->currentIdx = -1 ;
  return exampleSet->currentIdx ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getNext
 *	Description:	getNext action for the exampleSet class of objects 
 *			objects. It sets the network values and also
 *			sets the current example number.p
 *		ExampleSet	exampleSet - the exampleSet object ;
 *	Return Value:	
 *		int	getNext - the index of this example, -1 if none
 *				left
 ***********************************************************************/
typedef struct TraverseRec {
  int		idx ;
  Boolean	doTransform ;
  Real		shift ;
  Real		scale ;
  Real		**vector ;
  QArray	*qVector ;
} TraverseRec,  *Traverse ;
/**********************************************************************/
static int	getNext (exampleSet) 
  ExampleSet	exampleSet ;
{
  Net		net = exampleSet->net ;
  TraverseRec	traverse ;
  Example	example ;
  int		numClamps, numTargets ;

  if (exampleSet->numExamples == 0)
    return -1 ;

  ++exampleSet->currentIdx ;
  if (exampleSet->currentIdx == exampleSet->numExamples) {
    exampleSet->currentIdx = 0 ;
    if (exampleSet->permute) 
      exampleSetPermute(exampleSet);
  }

  example = &exampleSet->example[exampleSet->currentIdx] ;

  /* Make sure that the number of time slices is the same
   * in the example and the net */
  if (example->numVectors != net->timeSlices) {
    fprintf(stderr,
	    "Inconsistent number of time slices (%d:%d) in net \"%s\"\n", 
	    example->numVectors, net->timeSlices, net->name) ;
    return exampleSet->currentIdx ;
  }

  /* make sure that the clamped units exist and there is 
   * the same number as the exampleSet's input array */

  traverse.idx = 0 ;
  netForAllGroups(net, exampleSet->clampMask, sumUnits, &traverse) ;
  numClamps = traverse.idx ;

  /* maybe there are no clamped units - next test will catch errors anyway
  if (numClamps == 0) {
    fprintf(stderr, "No clamped units in net \"%s\"\n", 
	    net->name) ;
    return exampleSet->currentIdx ;
  }
  */

  if (numClamps != example->numInputs) {
    fprintf(stderr, "Inconsistent number of clamped units in net \"%s\"\n", 
	    net->name) ;
    return exampleSet->currentIdx ;
  }

  /* only do clamps if they exist */
  if (numClamps > 0) {
    traverse.idx = 0 ;
    traverse.shift = exampleSet->shift ;
    traverse.scale = exampleSet->scale ;
    traverse.vector  = example->inputVector ;
    traverse.qVector = example->inputQVector ;
    traverse.doTransform = (traverse.shift || traverse.scale != 1.0) ;
    netForAllUnits(net, exampleSet->clampMask, setInput, &traverse);
  }

  /* only do targets if they exist */
  if (example->numTargets == 0)
    return exampleSet->currentIdx ;

  /* make sure that the target units exist and there is 
   * the same number as the example's target array */
  traverse.idx = 0 ;
  netForAllGroups(net, exampleSet->targetMask, sumUnits, &traverse) ;
  numTargets = traverse.idx ;

  if (numTargets == 0) {
    fprintf(stderr, "No target units in net \"%s\"\n", net->name) ;
    return exampleSet->currentIdx ;
  }
  if (numTargets != example->numTargets) {
    fprintf(stderr, "Inconsistent number of target units in net \"%s\"\n", 
	    net->name) ;
    return exampleSet->currentIdx ;
  }

  traverse.idx = 0 ;
  traverse.shift = exampleSet->shift ;
  traverse.scale = exampleSet->scale ;
  traverse.vector  = example->targetVector ;
  traverse.qVector = example->targetQVector ;
  traverse.doTransform = (traverse.shift || traverse.scale != 1.0) ;
  netForAllUnits(net, exampleSet->targetMask, setTarget, &traverse) ;

  return exampleSet->currentIdx ;
}
/**********************************************************************/
static void	setInput (unit, data)
  Unit		unit ;
  void		*data ;
{
  Traverse	traverse = (Traverse)data ;
  QArray	*qVector ;
  Real		**vector ;

  if (unit->net->type & RECURRENT) {
    int		idx, maxIdx = unit->timeSlices ;

    if (vector = traverse->vector)
      for (idx = 0 ; idx < unit->timeSlices ; ++idx)
	unit->clampingHistory[idx] = vector[idx][traverse->idx] ;

    else if (qVector = traverse->qVector)
      for (idx = 0 ; idx < maxIdx ; ++idx)
	unit->clampingHistory[idx] = MQAretrieve(qVector[idx], traverse->idx) ;

    else
      return ;

    if (traverse->doTransform) {
      Real	shift = traverse->shift ;
      Real	scale = traverse->scale ;

      for (idx = 0 ; idx < maxIdx ; ++idx)
	unit->clampingHistory[idx] = shift + scale*unit->clampingHistory[idx] ;
    }

  } else {
    if (vector = traverse->vector)
      unit->output = vector[0][traverse->idx] ;

    else if (qVector = traverse->qVector)
      unit->output = MQAretrieve(qVector[0], traverse->idx) ;
    
    else
      return ;

    if (traverse->doTransform)
      unit->output = traverse->shift + traverse->scale * unit->output ;
  }
  ++traverse->idx ;
}
/**********************************************************************/
static void	setTarget (unit, data)
  Unit		unit ;
  void		*data ;
{
  Traverse	traverse = (Traverse)data ;
  QArray	*qVector ;
  Real		**vector ;

  if (unit->net->type & RECURRENT) {
    int		idx, maxIdx = unit->timeSlices ;

    if (vector = traverse->vector)
      for (idx = 0 ; idx < unit->timeSlices ; ++idx)
	unit->targetHistory[idx] = vector[idx][traverse->idx] ;

    else if (qVector = traverse->qVector)
      for (idx = 0 ; idx < maxIdx ; ++idx)
	unit->targetHistory[idx] = MQAretrieve(qVector[idx], traverse->idx);

    else
      return ;

    if (traverse->doTransform) {
      Real	shift = traverse->shift ;
      Real	scale = traverse->scale ;

      for (idx = 0 ; idx < unit->timeSlices ; ++idx)
	unit->targetHistory[idx] = shift + scale * unit->targetHistory[idx] ;
    }

  } else {
    if (vector = traverse->vector)
      unit->target = vector[0][traverse->idx] ;

    else if (qVector = traverse->qVector)
      unit->target = MQAretrieve(qVector[0], traverse->idx) ;

    else
      return ;

    if (traverse->doTransform)
      unit->target = traverse->shift + traverse->scale * unit->target ;
  }
  ++traverse->idx ;
}
/**********************************************************************/
static void	sumUnits (group, data)
  Group		group ;
  void		*data ;
{
  Traverse	traverse = (Traverse)data ;
  traverse->idx += group->numUnits ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getNumber
 *	Description:	getNumber action for the exampleSet class of 
 *			objects. It sets the network values and also
 *			sets the current example number.p
 *		ExampleSet	exampleSet - the exampleSet object ;
 *		int		number     - the index (base 0) of the
 *					required example.
 *	Return Value:	
 *		int	getNumber - the index of this example, -1 if
 *				it doesn't exist
 ***********************************************************************/
static int	getNumber (exampleSet, number)
  ExampleSet	exampleSet ;
  int		number ;
{
  if (number >= 0 && number < exampleSet->numExamples) {
    exampleSet->currentIdx = number - 1 ;
    return MgetNext(exampleSet) ;
  } else {
    return -1 ;
  }
}
/**********************************************************************/


/*********************************************************************
 *	Name:		quantizeVectors
 *	Description:	quantizes a real vector into a QArray (real 
 *			array quantized into <bits> bits representation)
 *	Parameters:
 *	  Real	**vector - the input vector
 *	  int	n1 - the first dimension
 *	  int	n2 - the second dimension
 *	  int	bits - the number of bits to use in the representation
 *	Return Value:
 *	  static QArray	*quantizeVectors - the quantized array
 *********************************************************************/
static QArray	*quantizeVectors(vector, n1, n2, bits)
  Real	**vector ;
  int	n1 ;
  int	n2 ;
  int	bits ;
{
  Real		*array ;
  QArray	*qVector, qArray ;
  int		idx, nextIdx, subIdx ; 
  Real		max, min ;

  qVector = (QArray *)calloc(n1, sizeof(QArray)) ;

  for (idx = 0 ; idx < n1 ; ++idx) {
    array = vector[idx] ;
    if (qVector[idx] || array == NULL || n2 == 0)
      continue ;

    min = max = array[0] ;
    for (subIdx = 1 ; subIdx < n2 ; ++subIdx) {
      max = MAX(max, array[subIdx]) ;
      min = MIN(min, array[subIdx]) ;
    }

    qArray = createQArray(n2, bits, min, max) ;
    for (subIdx = 0 ; subIdx < n2 ; ++subIdx)
      MQAstore(qArray, subIdx, array[subIdx]) ;
    
    qVector[idx] = qArray ;
    for (nextIdx = idx + 1 ; nextIdx < n1 ; ++nextIdx) {
      if (vector[nextIdx] == array)
	qVector[idx] = qArray ;
    }
  }
  return qVector ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		freeOldVector
 *	Description:	frees a 2d vector of reals
 *	Parameters:
 *	  Real		**vector - the vector
 *	  int		n - the first dimension
 *	Return Value:
 *	  static void	freeOldVector  - NONE
 *********************************************************************/
static void	freeOldVector (vector, n)
  Real		**vector ;
  int		n ;
{
  int	idx, nextIdx ;

  if (vector == NULL)
    return ;

  for (idx = 0 ; idx < n ; ++idx) {
    Real	*vectorPtr = vector[idx] ;

    if (vectorPtr) {
      free(vectorPtr) ;

      /* if the vector was shared, set all other pointers to NULL */
      for (nextIdx = idx + 1 ; nextIdx < n ; ++nextIdx) {
	if (vectorPtr == vector[nextIdx])
	  vector[nextIdx] = NULL ;
      }
    }
  }
  free(vector) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		freeOldQVector
 *	Description:	frees a 2d vector of quantized reals
 *	Parameters:
 *	  QArray	*vector - the vector
 *	  int		n - the first dimension
 *	Return Value:
 *	  static void	freeOldQVector  - NONE
 *********************************************************************/
static void	freeOldQVector (vector, n)
  QArray	*vector ;
  int		n ;
{
  int	idx, nextIdx ;

  if (vector == NULL)
    return ;

  for (idx = 0 ; idx < n ; ++idx) {
    QArray	vectorPtr = vector[idx] ;

    if (vectorPtr) {
      MQAdestroy(vectorPtr) ;

      /* if the vector was shared, set all other pointers to NULL */
      for (nextIdx = idx + 1 ; nextIdx < n ; ++nextIdx) {
	if (vectorPtr == vector[nextIdx])
	  vector[nextIdx] = NULL ;
      }
    }
  }
  free(vector) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		exampleSetPermute
 *	Description:	permutes the examples in an example set
 *			fast, since only pointers are swapped
 *	Parameters:
 *	  ExampleSet	set - the example set to permute
 *	Return Value:
 *	  void		exampleSetPermute - NONE
 *********************************************************************/
void		exampleSetPermute(set)
  ExampleSet	set ;
{
  int		i, j ;
  ExampleRec	tmp ;

  for (i = 0; i < set->numExamples; i++) {
    j = (int)(simRand() * set->numExamples) % set->numExamples ;
    tmp = set->example[i] ;
    set->example[i] = set->example[j] ;
    set->example[j] = tmp ;
  }
}
/********************************************************************/
