
/**********************************************************************
 * $Id: rudisearch.c,v 1.2 92/11/30 11:30:58 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <errno.h>

#include <xerion/useful.h>
#include "minimize.h"
#include "linesearch.h"
#include "wobbly.h"
#include "nanstuff.h"

#include "rudisearch.h"

/***********************************************************************
 *  Name:		rudisLineSearch
 *  Description:	
 *			
 *  Parameters:	
 *I	Minimize	mz,	- minimize record
 *I	int		n,	- dimensionality of the vectors
 *I	Real		*start	- start point from which to search
 *I	Real		*search	- search direction
 *O	Real		*x,	- returns the end point
 *O	Real		*grad,	- returns the gradient at chosen step
 *I	RealVecFunc	fEval,	- function to evaluate f
 *I	VecProc		gEval,	- function to evaluate g
 *I	Real2VecFunc	fgEval,	- function to evaluate f ang g
 *I	double		s_len	- |search|
 *I	double		x_len	- |x|
 *IO	double		*ap,	- suggested step size, returns chosen stepsize
 *IO	double		*fp,	- function value at x, returns value at new x
 *IO	double		*dp	- slope at x, returns slope at new x
 *
 *  Return Value:
 *	int	rudisLineSearch - return code:
 *	Possible values:
 *	MZSUCCEED    - the line search found a lower value that
 *		       meets all criteria within the allowed number
 *		       of function evaluations.
 *	MZMAXF	     - the total number of function evaluations
 *		       allowed was exceeded.
 *	MZMAXFLINE   - the number of functions evaluations allowed
 *		       in one line search was exceeded.  A lower
 *		       function value was found, but it did not
 *		       satisfy the criterion.  *ap is set to the
 *		       step found and it is  possible to proceed
 *		       as normal.
 *	MZMAXFLINEFAIL - the number of functions evaluations allowed
 *		       in one line search was exceeded.  No lower
 *		       function value was found.
 *	MZSTOPPED    - the stop flag was set to a value > 1
 *	MZWOBBLY     - the function values are wobbly (i.e. small
 *		       and inconsistent with gradients).
 *	MZNOGRADIENT - the slope disappeared during the line search
 *	MZFAILIMPROVE- the step size has become too small and will
 *		       result in a change in x too small to represent
 *	The last two return codes are abnormal conditions.
 *
 ***********************************************************************/

int	rudisLineSearch (mz, n, start, search, x, grad, 
			 fEval, gEval, fgEval, s_len, x_len, ap, fp, dp)
  Minimize		mz ;
  int			n ;
  Real			*start ;
  Real			*search ;
  Real			*x ;
  Real			*grad ;
  RealVecFunc		fEval ;
  VecProc		gEval ;
  Real2VecFunc		fgEval ;
  double		s_len ;
  double		x_len ;
  double		*ap ;
  double		*fp ;
  double		*dp ;
{
  double a = *ap;  /* The initial step size */
  double a2 = 0.0; /* The next step size to take */
  double f2 = *fp; /* f(start) */
  double d2 = *dp; /* d(start) */
  double f0 = *fp; /* The value of f at start (used to remember start value)*/
  double d0 = *dp; /* The value of f at start (used to remember start value)*/
  double a1, f1, d1, u1, u2, z, r;
  int ok, low_delta_f;
  int vb = mz->lsVerbosity;
  double delta_f;

  mz->evalReason = "Starting line search";
  mz->lsnFuncEvals = 0;

  initBest(mz, grad);

  if (a<=0.0)
    IErrorAbort("rudisLineSearch: no initial stepsize supplied");

  VB(1, vb, "a= %-12g f= %-12g f-f0= %-12g d= %-12g\n", 0.0, f2, f2-f0, d2);

  for (;;) {
    f1 = f2;
    d1 = d2;
    a1 = a2;
    a2 = a;

    if (mz->askStep>0 && mz->nFuncEvals>mz->askStep) {
      char buf[100];
      buf[0] = '\0';
      fprintf(dout, "Enter the step size [%g]: ", a2);
      fgets(buf, 100, din);
      if (atof(buf)>0.0) {
	a2 = atof(buf);
	fprintf(dout, "Taking step %g\n", a2);
      }
    }

    /* check whether to stop now */
    if (a2*s_len < mz->machinePrecision*mz->machinePrecision*x_len) {
      VB(2, vb, "Stopping line search: step too small\n");
      VB(2, vb, "a2= %g d2= %g mp=%g f1=%g\n", a2, d2,
	 mz->machinePrecision, f1);
      mz->lsResultCode = MZFAILIMPROVE;
      goto getout;
    }
    if (mz->stopFlag>1) {
      VB(2, vb, "Stopping line search: stop flag set\n");
      mz->lsResultCode = MZSTOPPED;
      goto getout;
    }
    if (mz->lsnFuncEvals >= mz->lsMaxFuncEvals) {
      VB(2, vb, "Stopping line search: too many f evals in line search\n");
      mz->lsResultCode = (mz->lsBestByF>0 ? MZMAXFLINE : MZMAXFLINEFAIL);
      goto getout;
    }
    if (mz->maxFuncEvals>0
	&& (  mz->lsnFuncEvals
	    > (mz->maxFuncEvals + mz->lsFlexFuncEvals))) {
      VB(2, vb, "Stopping line search: too many f evals in total\n");
      mz->lsResultCode = MZMAXF;
      goto getout;
    }

    low_delta_f = 0;
    if (mz->wobbleTest || mz->wobbleWatch) {
      if (-a2*d0 < fPrecision(mz, f0)) {
	VB(2, vb, "Expected low change in func value: requesting gradient\n");
	low_delta_f = 1;
      } else if (mz->lsnFuncEvals>0 && fabs(delta_f) < fPrecision(mz, f0)) {
	VB(2, vb, "Observed low change in func value: requesting gradient\n");
	low_delta_f = 1;
      }
    }
    if (mz->wobbleWatch) {
      int good, total;
      gradientsConsistent(mz, &good, &total);
      if (low_delta_f || good < total)
	fprintf(dout,
		"Wobbliness diagnostics: deltas: %s, %d slopes ok out of %d\n",
		low_delta_f ? "low" : "ok", good, total);
    }
    if (mz->wobbleTest && low_delta_f &&
	(   (mz->wobbleTest<2)
	 || (mz->wobbleTest<3 && mz->lsnFuncEvals>0)
	 || (mz->lsnFuncEvals>0 && !gradientsConsistent(mz, NULL, NULL)))) {
      VB(2, vb, "Stopping line search: f wobbly at level %d\n",mz->wobbleTest);
      mz->lsResultCode = MZWOBBLY;
      goto getout;
    }

    /* make a step of distance a2 along search */
    moveInDirection(n, start, search, a2, x);
    
    if (fgEval)
      f2 = fgEval(mz, n, x, grad);
    else {
      f2 = fEval(mz, n, x);
      gEval(mz, n, grad);
    }
    d2 = dotProduct(n, grad, search);

    VB(1, vb, "a= %-12g f= %-12g f-f0= %-12g d= %-12g\n", a2, f2, f2-f0, d2);
    mz->nFuncEvals++;
    mz->lsnFuncEvals++;
    
    insertLSData(mz, mz->lsnFuncEvals, a2, f2, d2, 1, 0);
    if (isNaN(f2) || isNaN(d2)) {
      VB(2, vb, "Computed NaN f=%g d=%g - backing up by 1/%g\n",
	 f2, d2, mz->backUpFactor);
      a /= mz->backUpFactor;
      a2 = 0.0; f2 = f0; d2 = d0;
      continue;
    }
    
    updateBest(mz, mz->lsnFuncEvals, grad);
    
    delta_f = f2 - f0;
    if (f2>f0 && d2<0.0) {
      VB(2, vb, "Continuing search: f>f0 & slope -ve, cutting step by 1/%g\n",
	 mz->backUpFactor);
      VB(3, vb, "f-f0= %g d= %g\n", f2-f0, d2);
      a /= mz->backUpFactor;
      a2 = 0.0; f2 = f0; d2 = d0;
      continue;
    } else {
      r = fabs(d2/d0);	/* ratio of new & old slope */
    }

    /***
     * If decrease in function is sufficient & slope is not as
     * steep as at start, accept the step.  (The slope is the
     * dot product of the search direction and the gradient.
     * The slope at the start is d0, and the slope at the current
     * point is d2.  Their ratio is r.)
     */
    if (f2>=f0) {
      VB(2, vb, "Continuing search: f has increased to %g\n", f2);
    } else if (f2>=f0+mz->minFuncReduction*a2*d0) {
      VB(2, vb, "Continuing search: f reduction %g not sufficient\n",
	 (f0-f2)/(a2*d0));
    } else if (r>=mz->maxSlopeRatio) {
      VB(2, vb, "Continuing search: slope ratio %g too large\n", r);
      VB(3, vb, "Slope at start = %g, at end = %g\n", d0, d2);
    } else {
      VB(2, vb,
	 "End line search: f reduction %g sufficient & slope ratio %g OK\n",
	 (f0-f2)/(a2*d0), r);
      VB(1, vb, "Line search %d steps: f reduction factor= %g\n",
	 mz->lsnFuncEvals, (f0-f2)/(a2*d0));
      *fp = f2; *ap = a2; *dp = d2;
      return mz->lsResultCode = MZSUCCEED;
    }
    /***
     * Use cubic interpolation to find a better step
     */
    VB(3, vb, "Using cubic interpolation (%g %g %g) (%g %g %g)\n",
       a1, f1, d1, a2, f2, d2);
    u1 = d1+d2-3.0*(f1-f2)/(a1-a2);
    if ((z = u1*u1-d1*d2) < 0.0)
      u2 = 0.0;
    else
      u2 = sqrt(z);
    if (d2==0.0 && d1==0.0 && u2==0.0) {
      mz->lsResultCode = MZNOGRADIENT;
      goto getout;
    }
    /* set a to minimum according to cubic interpolation/extrapolation */
    a = a2-(a2-a1)*(d2+u2-u1)/(d2-d1+u2+u2);
    /* set u1 and u2 to upper and lower of a1 and a2 */
    u1 = (a1<a2) ? a1 : a2;
    u2 = (a1>a2) ? a1 : a2;
    VB(3, vb, "Interpolated minimum is %g\n", a);
    /***
     * Do some checks that the minimum is sensible
     */
    if (d2/d1 <= 0.0) {	/* changed from (< 0.0) */
      if (a<1.01*u1 || a>0.99*u2) {
	VB(3, vb, "Rejecting cubic min: outside points: choosing midpoint\n");
	a = (a1+a2)/2.0;
      }
    } else if ((a<0.0 || a>0.99*u1) && d2>0.0) {
      /* both gradients positive, so we should end up between 0 and u1 */
      VB(3, vb, "Rejecting cubic min: both gradients +ve: setting to u1/2\n");
      a = u1/2.0;
    } else if (a<1.01*u2 && d2<0.0) {
      /* both gradients negative, so we should end up beyond u2 */
      VB(3, vb, "Rejecting cubic min: both gradients -ve: setting to 2*u2\n");
      a = 2.0*u2;
    }
    mz->evalReason = "Retrying line search";
  }

  IErrorAbort("rudisLineSearch: should not reach here.");
 getout:
  ok = getBestByF(mz, n, start, search, x, grad,
		  fEval, gEval, fgEval, ap, fp, dp);
  if (!ok) {
    VB(1, vb, "Failed to restore best point of line search\n");
    return mz->lsResultCode = MZFAIL;
  } else
    return mz->lsResultCode;
}

