
/**********************************************************************
 * $Id: minimize.c,v 1.9 93/03/29 14:13:12 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

/**********************************************************************
 * Written by Tony Plate in June 1991, based on code written by
 * Rudi Mathon.
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <xerion/useful.h>
#include <signal.h>

#include "minimize.h"
#include "initialstep.h"
#include "linesearch.h"
#include "fixedstep.h"
#include "raysearch.h"
#include "rudisearch.h"
#include "tapsearch.h"
#include "direction.h"
#include "datalog.h"
#include "nanstuff.h"

#ifndef MIN
#define MIN(x, y)	((x) < (y) ? (x) : (y))
#endif
#ifndef MAX
#define MAX(x, y)	((x) > (y) ? (x) : (y))
#endif

#define MAGIC_NUMBER	(Real)(1.2345e-10)

extern int IIsInteger ARGS((char *));

struct TRACE beginMinimize ;
struct TRACE minimizeEndIteration;
struct TRACE minimizeEndRepetition;
struct TRACE endMinimize ;


#define FreeIfNotNull(p) {if ((p)!=NULL) {free(p); p=NULL;}}

static Boolean interrupted = FALSE ;
static int	isCGMethod ARGS((int method)) ;
static char	*simpleName ARGS((Minimize, int i)) ;

void	NULL_proc() {
  IErrorAbort("NULL_proc: you didn't set one of the cgrad procedures!!") ;
}

/***********************************************************************
 * Name:		resultCodeStr
 * Description:
 *	Returns a string with a description of the result code.
 *
 * Parameters:
 *	int	i	- result code
 *
 * Return Value:	a string
 ***********************************************************************/
static char	*resultCodeStr(i)
  int		i ;
{
  switch (i) {
  case MZCONVERGE:
    return "algorithm has converged.";
  case MZMAXF:
    return "computed maximum # of function evaluations.";
  case MZMAXITER:
    return "computed maximum # of iterations.";
  case MZMAXFLINE:
    return "computed maximum # of function evaluations in line search but found lower function value.";
  case MZMAXFLINEFAIL:
    return "computed maximum # of function evaluations in line search and did not find lower function value.";
  case MZFAILIMPROVE:
    return "step for line search will not result in detectable lower function value.";
  case MZNOGRADIENT:
    return "gradient disappeared during line search.";
  case MZNOTDESCENT:
    return "search vector is not descent.";
  case MZSTOPPED:
    return "stop flag set - stopped normally.";
  case MZSYSTEMERROR:
    return "system error.";
  case MZBADMETHOD:
    return "bad method.";
  case MZBADRETURN:
    return "bad return value.";
  case MZSUCCEED:
    return "succeeded.";
  case MZOKFMIN:
    return "acceptably low value of function.";
  case MZINTERRUPTED:
    return "interrupt signal caught - stopped (ab)normally.";
  case MZWOBBLY:
    return "slope values inconsistent - function value is wobbly.";
  case MZFAIL:
    return "line search failed.";
  default:
    IErrorAbort("resultCodeStr: bad result code %d.", i);
    break ;
  }
  return NULL ;
}

/***********************************************************************
 * Name:		VB
 * Description:
 *	Prints a message if it should be printed at the
 *	current verbosity level (i.e., if verbosity>=level).
 *	The message is printed using printf, and VB can
 *	take any number of arguments.
 *	VB(msg_level, vb_level, format, x, y, z...) prints the same
 *	thing as fprintf(dout, format, x, y, z...).
 *
 * Parameters:	
 *	int	msg_level	- the verbosity level of this message
 *	int	vb_level	- the current verbosity level
 *	char*	format	- the format (message) string
 *	?	args... - the arguments for printf
 *
 * Return Value:	none
 ***********************************************************************/
void VB(va_alist)
  va_dcl
{
  va_list 	args;
  int		msg_level, vb_level ;
  int		i;
  char		*format ;

  va_start(args) ;
  msg_level	= va_arg(args, int) ;
  vb_level	= va_arg(args, int) ;
  format	= va_arg(args, char *) ;

  if (vb_level<msg_level)
    return;
  for (i=1; i<msg_level; i++)
    fprintf(dout, "  ");

  vfprintf(dout, format, args);
  va_end(args);
}

/***********************************************************************
 * Name:		DB
 * Description:
 *	Prints a message if it should be printed at the
 *	current debug level (i.e., if debug>=level).
 *	The message is printed using printf, and DB can
 *	take any number of arguments.
 *	VB(msg_level, db_level, format, x, y, z...) prints the same
 *	thing as fprintf(dout, format, x, y, z...).
 *
 * Parameters:	
 *	int	msg_level	- the debug level of this message
 *	int	db_level	- the current debug level
 *	char*	format	- the format (message) string
 *	?	args... - the arguments for printf
 *
 * Return Value:	none
 ***********************************************************************/
void DB(va_alist)
  va_dcl
{
  va_list 	args;
  int		msg_level, db_level ;
  int		i;
  char		*format ;

  va_start(args) ;
  msg_level	= va_arg(args, int) ;
  db_level	= va_arg(args, int) ;
  format	= va_arg(args, char *) ;

  if (db_level<msg_level)
    return;

  vfprintf(dout, format, args);
  va_end(args);
}


/***********************************************************************
 * Name:		simpleName
 * Description:
 *	Creates a name for parameter <i> of the form "p<i>".
 *	The name is written in static storage, so calling it
 *	twice will result in the first name being overwritten.
 *
 * Parameters:
 *	int	i	- number of the parameter
 *
 * Return Value:	
 *	char *	simpleName
 *			- the name
 ***********************************************************************/
static char	*simpleName (mz, i)
  Minimize	mz ;
  int		i ;
{
  static char name[10];
  sprintf(name, "p%d", i);
  return name;
}

/***********************************************************************
 * Name:		methodName
 * Description:
 *	Returns a string with a description of the method
 *
 * Parameters:
 *	int	i	- method code
 *
 * Return Value:	a string
 ***********************************************************************/

static char	*methodName(i)
  int		i ;
{
  switch (i) {
  case MZFIXEDSTEP:
    return "fixed step size";
  case MZRUDISLS:
    return "Rudi's line search";
  case MZRAYSLS:
    return "Ray's line search";
  case MZSLOPESEARCH:
    return "Tap's slope search";
  case MZISNEW:
    return "customizable";
  case MZISASK:
    return "ask user";
  case MZISSLEN:
    return "reciprocal of search length";
  case MZSHOOT:
    return "shoot for previous change";
  case MZDRATIO:
    return "slope ratio";
  case MZDRATIOSHOOT:
    return "slope ratio with shoot at first step";
  case MZSTEEPEST:
    return "steepest descent";
  case MZMOMENTUM:
    return "momentum descent";
  case MZQUICKPROP:
    return "quickprop";
  case MZDELTABARDELTA:
    return "delta-bar-delta";
  case MZCONJGRAD:
    return "conjugate gradients";
  case MZRUDICG:
    return "Rudi's conjugate gradients";
  case MZCONJGRADRST:
    return "conjugate gradients with restarts";
  default:
    IErrorAbort("methodName: bad method %d", i);
  }
  return NULL ;
}

void	*setMinimizeUserData(mz, userData)
  Minimize	mz ;
  void		*userData ;
{
  void	*oldUserData = mz->userData ;
  mz->userData = userData ;
  return oldUserData ;
}

UnknownProc	setMinimizeMethod(mz, which, value)
  Minimize	mz ;
  unsigned	which ;
  UnknownProc	value ;
{
  UnknownProc	oldValue = NULL ;
  
  switch(which) {
  case MZGETNVARS:
    oldValue = (UnknownProc)mz->getNVars ;
    mz->getNVars = (intFunc)value ;
    break ;
  case MZGETVALUES:
    oldValue = (UnknownProc)mz->getValues ;
    mz->getValues = (VecProc)value ;
    break ;
  case MZSETVALUES:
    oldValue = (UnknownProc)mz->setValues ;
    mz->setValues = (VecProc)value ;
    break ;
  case MZFEVAL:
    oldValue = (UnknownProc)mz->fEval ;
    mz->fEval = (RealVecFunc)value ;
    break ;
  case MZGEVAL:
    oldValue = (UnknownProc)mz->gEval ;
    mz->gEval = (VecProc)value ;
    break ;
  case MZFGEVAL:
    oldValue = (UnknownProc)mz->fgEval ;
    mz->fgEval = (Real2VecFunc)value ;
    break ;
  case MZVALUENAME:
    oldValue = (UnknownProc)mz->valueName ;
    mz->valueName = (strFunc)value ;
    break ;
  case MZINCITER:
    oldValue = (UnknownProc)mz->incIter ;
    mz->incIter = (voidFunc)value ;
    break ;
  default:
    break ;
  }
  return oldValue ;
}

/***********************************************************************
 * Name:		initMinimize
 * Description:
 *	Allocates vectors in the Minimize record.  If vectors are
 *	already there and of the right size, they are left untouched.
 *	A checksum of the vector addresses is maintained.  It is used
 *	to decide whether to free or discard vectors that are of the
 *	wrong size.  If the checksum is not consistent, it is assumed
 *	that something has got overwritten (or badly initialized), and
 *	thus it is not safe to free the vectors.
 *
 * Parameters:
 *	Minimize mz	- minimize record - if NULL one is allocated
 *	int	n	- number of parameters - if non-zero vectors will
 *			  be allocated and mz->epsilon set to
 *			  default value if zero.
 *
 * Return Value:	none
 ***********************************************************************/
Minimize	initMinimize(mz, n)
  Minimize	mz ;
  int		n ;
{
  int chksum = 0;
  if (mz!=NULL && (mz->mep1!=mz || mz->mep2!=mz)) {
    if (mz->mep1)
      fprintf(stderr,
	      "Warning: initMinimize: start of minimize record corrupted.\n");
    if (mz->mep2)
      fprintf(stderr,
	      "Warning: initMinimize: end of minimize record corrupted.\n");
    fprintf(stderr, "Minimize record will be discarded and reallocated.\n");
    mz = NULL;
  }

  if (mz==NULL) {
    mz = CallocOrAbort(struct minimizeRec, 1);
  }

  if (n == 0) {
    mz->mep1 = mz->mep2 = mz;
    mz->mzVerbosity = 1;
    mz->wolfeTest = 1;
    mz->localMinima = 1;
    mz->machinePrecision = machineEpsilon();

    mz->tolerance = 1e-6;

    mz->wobbleTest = 4;
    mz->wobbleTolerance = 10;
    mz->wobbleFlex = 2;
    mz->funcPrecision = 10;
    mz->funcValueScale = 1.0;

    mz->maxGrowthFactor = 1.75;

    mz->gainIncrement = 0.1;
    mz->gainDecrement = 0.9;
  
    mz->maxSlopeRatio = 0.5;
    mz->minFuncReduction = 1e-6;
    mz->interpLimits = 0.05;
    mz->extrapLimits = 0.35;
    mz->maxExtrapol = 9.0;
    mz->extensor = 2.0;
    mz->stepBound = 10;

    mz->initialStepFactor = 2;
    mz->backUpFactor = 3;
    mz->lsMaxFuncEvals = 10;

    mz->initialStepMethod = MZDRATIO;
    mz->directionMethod = MZRUDICG;
    mz->stepMethod = MZRAYSLS;

    mz->alpha = 0.9;
    mz->epsilon   = MAGIC_NUMBER ;
    mz->qpEpsilon = MAGIC_NUMBER ;

    mz->getNVars  = (intFunc)NULL_proc ;
    mz->getValues = (VecProc)NULL_proc ;
    mz->setValues = (VecProc)NULL_proc ;
    mz->fEval     = (RealVecFunc)NULL_proc ;
    mz->gEval     = (VecProc)NULL_proc ;
    mz->fgEval    = (Real2VecFunc)NULL_proc ;
    mz->valueName = (strFunc)NULL_proc ;
    mz->incIter   = (voidFunc)NULL_proc ;
  }

  if (n>0) {
    chksum = (  (int)mz->x ^ (int)mz->bestx ^ (int)mz->gradient
	      ^ (int)mz->bestGradientF ^ (int)mz->bestGradientD
	      ^ (int)mz->prevGradient ^ (int)mz->update ^ (int)mz->search
	      ^ (int)mz->start ^ (int)mz->rsty ^ (int)mz->rstd
	      ^ (int)mz->lsStep ^ (int)mz->lsFuncValue ^(int)mz->lsSlope
	      ^ (int)mz->lsOrder ^ (int)mz->lsHaveSlope);
    if (chksum != mz->chksum) {
      fprintf(stderr, "%s\n%s\n",
	      "Warning: initMinimize: checksum failed for pointers.",
	      "Vectors will be discarded (not freed) and reallocated.");
      mz->nInit = 0;
      mz->n = 0;
      
      mz->x = mz->bestx = mz->gradient = mz->bestGradientF = mz->bestGradientD
	= mz->prevGradient = mz->update = mz->search = mz->start
	= mz->rsty = mz->rstd = NULL;
      mz->lsStep = mz->lsFuncValue = mz->lsSlope = NULL;
      mz->lsOrder = mz->lsHaveSlope = NULL;
    }
    
    if (n != mz->nInit) {
      FreeIfNotNull(mz->x);
      FreeIfNotNull(mz->bestx);
      FreeIfNotNull(mz->gradient);
      FreeIfNotNull(mz->bestGradientF);
      FreeIfNotNull(mz->bestGradientD);
      FreeIfNotNull(mz->prevGradient);
      FreeIfNotNull(mz->update);
      FreeIfNotNull(mz->search);
      FreeIfNotNull(mz->start);
      FreeIfNotNull(mz->rsty);
      FreeIfNotNull(mz->rstd);
      
      /* set the N's to zero first in case of problems in the allocs */
      mz->n = 0;
      mz->x = CallocOrAbort(Real, n);
      mz->bestx = CallocOrAbort(Real, n);
      mz->gradient = CallocOrAbort(Real, n);
      mz->bestGradientF = CallocOrAbort(Real, n);
      mz->bestGradientD = CallocOrAbort(Real, n);
      mz->prevGradient = CallocOrAbort(Real, n);
      mz->update = CallocOrAbort(Real, n);
      mz->search = CallocOrAbort(Real, n);
      mz->start = CallocOrAbort(Real, n);
      mz->rsty = CallocOrAbort(Real, n);
      mz->rstd = CallocOrAbort(Real, n);
      mz->n = n;
      mz->nInit = n;
      
      mz->nIterations = 0;
      mz->nFuncEvals = 0;
      mz->lsnFuncEvals = 0;
      mz->E = 0.0;
      mz->lastE = 0.0;
      mz->bestE = 0.0;
    }
    
    if (mz->lsnPoints!=mz->lsMaxFuncEvals+1) {
      FreeIfNotNull(mz->lsStep);
      FreeIfNotNull(mz->lsFuncValue);
      FreeIfNotNull(mz->lsSlope);
      FreeIfNotNull(mz->lsHaveSlope);
      FreeIfNotNull(mz->lsOrder);
      mz->lsnPoints = 0;
      mz->lsStep = CallocOrAbort(double, mz->lsMaxFuncEvals+1);
      mz->lsFuncValue = CallocOrAbort(double, mz->lsMaxFuncEvals+1);
      mz->lsSlope = CallocOrAbort(double, mz->lsMaxFuncEvals+1);
      mz->lsOrder = CallocOrAbort(int, mz->lsMaxFuncEvals+1);
      mz->lsHaveSlope = CallocOrAbort(int, mz->lsMaxFuncEvals+1);
      mz->lsnPoints = mz->lsMaxFuncEvals+1;
    }
    
    mz->chksum
      = (  (int)mz->x ^ (int)mz->bestx ^ (int)mz->gradient
	 ^ (int)mz->bestGradientF ^ (int)mz->bestGradientD
	 ^ (int)mz->prevGradient ^ (int)mz->update ^ (int)mz->search
	 ^ (int)mz->start ^ (int)mz->rsty ^ (int)mz->rstd
	 ^ (int)mz->lsStep ^ (int)mz->lsFuncValue ^ (int)mz->lsSlope
	 ^ (int)mz->lsOrder ^ (int)mz->lsHaveSlope);

    if (mz->epsilon == MAGIC_NUMBER)
      mz->epsilon = (mz->directionMethod==MZQUICKPROP
		     ? 1.0 : 4.0/sqrt((Real)mz->n));
    
    if (mz->qpEpsilon == MAGIC_NUMBER && mz->n>0)
      mz->qpEpsilon = 0.3/sqrt((Real)mz->n);

  }

  return mz;
}


/***********************************************************************
 * Name:		checkParameters
 * Description:
 *	Checks that all the parameters in the Minimize record are
 *	set to (semi) sensible values.  If parameters are zero, they
 *	are set to a default value (those for which zero is not the
 *	default).
 *
 * Parameters:
 *	Minimize mz	- minimize record
 *
 * Return Value:	none
 ***********************************************************************/

void		checkParameters(mz)
  Minimize	mz ;
{
  int using_conjgrad, using_linesearch;

  mz->stopFlag = 0;

  if (mz->directionMethod<=0 || mz->directionMethod>MZMAXMETHOD) {
  }

  using_conjgrad = isCGMethod(mz->directionMethod);

  using_linesearch = (mz->stepMethod==MZRAYSLS || mz->stepMethod==MZRUDISLS
		      || mz->stepMethod==MZSLOPESEARCH);

  if (using_conjgrad && !using_linesearch)
    fprintf(dout, "Warning: conjugate gradient methods should be used with a line search.\n");

  if ((mz->directionMethod==MZMOMENTUM || mz->directionMethod==MZDELTABARDELTA)
      && mz->stepMethod!=MZFIXEDSTEP)
    fprintf(dout, "Warning: %s should be used with fixed step size\nbecause line searches cannot handle non non-descent search directions.\n",
	    methodName(mz->directionMethod));

  if (mz->directionMethod==MZQUICKPROP && mz->stepMethod==MZFIXEDSTEP
      && mz->epsilon!=1.0)
    fprintf(dout, 
	    "Warning: quickprop should be used with a stepsize (epsilon) of 1.0.\n");

  if (mz->minFuncReduction<0 || mz->minFuncReduction>0.5)
    fprintf(dout, "Warning: mz.minFuncReduction= %g, should be in (0, 0.5).\n",
	    mz->minFuncReduction);

  if (mz->maxSlopeRatio<mz->minFuncReduction || mz->maxSlopeRatio>1)
    fprintf(dout,
      "Warning: mz.maxSlopeRatio= %g, should be in (mz.minFuncReduction,1).\n",
	    mz->maxSlopeRatio);

  if (mz->extrapLimits<=0 || mz->extrapLimits>=0.5)
    fprintf(dout, "Warning: mz.extrapLimits= %g, should be in (0, 0.5).\n",
	    mz->extrapLimits);

  if (mz->interpLimits<=0 || mz->interpLimits>=0.5)
    fprintf(dout, "Warning: mz.interLimits= %g, should be in (0, 0.5).\n",
	    mz->interpLimits);

  if (mz->stepBound<2.0)
    fprintf(dout, "Warning: mz.stepBound= %g, should be >= 2\n.",
	    mz->stepBound);

  if (mz->maxExtrapol<1.0)
    fprintf(dout, "Warning: mz.maxExtrapol= %g, should be >= 1\n.",
	    mz->maxExtrapol);

  if (mz->extensor<1.0)
    fprintf(dout, "Warning: mz.extensor= %g, should be >= 1\n.",
	    mz->extensor);

  if (mz->lsFlexFuncEvals==0 || mz->lsFlexFuncEvals>mz->lsMaxFuncEvals)
    mz->lsFlexFuncEvals = mz->lsMaxFuncEvals;

}

/***********************************************************************
 *  Name:		summarizeLineSearch
 *  Description:
 *	Prints a summary of the line search to dout (stdout).
 *	The data is the stepsize, function value, and slope (if
 *	available.)
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *
 *  Return Value: none
 *
 ***********************************************************************/
static void	summarizeLineSearch(mz)
  Minimize	mz ;
{
  int i;
  if (mz->lsnFuncEvals > mz->lsnPoints)
    fprintf(dout, "Warning: line search (%d) exceeded max number of func evals (%d)\n.",
	    mz->lsnFuncEvals, mz->lsMaxFuncEvals);
  fprintf(dout, "  Step:     ");
  for (i=0; i<=mz->lsnFuncEvals; i++)
    fprintf(dout, " %-12g", mz->lsStep[i]);
  fprintf(dout, "\n  FuncValue:");
  for (i=0; i<=mz->lsnFuncEvals; i++)
    fprintf(dout, " %-12g", mz->lsFuncValue[i]);
  fprintf(dout, "\n  Slope:    ");
  for (i=0; i<=mz->lsnFuncEvals; i++)
    fprintf(dout, " %-12g", mz->lsSlope[i]);
  fprintf(dout, "\n");
}

/***********************************************************************
 *  Name:		printMethodInfo
 *  Description:
 *	Prints the methods being used and their parameters.
 *	The output is controlled by the mz->mzVerbosity flag.
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *
 *  Return Value: none
 *
 ***********************************************************************/

static void	printMethodInfo(mz)
  Minimize	mz ;
{
  double chksum;
  /* do a checksum on the parameters to see if they have changed */
  chksum = (mz->tolerance + mz->acceptableFuncMin + mz->alpha
	    + mz->qpEpsilon + mz->maxGrowthFactor + mz->gainIncrement
	    + mz->gainDecrement + mz->epsilon
	    + mz->initialStepFactor + mz->expectedFuncMin
	    + mz->backUpFactor
	    + mz->quasiNewton + mz->localMinima + mz->wolfeTest
	    + mz->stepBound + mz->extrapLimits + mz->extensor
	    + mz->maxExtrapol + mz->maxSlopeRatio + mz->minFuncReduction
	    + mz->machinePrecision);

  if (   mz->mzVerbosity>2
      || mz->mzVerbosity>0 && (   mz->nIterations==0
			       || chksum != mz->paramChksum
			       || mz->prevDirectionMethod!=mz->directionMethod
			       || mz->prevStepMethod!=mz->stepMethod)) {
    fprintf(dout, "Using %s and %s.\n",
	    methodName(mz->directionMethod), methodName(mz->stepMethod));
    if (   mz->stepMethod==MZRAYSLS 
	|| mz->stepMethod==MZRUDISLS 
	|| mz->stepMethod==MZSLOPESEARCH)
      fprintf(dout, "Initial step for line search is %s.\n",
	      methodName(mz->initialStepMethod));
    if (mz->continueDirection)
      fprintf(dout, "Continuing previous direction.\n");
    else if (mz->nIterations>0)
      fprintf(dout, "Starting new direction.\n");
    mz->paramChksum = chksum;
    fprintf(dout, "Minimize parameters: tolerance= %g acceptableFuncMin= %g\n",
	    mz->tolerance, mz->acceptableFuncMin);
    fprintf(dout, "Parameters for %s: ", methodName(mz->directionMethod));
    switch (mz->directionMethod) {
    case MZSTEEPEST:
    case MZCONJGRAD:
    case MZRUDICG:
    case MZCONJGRADRST:
      /* no parameters to print */
      fprintf(dout, "(none)\n");
      break;
    case MZMOMENTUM:
      fprintf(dout, "momentum(alpha)= %g\n", mz->alpha);
      break;
    case MZQUICKPROP:
      fprintf(dout, "\n  qpEpsilon= %g maxGrowthFactor= %g\n",
	      mz->qpEpsilon, mz->maxGrowthFactor);
      break;
    case MZDELTABARDELTA:
      fprintf(dout,
	      "\n  gainIncrement= %g gainDecrement= %g momentum(alpha)= %g\n",
	      mz->gainIncrement, mz->gainDecrement, mz->alpha);
      break;
    default:
      IErrorAbort("printMethodInfo: bad directionMethod %d",
		  mz->directionMethod);
    }
    fprintf(dout, "Parameters for %s: ", methodName(mz->stepMethod));
    switch (mz->stepMethod) {
    case MZFIXEDSTEP:
      fprintf(dout, "epsilon= %g\n", mz->epsilon);
      break;
    case MZRUDISLS:
      fprintf(dout,
	      "\n  maxSlopeRatio= %g minFuncReduction= %g machinePrecision= %.2g\n",
	      mz->maxSlopeRatio, mz->minFuncReduction, mz->machinePrecision);
      fprintf(dout, "  backUpFactor= %g\n", mz->backUpFactor);
      break;
    case MZRAYSLS:
    case MZSLOPESEARCH:
      fprintf(dout,
	      "\n  maxSlopeRatio= %g minFuncReduction= %g machinePrecision= %.2g\n",
	      mz->maxSlopeRatio, mz->minFuncReduction, mz->machinePrecision);
      fprintf(dout, "  stepBound= %g extrapLimits= %g interpLimits= %g\n",
	      mz->stepBound, mz->extrapLimits, mz->interpLimits);
      fprintf(dout, "  extensor= %g maxExtrapol= %g backUpFactor= %g\n",
	      mz->extensor, mz->maxExtrapol, mz->backUpFactor);
      fprintf(dout, "  quasiNewton= %d localMinima= %d wolfeTest= %d\n",
	      mz->quasiNewton, mz->localMinima, mz->wolfeTest);
      break;
    default:
      IErrorAbort("printMethodInfo: bad stepMethod %d",
		  mz->stepMethod);
    }
    if (mz->stepMethod!=MZFIXEDSTEP) {
      fprintf(dout, "Parameters for %s initial step: ",
	      methodName(mz->initialStepMethod));
      switch (mz->initialStepMethod) {
      case MZDRATIO:
      case MZISSLEN:
	fprintf(dout, "(none)\n");
	break;
      case MZISNEW:
      case MZISASK:
      case MZSHOOT:
      case MZDRATIOSHOOT:
	fprintf(dout, "\n  initialStepFactor= %g expectedFuncMin = %g\n",
		mz->initialStepFactor, mz->expectedFuncMin);
	break;
      default:
	IErrorAbort("printMethodInfo: bad initialStepMethod %d",
		    mz->initialStepMethod);
      }
    }
  }
}

static int	isCGMethod(method)
  int		method;
{
  return method==MZCONJGRAD ||method==MZRUDICG ||method==MZCONJGRADRST;
}

/***********************************************************************
 *  Name:		minimize0
 *  Description:
 *	This function contains the minimization code - it has a
 *	loop in which it calls the appropriate methods for directions
 *	and line searches and tests for termination.
 *
 *	(This function is a rewritten version of Rudi Mathon's varmin
 *	with more comments and more informatative variable names.)
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *I			fEval	- function to evaluate f (but not gradients)
 *I			gEval	- function to evaluate gradients (but not f)
 *I			fgEval	- function to evaluate both f and gradients
 *I	int		n	- the dimensionality of the vectors
 *IO	Real		*x	- current position
 *IO	Real		*grad	- gradient
 *IO	Real		*bestF	- work vector (best grad in LS by F)
 *IO	Real		*bestD	- work vector (best grad in LS by D)
 *IO	Real		*search	- search direction
 *IO	Real		*start	- work vector
 *IO	Real		*grad	- work vector
 *IO	Real		*rstd	- work vector
 *IO	Real		*rsty	- work vector
 *
 *  Return Value: none
 *
 ***********************************************************************/

static int minimize0 (mz, fEval, gEval, fgEval, n, x, 
		      grad, bestF, bestD, search, start, pgrad, rstd, rsty)
  Minimize		mz ;
  RealVecFunc		fEval ;
  VecProc		gEval ;
  Real2VecFunc		fgEval ;
  int			n ;
  Real			*x ;
  Real			*grad ;
  Real			*bestF ;
  Real			*bestD ;
  Real			*search ;
  Real			*start ;
  Real			*pgrad ;
  Real			*rstd ;
  Real			*rsty ;
{
  int 	 printed_result = 0 ;
  double f;		/* function value */
  double a;		/* step size */
  double d;		/* slope of current direction
			   (dot product of gradient and direction) */
  double d_prev;	/* previous slope */
  double x_len;		/* |x| */
  double g_len;		/* |grad| */
  double s_len;		/* |search| */
  double d_ratio;	/* ratio of current and previous slope */
  double delta_f;	/* change in function value achieved by line search */
  double f_prev;	/* previous function value */
  double start_cos;	/* cosine of gradient & direction at start of step */
  double end_cos;	/* cosine of gradient & direction at end of step */
  int vb = mz->mzVerbosity;
  int db = mz->mzDebug;

  copyVector(n, x, start);
  printMethodInfo(mz);

  /***
   * First see if we are at the same position as before, and if not,
   * intialize all the local variables.  Otherwise, restore all the
   * local variables from saved values.
   */
  if (!mz->samePosition) {
    DB(1, db, "minimize0: at new position - initializing local variables\n");
    if (mz->maxFuncEvals>0 && mz->nFuncEvals>=mz->maxFuncEvals) {
      fprintf(dout, "Warning: already computed maximum number of function evaluations (%d).\n",
	 mz->maxFuncEvals);
      mz->mzResult = resultCodeStr(mz->mzResultCode = MZMAXF);
      return MZMAXF;
    }
    if (mz->maxIterations>0 && mz->nIterations>=mz->maxIterations) {
      fprintf(dout, "Warning: already computed maximum number of iterations (%d).\n",
	 mz->maxIterations);
      mz->mzResult = resultCodeStr(mz->mzResultCode = MZMAXITER);
      return MZMAXITER;
    }

    IDoTrace(&beginMinimize) ;
    mz->evalReason = "Initial evaluation";
    if (fgEval)
      mz->E = f = fgEval(mz, n, x, grad);
    else {
      mz->E = f = fEval(mz, n, x);
      gEval(mz, n, grad);
    }
    mz->nFuncEvals++;
    a		=	0.0;
    delta_f	=	0.0;
    d		=	d_prev	=	-dotProduct(n, grad, grad);
    x_len	=	vectorLength(n, x);
    s_len	=	g_len	=	sqrt(-d);
    start_cos	=	1.0;
    end_cos	=	1.0;
    if (mz->nIterations) /* continuing on same problem, new direction */
      mz->nIterations++;
    mz->lsResultCode	=	0;
    mz->lsResult	=	"";
  } else {
    IDoTrace(&beginMinimize) ;
    DB(1, db, "minimize0: continuing previous minimization - restoring local variables\n");
    f		=	mz->E;
    a		=	mz->a;
    delta_f	=	mz->delta_f;
    d		=	mz->d;
    d_prev	=	mz->d_prev;
    x_len	=	mz->x_len;
    g_len	=	mz->g_len;
    s_len	=	mz->s_len;
    start_cos	=	mz->start_cos;
    end_cos 	=	mz->end_cos;
  }


  if (g_len < mz->tolerance*MAX(1.0, x_len)) {
    fprintf(dout, "Warning: gradient too small at starting point.\n");
    mz->mzResultCode = MZCONVERGE;
    goto getout;
  }

  VB(1, vb,
     "iter= %-4d nFE= %-4d f= %-14.8g |g|= %-7.2g d= %-9.3g dr= %-8.2g\n",
     mz->nIterations, mz->nFuncEvals, mz->E, g_len, d_prev, d/d_prev);

  logParameters(mz);

  /* main iteration loop */
  for (;;) {
    /* tests at the beginning so that they get performed when continuing */
    DB(1, db, "minimize0: testing termination conditions.\n");
    if (!(mz->lsResultCode==MZSUCCEED || mz->lsResultCode==0
	  || mz->lsResultCode==MZMAXFLINE)) {
      DB(1, db, "minimize0: line search failed.\n");
      mz->mzResultCode = mz->lsResultCode;
      goto getout;
    }
    
    /* convergence test */
    
    if (g_len < mz->tolerance*MAX(1.0, x_len)) {
      DB(1, db, "minimize0: search converged.\n");
      mz->mzResultCode = MZCONVERGE;
      goto getout;
    }
    
    if (f <= mz->acceptableFuncMin) {
      DB(1, db, "minimize0: achieved acceptable function value.\n");
      mz->mzResultCode = MZOKFMIN;
      goto getout;
    }
    
    if (mz->maxFuncEvals>0 && mz->nFuncEvals >= mz->maxFuncEvals) {
      DB(1, db, "minimize0: too many function evaluations.\n");
      mz->mzResultCode = MZMAXF;
      goto getout;
    }
    
    if (mz->maxIterations>0 && mz->nIterations >= mz->maxIterations) {
      DB(1, db, "minimize0: too many iterations.\n");
      mz->mzResultCode = MZMAXITER;
      goto getout;
    }
    
    if (mz->lsResultCode == MZMAXFLINEFAIL) {
      DB(1, db, "minimize0: too many function evaluations in line search.\n");
      mz->mzResultCode = MZMAXFLINEFAIL;
      goto getout;
    }
    
    if (mz->stopFlag) {
      DB(1, db, "minimize0: stopFlag set.\n");
      if (interrupted)		/* static global variable set by interrupt
				 * handler */
	mz->mzResultCode = MZINTERRUPTED;
      else
	mz->mzResultCode = MZSTOPPED;
      goto getout;
    }

    DB(1, db, "minimize0: not terminating, now choosing direction.\n");
    switch (mz->directionMethod) {
      
    case MZSTEEPEST :  /* gradient descent method */
      steepestDescent(mz, n, search, grad, &d);
      break;
      
    case MZMOMENTUM :  /* gradient descent method */
      momentumDirection(mz, n, search, grad, &d);
      break;
      
    case MZQUICKPROP :
      quickPropDirection(mz, n, search, grad, pgrad, &d);
      break;
      
    case MZDELTABARDELTA :
      deltaBarDeltaDirection(mz, n, search, grad, rstd, rsty, &d);
      break;
      
    case MZRUDICG :  /* conjugate gradient method */
      rudisConjugateGradient(mz, n, a, search, grad, pgrad, &d);
      if (d>0.0 || isNaN(d)) {
	VB(2, vb, "Search direction not descent (%g) - resetting\n", d);
	mz->continueDirection = 0;
	rudisConjugateGradient(mz, n, 0.0, search, grad, pgrad, &d);
      }
      break;
      
    case MZCONJGRAD :  /* conjugate gradient method */
      conjugateGradient(mz, n, search, grad, pgrad, &d);
      if (d>0.0 || isNaN(d)) {
	VB(2, vb, "Search direction not descent (%g) - resetting\n", d);
	mz->continueDirection = 0;
	conjugateGradient(mz, n, search, grad, pgrad, &d);
      }
      break;
      
    case MZCONJGRADRST :  /* conjugate gradient method with restarts */
      conjugateGradientRestart(mz, n, a, search, grad, pgrad, bestF,
			       rstd, rsty, &d, g_len);
      if (d>0.0) {
	VB(2, vb, "Search direction not descent (%g) - resetting\n", d);
	mz->continueDirection = 0;
	conjugateGradientRestart(mz, n, 0.0, search, grad, pgrad, bestF,
				 rstd, rsty, &d, g_len);
      }
      break;
      
    default:
      fprintf(dout, "Warning: direction method %d not implemented.\n",
	      mz->directionMethod);
      mz->mzResultCode = MZBADMETHOD;
      goto getout;
    }
    
    mz->continueDirection = 1;
    s_len = vectorLength(n, search);
    
    if (d > 0.0) {
      if (isCGMethod(mz->directionMethod)) {
	DB(1, db, "minimize0: terminating, search vector not descent for CG method.\n");
	mz->mzResultCode = MZNOTDESCENT;
	goto getout;
      }
    }
    
    mz->nIterations++;
    
    d_ratio = d_prev/d; /* old_d/new_d */

    logInitialStepData(mz, a, f, d, d_ratio, delta_f, g_len, s_len);
    DB(1, db, "minimize0: choosing initial step.\n");

    switch (mz->initialStepMethod) {
    case MZISASK:
      a = askStep(mz, a, f, d, d_ratio, delta_f, g_len, s_len);
      break;
    case MZISNEW:
      a = newStep(mz, a, f, d, d_ratio, delta_f, g_len, s_len);
      break;
    case MZSHOOT:
      a = shootStep(mz, f, d, delta_f);
      break;
    case MZISSLEN:
      a = sLenStep(s_len);
      break;
    case MZDRATIO:
      a = dRatioStep(a, g_len, d_ratio);
      break;
    case MZDRATIOSHOOT:
      a = (a<=0.0) ?shootStep(mz, f, d, delta_f):dRatioStep(a, g_len, d_ratio);
      break;
    default:
      fprintf(dout, "Warning: initial step method %d not implemented.",
		  mz->initialStepMethod);
      mz->mzResultCode = MZBADMETHOD;
      goto getout;
    }
    
    if (mz->nrs == 1 && mz->directionMethod==MZCONJGRADRST) {
      VB(2, mz->lsVerbosity,
	 "Resetting initial step (a) to 1 because nrs==0\n");
      a = 1.0;
    }
    
    d_prev = d;
    f_prev = f;
    copyVector(n, x, start);
    copyVector(n, grad, pgrad);

    insertLSData(mz, 0, 0.0, f, d, 1, 0);
    DB(1, db, "minimize0: taking step.\n");

    switch (mz->stepMethod) {
      /***
       * The step method takes as input:
       *   a		the initial step (can be ignored by the method)
       *   start	the start point
       *   search	the direction to move in
       *   d		slope (dotproduct of direction and gradient) at start
       * The step method must set:
       *   a		the actual step taken
       *   f		the value of the function at the new point
       *   x		= start + a*search
       *   d		slope at the new point (only used for cos diagnostic)
       *   grad	the gradient at the new point
       */
    case MZFIXEDSTEP:
      fixedStep(mz, n, start, search, x, grad, fEval, gEval, fgEval,
		mz->epsilon, &a, &f, &d);
      mz->lsResult = resultCodeStr(mz->lsResultCode);
      break;
      
    case MZRUDISLS:
      rudisLineSearch(mz, n, start, search, x, grad, fEval, gEval, fgEval,
		      s_len, x_len, &a, &f, &d);
      mz->lsResult = resultCodeStr(mz->lsResultCode);
      VB(3, vb, "rudisLineSearch: %s\n", mz->lsResult);
      break;
      
    case MZRAYSLS:
      raysLineSearch(mz, n, start, search, x, grad, fEval, gEval, fgEval,
		     &a, &f, &d);
      mz->lsResult = resultCodeStr(mz->lsResultCode);
      VB(3, vb, "raysLineSearch: %s\n", mz->lsResult);
      break;

    case MZSLOPESEARCH:
      tapsLineSearch(mz, n, start, search, x, grad, fEval, gEval, fgEval,
		     &a, &f, &d);
      mz->lsResult = resultCodeStr(mz->lsResultCode);
      VB(3, vb, "tapsLineSearch: %s\n", mz->lsResult);
      break;

    default:
      fprintf(dout, "Warning: step method %d not implemented.",
	      mz->stepMethod);
      mz->lsResultCode = mz->mzResultCode = MZBADMETHOD;
      goto getout;
    }
    
    mz->evalReason = "Line search ended";
    mz->E = f;
    delta_f = f_prev - f;
    start_cos = -d_prev/(s_len*g_len);
    x_len = vectorLength(n, x);
    g_len = vectorLength(n, grad);
    end_cos = -d/(s_len*g_len);

    if (mz->lsnFuncEvals > mz->lsnPoints)
      fprintf(dout,
	"Warning: line search (%d) exceeded max number of func evals (%d)\n.",
	      mz->lsnFuncEvals, mz->lsMaxFuncEvals);
    if (mz->lsSummarize)
      summarizeLineSearch(mz);
    if (mz->lsPlot)
      plotLineSearch(mz, n, start, search, fEval, gEval, fgEval, a);
    VB(2, vb, "Line search %d steps, initial a= %g final a = %g dratio= %g\n",
       mz->lsnFuncEvals, mz->lsStep[1], a, d/d_prev);
    
    logLineSearchData(mz);
    if (mz->incIter && mz->incIter != (voidFunc)NULL_proc)
      (*mz->incIter)(mz) ;

    IDoTrace(&minimizeEndIteration) ;
    
    printed_result = 0;
    if (mz->repetitionCount==0 || mz->nIterations%mz->repetitionCount==0) {
      VB(1, vb,
	 "iter= %-4d nFE= %-4d f= %-14.8g |g|= %-7.2g d= %-9.3g dr= %-8.2g\n",
	 mz->nIterations, mz->nFuncEvals, mz->E, g_len, d_prev, d/d_prev);
      IDoTrace(&minimizeEndRepetition) ;
      printed_result = 1;
    }
  }

 getout:
  if (!printed_result) {
    VB(1, vb,
       "iter= %-4d nFE= %-4d f= %-14.8g |g|= %-7.2g d= %-9.3g dr= %-8.2g\n",
       mz->nIterations, mz->nFuncEvals, mz->E, g_len, d_prev, d/d_prev);
    IDoTrace(&minimizeEndRepetition) ;
  }
  mz->a = a;
  mz->delta_f = delta_f;
  mz->d = d;
  mz->d_prev = d_prev;
  mz->x_len = x_len;
  mz->g_len = g_len;
  mz->s_len = s_len;
  mz->start_cos = start_cos;
  mz->end_cos = end_cos;

  return mz->mzResultCode;
}

/***********************************************************************
 *  Name:		interruptMinimize
 *  Description:
 *	Handle an interrupt while in minimize.  The pointer stop_flag_p
 *	must be set to point to the stopFlag field in the current
 *	minimize record.
 *	The action taken on getting an interrupt is to print a message,
 *	increment the stop flag, and return (continuing processing).
 *	If more than three interrupts in a row are recieved, a long
 *	jump is made to a point set from minimize().
 *			
 *  Parameters:	none
 *  Return Value: none
 *
 ***********************************************************************/

static jmp_buf jmp_buf_env;
static int *stop_flag_p = NULL;

static int	interruptMinimize()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGINT, interruptMinimize);
#endif
  interrupted = TRUE ;
  if (stop_flag_p==NULL)
    IErrorAbort("interruptMinimize: stop flag address is NULL");
  switch (*stop_flag_p) {
  case 0:
    fprintf(dout, "Interrupt detected: will stop at end of step or linesearch\n");
    (*stop_flag_p)++;
    break;
  case 1:
    fprintf(dout, "2nd Interrupt detected: will stop at end of current epoch\n");
    (*stop_flag_p)++;
    break;
  case 2:
    fprintf(dout, "3rd Interrupt detected: aborting immediately\n");
    (*stop_flag_p)++;
    longjmp(jmp_buf_env, 1);
    IErrorAbort("interruptMinimize: failed longjump");
    break;
  default:
    fprintf(dout, "interruptMinimize: bad value for stop flag (%d)",
	    *stop_flag_p);
    longjmp(jmp_buf_env, 1);
    IErrorAbort("interruptMinimize: failed longjump");
    break;
  }
  return -1 ;
}

/***********************************************************************
 * Name:		minimize
 * Description:
 *	A wrapper for minimize0.  Takes care of some book-keeping,
 *	e.g. setting interrupts and long jumps, calling functions
 *	to check parameters.  Also checks if it is OK to continue
 *	a previous search direction.
 *
 *	This is the function to call from C code.
 *
 * Parameters:
 *	Minimize	mz	- the minimize record
 *	int		nEvals - the number of function evals allowed
 *	int		iterations - the number of iterations allowed
 *
 * The following functions must be supplied in the minimize record.
 *
 *	int		getNVars()
 *	  returns the number of values
 *
 *	int		getValues(n, x)
 *	  loads the values into x
 *
 *	int		setValues(n, x)
 *	  loads the values from x 
 *
 *	Real		fEval(n, x)
 *	  returns the value of the function at the given values.
 *
 *	void		gEval(n, grad)
 *	  evaluates the gradient for the previous call of fEval (no return val)
 *
 *	Real		fgEval(n, x, grad)
 *	  evaluates the function and the gradient at the given values.
 *
 *	char *		valueName(n)
 *	  returns a string containing the name for value n (for debugging)
 *
 *	It is not necessary to supply all 3 network evaluation functions -
 *	it is OK to supply just fEval and gEval, or just fgEval.  Ray's
 *	line search does not need the gradient if the function value is
 *	not OK, thus computation can be saved by only evaluating f.
 *	Here are rules for which to supply.
 *	Supply fEval if it is cheaper than fgEval.
 *	Supply gEval if fEval is supplied and {fEval();gEval();}
 *        is cheaper than {fEval();fgEval();}, or if fgEval is not
 *	  available.
 *	Supply fgEval if fgEval() is cheaper than fEval();gEval(),
 *	  or if fEval and gEval are not available separately.
 *
 *	The same function should not be supplied to do more than one task.
 *
 * Return Value:	none
 ***********************************************************************/

void minimize (mz, evaluations, iterations)
  Minimize		mz ;
  int			evaluations ;
  int			iterations ;
{
  char *because = " (no previous direction)";
  SignalHandler	old_signal_handler ;
  int new_method = 0;
  Real x_len;

  intFunc		getNVars  = mz->getNVars;
  VecProc		getValues = mz->getValues;
  VecProc		setValues = mz->setValues;
  RealVecFunc		fEval     = mz->fEval;
  VecProc		gEval     = mz->gEval;
  Real2VecFunc		fgEval    = mz->fgEval;
  strFunc		valueName = mz->valueName;
  int db;

  if (valueName==NULL)
    valueName = simpleName;

  /* call check parameters first so that max # of function evals in
     line search can get set */
  checkParameters(mz);
  mz = initMinimize(mz, getNVars(mz));
  db = mz->mzDebug;

#ifdef CHECK_FUNCTIONS
  if (mz->doNotCheckFunctions != 1 &&
      mz->doNotCheckFunctions != ((int)fEval ^ (int)gEval ^ (int)fgEval)) {
    checkOutFunctions(getNVars, getValues, setValues,
		      fEval, gEval, fgEval, valueName, "minimize",
		      mz->checkOutVerbosity);
    mz->doNotCheckFunctions = ((int)fEval ^ (int)gEval ^ (int)fgEval);
    fprintf(dout, "%s\n%s\n%s\n%s\n\n",
	    "Checks completed (and will not be repeated for this network).",
	    "To prevent checks being performed on other networks, set the",
	    "field currentNet.mz.doNotCheckFunctions to 1,",
	    "or use the check option of the command \"minimize\".");
  }
#endif				/* CHECK_FUNCTIONS */
  
  stop_flag_p = &mz->stopFlag;
  interrupted = FALSE ;
  old_signal_handler = (SignalHandler)signal(SIGINT, interruptMinimize);
  if (setjmp(jmp_buf_env)) {
    signal(SIGINT, old_signal_handler);
    fprintf(dout, "\nMinimize: interrupted.  Net values in unknown state.\n");
    stop_flag_p = NULL;
    mz->mzResult = resultCodeStr(mz->mzResultCode = MZINTERRUPTED);
    return;
  }

  getValues(mz, mz->n, mz->x);

  if (mz->mzResultCode == MZINTERRUPTED) {
    because = " (because minimize was interrupted)";
    DB(1, db, "minimize: previous invocation appears to have been interrupted.\n");
  }

  if (mz->mzResultCode == 0) {
    because = " (first invocation or abnormal termination)";
    DB(1, db, "minimize: first invocation or previous invocation terminated abnormally.\n");
  }

  if (mz->prevDirectionMethod != mz->directionMethod) {
    new_method = 1;
    because = " (because of changed method)";
    DB(1, db, "minimize: new method being used.\n");
  }

  /***
   * check if we have what looks like the same x vector as before - if
   * so, don't reset the counters
   */
  x_len = dotProduct(mz->n, mz->x, mz->x);
  if (mz->x_len == -1.0) {
    because = " (because mz->xLen == -1)";
  } else if (x_len != mz->xLen) {
    if (mz->nIterations!=0) {
      fprintf(dout, "Warning: new x vector - resetting iteration and evaluation counters.\n");
      because = " (because of new x vector)";
    }
    mz->nIterations = 0;
    mz->nFuncEvals = 0;
  }

  if (mz->nIterations==0 || new_method==1 || x_len != mz->xLen
      || mz->mzResultCode==MZINTERRUPTED || mz->mzResultCode==0) {
    if (mz->continueDirection)
      fprintf(dout, "Warning: cannot continue previous direction%s.\n",
	      because);
    mz->continueDirection = 0;
  }

  if (mz->nIterations==0 || x_len != mz->xLen || mz->xLen == -1
      || mz->mzResultCode==MZINTERRUPTED || mz->mzResultCode==0) {
    mz->samePosition = 0;
    DB(1, db, "minimize: new position.\n");
  } else {
    mz->samePosition = 1;
    DB(1, db, "minimize: same position as previous invocation.\n");
  }

  mz->evalReason = "";
  mz->mzResultCode = 0;
  mz->mzResult = "";
  mz->lsResultCode = 0;
  mz->lsResult = "";

  if (evaluations>0) {
    DB(1, db, "minimize: setting maxFuncEvals to nFuncEvals (%d) + %d.\n",
       mz->nFuncEvals, evaluations);
    mz->maxFuncEvals = mz->nFuncEvals + evaluations;
  }
  if (iterations>0) {
    DB(1, db, "minimize: setting maxIterations to nIterations (%d) + %d.\n",
       mz->nIterations, iterations);
    mz->maxIterations = mz->nIterations + iterations;
  }

  DB(1, db, "minimize: calling minimize0.\n");
  minimize0(mz, fEval, gEval, fgEval, mz->n, mz->x, mz->gradient,
	    mz->bestGradientF, mz->bestGradientD,
	    mz->search, mz->start, mz->prevGradient,
	    mz->rstd, mz->rsty);

  DB(1, db,
     "minimize: setting values in net to those returned by minimize0.\n");
  setValues(mz, mz->n, mz->x);

  mz->xLen = dotProduct(mz->n, mz->x, mz->x);
  mz->prevDirectionMethod = mz->directionMethod;
  mz->prevStepMethod = mz->directionMethod;

  mz->mzResult = resultCodeStr(mz->mzResultCode);
  stop_flag_p = NULL;
  interrupted = FALSE ;

  IDoTrace(&endMinimize) ;

  signal(SIGINT, old_signal_handler);
}
