
/**********************************************************************
 * $Id: initialstep.c,v 1.2 92/11/30 11:30:47 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <xerion/useful.h>
#include "minimize.h"
#include "initialstep.h"

/***********************************************************************
 *  Name:		shootStep
 *  Description:
 *	Computes an initial step, based on shooting for the previous
 *	reduction in f.
 *	Originally from Ray's line search.
 *			
 *  Parameters:	
 *I	Minimize	mz,	- the minimize record
 *I	double		f	- the current value of f
 *I	double		d	- the slope
 *I	double		delta_f	- the previous reduction in f
 *
 *  Return Value:
 * 	double		shootStep
 *				- the size of the step
 *
 ***********************************************************************/

double shootStep(mz, f, d, delta_f)
  Minimize mz;
  double f;
  double d;
  double delta_f;
{
  double a;
  if (delta_f == 0.0)
    delta_f = f - mz->expectedFuncMin;
  a = - delta_f / d * mz->initialStepFactor;
  return a;
}

/***********************************************************************
 *  Name:		dRatioStep
 *  Description:
 *	Computes an initial step, based on the previous step and
 *	current and previous slopes.
 *	The first step (indicated by a==0.0) is based on the euclidean
 *	length of the gradient vector (which for first steps is the
 *	same as the search vector.)
 *	Originally from Rudi's line search.
 *			
 *  Parameters:	
 *I	double		a	- the previous stepsize
 *I	double		g_len	- the length of the g vector
 *I	double		d_ratio	- the slope ratio
 *
 *  Return Value:
 * 	double		dRatioStep
 *				- the size of the step
 *
 ***********************************************************************/

double dRatioStep(a, g_len, d_ratio)
  double a;
  double g_len;
  double d_ratio;
{
  if (a<=0.0)
    a = 1.0/g_len;
  else
    a *= d_ratio;
  return a;
}

/***********************************************************************
 *  Name:		sLenStep
 *  Description:
 *	Computes an initial step, based on the length of the search vector.
 *			
 *  Parameters:	
 *I	double		s_len	- the length of the search vector
 *
 *  Return Value:
 * 	double		sLenStep
 *				- the size of the step
 *
 ***********************************************************************/

double sLenStep(s_len)
  double s_len;
{
  double a;
  a = 1.0/s_len;
  return a;
}

/***********************************************************************
 *  Name:		askStep
 *  Description:
 *	Prints out various values that could be used to estimate the
 *	initial step.  The steps that the three other methods
 *	would produce are also printed.  The user can enter a value.
 *	The default value is the value produced by dRatioStep.
 *			
 *  Parameters:
 *I	Minimize	mz	- the minimize record
 *I	double		a	- the previous stepsize
 *I	double		f	- the current value of f
 *I	double		d	- the slope
 *I	double		d_ratio	- the slope ratio
 *I	double		delta_f	- the previous reduction in f
 *I	double		g_len	- the length of the g vector
 *I	double		s_len	- the length of the search vector
 *
 *  Return Value:
 * 	double		askStep
 *				- the size of the step
 *
 ***********************************************************************/

double askStep(mz, a, f, d, d_ratio, delta_f, g_len, s_len)
  Minimize mz;
  double a;
  double f;
  double d;
  double d_ratio;
  double delta_f;
  double g_len;
  double s_len;
{
  char buf[100];
  fprintf(dout, "Initial step data: previous step= %.6g f= %.6g d= %.6g dratio= %.6g\n  delta_f= %.6g g_len= %.6g s_len= %.6g\n",
	  a, f, d, d_ratio, delta_f, g_len, s_len);
  fprintf(dout, "sLenStep= %g dRatioStep= %g shootStep= %g newStep= %g\n",
	  sLenStep(s_len), dRatioStep(a, g_len, d_ratio),
	  shootStep(mz, f, d, delta_f),
	  newStep(mz, a, f, d, d_ratio, delta_f, g_len, s_len));
  a = dRatioStep(a, g_len, d_ratio);
  buf[0] = '\0';
  fprintf(dout, "Enter the step size [%g]: ", a);
  fgets(buf, 100, din);
  if (atof(buf)>0.0) {
    a = atof(buf);
    fprintf(dout, "Taking step %g\n", a);
  }
  return a;
}



