
/**********************************************************************
 * $Id: fixedstep.c,v 1.2 92/11/30 11:30:46 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <xerion/useful.h>
#include "minimize.h"
#include "linesearch.h"
#include "fixedstep.h"

/***********************************************************************
 *  Name:		fixedStep
 *  Description:
 *	Makes a move from start along the search direction.  The size
 *	of the step is specified by epsilon.  The function is evaluated
 *	at the end of the step, and the value and gradient returned.
 *			
 *  Parameters:	
 *I	Minimize	mz,	- minimize record
 *I	int		n,	- dimensionality of the vectors
 *I	Real		*start	- start point from which to search
 *I	Real		*search	- search direction
 *O	Real		*x,	- returns the end point
 *O	Real		*grad,	- returns the gradient at chosen step
 *I	RealVecFunc	fEval,	- function to evaluate f
 *I	VecProc		gEval,	- function to evaluate g
 *I	Real2VecFunc	fgEval,	- function to evaluate f ang g
 *I	double		epsilon - stepsize
 *O	double		*ap,	- returns chosen stepsize
 *IO	double		*fp,	- function value at x, returns value at new x
 *IO	double		*dp	- slope a x, returns slope at new x
 *
 *  Return Value:
 *	int	fixedStep - return code:
 *	Possible values:
 *	MZSUCCEED    - the step was made.
 *
 ***********************************************************************/

int fixedStep (mz, n, start, search, x, grad, fEval, gEval, fgEval, 
	       epsilon, ap, fp, dp)
  Minimize		mz ;
  int			n ;
  Real			*start ;
  Real			*search ;
  Real			*x ;
  Real			*grad ;
  RealVecFunc		fEval ;
  VecProc		gEval ;
  Real2VecFunc		fgEval ;
  double		epsilon ;
  double		*ap ;
  double		*fp ;
  double		*dp ;
{
  double f;
  int i;
  mz->evalReason = "Taking fixed step";
  
  if (mz->lsVerbosity>3) {
    fprintf(dout, "%-12s %-12s %-12s %-12s epsilon = %g\n",
	    "Old", "grad", "update", "New", epsilon);
    for (i=0; i<n; i++)
      fprintf(dout, "%-12g %-12g %-12g %-12g\n",
	      x[i], grad[i], epsilon*search[i], x[i]+epsilon*search[i]);
  }

  moveInDirection(n, start, search, epsilon, x);

  if (fgEval)
    f = fgEval(mz, n, x, grad);
  else {
    f = fEval(mz, n, x);
    gEval(mz, n, grad);
  }

  mz->nFuncEvals++;

  *fp = f;
  *ap = epsilon;
  *dp = dotProduct(n, grad, search);

  insertLSData(mz, 1, epsilon, f, *dp, 1, 0);

  return mz->lsResultCode = MZSUCCEED;
}

