
/**********************************************************************
 * $Id: datalog.c,v 1.2 92/11/30 11:30:44 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <errno.h>

extern char	*sys_errlist[] ;

#include <xerion/useful.h>

#include "minimize.h"
#include "datalog.h"

static int warned_no_open = 0;	/* used to remember that we warned
				   about not being able to open a file */

static int hash(str)
  char *str;
{
  int i, a = 0;
  for (i=0; str[i]!='\0'; i++)
    a = a*10 + str[i];
  return a;
}

/***********************************************************************
 *  Name:		logInitialStepData
 *  Description:
 *	Logs data which is (or could be) used to estimate the initial
 *	step size.  The file is mz->logFile.  If the file cannot be
 *	openned a warning is given, and the file name is remembered
 *	as a hash value (in warned_no_open) so that the warning is
 *	only given once.
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *I	double		a	- the previous stepsize
 *I	double		f	- the function value
 *I	double		d	- the slope
 *I	double		d_ratio	- the slope ratio
 *I	double		delta_f	- the reduction in f on the last step
 *I	double		g_len	- the length of the g vector
 *I	double		s_len	- the length of the search vector
 *
 *  Return Value: none
 *
 ***********************************************************************/

void logInitialStepData(mz, a, f, d, d_ratio, delta_f, g_len, s_len)
  Minimize mz ;
  double a;
  double f;
  double d;
  double d_ratio;
  double delta_f;
  double g_len;
  double s_len;
{
  FILE *file;
  if (mz->lsLogFile==NULL || mz->lsLogFile[0]=='\0')
    return;
  file = fopen(mz->lsLogFile, "a");
  if (file==NULL) {
    if (warned_no_open!=hash(mz->lsLogFile)) {
      fprintf(dout, "Warning: cannot open line search logfile \"%s\": %s.\n",
	      mz->lsLogFile, sys_errlist[errno]);
      warned_no_open = hash(mz->lsLogFile);
    }
  }
  fprintf(file, "I %.6g %.6g %.6g %.6g %.6g %.6g %.6g\n",
	  a, f, d, d_ratio, delta_f, g_len, s_len);
  fclose(file);
}

/***********************************************************************
 *  Name:		logLineSearchData
 *  Description:
 *	Logs data about the line search to the file mz->logFile.
 *	The data is the stepsize, function value, and slope (if
 *	available.)
 *	If the file cannot be openned a warning is given, and the
 *	file name is remembered as a hash value (in warned_no_open)
 *	so that the warning is only given once.
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *
 *  Return Value: none
 *
 ***********************************************************************/

void logLineSearchData(mz)
  Minimize mz;
{
  FILE *file;
  int i;
  if (mz->lsLogFile==NULL || mz->lsLogFile[0]=='\0')
    return;
  file = fopen(mz->lsLogFile, "a");
  if (file==NULL) {
    if (warned_no_open!=hash(mz->lsLogFile)) {
      fprintf(dout, "Warning: cannot open line search logfile \"%s\": %s.\n",
	      mz->lsLogFile, sys_errlist[errno]);
      warned_no_open = hash(mz->lsLogFile);
    }
  }
  fprintf(file, "L %d", mz->lsnFuncEvals);
  for (i=0; i<=mz->lsnFuncEvals; i++)
    fprintf(file, " %.6g %.6g %.6g",
	    mz->lsStep[i], mz->lsFuncValue[i], mz->lsSlope[i]);
  fprintf(file, "\n");
  fclose(file);
}

/***********************************************************************
 *  Name:		logParameters
 *  Description:
 *	Logs method parameters to the file mz->logFile.
 *	Headers for the columns of initial step data and line
 *	search data are also written.
 *	If the file cannot be openned a warning is given, and the
 *	file name is remembered as a hash value (in warned_no_open)
 *	so that the warning is only given once.
 *			
 *  Parameters:	
 *I	Minimize	mz	- the minimize record
 *
 *  Return Value: none
 *
 ***********************************************************************/

void logParameters(mz)
  Minimize mz;
{
  FILE *file;
  if (mz->lsLogFile==NULL || mz->lsLogFile[0]=='\0')
    return;
  file = fopen(mz->lsLogFile, "a");
  if (file==NULL) {
    if (warned_no_open!=hash(mz->lsLogFile)) {
      fprintf(dout, "Warning: cannot open line search logfile \"%s\": %s.\n",
	      mz->lsLogFile, sys_errlist[errno]);
      warned_no_open = hash(mz->lsLogFile);
    }
  }

  fprintf(file, "P dm=%d sm=%d ism=%d tol=%g mp=%g mfr=%g msr=%g isf=%g sb=%g el=%g qn=%d nlm=%d nwt=%d\n",
	  mz->directionMethod, mz->stepMethod, mz->initialStepMethod,
	  mz->tolerance, mz->machinePrecision, mz->minFuncReduction,
	  mz->maxSlopeRatio, mz->initialStepFactor, mz->stepBound,
	  mz->extrapLimits, mz->quasiNewton, mz->localMinima,
	  mz->wolfeTest);
  fprintf(file, "IF a f d d_ratio delta_f |g| |s|\n");
  fprintf(file, "LF n-steps {a f d}\n");
  fclose(file);
}
