
/**********************************************************************
 * $Id: layoutUtils.c,v 1.4 92/11/30 11:30:08 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include "layoutUtils.h"

int	layoutInputLine ;

#define GRANULARITY	10

typedef struct Pair {
  Group		group ;
  int		lastLaid ;
} Pair ;
static Pair	*lastLaid ;
static int	numGroups ;
static int	maxGroups ;

static Section	*sectionList ;
static int	numSections = 0 ;
static int	maxSections = 0 ;

static Section	*addSection  ARGS((SectionType	type)) ;

static int	markLastLaid ARGS((Net, Group	group, int	num)) ;
static void	buildPairs   ARGS((Group	group, void	*data)) ;
static int	getLastLaid  ARGS((Group	group)) ;

static int	currentRow ;

/***********************************************************************
 *	Name:		resetCurrentRow
 *	Description:	resets the current row to being 0
 *	Parameters:	NONE
 *	Return Value:	NONE
 ***********************************************************************/
void	resetCurrentRow () {
  currentRow = 0 ;
}
/**********************************************************************/



/***********************************************************************
 *	Name:		getCurrentRow
 *	Description:	returns the current row
 *	Parameters:	NONE
 *	Return Value:	NONE
 ***********************************************************************/
int	getCurrentRow() {
  return currentRow ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		incrementCurrentRow
 *	Description:	increments the current row by 1
 *	Parameters:	NONE
 *	Return Value:	NONE
 ***********************************************************************/
void	incrementCurrentRow() {
  ++currentRow ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		buildSections
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
Section		*buildSections (argvIn, argcIn)
  char		**argvIn ;
  int		argcIn ;
{
  int		argc, argcBase, value ;
  char		**argv, *ptr ;
  Section	*section ;

  numSections = 0 ;
  argcBase = 0 ;
  argc     = 0 ;
  do {				/* break up line into "," separated lists */
    /* skip leading comma's */
    argcBase += argc ;
    argv      = argvIn + argcBase ;
    for (argc = 0 ; 
	 argcBase + argc < argcIn && strcmp(argv[argc], ",") == 0 ; ++argc)
      ;
    /* read up to next comma or end of line */
    argcBase += argc ;
    argv      = argvIn + argcBase ;
    for (argc = 0 ;
	 argcBase + argc < argcIn && strcmp(argv[argc], ",") ; ++argc)
      ;

    if (argc == 0)
      break ;

    /* Now parse the sub list */
    if (strcmp(argv[0], "all") == 0) {
      if (argc != 2)
	IErrorAbort("Line %d: wrong number of arguments for \"all\"", 
		    layoutInputLine) ;
      section = addSection(All) ;
      section->name = strdup(argv[1]) ;

    } else if (strcmp(*argv, "section") == 0) {
      if (argc != 4)
	IErrorAbort("Line %d: wrong number of arguments for \"section\"",
		    layoutInputLine) ;
      section = addSection(Sect) ;
      section->name = strdup(argv[1]) ;

      value = strtol(argv[2], &ptr, 0) ;
      if (*ptr != '\0') 
	IErrorAbort("Line %d: bad first unit for \"section\" --- \"%s\"",
		    layoutInputLine, argv[2]) ;
      section->from = value ;

      value = strtol(argv[3], &ptr, 0) ;
      if (*ptr != '\0') 
	IErrorAbort("Line %d: bad last unit for \"section\" --- \"%s\"",
		    layoutInputLine, argv[3]) ;
      section->size = value - section->from + 1 ;

    } else if (strcmp(*argv, "next") == 0) {
      if (argc != 3)
	IErrorAbort("Line %d: wrong number of arguments for \"next\"",
		    layoutInputLine) ;
      section = addSection(Next) ;
      section->name = strdup(argv[1]) ;

      value = strtol(argv[2], &ptr, 0) ;
      if (*ptr != '\0') 
	IErrorAbort("Line %d: bad size for \"section\" --- \"%s\"",
		    layoutInputLine, argv[2]) ;
      section->size = value ;

    } else if (strcmp(*argv, "blank") == 0) {
      if (argc != 2)
	IErrorAbort("Line %d: wrong number of arguments for \"blank\"",
		    layoutInputLine) ;
      section = addSection(Blank) ;
      value = strtol(argv[1], &ptr, 0) ;
      if (*ptr != '\0') 
	IErrorAbort("Line %d: bad size for \"blank\" --- \"%s\"",
		    layoutInputLine, argv[1]) ;
      section->size = value ;

    } else if (strcmp(*argv, "fill") == 0) {
      if (argc != 1)
	IErrorAbort("Line %d: wrong number of arguments for \"fill\"",
		    layoutInputLine) ;
      section = addSection(Fill) ;

    } else {
      IErrorAbort("Line %d: unknown command \"%s\"", layoutInputLine, *argv) ;
      break ;
    }
  } while (argcIn > argcBase + argc) ;
    
  addSection(End) ;

  return sectionList ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		fillSections
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
Section		*fillSections(net, section, perRow)
  Net		net ;
  Section	*section ;
  int		perRow ;
{
  int		idx ;
  int		numKnown, numFills, fillSize ;
  Group		group ;

  numKnown = 0 ;
  numFills = 0 ;
  for (idx = 0 ; idx < numSections ; ++idx) {
    switch (section[idx].type) {
    case All:
      group = groupFromName(net, section[idx].name) ;
      if (group == NULL)
	IErrorAbort("Line %d: unknown group \"%s\"",
		    layoutInputLine, section[idx].name) ;
      section[idx].from = 0 ;
      section[idx].size = group->numUnits ;
      numKnown += section[idx].size ;
      markLastLaid(net, group, section[idx].from + section[idx].size) ;
      break ;

    case Sect:
      group = groupFromName(net, section[idx].name) ;
      if (group == NULL)
	IErrorAbort("Line %d: unknown group \"%s\"",
		    layoutInputLine, section[idx].name) ;
      section[idx].from = MIN(section[idx].from, group->numUnits) ;
      section[idx].size = MIN(section[idx].size,
			      group->numUnits - section[idx].from) ;
      numKnown += section[idx].size ;
      markLastLaid(net, group, section[idx].from + section[idx].size) ;
      break ;

    case Next:
      group = groupFromName(net, section[idx].name) ;
      if (group == NULL)
	IErrorAbort("Line %d: unknown group \"%s\"",
		    layoutInputLine, section[idx].name) ;

      section[idx].from = MIN(getLastLaid(group), group->numUnits) ;
      section[idx].size = MIN(section[idx].size,
			      group->numUnits - section[idx].from) ;
      numKnown += section[idx].size ;
      markLastLaid(net, group, section[idx].from + section[idx].size) ;
      break ;

    case Blank:
      numKnown += section[idx].size ;
      break ;

    case Fill:
      ++numFills ;
      break ;

    default:
      break ;
    }
  }
 
  if (numFills > 0)
    fillSize = (perRow - numKnown)/numFills ;
  else
    fillSize = 0 ;

  for (idx = 0 ; idx < numSections ; ++idx) {
    if (section[idx].type == Fill) {
      section[idx].size = fillSize ;
    }
  }

  return section ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		addSection
 *	Description:	adds an argument to the arglist for this command
 *	Parameters:	
 *		char	*section - pointer to the sectionument string
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static Section	*addSection(type)
  SectionType	type ;
{
  if (numSections >= maxSections) {
    if (maxSections == 0) {
      maxSections = numSections + GRANULARITY ;
      sectionList = (Section *)malloc(maxSections*sizeof(*sectionList)) ;
    } else {
      maxSections = numSections + GRANULARITY ;
      sectionList = (Section *)realloc(sectionList, 
				       maxSections*sizeof(*sectionList)) ;
    }	
    memset(sectionList + numSections, 0, 
	   (maxSections - numSections)*sizeof(*sectionList)) ;
  }
  if (sectionList[numSections].name != NULL) {
    free(sectionList[numSections].name) ;
  }
  memset(sectionList + numSections, 0, sizeof(*sectionList)) ;
  sectionList[numSections].type = type ;

  return (sectionList + numSections++) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		resetLastLaid
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
void	resetLastLaid() {
  if (lastLaid != NULL) {
    free(lastLaid) ;
    lastLaid = NULL ;
  }
  numGroups = 0 ;
  maxGroups = 0 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		markLastLaid
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static int	markLastLaid(net, group, num)
  Net		net ;
  Group		group ;
  int		num ;
{
  int	idx ;

  if (lastLaid == NULL) {
    maxGroups = GRANULARITY ;
    numGroups = 0 ;
    lastLaid = (Pair	*)malloc(maxGroups*sizeof(Pair)) ;
    netForAllGroups(net, ALL, buildPairs, NULL) ;
  }

  for (idx = 0 ; idx < numGroups ; ++idx) {
    if (group == lastLaid[idx].group) {
      lastLaid[idx].lastLaid = num ;
      break ;
    }
  }
  return 1 ;
}
/**********************************************************************/
static void	buildPairs(group, data)
  Group		group ; 
  void		*data ;
{
  if (numGroups >= maxGroups) {
    maxGroups = numGroups + GRANULARITY ;
    lastLaid  = (Pair	*)realloc(lastLaid, maxGroups*sizeof(Pair)) ;
  }
  lastLaid[numGroups].group    = group ;
  lastLaid[numGroups].lastLaid = 0 ;
  ++numGroups ;
}
/**********************************************************************/

/***********************************************************************
 *	Name:		getLastLaid
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static int	getLastLaid(group)
  Group		group ;
{
  int		idx ;

  for (idx = 0 ; idx < numGroups ; ++ idx) {
    if (group == lastLaid[idx].group) {
      break ;
    }
  }
  if (idx < numGroups)
    return lastLaid[idx].lastLaid ;
  else
    return 0 ;
}
/**********************************************************************/
