
/**********************************************************************
 * $Id: varsub.c,v 1.13 93/04/14 10:51:12 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <setjmp.h>
#include <signal.h>
#include "itf.h"
#include "lex.h"
#include "loop.h"
#include "varsub.h"

/* #define DBP(X) fprintf X */
#define DBP(X)

#define MIN(x,y)	((x) < (y) ? (x) : (y))

/*
 * VarSubstitute()
 *
 * Do variable subsitution on a line.  Neither quotes nor backslashes are
 * removed from the line.  The line is terminated by null.
 *
 * A variable is identified by the $ sign in front of it
 * Quotes are noticed, variables are substituted inside double quotes but
 * not inside single quotes.
 * A variable may be written as $var or ${var}. Variables must be composed
 * solely of alphanumeric characters and (including _).
 */
#define VarNameSpace(start,pos)	(MAX_VAR_NAME - 1 - ((pos) - (start)))

static String	getCommandOutput ARGS((String	command, int echo)) ;
static char	*variableValue ARGS((char *, char *, int, int)) ;
static char	*parenValue    ARGS((char *, char *, int, int)) ;

static char	*parenValue(inline, outline, max_value_len, echo) 
  char	*inline ;
  char	*outline ;
  int	max_value_len ;
  int	echo ;
{
  int	escape_pending = 0 ;	/* 1 if last char was a '\' 		*/
  char	var_name[MAX_VAR_NAME] ;/* name of variable			*/
  char	*var_char ;		/* current position in the var name	*/
  char	*value ;		/* the value of the command		*/
  int	in_paren ;		/* the paren we're inside		*/
  int	look_for_paren ;	/* the closing paren to look for	*/
  int	in_quote ;		/* what quote we're in			*/

  DBP((stderr, "parenValue: Raw input line is \"%s\"\n",inline));

  in_paren = *(inline++) ;
  if (in_paren != '{' && in_paren != '(')
    return NULL ;

  look_for_paren = in_paren == '(' ? ')' : '}' ;
  in_quote = 0 ;
  var_char = var_name ;

  while(*inline && look_for_paren) {
    int	   escaped = escape_pending ;
    escape_pending = 0 ;

    if (*inline == '$' && !escaped && in_quote != '\'') {
      /* start of variable */
      inline = variableValue(inline, var_char, 
			     VarNameSpace(var_name, var_char), echo) ;
      if (inline == NULL)
	return NULL ;
      var_char = var_name + strlen(var_name) ;
    } else if (*inline == '#'  && !escaped && !in_quote) {
      while (*(++inline))
	;
    } else {
      if (!escaped) {
	switch (*inline) {
	case '\\':		/* escape character - set escape pending */
	  escape_pending = 1;
	  break;

	case '\'':		/* single or double quote character */
	case '"':	
	  if (!in_quote) 
	    in_quote = *inline ;
	  else if (*inline == in_quote) 
	    in_quote = 0 ;
	  break ;

	case ')':
	case '}':		/* closing parenthesis */
	  if (!in_quote && *inline == look_for_paren)
	    look_for_paren = 0 ;
	  break;

	default:		/* everything else */
	  break ;
	}
      }

      *(var_char++) = *(inline++);
    }
  }

  if (look_for_paren == 0)
    --var_char ;
  *var_char = '\0' ;

  if (look_for_paren) {
    itf_errno = SYNTAX_ERROR ;
    IError("Missing right '%c' on: \"%s\"", look_for_paren, var_name);
    return NULL;
  }

  DBP((stderr, "parenValue: Raw variable name is %s\n",var_name));
  if (in_paren == '(')
    value = getCommandOutput(var_name, echo) ;
  else
    value = IGetValue(var_name, NULL) ;
  DBP((stderr, "parenValue: Value is %s\n", value));

  if (value == NULL || itf_errno)
    return NULL;

  if (strlen(value) >= max_value_len) {
    IError("Variable-substituted command is too long");
    return NULL;
  } 

  value = VarSubstitute(value, outline, max_value_len, echo) ;
  
  if (value == NULL) {
    return NULL ;
  } else {
    if (value != outline)	/* value string unchanged by VarSubstitute */
      strcpy(outline, value) ;
    return inline ;
  }
}

/*********************************************************************
 *	Name:		VarNameSpace
 *	Description:	given a pointer to a string containing
 *			"$var" or "${var}" (plus trailing garbage), copies
 *			the value of $var into outline, (up to max_value_len
 *			characters, and returns a pointer to the trailing
 *			garbage.
 *	Parameters:
 *	  char	*inline 	- the input string
 *	  char	*outline	- the output value
 *	  int	max_value_len 	- maximum length for the value
 *
 *	char	*variableValue
 *	  Return Value:	variableValue - ptr to trailing garbage, NULL
 *				on any type of error.
 *********************************************************************/
static char	*variableValue(inline, outline, max_value_len, echo) 
  char	*inline ;
  char	*outline ;
  int	max_value_len ;
  int	echo ;
{
  char	var_name[MAX_VAR_NAME] ; /* name of variable */
  char	*var_char ;		/* current  position in the var name */
  char	*value ;		/* value of the variable */

  DBP((stderr, "variableValue: Raw input line is \"%s\"\n",inline));

  if (*(inline++) != '$')
    return NULL ;

  if (*inline == '{' || *inline == '(')
    return parenValue(inline, outline, max_value_len, echo) ;

  var_char = var_name ;
  while (isalnum(*inline) || *inline=='_' || *inline=='-' || *inline=='*') {
    *(var_char++) = *(inline++) ;

    if (VarNameSpace(var_name, var_char) <= 0) {
      var_name[MAX_VAR_NAME-1] = '\0';
      IError("Variable name too long: \"%s...\"", var_name) ;
      return NULL ;
    }
  }
  *var_char = '\0' ;

  if (var_char == var_name) {	/* a single '$' sign */
      value = "$" ;
  } else {
    DBP((stderr, "variableValue: Raw variable name is %s\n",var_name));
    value = IGetValue(var_name, NULL) ;
    DBP((stderr, "variableValue: Value is %s\n", value));
    if (value == NULL || itf_errno)
      return NULL;
  }

  if (strlen(value) >= max_value_len) {
    IError("Variable-substituted command is too long");
    return NULL;
  } 

  value = VarSubstitute(value, outline, max_value_len, echo) ;
  
  if (value == NULL) {
    return NULL ;
  } else {
    if (value != outline)	/* value string unchanged by VarSubstitute */
      strcpy(outline, value) ;
    return inline ;
  }
}
/********************************************************************/

char *VarSubstitute(line,output,outlen,echo)
char *line;
char *output;
int outlen;
int echo;
{
  int in_quote=0;
  int escape_pending=0;
  char *outend = output+outlen;
  char *outstart = output;

  DBP((stderr, "VarSubstitute: Raw input line is \"%s\"\n",line));

  if (strchr(line,'$') == NULL) {	/* most times return here */
    DBP((stderr, "VarSubstitute: Substituted line is \"%s\"\n",line));
    return line;
  }

  while (*line) {
    int	  escaped = escape_pending ;
    escape_pending=0;
    if (*line == '$' && !escaped && in_quote != '\'') {
      /* start of variable */
      line = variableValue(line, output, outend - output, echo) ;
      if (line == NULL)
	return NULL ;
      output = outstart + strlen(outstart) ;

    } else if (*line == '#'  && !escaped && !in_quote) {
      while (*(++line))
	;

    } else {
      if (!escaped) {
	switch (*line) {
	case '\\':		/* escape character - set escape pending */
	  escape_pending = 1;
	  break;

	case '\'':		/* single or double quote character */
	case '"':	
	  if (!in_quote) 
	    in_quote = *line ;
	  else if (*line == in_quote) 
	    in_quote = 0 ;
	  break ;
	    
	default:		/* everything else */
	  break ;
	}
      } 
      *(output++) = *(line++);
    }

    if (output >= outend) {
      IError("Variable substituted command too long");
      return NULL;
    }
  }

  if (in_quote) {
    /* unmatched something */
    itf_errno = SYNTAX_ERROR ;
    IError("Unmatched quote '%c'", in_quote) ;
    return NULL;
  }

  *output = '\0';
  DBP((stderr, "VarSubstitute: Substituted line is \"%s\"\n",output));
  return outstart;
}

int	commandLexSubstitute(lex, tokc, tokv, echo)
  Lex	lex ;
  int	tokc ;
  char	**tokv ;
  int	echo ;
{
  int	idx ;
  char	*output ;

  for (idx = 0 ; idx < tokc ; ++idx) {
    if (LexQuotedTok(tokv[idx]) == '`') {
      output = getCommandOutput(tokv[idx], echo) ;
      if (output == NULL || LexObjChangeArg(lex, &tokv[idx], output) == NULL)
	return 0 ;
    }
  }
  return 1 ;
}

static String	getCommandOutput(command, echo)
  String	command ;
  int		echo ;
{
  static char	buffer[BUFSIZ] ;
  int		pipedesc[2] ;
  int		idx, nBytes, end ;
  jmp_buf	env ;
  void		*oldEnv ;
  int		oldfildes = -1 ;
  char		*ptr, *nextChar ;
  Boolean	error ;
  Loop 		*loop ;

  /* open a pipe to write to and read from */
  if (pipe(pipedesc) == -1) {
    IError("Pipe error \"%s\"", command);
    return NULL ;
  }
  /* temporarily change fileno(dout) to write to the pipe */
  oldfildes = dup(fileno(dout)) ;
  close(fileno(dout)) ;
  dup(pipedesc[1]) ;
  close(pipedesc[1]) ;

  /* assume pipe can hold infinite data */
  loop	= createLoop() ;
  if (setjmp(env) == 0) {
    oldEnv = ISetAbortEnv(env);
    error = IDoCommandPipeline(command, loop, echo) ;
  } else {
    error = TRUE ;
  }
  ISetAbortEnv(oldEnv);
  destroyLoop(loop) ;

  /* restore fildes(dout) to original value */
  close(fileno(dout)) ;
  dup(oldfildes) ;
  close(oldfildes) ;

  if (error) {
    close(pipedesc[0]) ;
    IReportError(stderr) ;
    itf_errno = ERROR_REPORTED;
    return NULL ;
  }
  
  /* read from read side of the pipe */
  end = 0 ;
  do {
    nBytes = read(pipedesc[0], buffer + end, BUFSIZ) ;
    end += nBytes ;
  } while (nBytes && end < BUFSIZ) ;
  end = MIN(end, BUFSIZ - 1) ;
  buffer[end] = '\0' ;
  close(pipedesc[0]) ;

  /* compress and strip leading and trailing whitespace */
  ptr = nextChar = buffer ;
  while (*nextChar) {
    if (isspace(*nextChar)) {
      while (*nextChar && isspace(*nextChar))
	++nextChar ;
      if (*nextChar)
	--nextChar ;
    }

    if (!isspace(*nextChar))
      *(ptr++) = *nextChar ;
    else if (ptr != buffer)
      *(ptr++) = ' ' ;
    ++nextChar ;
  }
  *(ptr) = '\0' ;
  return buffer ;
}
