
/**********************************************************************
 * $Id: trace.c,v 1.3 92/11/30 11:40:16 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <signal.h>
#include "itf.h"
#include "lex.h"
#include "trace.h"

struct TRACE *first_trace = NULL;
struct TRACE *last_trace = NULL;

int	autoinit_traces()
{
  /***
   * Find all the traces and make a linked list
   */
  int id=0;
  int trace_type;
  int size;
  /* int count=0; */
  TBL_ITEM *item;
  struct TRACE *trace;
  trace_type = IGetType("TRACE", &size);
  while ((item=ITableNextByContext(itf_table, &id, itf_obj_context))) {
    if (item->data.binding->var_type == trace_type) {
      trace=(struct TRACE*)IGetObjectPtr("TRACE", item->name);
      if (trace==NULL) {
	fprintf(stderr, "Found trace \"%s\", but it is a null pointer\n", item->name);
	continue;
      }
      trace->name = item->name;
      if (first_trace==NULL)
	first_trace = trace;
      else
	last_trace->next = trace;
      last_trace = trace;
      trace->active = 1;
      trace->next = NULL;
      /* count++; */
    }
  }
  /* fprintf(stderr, "Traces initialized - found %d trace%s\n", count, plural(count)); */
  return 1 ;
}

int command_showTrace(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  enum SHOW_MODE mode = BRIEF;
  int verbose = 1;
  int summary = 0;
  int status = 0;
  char *trace_name = NULL;
  char *trace_num = NULL;
  struct ARG_DESC *argd;
  argd = StartArgs(tokv[0]);
  Args(argd, "[-Sverbose %S %?]", &verbose, "print commands for traces");
  Args(argd, "[-Pstatus %P %?]", &status, "show error status");
  Args(argd, "[-Psummary %P %?]", &summary, "only show a summary");
  Args(argd, "[<trace-name>%s [<trace-num>...%R]]", &trace_name, &trace_num);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("show all of or one of the traces");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Print out details of the named traces or  all traces.  The %pverbose",
       "format is suitable for saving  to a file,  editing, and then reading",
       "back in.  This is a convenient way to change traces.",
       "SEE ALSO",
       "redirection",
       NULL);
    return 1;
  }

  (void) ParseArgs(argd, tokc, tokv, 0);
  
  if (verbose)
    mode = VERBOSE;
  else if (status)
    mode = STATUS;
  else if (summary)
    mode = SUMMARY;

  if (trace_name!=NULL) {
    if ((trace = (struct TRACE*)IGetObjectPtr("TRACE", trace_name))!=NULL)
      IShowTrace(trace, mode, (char **)trace_num);
    else {
      itf_errno = COMMAND_ERROR;
      IErrorAbort("\"%s\" is not a trace", trace_name);
    }
  } else
    for (trace=first_trace;trace!=NULL;trace = trace->next) {
      /* if (mode==BRIEF&&trace->n_traces>0)
	fprintf(dout, "trace %s\n", trace->name); */
      IShowTrace(trace, mode, (char **)NULL);
    }

  return 1 ;
}

int		IShowTrace(trace, mode, tokv)
TRACE		*trace;
enum SHOW_MODE	mode;
char		**tokv;
{
  int i;
  struct RANGE range;
  for (i=0;i<trace->n_traces;i++) trace->trace[i].mark = (tokv==NULL);
  SetDefaultRange(&range, NULL);
  for (;tokv!=NULL&&*tokv!=NULL;tokv++) {
    if (!ParseRange(*tokv, &range, NULL)) IAbort();
    for (i=0;i<trace->n_traces;i++)
      if (InRange(i, &range, NULL))
	trace->trace[i].mark = 1;
  }

  /* if (mode==BRIEF && (mode==SUMMARY || trace->n_traces>0)) */
    fprintf(dout, "trace %-20s %cactive +delete # contains %d traces\n",
	    trace->name, (trace->active ? '+' : '-'), trace->n_traces);
  
  for (i=0;i<trace->n_traces;i++) if (trace->trace[i].mark) {
    if (mode==BRIEF)
      fprintf(dout, "(%d) %s\"%s\"\n", i, (trace->trace[i].active ? "":"(off) "),
	      trace->trace[i].command);
    else if (mode==VERBOSE || mode==STATUS){
      fprintf(dout, "trace %s %cactive -freq %d -count %d # number %d\\\n  ",
	      trace->name,
	      trace->trace[i].active?'+':'-',
	      trace->trace[i].freq, trace->trace[i].count, i);
      IPutQuotedString(trace->trace[i].command, '"', dout);
      fputc('\n', dout);
      if (mode==STATUS)
	if (trace->trace[i].stream_not_open)
	  fprintf(dout, "# %s: stream not open on most recent call\n",
		  trace->name);
	else
	  fprintf(dout, "# %s: status OK\n", trace->name);
    }
  }
  /*
  if (mode==SUMMARY || mode==ALL)
    fprintf(dout, "%-10s : %s, contains %d traces\n", trace->name,
	    (trace->active ? "active" : "inactive"), trace->n_traces);
  if (mode==ALL) {
    for (i=0;i<trace->n_traces;i++)
      fprintf(dout, "(%d) (%sactive) \"%s\"\n", i,
	      (trace->trace[i].active ? "" : "in"),
	      (trace->trace[i].command ? trace->trace[i].command : "(null)"));
  }
  if (mode==SET) {
    for (i=0;i<trace->n_traces;i++) {
      fprintf(dout, "trace-add %s %s", trace->name,
	      trace->trace[i].active?"":"+off ");
      IPutQuotedString(trace->trace[i].command, '"', dout);
      fprintf(dout, "\n");
    }
    fprintf(dout, "trace-%s %s\n", trace->active?"on":"off", trace->name);
  }
  */
  return 1 ;
}

int command_doTrace(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  struct RANGE range;
  int all = 0;
  int i,j;
  char *name = tokv[0];
  IUsage("[+all] <trace-name> [trace-number]");
  if (GiveHelp(tokc)) {
    ISynopsis("execute the commands associated with a trace");
    IHelp(IHelpArgs,
	  "Some of the commands  associated with the  named trace are executed.",
	  "If a trace  number is  supplied then just that  numbered  command is",
	  "executed, otherwise all the commands are executed.",
	  "OPTIONS",
	  "+all   : execute traces even if they are switched off.",
	  NULL);
    return 1;
  }

  tokv++;
  tokc--;
  while (*tokv && (**tokv=='-' || **tokv=='+')) {
    if (!strcmp("+all", *tokv)) all = 1;
    else {
      IErrorAbort("bad option: %s", *tokv);
    }
    tokv++;
    tokc--;
  }

  if (tokc<1)
    IErrorAbort(IPrintUsage(name, usage));

  trace = (struct TRACE*)IGetObjectPtr("TRACE", tokv[0]);
  if (trace==NULL) {
    itf_errno = COMMAND_ERROR;
    IErrorAbort("\"%s\" is not a trace", tokv[0]);
  }

  tokv++,tokc--;
  for (i=0;i<trace->n_traces;i++)
    trace->trace[i].mark = (*tokv==NULL);
  for (j=0;j<tokc;j++) {
    if (!ParseRange(tokv[j], &range, NULL)) IAbort();
    for (i=0;i<trace->n_traces;i++)
      if (InRange(i, &range, NULL))
	trace->trace[i].mark = 1;
  }

  for (i=0;i<trace->n_traces;i++)
    if (trace->trace[i].mark && (trace->trace[i].active || all))
      (void) IDoCommandLine(trace->trace[i].command);
  return 1 ;
}

void ChangeTraceFlags(trace, active, freq, count, delete, tokv)
struct TRACE *trace;
int active;
int freq;
int count;
int delete;
char *tokv[];
{
  struct RANGE range;
  char *range_str = "-";
  int i, j;
  if (trace==NULL) return;
  for (i=0;i<trace->n_traces;i++)
    trace->trace[i].mark = 0; /* used to say whether or not to delete */
  if (tokv && *tokv) range_str = *tokv;
  while (1) {
    if (!ParseRange(range_str, &range, NULL)) IAbort();
    else {
      for (i=0;i<trace->n_traces;i++)
	if (InRange(i, &range, NULL)) {
	  if (delete) trace->trace[i].mark = 1;
	  if (active!=-1) trace->trace[i].active = active;
	  if (freq!=-1) trace->trace[i].freq = freq;
	  if (count!=-1) trace->trace[i].count = count;
	}
    }
    if (tokv==NULL || tokv[0]==NULL || tokv[1]==NULL) break;
    range_str = *++tokv;
  }
  j = 0;
  for (i=0;i<trace->n_traces;i++) {
    if (trace->trace[i].mark) {
      free(trace->trace[i].command);
      trace->trace[i].command=NULL;
    } else {
      if (i>j) trace->trace[j] = trace->trace[i];
      j++;
    }
  }
  trace->n_traces = j;
}

static int commandTrace(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  struct RANGE range;
  char *command, *name;
  int pos;
  int j;
  int active = -1;
  int freq = -1;
  int insert_point = -1;
  int already_present = 0;
  int count = 0;
  int delete = 0;
  /* char *my_name = tokv[0]; */
  IUsage("<trace-name> [+/-active] [+off] [+on] [-freq <f>] [+/-N]\n\t[-resetcount] [-count <n>] [+delete] command...");
  if (GiveHelp(tokc)) {
    ISynopsis("add a command to a trace");
    IHelp
      (IHelpArgs,
       "Add a command to the specified trace.  If the  command is already on",
       "the trace, it is not added, but is turned on  or off as appropriate,",
       "and moved to the end of the trace.",
       "OPTIONS",
       "+off         : turn this trace off",
       "+on          : turn this trace on",
       "+/-active    : turn this trace on or off",
       "-freq <freq> : set this trace to execute once every <freq>",
       "               times it is called",
       "-count <n>   : set the current count on this trace to <n>",
       "-resetcount  : don't reset the count on this trace",
       "+/-N         : add this trace after/before command N",
       "+delete      : delete all the commands in this trace",
       "EXAMPLE",
       "xerion-> addTrace train4 \"print currentNet->error \\\\%9.4g\"",
       "",
       "Note the  double  backslash before the  percent.  This  is necessary",
       "because this command here is parsed twice - once when the \"addTrace\"",
       "command  is parsed, and  again  when the  trace  is executed.  If  a",
       "blackslash is not  before the   percent  found when  the command  is",
       "executed, the percent will be interpreted as a redirection symbol.",
       "SEE ALSO",
       "showTrace, deleteTrace",
       NULL);
    return 1;
  }

  if (tokc<3) IErrorAbort(IPrintUsage(tokv[0], usage));

  tokv++;
  tokc--;
  name = NULL;
  if (IIsObjectName(tokv[0])) {
    name = tokv[0];
    tokv++;
    tokc--;
  }
  while (*tokv && (**tokv=='+'||**tokv=='-') && !LexQuotedTok(*tokv)) {
    if (!strcmp(*tokv, "+off")) active = 0;
    else if (!strcmp(*tokv, "+on")) active = 1;
    else if (strprefix(1+*tokv, "active")) active = (**tokv=='+');
    else if (strprefix(*tokv, "-resetcount")) count = -1;
    else if (strprefix(*tokv, "+delete")) delete = 1;
    else if (strprefix(*tokv, "-freq")) {
      tokv++; tokc--;
      if (*tokv && IIsInteger(*tokv)) freq = atoi(*tokv);
      else IErrorAbort("expected integer after \"-freq\"");
    } else if (strprefix(*tokv, "-count")) {
      tokv++; tokc--;
      if (*tokv && IIsInteger(*tokv)) count = atoi(*tokv);
      else IErrorAbort("expected integer after \"-count\"");
    } else if (IIsNumber(1+*tokv)) {
      insert_point = atoi(1+*tokv);
      if (**tokv=='-') insert_point--;
    } else
      IErrorAbort("bad option \"%s\"", *tokv);
    if (*tokv) {
      tokv++;
      tokc--;
    }
  }
  if (name==NULL) {
    if (tokc<1 || LexQuotedTok(tokv[0]) || isdigit(**tokv))
      IErrorAbort("need a trace name");
    name = tokv[0];
    tokv++;
    tokc--;
  }

  trace = (struct TRACE*)IGetObjectPtr("TRACE", name);
  if (trace==NULL) {
    itf_errno = COMMAND_ERROR;
    IErrorAbort("\"%s\" is not a trace", name);
  }
  /* now see whether we have a command or a trace number */

  /***
   * Four possibilities for rest of tokens now:
   *     case			action
   * (1) a command		insert or move to end
   * (2) a single number & move	change flags/count & move
   * (3) a single number	call ChangeTraceFlags
   * (4) a range of numbers	call ChangeTraceFlags
   * (5) nothing                turn trace on/off, delete all
   */
  if (insert_point==-1
      && (*tokv==NULL || (isdigit(**tokv) && ParseRange(*tokv, &range, NULL)))) {
    if (*tokv==NULL && active!=-1) trace->active = active;
    ChangeTraceFlags(trace, active, freq, count, delete, tokv);
    return 1;
  }
  if (IIsInteger(*tokv)) {
    command = NULL;
    already_present = 1;
    pos = atoi(*tokv);
    if (pos<0 || pos>=trace->n_traces)
      IErrorAbort("command number %d out of range", pos);
  } else {
    command = IComposeTokvLine(tokv, (tokc>1));
    if (command==NULL || !*command)
      IErrorAbort("no command specified");
    for (pos=0;pos<trace->n_traces;pos++)
      if (!strcmp(trace->trace[pos].command, command)) {
	already_present = 1;
	break;
      }
  }
  if (!already_present) {
    /* add the new trace */
    pos = trace->n_traces;
    if (trace->n_traces>0)
      trace->trace = (struct TRACE_RECORD*)
	realloc((char*) trace->trace,
		(unsigned)(sizeof(struct TRACE_RECORD)*(trace->n_traces+1)));
    else
      trace->trace = Calloc(struct TRACE_RECORD, 1);
    if (trace->trace==NULL) {
      IOutOfMemory("trace-add", "allocating trace record");
      return 0;
    }
    trace->n_traces++;
    trace->trace[pos].command = malloc((unsigned) strlen(command)+1);
    if (trace->trace[pos].command==NULL) {
      IOutOfMemory("trace-add", "allocating command string");
      return 0;
    }
    strcpy(trace->trace[pos].command, command);
    trace->trace[pos].active = 1;
    trace->trace[pos].freq = 0;
    trace->trace[pos].count = 0;
    trace->trace[pos].stream_not_open = 0;
  } else if (command!=NULL) {
    /* move to end */
    struct TRACE_RECORD tp;
    tp = trace->trace[pos];
    for (j=pos;j<trace->n_traces-1;j++) trace->trace[j] = trace->trace[j+1];
    trace->trace[trace->n_traces-1] = tp;
    pos = trace->n_traces-1;
  }
  /* the command is now at position pos */
  if (active!=-1) trace->trace[pos].active = active;
  if (freq!=-1) trace->trace[pos].freq = freq;
  if (count!=-1) trace->trace[pos].count = count;
  /* if it should be moved to somewhere else, do it! */
  if (insert_point>=0 && insert_point<trace->n_traces) {
    struct TRACE_RECORD tp;
    tp = trace->trace[pos];
    if (pos>insert_point)
      for (j=pos;j>insert_point;j--)
	trace->trace[j] = trace->trace[j-1];
    else if (pos<insert_point)
      for (j=pos;j<insert_point;j++)
	trace->trace[j] = trace->trace[j+1];
    trace->trace[insert_point] = tp;
  }
  return 1 ;
}

int command_addTrace(tokc, tokv)
int tokc;
char *tokv[];
{
  return commandTrace(tokc, tokv);
}

int command_traceOff(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  IUsage("<trace-name> [trace#] [trace#] ...");
  if (GiveHelp(tokc)) {
    ISynopsis("turn a trace off");
    IHelp
      (IHelpArgs,
       "If no trace number is supplied the whole trace  is turned off.  If a",
       "trace number is supplied, just  that command in the trace  is turned",
       "off.  A range of values may be  supplied for the trace numbers, e.g.",
       "\"1-4\".",
       "SEE ALSO",
       "addTrace, showTrace, doTrace",
       "",
       NULL);
    return 1;
  }

  if (tokc<2)
    IErrorAbort(IPrintUsage(tokv[0], usage));
  trace = (struct TRACE*)IGetObjectPtr("TRACE", tokv[1]);
  if (trace==NULL) {
    itf_errno = COMMAND_ERROR;
    IErrorAbort("\"%s\" is not a trace", tokv[1]);
  }
  if (tokc==2) {
    trace->active = 0;
  } else
    ChangeTraceFlags(trace, 0, -1, -1, 0, tokv+2);
  return 1 ;
}

int command_traceOn(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  IUsage("<trace-name> [trace#] [trace#] ...");
  if (GiveHelp(tokc)) {
    ISynopsis("turn a trace on");
    IHelp
      (IHelpArgs,
       "If no trace number is supplied  the whole  trace is turned on.  If a",
       "trace number is  supplied, just that command in  the trace is turned",
       "on.  A range of values may be supplied for the  trace  numbers, e.g.",
       "\"1-4\".",
       "SEE ALSO",
       "addTrace, deleteTrace, showTrace",
       NULL);
    return 1;
  }
  if (tokc<2)
    IErrorAbort(IPrintUsage(tokv[0], usage));
  trace = (struct TRACE*)IGetObjectPtr("TRACE", tokv[1]);
  if (trace==NULL) {
    itf_errno = COMMAND_ERROR;
    IErrorAbort("\"%s\" is not a trace", tokv[1]);
  }
  if (tokc==2) {
    trace->active = 1;
  } else
    ChangeTraceFlags(trace, 1, -1, -1, 0, tokv+2);
  return 1 ;
}

int command_deleteTrace(tokc, tokv)
int tokc;
char *tokv[];
{
  struct TRACE *trace;
  int i;
  IUsage("<trace-name> <command#>|<trace-name> all");
  if (GiveHelp(tokc)) {
    ISynopsis("delete one or all commands from a trace");
    IHelp
      (IHelpArgs,
       "The specified command is deleted permanently from  the trace.  If no",
       "command number is given, all commands are deleted from that trace.",
       "SEE ALSO",
       "addTrace, showTrace, doTrace",
       NULL);
    return 1;
  }

  if (tokc!=3 || (strcmp("all", tokv[2]) && !IIsInteger(tokv[2]))) {
    IErrorAbort(IPrintUsage(tokv[0], usage));
  }
  trace = (struct TRACE*)IGetObjectPtr("TRACE", tokv[1]);
  if (trace==NULL) {
    itf_errno = COMMAND_ERROR;
    IErrorAbort("\"%s\" is not a trace", tokv[1]);
  }
  if (strcmp("all", tokv[2]) == 0) {
    for (i=0;i<trace->n_traces-1;i++)
      free(trace->trace[i].command);
    free(trace->trace);
    trace->n_traces = 0;
  } else
    ChangeTraceFlags(trace, -1, -1, -1, 1, tokv+2);
  return 1 ;
}

int IDoTrace(trace)
struct TRACE *trace;
{
  int i;
  /* fprintf(stderr, "Doing trace \"%s\"\n", trace->name); */
  if (trace->active) {
    for (i=0;i<trace->n_traces;i++) {
      if (trace->trace[i].active) {
	if (   trace->trace[i].freq==0
	    || ((trace->trace[i].count % trace->trace[i].freq)==0)) {
	  if (!IDoCommandLine(trace->trace[i].command)) {
	    if (itf_errno==INTERRUPT)
	      kill(mypid, SIGINT);/*SIGINT -follow on*/
	    else if (itf_errno!=STREAM_NOT_OPEN) {
	      IReportError(stderr);
	      fprintf(stderr,
      "Command %d on trace \"%s\" has been turned off because of errors\n",
		      i, trace->name);
	      trace->trace[i].active = 0;
	    }
	    IResetError();
	  }
	}
	trace->trace[i].count++;
      }
    }
  }
  dispatchExposures() ;
  return 1 ;
}

