
/**********************************************************************
 * $Id: stream.c,v 1.3 93/02/23 13:17:37 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include "itf.h"
#include "lex.h"

#define DB(X)
#define IsPipe(type) ((type)==STREAM_REMOTE_FILE||(type)==STREAM_COMMAND||(type)==STREAM_REMOTE_APPEND)

#define MAX_PORT_TRIES 10

static struct STREAM _current_outstream;
static struct STREAM _current_instream;
struct STREAM *current_outstream = &_current_outstream;
struct STREAM *current_instream = &_current_instream;
struct STREAM *stdout_stream, *stdin_stream;
struct STREAM *command_stdin, *command_stdout;

static struct STREAM *first_stream, *last_stream;

extern char *sys_errlist[];
static char *ignore_pipe_error;

static int IRedirectToCommand ARGS((struct STREAM *, char *, char *, char *)) ;
static int IRedirectToNumberedSocket ARGS((struct STREAM *, int, int, char *, char *)) ;
static int IRedirectToNamedSocket ARGS((struct STREAM *, char *, char *)) ;

int	IInitStream()
{
  first_stream = last_stream = NULL;

  /***
   * Set up the current stream to be stdout
   */
  current_outstream = &_current_outstream;
  current_instream  = &_current_instream;

  current_outstream->status = STREAM_OPEN;
  current_outstream->type = STREAM_FILE;
  current_outstream->desc = "stdout";
  current_outstream->mode = "w";
  current_outstream->file = stdout;
  current_outstream->name = current_outstream->desc;
  current_outstream->permanent = 1;
  IBindStruct(current_outstream->name, "STREAM", (char*)current_outstream,
	      P_OBJECT, 0);
  stdout_stream = current_outstream;
  IAddStream(current_outstream);

  current_instream->status = STREAM_OPEN;
  current_instream->type = STREAM_FILE;
  current_instream->desc = "stdin";
  current_instream->file = stdin;
  current_instream->mode = "r";
  current_instream->name = current_instream->desc;
  current_instream->permanent = 1;
  IBindStruct(current_instream->name, "STREAM", (char*)current_instream,
	      P_OBJECT, 0);
  stdin_stream = current_instream;
  IAddStream(current_instream);
  (void) IBindString("ignore-pipe-error", &ignore_pipe_error, P_OBJECT, 0, "");
  ISetValue("ignore-pipe-error", ":more:less:");

  /* what the $#%@ is the following for??? we want to catch these
     signals to avoid useless processing!
  (void) signal(SIGPIPE, SIG_IGN); */ /* we check for errors on streams */
  return 1 ;
}

int		IAddStream(stream)
struct STREAM	*stream;
{
  if (first_stream==NULL)
    first_stream = stream;
  else
    last_stream->next = stream;
  last_stream = stream;
  stream->next = NULL;
  return 1 ;
}

int	IShowStream(stream, file, name)
struct STREAM	*stream;
FILE		*file;
int name; /* Boolean, print the name */
{
  if (name)
    fprintf(file, "%-10s : %s, %s, ",
	    (stream->name==NULL ? "(anonymous)" : stream->name),
	    (  stream->status==STREAM_OPEN ? "open"
	     : stream->status==STREAM_ERROR ? "error"
	     : stream->status==STREAM_CLOSED ? "closed"
	     : "bad status"),
	    (stream->permanent ? "permanent" : "temporary"));
  fprintf(file, " %s ",
	  (  stream->type==STREAM_FILE ? "File:"
	   : stream->type==STREAM_APPEND ? "File (appending):"
	   : stream->type==STREAM_REMOTE_FILE ? "File:"
	   : stream->type==STREAM_REMOTE_APPEND ? "File (appending):"
	   : stream->type==STREAM_SOCKET ? "Socket:"
	   : stream->type==STREAM_PORT ? "Socket:"
	   : stream->type==STREAM_COMMAND ? "Pipe with command:"
	   : "bad type")
	  );
  if (stream->type==STREAM_PORT)
    fprintf(file, "port #%d on %s", stream->real_port, stream->desc);
  else if (stream->type==STREAM_REMOTE_FILE || stream->type==STREAM_REMOTE_APPEND)
    fprintf(file, "\"%s\" on \"%s\"", stream->desc, stream->machine);
  else
    fprintf(file, "\"%s\"", stream->desc);
  fprintf(file, " mode=%s", stream->mode);
  fprintf(file, "\n");
  return 1 ;
}

int	command_showStream(tokc, tokv)
int 	tokc;
char 	*tokv[];
{
  struct STREAM *stream;
  if (GiveHelp(tokc)) {
    IUsage("[stream-name] ...");
    ISynopsis("show all of or one of the streams");
    IHelp
      (IHelpArgs,
       "Print out  details  of  the  named stream,  or if no  stream name is",
       "supplied, of all streams.   See the help  for `redirection' for more",
       "information about streams",
       "",
       NULL);
    return 1;
  }
  
  if (tokc>1)
    while (*++tokv) {
      if ((stream = (struct STREAM*)IGetObjectPtr("STREAM", *tokv))!=NULL)
	IShowStream(stream, dout, 1);
      else {
	IError("\"%s\" is not a stream", *tokv);
	return 0;
      }
    }
  else
    for (stream=first_stream;stream!=NULL;stream = stream->next)
      IShowStream(stream, dout, 1);
  return 1 ;
}

struct STREAM *IGetStream(name)
char *name;
{
  struct STREAM *stream;
  int status = 1;
  stream = (struct STREAM*)IGetObjectPtr("STREAM", name);
  if (stream==NULL) {
    IResetError();
    itf_errno = REDIRECTION_ERROR;
    IError("\"%s\" is not a stream name", name);
    status = 0;
  }

  if (status && stream->status != STREAM_OPEN) {
    itf_errno = STREAM_NOT_OPEN;
    IError("\"%s\" is not currently open", name);
    status = 0;
  }

  if (status && ferror(stream->file)) {
    itf_errno = REDIRECTION_ERROR;
    IError("Error on stream \"%s\": %s", stream->name, sys_errlist[errno]);
    fclose(stream->file);
    stream->file = NULL;
    stream->status = STREAM_CLOSED;
    status = 0;
  }

  if (status) return stream;
  else return NULL;
}

/***
 * IMachineName()
 * Check if the machine name should be translated:
 * host - get from $HOST
 * display - get from $DISPLAY
 */
static char	*IMachineName(name)
char		*name;
{
  char *newname;
  static char buf[MAXHOSTNAMELEN];
  if (!strcmp(name, "host")) {
    newname = getenv("HOST");
    if (newname==NULL)
      IError("Environment variable HOST not set: cannot get host machine");
    else
      name = newname;
  } else if (!strcmp(name, "display")) {
    newname = getenv("DISPLAY");
    if (newname==NULL) {
      IError("Environment variable DISPLAY not set: cannot get display machine");
      name = NULL;
    } else {
      strcpy(buf, newname);
      newname = strchr(buf, ':');
      if (newname==NULL) {
	IError("Bad format in environment variable DISPLAY (%s): can't find ':'", buf);
	name = NULL;
      } else {
	name = buf;
	*newname = 0; /* end the string at the ':' */
      }
    }
  }
  return name;
}

/***
 * IDoRedirection - looks at and modifies tokv, and returns 1 if
 * everything goes ok
 */
int IDoRedirection(tokc, tokv, old_instream, old_outstream)
int *tokc;
char *tokv[];
struct STREAM **old_instream;
struct STREAM **old_outstream;
{
  int i;
  int status=1;
  char *sh_command;
  char *machine;
  char *port_num;
  char *mode = "w";
  char *sh_command_name = NULL;
  int search;
  *old_instream = current_instream;
  *old_outstream = current_outstream;
  for (i=1;i<*tokc;i++) {
    if (LexIsOperator(tokv[i])) {
      if (!strcmp("|", tokv[i])) {
	int j=i;
	sh_command_name = tokv[i+1];
	sh_command = IComposeTokvLine(tokv+i+1, 1);
	if (sh_command==NULL) {
	  status = 0;
	} else {
	  for (;i<*tokc;i++) tokv[i] = NULL;
	  *tokc = j;
	  DB( fprintf(stderr, "Redirecting to command \"%s\"\n", sh_command); );
	  status = IRedirectToCommand(NULL, sh_command, mode, sh_command_name);
	}
	break;
      } else if (!strcmp(">", tokv[i]) || !strcmp(">>", tokv[i])) {
	if (*tokc == i+2) {
	  DB( fprintf(stderr, "Redirecting to %sfile \"%s\"\n",
		      (!strcmp(">>", tokv[i]) ? "append to ":""), tokv[i+1]););
	  status = IRedirectToFile(NULL, tokv[i+1], !strcmp(">>", tokv[i]),
				   true(noclobber), mode);
	  *tokc -= 2;
	  tokv[i] = NULL;
	  tokv[i+1] = NULL;
	} else if (*tokc == i+4) {
	  machine = (LexQuotedTok(tokv[i+1])
		     ? tokv[i+1]
		     : IMachineName(tokv[i+1]));
	  DB( fprintf(stderr, "Redirecting to %sfile \"%s\" on \"%s\"\n",
		      (!strcmp(">>", tokv[i]) ? "append to ":""), tokv[i+3],
		      machine); );
	  if (strcmp(":", tokv[i+2])) {
	    status = 0;
	    itf_errno = REDIRECTION_ERROR;
	    IExpectedToken(":", tokv[i+2], tokv[i+1]);
	  } else {
	    status=IRedirectToRemoteFile(NULL, tokv[i+3], machine,
					 !strcmp(">>", tokv[i]), true(noclobber), mode);
	    tokv[i] = tokv[i+1] = tokv[i+2] = tokv[i+3] = NULL;
	    *tokc -= 4;
	  }
	} else {
	  status = 0;
	  itf_errno = REDIRECTION_ERROR;
	  IError("Expected a filename after \"%s\"", tokv[i]);
	}
	break;
      } else if (!strcmp("%", tokv[i]) || !strcmp("%<", tokv[i])) {
	mode = (tokv[i][1]=='<' ? "r" : "w");
	if (*tokc != i+2 && *tokc != i+3) {
	  status = 0;
	  itf_errno = REDIRECTION_ERROR;
	  IExpected("<port#> <machine> or <servername> after \"%\"");
	} else {
	  if (*tokc == i+2) {
	    DB( fprintf(stderr, "Redirecting to server \"%s\"\n", tokv[i+1]); )
	      status = IRedirectToNamedSocket(NULL, tokv[i+2], mode);
	    *tokc -= 2;
	    tokv[i] = NULL;
	    tokv[i+1] = NULL;
	  } else {
	    port_num = tokv[i+1];
	    search = 0;
	    if (*port_num=='-') {
	      port_num++;
	      search = -1;
	    } else if (*port_num=='+') {
	      port_num++;
	      search = 1;
	    }
	    if (!IIsInteger(port_num)) {
	      itf_errno = REDIRECTION_ERROR;
	      IExpectedToken("<Integer port #>", port_num, tokv[i]);
	      status = 0;
	    } else {
	      machine = (LexQuotedTok(tokv[i+2])
			 ? tokv[i+2]
			 : IMachineName(tokv[i+2]));
	      DB( fprintf(stderr, "Redirecting to port #%d on \"%s\"\n",
			  atoi(port_num), machine); )
		status = IRedirectToNumberedSocket(NULL, atoi(port_num),
						   search, machine, mode);
	      *tokc -= 3;
	      tokv[i] = NULL;
	      tokv[i+1] = NULL;
	      tokv[i+2] = NULL;
	    }
	  }
	}
	break;
      } else if (!strcmp("@", tokv[i])) {
	if (*tokc != i+2) {
	  status = 0;
	  itf_errno = REDIRECTION_ERROR;
	  IExpected("a stream-name after \"@\"");
	} else {
	  DB( fprintf(stderr, "Redirecting to stream \"%s\"\n", tokv[i+1]); )
	    status = IRedirectToStream(tokv[i+1]);
	  *tokc -= 2;
	  tokv[i] = NULL;
	  tokv[i+1] = NULL;
	}
	break;
      }
    }
  }
  if (status && current_outstream!=*old_outstream)
    dout = current_outstream->file;
  else if (status && auto_more_enabled && current_outstream->file==stdout) {
    char *command_automore;
    command_automore = IGetValueCompositeName(tokv[0], "-automore", NULL);
    if (command_automore==NULL ? true(automore) : true(command_automore)) {
      char *more = IGetValue("PAGER", NULL);
      if (more==NULL||*more=='\0')
	more = "more";
      status = IRedirectToCommand(NULL, more, "w", more);
      if (status)
	dout = current_outstream->file;
    }
  }
  if (status && current_instream!=*old_instream)
    din = current_instream->file;

  if (!status && !itf_errno) itf_errno = REDIRECTION_ERROR;
  return status;
}

/***
 * INewStream()
 */
static struct STREAM *INewStream()
{
  struct STREAM *stream;
  stream = (struct STREAM*)calloc(sizeof(struct STREAM), 1);
  if (!stream) {
    IOutOfMemory("INewStream", NULL);
    IAbort();
  }
  stream->name = NULL;
  stream->desc = NULL;
  stream->machine = NULL;
  stream->file = NULL;
  stream->type = STREAM_TYPE_XX;
  stream->status = STREAM_STATUS_XX;
  stream->permanent = 0;
  stream->next = NULL;
  return stream;
}

/***
 * IRedirectToCommand
 */
static int IRedirectToCommand(stream, sh_command, mode, sh_command_name)
struct STREAM *stream;
char *sh_command;
char *mode;
char *sh_command_name;
{
  FILE *f;
  int status = 1;
  int new_stream = 1;
  char *tmp_command;
  if (stream==NULL)
    stream = INewStream();
  else {
    new_stream = 0;
    sh_command = stream->desc;
    mode = stream->mode;
  }
  if (!sh_command || !*sh_command) {
    itf_errno = REDIRECTION_ERROR;
    IError("No command for pipe redirection");
    status = 0;
  }

  if (status) {
    tmp_command = ITempCalloc(char, strlen(sh_command)+20);
    strcpy(tmp_command, "trap '' 2; ");
    strcat(tmp_command, sh_command);
    if ((f = (FILE *)popen(tmp_command, "w"))==NULL) {
      itf_errno = REDIRECTION_ERROR;
      IError("Couldn't execute pipe command: %s", sys_errlist[errno]);
      status = 0;
    }
  }

  if (status) {
    if (new_stream) {
      stream->type = STREAM_COMMAND;
      stream->desc = stralloc(sh_command);
      stream->name = stralloc(sh_command_name);
      stream->mode = mode;
    }
    stream->file = f;
    stream->status = STREAM_OPEN;
  }

  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  else if (new_stream)
    free((char*)stream);
  return status;
}

/***
 * IRedirectToNumberedSocket
 */
static int IRedirectToNumberedSocket(stream, port, search, host, mode)
struct STREAM *stream;
int port;
int search;
char *host;
char *mode;
{
  FILE *f;
  int status = 1;
  int new_stream = 1;
  int n_tries;
  int try_port;
  int waited = 0;
  if (stream==NULL)
    stream = INewStream();
  else {
    new_stream = 0;
    host = stream->desc;
    port = stream->port;
    search = stream->search;
    mode = stream->mode;
  }
  if (!host || !*host) {
    host = "localhost";
    stream->desc = host;
  }

  try_port = port;
  n_tries = 0;
  if (status) {

  try_again:
    f = IOpenNumberedSocket(try_port, host, mode);
    if (f==NULL) {
      if (itf_errno) {
	status = 0;
      } else {
	if (!waited) {
	  waited = 1;
	  sleep(1);
	  goto try_again;
	}
	if (search!=0 && n_tries<MAX_PORT_TRIES) {
	  fprintf(stderr, "port #%d at \"%s\" not accepting, trying port #%d\n",
		  try_port, host, try_port+search);
	  try_port += search;
	  n_tries++;
	  goto try_again;
	}
	itf_errno = REDIRECTION_ERROR;
	IError("Couldn't open port #%d at \"%s\": %s",
	       try_port, host, sys_errlist[errno]);
	status = 0;
      }
    }
  }

  if (status) {
    if (new_stream) {
      stream->type = STREAM_PORT;
      stream->desc = stralloc(host);
      stream->name = stralloc(strcmp(mode, "w")?"din":"dout");
      stream->port = port;
      stream->search = search;
      stream->mode = mode;
    }
    stream->real_port = try_port;
    stream->file = f;
    stream->status = STREAM_OPEN;
  }

  
  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  else if (new_stream)
    free((char*)stream);
  return status;
}

/***
 * IRedirectToNamedSocket
 */
static int IRedirectToNamedSocket(stream, name, mode)
struct STREAM *stream;
char *name;
char *mode;
{
  FILE *f;
  int status = 1;
  int new_stream = 1;
  if (stream==NULL)
    stream = INewStream();
  else {
    new_stream = 0;
    name = stream->desc;
    mode = stream->mode;
  }
  if (!name || !*name) {
    itf_errno = REDIRECTION_ERROR;
    IError("No name for named socket redirection");
    status = 0;
  }

  if (status) {
    if ((f = IOpenNamedSocket(name, mode))==NULL) {
      itf_errno = REDIRECTION_ERROR;
      IError("Couldn't open named socket \"%s\": %s",
		    name, sys_errlist[errno]);
      status = 0;
    }
  }

  if (status) {
    if (new_stream) {
      stream->type = STREAM_SOCKET;
      stream->desc = stralloc(name);
      stream->mode = mode;
      stream->name = stralloc(strcmp(mode, "w")?"din":"dout");
    }
    stream->file = f;
    stream->status = STREAM_OPEN;
  }

  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  else if (new_stream)
    free((char*)stream);
  return status;
}

/***
 * IRedirectToStream
 */
int IRedirectToStream(name)
char *name;
{
  struct STREAM *stream;
  int status = 1;
  stream = (struct STREAM*)IGetObjectPtr("STREAM", name);
  if (stream==NULL) {
    IResetError();
    itf_errno = REDIRECTION_ERROR;
    IError("\"%s\" is not a stream name", name);
    status = 0;
  }

  if (status && stream->status != STREAM_OPEN) {
    itf_errno = STREAM_NOT_OPEN;
    IError("\"%s\" is not currently open", name);
    status = 0;
  }
/*
  if (status && ferror(stream->file)) {
    itf_errno = REDIRECTION_ERROR;
    IError("Error on stream \"%s\": %s", stream->name,
	    sys_errlist[errno]);
    fclose(stream->file);
    stream->file = NULL;
    stream->status = STREAM_CLOSED;
    status = 0;
  }
  */
  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  return status;
}

/***
 * IRedirectToFile
 */
int IRedirectToFile(stream, name, append, noclobber_file, mode)
struct STREAM *stream;
char *name;
int append, noclobber_file;
char *mode;
{
  FILE *f;
  int status = 1;
  int new_stream=1;
  if (stream==NULL)
    stream = INewStream();
  else {
    new_stream=0;
    name = stream->desc;
    append = stream->type==STREAM_APPEND;
    mode = stream->mode;
  }

  if (noclobber_file && !append) {
    struct stat buf;
    if (stat(name, &buf)==0) {
      if (!(buf.st_mode & (S_IFIFO|S_IFCHR))) {
	itf_errno = REDIRECTION_ERROR;
	IError("file \"%s\" already exists", name);
	status = 0;
      }
    }
  }

  if (status) {
    f = fopen(name, (append ? "a" : "w"));
    if (f==NULL) {
      itf_errno = REDIRECTION_ERROR;
      IError("can't open \"%s\": %s", name, sys_errlist[errno]);
      status = 0;
    }
  }

  if (status) {
    if (new_stream) {
      stream->type = (append ? STREAM_APPEND : STREAM_FILE);
      stream->desc = stralloc(name);
      stream->name = stralloc(strcmp(mode, "w")?"din":"dout");
      stream->mode = mode;
    }
    stream->file = f;
    stream->status = STREAM_OPEN;
  }

  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  else if (new_stream)
    free((char*)stream);
  return status;
}

/***
 * IRedirectToFile
 */
int IRedirectToRemoteFile(stream, name, machine, append, noclobber_file, mode)
struct STREAM *stream;
char *name;
char *machine;
int append, noclobber_file;
char *mode;
{
  FILE *f;
  int status = 1;
  int new_stream=1;
  char sh_command[MAXPATHLEN+70];
  if (stream==NULL)
    stream = INewStream();
  else {
    new_stream=0;
    name = stream->desc;
    machine = stream->machine;
    append = stream->type==STREAM_REMOTE_APPEND;
    mode = stream->mode;
  }

  sh_command[0] = 0;
  sprintf(sh_command, "trap '' 2; /usr/ucb/rsh %s \"%scat %s %s\"", machine,
	  (noclobber_file ? "set noclobber;" : ""), (append ? ">>" : ">"), name);
  DB( fprintf(stderr, "Command is: %s\n", sh_command); );
  f = (FILE *)popen(sh_command, "w");
  if (f==NULL) {
    itf_errno = REDIRECTION_ERROR;
    IError("can't access \"%s\" on \"%s\": %s", name, machine,
		  sys_errlist[errno]);
    status = 0;
  }

  if (status) {
    if (new_stream) {
      stream->type = (append ? STREAM_REMOTE_APPEND : STREAM_REMOTE_FILE);
      stream->desc = stralloc(name);
      stream->machine = stralloc(machine);
      stream->name = stralloc(strcmp(mode, "w")?"din":"dout");
      stream->mode = mode;
    }
    stream->file = f;
    stream->status = STREAM_OPEN;
  }

  if (status)
    if (!strcmp("w", stream->mode))
      current_outstream = stream;
    else
      current_instream = stream;
  else if (new_stream)
    free((char*) stream);
  return status;
}


/***
 * Ends the redirection for an old-current-d triple, one of:
 * (1) old_instream current_instream din
 * (2) old_outstream current_outstream dout
 */
void IEndRedirection2(old, current, d)
struct STREAM *old, **current;
FILE **d;
{
  extern char *sys_errlist[];
  char buf[100];
  if ((*current) == old) {
    DB( fprintf(stderr, "Don't need to reset current stream\n"); );
    if ((*current)->file!=NULL && !strcmp((*current)->mode, "w"))
      fflush((*current)->file);
  } else {
    DB( fprintf(stderr, "Resetting stream to \"%s\" from \"%s\"\n", 
		old->desc, (*current)->desc); );
    if (!ferror((*current)->file) && !strcmp((*current)->mode, "w"))
      fflush((*current)->file);
    if (ferror((*current)->file)) {
      itf_errno = REDIRECTION_ERROR;
      IError("Error on stream \"%s\": %s",
	     (*current)->name!=NULL ? (*current)->name : "(anonymous)",
		sys_errlist[errno]);
      if ((*current)->name && strlen((*current)->name)<100) {
	int i;
	/* put ":<command-name>:" in buf to search for in ignore-pipe-error */
	/* want to get just the first word out of (*current)->name */
	buf[0] = ':';
	for (i=0;
	     isprint((*current)->name[i]) && !isspace((*current)->name[i]);
	     i++)
	  buf[i+1] = (*current)->name[i];
	buf[i] = ':';
	buf[i+1] = '\0';
	/* there could be other error messages stored as well, clear all */
	if (substr(buf, ignore_pipe_error))
	  IResetError();
      }
      if (IsPipe((*current)->type))
	pclose((*current)->file);
      else
	fclose((*current)->file);
      (*current)->file = NULL;
      (*current)->status = STREAM_CLOSED;
    } else if (!(*current)->permanent) {
      DB( fprintf(stderr, "Closing stream: "); )
      DB( IShowStream((*current), stderr, 1); )
      /* assert((*current)->name == NULL); temporary streams have names now */
      if (IsPipe((*current)->type))
	pclose((*current)->file);
      else
	fclose((*current)->file);
      if ((*current)->desc!=NULL) free((*current)->desc);
      if ((*current)->name!=NULL) free((*current)->name);
      if ((*current)->machine!=NULL) free((*current)->machine);
      free((char*) (*current));
    }
    (*current) = old;
    *d = (*current)->file;
    /* what to do if this dout has an error? */
  }
}

/***
 * IEndRedirection - sets itf_errno in case of error
 *
 * current_instream  - the input stream this command is using
 * command_stdin     - the input stream this command inherited & uses
 *                     except when there is redirection
 * old_instream      - will be equal to command_stdin
 */
void IEndRedirection(old_instream, old_outstream)
struct STREAM *old_instream, *old_outstream;
{
  IEndRedirection2(old_instream, &current_instream, &din);
  IEndRedirection2(old_outstream, &current_outstream, &dout);
}

int command_open(tokc, tokv)
int tokc;
char *tokv[];
{
  struct STREAM *stream;
  IUsage("<stream-name> <redirection>");
  if (GiveHelp(tokc)) {
    ISynopsis("open a stream permanently");
    IHelp
      (IHelpArgs,
       "Open a stream permanently, and associate it with  a name.  To resuse",
       "a stream  name,  the stream  must  be deleted  with  \"deleteStream\",",
       "otherwise it hangs around waiting to be reopenned.",
       "",
       "SEE ALSO",
       "redirection, reopen, close, deleteStream, reopen",
       NULL);
    return 1;
  }
  
  if (tokc!=2) {
    IError(IPrintUsage(tokv[0], usage));
    return 0;
  }
  if ((stream=IOpenPermanentStream(tokv[1]))!=NULL) {
    fprintf(stderr, "Openned stream \"%s\": ", stream->name);
    IShowStream(stream, stderr, 0);
  }
  return 1;
}

int	ICloseStream(name, ignore_closed, ignore_error)
char	*name;
int	ignore_closed; /* if this is 1, doesn't return an error if the stream is closed */
int	ignore_error;
{
  struct STREAM *stream;
  int status = 1;
  stream = (struct STREAM*)IGetObjectPtr("STREAM", name);
  if (stream==NULL) {
    if (!ignore_error) {
      itf_errno = REDIRECTION_ERROR;
      IError("\"%s\" is not a stream", name);
    } else {
      IResetError();
    }
    return 0;
  }
  if (stream==stdout_stream || stream==stdin_stream) {
    if (!ignore_error) {
      itf_errno = REDIRECTION_ERROR;
      IError("cannot close stdout or stdin");
    }
    return 0;
  }
  if (stream==current_instream || stream==current_outstream) {
    if (!ignore_error) {
      itf_errno = REDIRECTION_ERROR;
      IError("cannot close current stream");
    }
    return 0;
  }
  if (stream->status!=STREAM_OPEN && !ignore_closed) {
    if (!ignore_error) {
      itf_errno = STREAM_NOT_OPEN;
      IError("stream \"%s\" is not currently open", name);
    }
    return 0;
  }
  if (stream->status==STREAM_OPEN) {
    if (ferror(stream->file)) {
      itf_errno = REDIRECTION_ERROR;
      IError("Error on stream \"%s\": %s",
	      stream->name!=NULL ? stream->name:"(anonymous)",
	      sys_errlist[errno]);
      status = 0;
    }
    if (stream->type==STREAM_REMOTE_FILE||stream->type==STREAM_REMOTE_APPEND)
      pclose(stream->file);
    else
      fclose(stream->file);
    stream->file = NULL;
    stream->status = STREAM_CLOSED;
  }
  return status;
}

struct STREAM *IOpenPermanentStream(name)
char *name;
{
  struct STREAM *stream;
  if (current_instream!=command_stdin && current_outstream!=command_stdout) {
    IError("confused! which stream? - can only open one stream at a time");
    return NULL;
  }
  if (current_instream!=command_stdin)
    stream = current_instream;
  else
    stream = current_outstream;
  
  if (stream->permanent) {
    IError("stream \"%s\" is already open permanently", stream->name);
    return NULL;
  }

  if (!IIsLegalVarName(name)) {
    itf_errno = ILLEGAL_VARNAME;
    IError("%s", name);
    return NULL;
  }

  if (IIsObjectName(name)) {
    itf_errno = VARNAME_IS_USED;
    IError("%s", name);
    return NULL;
  }

  if (!name || !*name) {
    itf_errno = NULL_VARNAME;
    IError("%s", name);
    return NULL;
  }

  if (stream->name!=NULL) free(stream->name);
  stream->name =
    IBindStruct(name, "STREAM", (char*)stream, P_OBJECT, 0);
  if (stream->name == NULL) {
    itf_errno = REDIRECTION_ERROR;
    IError("couldn't add stream to name table");
    return NULL;
  }
  stream->permanent = 1;
  IAddStream(stream);
  return stream;
}

struct STREAM *IReopenStream(name)
char *name;
{
  int status = 1;
  struct STREAM *stream=NULL;
  stream = (struct STREAM*)IGetObjectPtr("STREAM", name);
  if (stream==NULL) {
    itf_errno = REDIRECTION_ERROR;
    IError("\"%s\" is not a stream", name);
    return NULL;
  }
  if (stream->status!=STREAM_CLOSED) {
    if (!ICloseStream(name, 0, 0))
      return NULL;
  }
  if (stream->status==STREAM_CLOSED) {
    switch (stream->type) {
    case STREAM_FILE:
    case STREAM_APPEND:
      status = IRedirectToFile(stream, NULL, 0, true(noclobber), NULL);
      break;
    case STREAM_REMOTE_FILE:
    case STREAM_REMOTE_APPEND:
      status = IRedirectToRemoteFile(stream, NULL, NULL, 0, true(noclobber), NULL);
      break;
    case STREAM_COMMAND:
      status = IRedirectToCommand(stream, NULL, NULL, NULL);
      break;
    case STREAM_SOCKET:
      status = IRedirectToNamedSocket(stream, NULL, NULL);
      break;
    case STREAM_PORT:
      status = IRedirectToNumberedSocket(stream, 0, 0, NULL, NULL);
      break;
    default:
      panic("IReopenStream", "bad type %d", stream->type);
    }
  }
  if (status)
    return stream;
  else
    return NULL;
}

int command_close(tokc, tokv)
int tokc;
char *tokv[];
{
  IUsage("<stream-name>");
  if (GiveHelp(tokc)) {
    ISynopsis("close a stream");
    IHelp
      (IHelpArgs,
       "Close   a  stream.  The stream  is  kept,  but is closed and remains",
       "inactive.  It can be reopenned with the command \"reopen\".",
       "SEE ALSO",
       "redirection, open, reopen, deleteStream, reopen",
       NULL);
    return 1;
  }

  if (tokc!=2) {
    IError(IPrintUsage(tokv[0], usage));
    return 0;
  }
  if (ICloseStream(tokv[1], 0, 0))
    return 1;
  else
    return 0;
}

int command_deleteStream(tokc, tokv)
int tokc;
char *tokv[];
{
  char *stream_name;
  int quiet = 0;
  struct ARG_DESC *argd;
  struct STREAM *stream, *p;
  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pquiet %P %?]", &quiet, "no complaining about non-existent memory");
  Args(argd, "<stream-name>%s", &stream_name);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("delete a stream permanently");
    IHelp(tokc, tokv[0], NULL, synopsis,
	  "The stream is closed and all record of it is deleted",
	  "SEE ALSO",
	  "redirection, open, reopen, close, reopen",
	  NULL);
    return 1;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (!ICloseStream(stream_name, 1, quiet) && !quiet)
    return 0;
  stream = (struct STREAM*)IDeleteObject(stream_name);
  if (itf_errno && quiet)
    IResetError();
  
  if (stream!=NULL) {
    if (stream->desc!=NULL) free(stream->desc);
    if (stream->name!=NULL) free(stream->name);
    if (stream->machine!=NULL) free(stream->machine);
    /* find the stream in the linked list */
    if (first_stream==stream) {
      first_stream = first_stream->next;
      if (last_stream==stream) last_stream = NULL;
    } else {
      p = first_stream;
      while (p && p->next!=stream) p = p->next;
      if (p) p->next = p->next->next;
      if (last_stream==stream) last_stream=p;
    }
    free(stream);
    return 1 ;
  } else if (!quiet) {
    warn("stream_del",
	 "IDeleteObject returned NULL - this is not very tragic");
  }
  return 0 ;
}

int command_reopen(tokc, tokv)
int tokc;
char *tokv[];
{
  struct STREAM *stream;
  IUsage("<stream-name>");
  if (GiveHelp(tokc)) {
    ISynopsis("reopen a closed stream");
    IHelp(IHelpArgs,
	  "Reopen a  currently  inactive stream.  The  stream created is of the",
	  "same type as previously.",
	  "SEE ALSO",
	  "redirection, open, close, deleteStream, reopen",
	  NULL);
    return 1;
  }

  if (tokc!=2) {
    IError(IPrintUsage(tokv[0], usage));
    return 0;
  }
  if ((stream=IReopenStream(tokv[1]))!=NULL) {
    fprintf(stderr, "Opened stream \"%s\": ", stream->name);
    IShowStream(stream, stderr, 0);
    return 1 ;
  } else
    return 0 ;
}

/* ARGSUSED */
FILE *IOpenNamedSocket(name, mode)
char *name;
char *mode;
{
  warn("IOpenNamedSocket", "not yet implemented");
  return NULL;
}

FILE *IOpenNumberedSocket(port, host, mode)
int port;
char *host;
char *mode;
{
  FILE *f;
  int sock;
  struct sockaddr_in server;
  struct hostent *hp ;
  if (host==NULL) host = "localhost";

  /* create the socket */
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if (sock<0) return NULL;

  /* connect the socket to port & host */
  server.sin_family = AF_INET;
  hp = gethostbyname(host);
  if (hp == 0) {
    itf_errno = REDIRECTION_ERROR;
    IError("Unknown host \"%s\"", host);
    return NULL;
  }
  memcpy((char*) &server.sin_addr, hp->h_addr, hp->h_length);
  server.sin_port = htons((unsigned short) port);

  if (connect(sock, &server, sizeof(server))<0) return NULL;

  f = (FILE *)fdopen(sock, mode);
  return f;
}

command_redirection(tokc, tokv)
int tokc;
char *tokv[];
{
  if (tokc!=SHORT_HELP) tokc = LONG_HELP;
  if (GiveHelp(tokc)) {
    IUsage("");
    ISynopsis("prints documentation about streams and I/O redirection");
    IHelp
      (IHelpArgs,
       "The following types of redirection are available:\n",
       "(1)  command ... > filename",
       "(2)  command ... >> filename",
       "(3)  command ... | shell-command",
       "(4)  command ... % port# hostname",
       "(4w) command ... %< port# hostname (read)",
       "(5)  command ... % socket-name (not yet implemented)",
       "(5w) command ... %< socket-name (read) (not yet implemented)",
       "(6)  command ... @ stream-name\n",
       "",
       "Streams  can  be  named   and opened  permanently with the   \"open\"",
       "command.",
       "",
       "The \"open\" command can  be used  with any form of redirection except",
       "the 6'th.  It \"catches\" the redirection and saves it for future use.",
       "Output can be sent to  a stream that has been  \"opened\" by using the",
       "6'th form of redirection.",
       "",
       "A stream  that  has  been opened  can be  closed  using the  \"close\"",
       "command.   A  closed   stream  can  be reopened  using  the \"reopen\"",
       "command, it will be  reopened with the  same  filename,  command, or",
       "port as before.",
       "",
       "For the   hostname,  two  special names  can  be  used:   \"host\" and",
       "\"display\".  If either of these are used,  the hostname will be taken",
       "from the HOST or DISPLAY environment variables",
       "EXAMPLE",
       "Keeping open a file stream:",
       "",
       "xerion-> open junk >> junk-file    (opens the stream named \"junk\")",
       "xerion-> help @ junk               (writes some help to the stream)",
       "xerion-> help -k stream @ junk     (and some more)",
       "xerion-> close junk                (close the stream)",
       "xerion-> reopen junk               (and reopen it, it will cat onto same file)",
       "xerion-> help -k trace @ junk      (some more junk)",
       "xerion-> close junk                (and close it again)",
       "",
       "The following example shows  how to use  redirection with a graphics",
       "program running in another  window, and communicating  with sockets.",
       "In  another   window,  on machine  ephemeral,   we have  started the",
       "\"bboard\" program.  It is waiting for connections on port# 2388...",
       "",
       "xerion-> open bb % 2388 ephemeral  (open the stream to be socket, port# 2388)",
       "xerion-> net-weights @ bb          (send some stuff to it)",
       "xerion-> net-weights @ bb          (and some more)",
       "xerion-> close bb",
       "",
       
       NULL);
    return 1;
  }
  return 0;
}
