
/**********************************************************************
 * $Id: range.c,v 1.2 92/11/30 11:40:08 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include "itf.h"

/***
 * Stuff to do ranges and area format, these work together
 * Ranges are specified as lo-hi+depth, where lo and hi give
 * the start and end of the top row, and depth gives the depth
 * of the area to display below it
 * Area formats are specified as RxC, where R is the number of
 * rows and C is the number of columns.
 * The functions which return the dimensions of areas can take both
 * the range and the format into account, and will return the
 * smallest dimensions that will fit the given sub-area.
 * The function which returns the position gives the position
 * relative to this sub-area.
 */
void SetDefaultRange(range, format)
struct RANGE *range;
struct AREA_FORMAT *format;
{
  range->lo = 0;
  range->nothing = 0;
  if (format==NULL) {
    range->hi = -1;
    range->depth = 1;
  } else {
    range->hi = format->cols-1;
    range->depth = format->rows;
  }
}

int IsDefaultRange(range, format)
struct RANGE *range;
struct AREA_FORMAT *format;
{
  if (   range->lo == 0 && range->nothing == 0
      && (   (format==NULL && range->hi == -1 && range->depth == 1)
	  || (range->hi == format->cols-1 && range->depth == format->rows)))
    return 1;
  else
    return 0;
}

/***
 * int InRangeString(n, range_string)
 * Returns 1 if n is in the range given by range_string, 0 otherwise.
 */
int InRangeString(n, range_string)
int n;
char *range_string;
{
  struct RANGE range;
  if (range_string==NULL || *range_string=='\0')
    return 0;
  if (!ParseRange(range_string, &range, NULL))
    IAbort();
  return InRange(n, &range, NULL);
}

/***
 * int CompareWithRangeString(n, range_string, &range_size)
 * Returns:
 *   0 if n is in the range given by range_string,
 *   1 if n is above the range
 *   2 if the range is inactive or a null range string
 *   -1 if n is below the range
 * If range_size is non-null, the size of the range is stored in
 * the integer pointed to by it.  If the range is inactive, 0 is
 * stored, and if the range is open-ended, -1 is stored.
 */
int CompareWithRangeString(n, range_string, range_size)
int n;
char *range_string;
int *range_size;
{
  struct RANGE range;
  if (range_string==NULL || *range_string=='\0') {
    if (range_size!=NULL)
      *range_size = 0;
    return 2;
  }
  if (!ParseRange(range_string, &range, NULL))
    IAbort();
  if (range_size!=NULL) {
    if (range.nothing)
      *range_size = 0;
    else if (range.hi==-1)
      *range_size = -1;
    else
      *range_size = range.hi - range.lo + 1;
  }
  if (range.nothing)
    return 2;
  if (n<range.lo)
    return -1;
  if (range.hi>-1 && n>range.hi)
    return 1;
  return 0;
}

int InRange(n, range, format)
int n;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  if (range->nothing)
    return 0;
  if (format==NULL || range->depth<=1) {
    if (n<range->lo)
      return 0;
    if (range->hi!=-1 && n>range->hi)
      return 0;
  } else {
    int lo = range->lo;
    int hi = range->hi;
    if (n<lo)
      return 0;
    n -= (lo/format->cols)*format->cols;
    if (n/format->cols > range->depth-1)
      return 0;
    n %= format->cols;
    if (n < lo%format->cols)
      return 0;
    if (hi >= 0 && n > hi%format->cols)
      return 0;
  }
  return 1;
}

void RangeDimensions(max, range, format, rows, cols)
int max;
struct RANGE *range;
struct AREA_FORMAT *format;
int *rows, *cols;
{
  if (format==NULL || range->depth==1) {
    int min=0;
    *rows = 1;
    if (range->lo>0)
      min = range->lo;
    if (range->hi>=0 && max>range->hi)
      max = range->hi;
    *cols = 1+max-min;
  } else {
    /*
     * This assumes hi & lo are on the same line, and depth is correct
     * for the current format
     * This will be so if the format was given to ParseRange
     */
    *rows = range->depth;
    *cols = 1+range->hi-range->lo;
  }
}

void RangeForControl(max, range, format, first_row, top_row, row_inc, row_len)
int max;
struct RANGE *range;
struct AREA_FORMAT *format;
int *first_row, *top_row, *row_inc, *row_len;
{
  int rows, cols;
  if (format==NULL || range->depth==1) {
    int min=0;
    rows = 1;
    if (range->lo>0) min = range->lo;
    if (range->hi>=0 && max>range->hi) max = range->hi;
    cols = 1+max-min;
    *first_row = min;
    *top_row = min+1;
    *row_inc = 1;
    *row_len = cols;
  } else {
    /*
     * This assumes hi & lo are on the same line, and depth is correct
     * for the current format
     * This will be so if the format was supplied to ParseRange when
     * the range was parsed
     */
    rows = range->depth;
    cols = 1+range->hi-range->lo;
    *first_row = range->lo;
    *top_row = format->cols * rows + range->lo;
    *row_inc = format->cols;
    *row_len = cols;
  }
}

/***
 * RangeSize()
 * returns the total number of objects in the range
 */
int RangeSize(max, range, format)
int max;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  int rows, cols;
  RangeDimensions(max, range, format, &rows, &cols);
  return rows*cols;
}

/***
 * RangePosition()
 * returns 1 and puts the row and col in the arguments if n is in range,
 * returns 0 if n is not in range
 */
int RangePosition(n, range, format, row, col)
int n;
struct RANGE *range;
struct AREA_FORMAT *format;
int *row, *col;
{
  if (format==NULL || range->depth==1) {
    *row = 0;
    if (n < range->lo)
      return 0;
    if (range->hi >= 0 && n > range->hi)
      return 0;
    *col = n - range->lo;
    return 1;
  } else {
    int r, c;
    /***
     * Same assumptions as in RangeDimensions
     */
    if (n < range->lo)
      return 0;
    /* work out what column */
    c = n % format->cols;
    if (c < range->lo % format->cols || c > range->hi % format->cols)
      return 0;
    c -= (range->lo % format->cols);
    /* work out what row */
    r = n / format->cols - range->lo / format->cols;
    if (r >= range->depth)
      return 0;
    *row = r;
    *col = c;
    return 1;
  }
}

int ParseRange(tok, range, format)
char *tok;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  int r, lo, hi, depth;
  char c;
  if (tok==NULL) {
    SetDefaultRange(range, format);
    return 1;
  }
  /*
   * The range can look like:
   * (0) -
   * (1) lo-hi
   * (2) lo-
   * (3) -hi
   * (4) lo
   * (5) lo-hi+d
   * (6) lo-+d
   * (7) -hi+d
   * (8) lo+d
   * (9) X
   * one sscanf can parse (1) thru (5), so we must try to identify others
   */
  r = sscanf(tok, "%d-%d+%d", &lo, &hi, &depth);
  if (r==1 && strchr(tok, '+')) {
    if (sscanf(tok, "%d-+%d", &lo, &depth)==2)	/* (6) */
      hi = -1;
    else if (sscanf(tok, "%d+%d", &lo, &depth)==2)	{/* (7) & (8) */
      if (lo<0) {
	hi = -lo;
	lo = 0;
      } else hi = lo;				/* (8) */
    } else {
      SetDefaultRange(range, format);
      IError("bad range format: \"%s\"", tok);
      return 0;
    }
  } else if (r==1) {				/* (2), (3), & (4) */
    depth = 1;
    if (lo<0) {					/* (3) */
      hi = -lo;
      lo = 0;
    } else {
      if (sscanf(tok, "%d%c", &lo, &c)==2 && c=='-')
	hi = -1;				/* (2) */
      else
	hi = lo;				/* (4) */
    }
  } else if (r==2) {
    depth = 1;					/* (1) */
  } else if ((r==0||r==-1) && !strcmp("-", tok)) { /* on mips, r=-1, sun r=0 */
    SetDefaultRange(range, format);		/* (0) */
    return 1;
  } else if ((r==0||r==-1) && !strcmp("X", tok)) { /* on mips, r=-1, sun r=0 */
    range->nothing = 1;
    return 1;
  } else if (r!=3) {
    SetDefaultRange(range, format);
    IError("bad range string: \"%s\"", tok);
    return 0;
  }
  if (hi<lo && hi!=-1) {
    SetDefaultRange(range, format);
    IError("bad range, hi < lo: \"%s\"", tok);
    return 0;
  }
  if (depth<1) {
    SetDefaultRange(range, format);
    IError("bad range, depth < 1: \"%s\"", tok);
    return 0;
  }
  /*
   * if we've got a format, adjust the range so that depth is correct
   * and lo and hi are on the same line
   */
  if (format!=NULL) {
    int first_line;
    int last_line;
    if (hi<0)
      hi = format->rows*format->cols-1;
    if (hi>=format->rows*format->cols) {
      SetDefaultRange(range, format);
      IError("bad range, hi (%d) > objects in format (%dx%d)",
	     hi, format->rows, format->cols);
      return 0;
    }
    first_line = lo/format->cols;
    last_line = hi/format->cols;
    /* get lo & hi on the same line, & the correct depth */
    hi -= (last_line-first_line)*format->cols;
    if (1+last_line-first_line > depth)
      depth = 1+last_line-first_line;
    if (hi<lo) {
      int t = hi;
      hi = lo;
      lo = t;
    }
    if (depth>format->rows) {
      SetDefaultRange(range, format);
      IError("bad range, depth (%d) > rows in format (%d)", depth, format->rows);
      return 0;
    }
    if (hi >= format->rows*format->cols) {
      SetDefaultRange(range, format);
      IError("bad range, high value (%d) > units in format (%d)",
	     hi, format->rows*format->cols-1);
      return 0;
    }
  }
  range->lo = lo;
  range->hi = hi;
  range->depth = depth;
  range->nothing = 0;
  return 1;
}

#ifdef NOTUSEDATALL
/***
 * Parse the range after the n'th colon in the token
 * Returns:
 * 1  - ok
 * 0  - some error parsing range
 * -1 - couldn't find n'th colon
 */
int ParseRangeColon(tok, colon_n, range, format)
char *tok;
int colon_n;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  char *c = tok;
  int n = colon_n;
  /* find the n'th colon */
  while (n>0 && c!=NULL)
    if (c = strchr(c, ':')) {c++;n--;}
  if (c==NULL) {
    SetDefaultRange(range, format);
    /* IError("can't find %s colon in \"%s\"", INumberth(colon_n), tok); */
    return -1;
  } else
    return ParseRange(c, range, format);
}
#endif				/* NOTUSEDATALL */

int ParseAreaFormat(format_str, format)
char *format_str;
struct AREA_FORMAT *format;
{
  int r, c, i;
  char code;
#define PROPLEN 10
  char prop[PROPLEN];
  for (i=0;i<PROPLEN;i++)
    prop[i] = '\0';
  format->rows = format->cols = 0;
  format->code = 0;
  format->join = 0;
  format->lowzero = 0;
  ISetDefaults("AREA-FORMAT", (char *)format);
  if (format_str!=NULL) {
    if (sscanf(format_str, "%d%c%d%c%c%c", &r, &code, &c, prop, prop+1, prop+2)<3) {
      IError("bad layer format \"%s\"", format_str);
      return 0;
    }
    if (code!='x') {
      IError("bad layer format type character '%c' in \"%s\"", code, format_str);
      return 0;
    }
    for (i=0;prop[i]!='\0'&&i<PROPLEN;i++) {
      switch (prop[i]) {
      case 'j':
	format->join = 1;
	break;
      case 'z':
	format->lowzero = 1;
	break;
      default:
	IError("bad layer format property character '%c' in \"%s\"",
	       prop[i], prop);
	return 0;
      }
    }
    format->rows = r;
    format->cols = c;
    format->code = code;
  }
  return 1;
}

static char buffer[30];
char *PrintRangeString(range)
struct RANGE *range;
{
  if (range->nothing)
    sprintf(buffer, "X");
  if (range->depth==1) {
    if (range->hi == -1)
      sprintf(buffer, "%d-", range->lo);
    else
      sprintf(buffer, "%d-%d", range->lo, range->hi);
  } else {
    if (range->hi == -1)
      sprintf(buffer, "%d-+%d", range->lo, range->depth);
    else
      sprintf(buffer, "%d-%d+d", range->lo, range->hi, range->depth);
  }
  return buffer;
}


/***
 * AreaToRange(n, range, format)
 * Translates a location in the total area to a location in the range,
 * where locations are specified by a single number.  Returns -1 if
 * n is not in the range.
 */
int AreaToRange(n, range, format)
int n;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  int rows, cols, row, col;
  if (RangePosition(n, range, format, &row, &col)) {
    RangeDimensions(n, range, format, &rows, &cols);
    return row*cols + col;
  } else
    return -1;
}

/***
 * Translates a location in the range to a location in the total area,
 * where locations are specified by a single number.  Returns -1 if
 * n is not in the range.
 */
int RangeToArea(n, range, format)
int n;
struct RANGE *range;
struct AREA_FORMAT *format;
{
  int acols;		/* dimensions of area */
  int rrows, rcols;	/* dimensions of range */
  int row, col;		/* 2d position in range */

  if (format==NULL)
    return range->lo + n;

  RangeDimensions(range->lo+n, range, format, &rrows, &rcols);

  if (n>=rrows * rcols)
    return -1;
  row = n/rcols;
  col = n%rcols;
  acols = format->cols;
  return range->lo + row * acols + n%rcols;
}
