
/**********************************************************************
 * $Id: loop.c,v 1.12 93/03/29 14:15:47 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <setjmp.h>
#include <xerion/config.h>
#include <xerion/useful.h>
#include "itf.h"
#include "lex.h"
#include "varsub.h"
#include "loop.h"

/********************************************************************/
#define UnknownMask	((unsigned int)(0))
#define WhileMask	((unsigned int)(1 << 0))
#define UntilMask	((unsigned int)(1 << 1))
#define ForMask		((unsigned int)(1 << 2))
#define IfMask		((unsigned int)(1 << 3))
#define ElseMask	((unsigned int)(1 << 4))
#define ElifMask	((unsigned int)(1 << 5))
#define DoMask		((unsigned int)(1 << 6))
#define DoneMask	((unsigned int)(1 << 7))
#define BlankMask	((unsigned int)(1 << 8))
#define NegateMask	((unsigned int)(1 << 9))
#define LoopMask	(WhileMask | UntilMask | ForMask | IfMask)
/********************************************************************/

/********************************************************************/
#define COMMAND_OK	 (itf_errno == 0 \
			  && quit_flag != QUIT && quit_flag != EXIT)
#define doCommand(c,e)	(IDoCommandPipeline(c, NULL, e), \
			 COMMAND_OK ? TRUE : FALSE) 
/********************************************************************/
#define same(s1,s2)	  (strcmp(s1, s2) == 0)
/********************************************************************/

/********************************************************************/
static unsigned int getMask	 ARGS((String	name)) ;
static int	loopError	 ARGS((Loop	*, String)) ;
static int	doLoopCommands	 ARGS((Lex, LoopCommand *, int, int *)) ;
static int	doWhileCommands	 ARGS((Lex, LoopCommand *, int)) ;
static int	doForCommands	 ARGS((Lex, LoopCommand *, int)) ;
static int	doIfCommands	 ARGS((Lex, LoopCommand *, int)) ;
static void	storeLoopCommand ARGS((Loop 	*, String command, 
				       int	tokc, String	*tokv)) ;
/********************************************************************/


/*********************************************************************
 *	Name:		createLoop
 *	Description:	
 *	Parameters:
 *	Return Value:
 *	  Loop	*createLoop - 
 *********************************************************************/
Loop	*createLoop()
{
  Loop	*this ;

  this = (Loop *)malloc(sizeof(Loop)) ;
  this->level	    = 0 ;
  this->numCommands = 0 ;
  this->command     = NULL ;
  this->lex	    = NULL ;	/* don't create yet (too much overhead
				 * if we don't use it) */
  return this ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroyLoop
 *	Description:	
 *	Parameters:
 *	  Loop	*this - 
 *	Return Value:
 *	  void	destroyLoop - 
 *********************************************************************/
void	destroyLoop(this)
  Loop	*this ;
{
  resetLoop(this) ;
  if (this->lex)
    destroyLexObj(this->lex, TRUE) ;
  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		inLoop
 *	Description:	
 *	Parameters:
 *	Return Value:
 *	  int	inLoop - 
 *********************************************************************/
int		inLoop(this)
  Loop		*this ;
{
  return this->level ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		isLoop
 *	Description:	
 *	Parameters:
 *	  String	command
 *	Return Value:
 *	  int	isLoop - 
 *********************************************************************/
int		isLoop(this, command)
  Loop		*this ;
  String	command ;
{
  String	*tokv ;
  int		tokc, returnVal ;

  if (this->lex == NULL) {
    this->lex	    = duplicateLexObj(defaultLex) ;
    LexObjSetSyntax(this->lex, command_syntax) ;
  }
  tokv = LexObjAnalysePush(this->lex, command, &tokc) ;

  if (tokc && tokv
      && (same(tokv[0], "while") || same(tokv[0], "until")
	  || same(tokv[0], "for") || same(tokv[0], "if")))
    returnVal = TRUE ;
  else
    returnVal = FALSE ;
  
  LexObjAnalysePop(this->lex) ;

  return returnVal ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		getMask
 *	Description:	
 *	Parameters:
 *	  String	name - 
 *	Return Value:
 *	  static unsigned int	getMask - 
 *********************************************************************/
static unsigned int	getMask(name)
  String		name ;
{
  if (name == NULL)
    return BlankMask ;

  if (same(name, "while"))
    return WhileMask ;
  else if (same(name, "until"))
    return UntilMask ;
  else if (same(name, "for"))
    return ForMask ;
  else if (same(name, "if"))
    return IfMask ;
  else if (same(name, "else"))
    return ElseMask ;
  else if (same(name, "elif"))
    return ElifMask ;
  else if (same(name, "do") || same(name, "then"))
    return DoMask ;
  else if (same(name, "done") || same(name, "fi"))
    return DoneMask ;
  else 
    return UnknownMask ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addLoopCommand
 *	Description:	
 *	Parameters:
 *	  char	*command - 
 *	Return Value:
 *	  int	addLoopCommand - 
 *********************************************************************/
int		addLoopCommand(this, command, echo)
  Loop		*this ;
  String	command;
  int		echo ;
{
  int		idx ;
  int		tokc ;
  String	*tokv ;
  unsigned int	mask, preMask ;

  if (this->level < 0)
    return 0 ;

  if (this->lex == NULL) {
    this->lex	    = duplicateLexObj(defaultLex) ;
    LexObjSetSyntax(this->lex, command_syntax) ;
  }

  tokv = LexObjAnalysePush(this->lex, command, &tokc) ;

  storeLoopCommand(this, command, tokc, tokv) ;
  mask = this->command[this->numCommands - 1].mask ;

  if (mask & BlankMask) {	/* Blank line */
    LexObjAnalysePop(this->lex) ;
    return this->level ;
  }

  /* check for "do", and be sure it's in the right place */
  if (this->level) {
    preMask = this->command[this->numCommands - 2].mask ;
    
    if ((preMask & (LoopMask | ElifMask)) && !(mask & DoMask))
      return loopError(this, "expected \"do\"/\"fi\"") ;
    else if (!(preMask & (LoopMask | ElifMask)) && (mask & DoMask))
      return loopError(this, "unexpected \"do\"/\"fi\"") ;

  } else if (mask & DoMask) {
    return loopError(this, "unexpected \"do\"/\"fi\"") ;
  }

  /* if it's a loop, make sure that it has proper syntax */
  if (mask & LoopMask) {
    if (mask & ForMask) {
      if (tokc < 4)
	return loopError(this, "too few arguments") ;
      if (strcmp(tokv[2], "in"))
	return loopError(this, "expected \"in\"") ;

    } else if (tokc < 2) {
      return loopError(this, "too few arguments") ;
    }

  /* if it's a do, make sure it has proper syntax */
  } else if ((mask & DoMask) && tokc != 1) {
    return loopError(this, "unexpected arguments") ;

  /* if it's a done, make sure it has proper syntax */
  } else if ((mask & DoneMask) && tokc != 1) {
    return loopError(this, "unexpected arguments") ;
  }

  if (mask & LoopMask)
    ++this->level ;
  else if (mask & DoneMask)
    --this->level ;

  LexObjAnalysePop(this->lex) ;

  if (this->level == 0) {
    jmp_buf	env ;
    void	*oldEnv ;
    if (setjmp(env) == 0) {
      oldEnv = ISetAbortEnv(env);
      idx = 0 ;
      doLoopCommands(this->lex, this->command, echo, &idx) ;
    } 
    ISetAbortEnv(oldEnv);
    resetLoop(this) ;
  }

  return this->level ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		resetLoop
 *	Description:	
 *	Parameters:
 *	Return Value:
 *	  void	resetLoop - 
 *********************************************************************/
int	_resetLoop(this, file, line)
  Loop		*this ;
  String	file ;
  int		line ;
{
#if 0
  fprintf(stderr, "Reseting loop. Called from %s line %d. itf_errno: %d\n",
	  file, line, itf_errno) ;
#endif
  if (this->numCommands) {
    int	idx ;
    while (this->numCommands > 0) {
      --this->numCommands ;
      if (this->command[this->numCommands].line)
	free(this->command[this->numCommands].line) ;
    }
    free((void *)this->command) ;
  }
  
  while (this->lex && this->lex->stackTop >= 0)
    LexObjAnalysePop(this->lex) ;

  this->level = 0 ;

  return this->level ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		storeLoopCommand
 *	Description:	
 *	Parameters:
 *	  Loop	*this - 
 *	  char	*string - 
 *	Return Value:
 *	  static void	storeLoopCommand - 
 *********************************************************************/
static void	storeLoopCommand(this, string, tokc, tokv)
  Loop		*this ;
  char		*string ;
  int		tokc ;
  String	*tokv ;
{
  unsigned int	mask ;

  if (this->numCommands == 0) {
    this->command = (LoopCommand *)malloc(sizeof(LoopCommand)) ;
  } else {
    this->command
      = (LoopCommand *)realloc(this->command,
			       (this->numCommands + 1)*sizeof(LoopCommand)) ;
  }

  mask = (tokv && tokc) ? getMask(tokv[0]) : BlankMask ;
  this->command[this->numCommands].mask = mask ;

  if (mask & (WhileMask | UntilMask | IfMask | ElifMask)) {
    char	buffer[BUFSIZ] ;
    --tokc, ++tokv ;
    if (tokc && strcmp(*tokv, "!") == 0) {
      this->command[this->numCommands].mask |= NegateMask ;
      --tokc, ++tokv ;
    }
    string = LexObjReconstruct(this->lex, buffer, BUFSIZ, tokc, tokv) ;
    this->command[this->numCommands].line = string ? strdup(string) : NULL ;

  } else {
    this->command[this->numCommands].line = strdup(string) ;
  }

  ++this->numCommands ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		doLoopCommands
 *	Description:	figures out what type of loop to do, does it,
 *			then returns the number of commands it did.
 *	Parameters:
 *	  Lex		lex - 
 *	  String	*command - 
 *	Return Value:
 *	  static int	doLoopCommands - 
 *********************************************************************/
static int	doLoopCommands(lex, command, echo, idxp)
  Lex		lex ;
  LoopCommand	*command ;
  int		echo ;
  int		*idxp ;
{
  String	*tokv ;
  int		tokc ;
  int		level, idx ;
  Boolean	commandOk = TRUE ;

  if (command->mask & LoopMask) {
    if (command->mask & ForMask)
      commandOk = doForCommands(lex, command, echo) ;
    else if (command->mask & IfMask)
      commandOk = doIfCommands(lex, command, echo) ;
    else
      commandOk = doWhileCommands(lex, command, echo) ;
  }

  /* count number of commands in the loop */
  level = 0 ;
  idx   = 0 ;
  do {
    unsigned int	mask = command[idx++].mask ;

    if (mask & LoopMask)
      ++level ;
    else if (mask & DoneMask)
      --level ;
  } while (level > 0) ;

  *idxp += idx - 1 ;

  return commandOk ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		doTestCommand
 *	Description:	
 *	Parameters:
 *	  Lex		lex - 
 *	  String	command - 
 *	Return Value:
 *	  static int	doTestCommand - 
 *********************************************************************/
static int	doTestCommand(lex, command, echo)
  Lex		lex ;
  LoopCommand	command ;
  int		echo ;
{
  Boolean	negate = FALSE ;
  Boolean	commandOk ;

  if (command.mask & UntilMask)
    negate = TRUE ;

  if (command.mask & NegateMask)
    negate = !negate ;

  commandOk = doCommand(command.line, echo) ;

  if (negate && itf_errno != INTERRUPT) {
    commandOk = !commandOk ;
    if (itf_errno) {
      IReportError(stderr);
      IResetError();
    } else {
      itf_errno = ERROR_REPORTED ;
    }
  }

  if (commandOk) {
    return TRUE ;
  } else {
    if (itf_errno != INTERRUPT) {
      IReportError(stderr);
      IResetError();
    }
    return FALSE ;
  }
}
/********************************************************************/

/*********************************************************************
 *	Name:		doWhileCommands
 *	Description:	
 *	Parameters:
 *	  LoopCommand	*this - 
 *	Return Value:
 *	  static void	doWhileCommands - 
 *********************************************************************/
static int	doWhileCommands(lex, command, echo)
  Lex		lex ;
  LoopCommand	*command ;
  int		echo ;
{
  Boolean	commandOk = TRUE ;

  while (commandOk && doTestCommand(lex, command[0], echo)) {
    Boolean	done = FALSE ;
    int		idx  = 1 ;

    do {
      unsigned int	mask = command[++idx].mask ;

      if (mask & BlankMask)
	continue ;

      if (mask & DoneMask)
	done = TRUE ;

      else if (mask & LoopMask)
	commandOk = doLoopCommands(lex, &command[idx], echo, &idx) ;

      else if (!(mask & DoMask))
	commandOk = doCommand(command[idx].line, echo);

    } while (commandOk && !done) ;
  } 

  return commandOk ;
} 
/********************************************************************/
static int	doForCommands(lex, command, echo)
  Lex		lex ;
  LoopCommand	*command ;
  int		echo ;
{
  int		loopIdx ;
  String	*tokv ;
  int		tokc ;
  Boolean	commandOk = TRUE ;
  char		buffer[BUFSIZ], *newCommand ;

  newCommand = VarSubstitute(command[0].line, buffer, BUFSIZ, echo);
  if (newCommand && (tokv = LexObjAnalysePush(lex, newCommand, &tokc)))
    commandLexSubstitute(lex, tokc, tokv, echo) ;

  if (!IIsObjectName(tokv[1])) {
    itf_errno = UNKNOWN_VAR ;
    commandOk = FALSE ;
  }
  if (!IIsSimple(tokv[1])) {
    itf_errno = BAD_VARTYPE ;
    commandOk = FALSE ;
  }

  for (loopIdx = 3 ; commandOk && loopIdx < tokc ; ++loopIdx) {
    Boolean	done = FALSE ;
    int		idx  = 1 ;

    ISetValue(tokv[1], tokv[loopIdx]) ;
    do {
      unsigned int	mask = command[++idx].mask ;

      if (mask & BlankMask)
	continue ;

      if (mask & DoneMask)
	done = TRUE ;

      else if (mask & LoopMask)
	commandOk = doLoopCommands(lex, &command[idx], echo, &idx) ;

      else if (!(mask & DoMask))
	commandOk = doCommand(command[idx].line, echo);

    } while (commandOk && !done) ;
  } 
  LexObjAnalysePop(lex) ;

  return commandOk ;
} 
/********************************************************************/
static int	doIfCommands(lex, command, echo)
  Lex		lex ;
  LoopCommand	*command ;
  int		echo ;
{
  int		idx, tokc ;
  String	*tokv ;
  Boolean	done = FALSE ;
  Boolean	doElse = FALSE ;
  Boolean	commandOk = TRUE ;

  idx = 0 ;
  while (!(doElse || doTestCommand(lex, command[idx], echo))) { 
    int		level	  = 1 ;
    Boolean	foundElse = FALSE ;

    do {	/* find then next 'elif' 'else' or 'done' */
      unsigned int	mask = command[++idx].mask ;

      if (mask & BlankMask)
	continue ;

      if (mask & LoopMask)
	++level ;
      else if (mask & DoneMask)
	--level ;

      if (level == 0) {
	--idx ;
	foundElse = doElse = TRUE ;
      } else if (level == 1) {
	if (mask & ElseMask) 
	  foundElse = doElse = TRUE ;
	else if (mask & ElifMask)
	  foundElse = TRUE ;
      }

    } while (!foundElse) ;
  }

  do {
    unsigned int	mask = command[++idx].mask ;

    if (mask & BlankMask)
      continue ;

    if (mask & (DoneMask | ElseMask | ElifMask))
      done = TRUE ;

    else if (mask & LoopMask)
      commandOk = doLoopCommands(lex, &command[idx], echo, &idx) ;

    else if (!(mask & DoMask))
      commandOk = doCommand(command[idx].line, echo);

  } while (commandOk && !done) ;

  return commandOk ;
} 
/********************************************************************/


/*********************************************************************
 *	Name:		loopError
 *	Description:	
 *	Parameters:
 *	  String	s - 
 *	Return Value:
 *	  static void	loopError - 
 *********************************************************************/
static int	loopError(this, s)
  Loop		*this ;
  String	s ;
{
  resetLoop(this) ;
  itf_errno = SYNTAX_ERROR ;

  IError(s) ;

  return this->level ;
}
/********************************************************************/
