
/**********************************************************************
 * $Id: lex.c,v 1.10 93/03/29 14:15:46 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include "itf.h"
#include "lex.h"

#define DB(X)

/*
 *  A general purpose lexical analyser.
 *
 *    This code is intended to be portable, it is known to compile and
 *  run under both XENIX on Intel machines and UNIX on Sun machines.
 *
 *    This lexical analyzer is run on one line at a time and
 *  produces an array of pointers to each of the items in the
 *  line, in a very similar manner to the C-shell passing arguments
 *  to programs via 'argc' and 'argv'.  The syntax is definable, so it is
 *  possible to use syntax rather than spaces to separate arguments.
 *
 *    An example of its use is given at the end of this file.
 *
 *    Arguments are words or operators.
 *    Words are composed entirely of word-characters and
 *  operaters are composed entirely of operater-characters.
 *    Word and operater characters are specified in the initialization
 *  and can be changed at any time, and the sets need not be disjoint.
 *    Strings are enclosed in any type of matching quotes and are returned
 *  as one argument.
 *    To make things easier multiple character operaters can be specified.
 *    This package returns only a couple of errors, but the preprocessor
 *  also calls the error routine from here.  If there is an error,
 *  longjmp() is called with the supplied environment, and if no
 *  environment has been supplied an error message is printed on
 *  stderr and zero arguments are returned.
 *    Where there is  ambiguity the longest strings are chosen first, thus
 *  if "==" is an operater then "===" will be parsed as "==" and "=".
 *    This package is intended to be used with the hash table package
 *  to provide hierarchical processing of commands and programs.
 *  Thus, no facilities are provide to identify numbers, this can be
 *  done by something that tries to make a number or a real number
 *  out of one or more arguments.
 *    To aid with this the argument strings have an extra feature, the
 *  character before the start of the string is a space if there was
 *  a space between that argument and the previous argument.
 *    Quotes are removed from arguments, unless they occur inside other
 *  quotes, or are escaped. Quotes always break operator arguments, and
 *  thus a quote char cannot be an operator char.
 *  Whether they break word arguments is controlled by the parameter
 *  'quote_break'.
 *    There are two bytes prefixing every argument that is returned;
 *    The first byte (tokv[-1]) is true if the argument was preceded
 *  by white space (or was at the beginning of a line) and the second
 *  (tokv[-2]) is true if the argument was quoted.  In fact it is more
 *  than true, its value is 0 if the argument was not quoted and is
 *  the value of the type of quote used if the argument was quoted.
 *  This indication of quoted arguments only applies if 'quote_break'
 *  is true, otherwise it is impossible to say some arguments are quoted
 *  or not quoted; they might have a quoted string in the middle.
 *    Once a new line is parsed the argumment strings of the previous
 *  line are overwritten.  This limitation can be avoided in two ways
 *  (1) by taking control over the memory space by calling
 *  init_lex() with a different memory space.
 *  (2) by using cont_lex() or stack_lex() to continue parsing without
 *  overwriting the arguments constructed by the previous parse.
 *    The command line given to this package is not altered.
 *
 *  The initialization functions are:
 *
 *  char *init_lex(size, mem)
 *  int size;
 *  char *mem;
 *
 *  char *reinit_lex(mem)
 *  char *mem;
 *
 *  int *set_lex_err(env)
 *  int *env;
 *
 *    Both of the above functions return a pointer to the previous
 *  memory space.  The first integer of the memory space is used
 *  to store the size of it, thus in the call to reinit_lex() mem
 *  should point to something returned by init_lex() or reinit_lex().
 *
 *  make_lex_chars(operator_type, quote_break,
 *			space_chars, word_chars, operater_chars,
 *			quote_chars, escape_chars, comm_chars, lookup_table)
 *  int operater_type, quote_break;
 *  char *space_chars, *word_chars, *operater_chars,
 *	 *quote_chars, *escape_chars, *comm_chars, *lookup_table;
 *
 *  char *lex_set_syntax(lookup_table)
 *  char *lookup_table;
 *
 *    Quote_chars, escape_chars, and comm_chars should consist of pairs
 *  of characters.
 *  The first character of a pair in the string quote_chars specifies that
 *  that character can begin a quote, and the second of the pair is the
 *  character that ends a quote.  A character that begins a quote can only
 *  have one possible character to end the quote, but one character may
 *  end any number of quote characters.  The first character of a pair in
 *  the string escape_chars is the character that is escaped, the second
 *  is the value of the escaped character (the value is just another character).
 *  These strings must be of even length, if they are not, disaster will ensue.
 *    make_lex_chars() makes a lookup table of the characters in store[].
 *    lookup_table[] must be at least LEX_TABLE chars long.
 *    set_lex_chars() sets that lookup table the one to be used by lex,
 *  and returns the address of the previous one.
 *    Operater strings are entered into the package by putting them
 *  in the hash table with the following function.
 *
 *  TBL_ITEM *ins_item(operater, operater_type)
 *  char *operater;
 *  int operater_type;
 *
 *    The memory that lex uses is specified by init_lex().  The maximum
 *  number of arguments is variable, string space and argument pointer
 *  are allocated from opposite ends of the storage space.
 *    The lex character environment is set up by first calling make_lex_chars()
 *  to create a lookup table and then calling set_lex_chars() to use it.  For
 *  details of the lookup table, see the comments with make_lex_chars()
 *    The operater_type is an integer which lex uses
 *  to look up operaters in the hash table.  This provides a mechanism
 *  for swapping between different sets of operaters.  Any character
 *  that is not a word, quote, space, or comment
 *  character is treated as an operater character,
 *  the only reason that there operater characters can be specified is
 *  so that word and operater characters can overlap.
 *    Lex can be set up to make a longjmp() when it runs out of memory,
 *  but care must be taken when using this feature.  The various
 *  restrictions on the use of longjmp() apply here, and it is not
 *  recommended that this feature be used unless the programmer is
 *  thoroughly familiar with setjmp() and longjmp().
 */

static void massage_string ARGS((Lex, char *original)) ;
static int new_arg ARGS((Lex, char *, int, int, int, int, int)) ;
static int parse_line ARGS((Lex this, char *line, int, int));

#define NO_ESCAPE ((char)0xFF)
#define SPACE_TYPE 1
#define WORD_TYPE 2
#define OP_TYPE 4
#define SPACE_CHAR(l, C) ((l)->lex_char_type[LEX_CM & (C)] & SPACE_TYPE)
#define WORD_CHAR(l, C) ((l)->lex_char_type[LEX_CM & (C)] & WORD_TYPE)
#define OP_CHAR(l, C) ((l)->lex_char_type[LEX_CM & (C)] & OP_TYPE)
#define SET_LEX_TYPE(l, TYPE, C, VALUE) (l)->lex_char_type[LEX_CM & (C)] = (VALUE ? (l)->lex_char_type[LEX_CM & (C)] | TYPE : (l)->lex_char_type[LEX_CM & (C)] & ~TYPE)

Lex	defaultLex = NULL ;

/***********************************************************************
 * LexInit(size, space, ht)
 * give some memory and a hash table to lex - return the address
 * of the previous memory
 */
Lex	createLexObj(size, space, ht)
int size;
char *space;
HASH_TABLE *ht;
{
  Lex	this ;

  this = (Lex)calloc(1, sizeof(LexRec)) ; /* use calloc to zero all data */
  if (this == NULL)
    return NULL ;

  LexObjMem(this, size, space) ;
  this->lex_ht = ht;

  this->stackTop = -1 ;

  return this ;
}
/**********************************************************************/
Lex	duplicateLexObj(original) 
Lex	original ;
{
  Lex	this ;
  char	*mem = malloc(original->mem_size) ;

  this = createLexObj(original->mem_size, mem, original->lex_ht) ;
  if (this)
    LexObjSetSyntax(this, original->syntax) ;
  return this ;
}
/**********************************************************************/
void	destroyLexObj(this, freeMemory) 
Lex	this ;
int	freeMemory ;
{
  if (this == defaultLex)
    defaultLex = NULL ;

  if (freeMemory && this->mem)
    free((void *)this->mem) ;
  if (this)
    free((void *)this) ;
}
/**********************************************************************/
char *LexInit(size, space, ht)
int size;
char *space;
HASH_TABLE *ht;
{
  char	*old_mem ;

  if (defaultLex == NULL) {
    defaultLex = createLexObj(size, space, ht) ;
    old_mem = NULL ;
  } else {
    old_mem = defaultLex->mem ;
  }

  return old_mem ;
}
/**********************************************************************/

/***********************************************************************
 * LexMem(size, space)
 * give some memory to lex to use - return the address
 * of the previous memory
 */
char	*LexObjMem(this, size, space)
Lex this;
int size;
char *space;
{
  char *old_mem = this->mem;

  this->mem	   = space;
  this->mem_size   = size;
  *(int*)(this->mem) = this->mem_size;
  this->arg_start  = (String *)(this->mem + sizeof(long));
  this->arg_end    = this->arg_start ;
  *this->arg_end   = NULL;
  this->arg_count  = 0;
  this->string_end = this->mem + this->mem_size;

  return old_mem;
}
/**********************************************************************/
char *LexMem(size, space)
int size;
char *space;
{
  char *old_mem;
  if (defaultLex == NULL) {
    defaultLex = createLexObj(size, space, NULL) ;
    old_mem = NULL ;
  } else {
    old_mem = LexObjMem(defaultLex, size, space) ;
  }

  return old_mem;
}
/**********************************************************************/

/***********************************************************************
 * LexReinit(space)
 * return some memory to lex - return the address
 * of the previous memory
 */
char *LexObjReinit(this, space)
Lex this ;
char *space;
{
  return LexObjMem(this, *(int *)space, space) ;
}
/**********************************************************************/
char *LexReinit(space)
char *space;
{
  if (defaultLex == NULL)
    defaultLex = createLexObj(0, NULL, NULL) ;
  
  return LexObjReinit(defaultLex, space) ;
}
/**********************************************************************/

/***********************************************************************
 *  make_lex_chars() -- make a lex lookup table
 *  o_type	- the context to use when looking up operaters
 *  q_break	- if true then quotes break words
 *  massage	- if true then massage the string (remove quotes, escapes)
 *  The rest of the arguments specify the various classes of characters
 *  s_chars	- space
 *  w_chars	- word
 *  o_chars	- operater
 *  q_chars	- pairs of characters to delimit strings (quote chars)
 *  e_chars	- values for characters that are escaped within quotes
 *  c_chars	- pairs of characters to delimit comments
 *
 *  A lex lookup table contains four arrays and one integer, the
 *  integer is at the beginning of the table and the four arrays follow.
 *  All the arrays are of size LEX_N_CHARS, (LEX_CM is used to mask indexes
 *  to keep within bounds).
 *  The first array is a simple TRUE/FALSE table to identify space characters,
 *  word characters and operator characters.
 *  The last three arrays all contain pairs of characters, in the following
 *  manner: the value of the location indexed by the first member of a pair
 *  is the second member of the pair.  Thus if "ab" was a pair in quote_chars,
 *  quote_chars['a'] = 'b';  This makes it easy to look for the closing quote,
 *  etc, and to translate escape characters.  Note that it does not allow us
 *  to have multi-character comment or quote delimiters.  Since lex has no
 *  errors except out-of-memory, end of line also serves to delimit quoted
 *  strings and comments.  Characters only have escaped values inside strings.
 */
char *LexObjMakeSyntax(this, o_type, q_break, massage,
		       s_chars, w_chars, o_chars, q_chars, e_chars, c_chars, 
		       lookup)
Lex this ;
int o_type, q_break, massage;
char *s_chars, *w_chars, *o_chars, *q_chars, *e_chars, *c_chars, *lookup;
{
  int i;
  char *old_lookup;
  char *lex_message = "string should be pairs: %s\n";
  old_lookup = LexObjSetSyntax(this, lookup);
  if (strlen(q_chars) & 1) {
    warn("LexObjMakeSyntax", lex_message, q_chars);
    q_chars[strlen(q_chars)-1]=0;
  }
  if (strlen(c_chars) & 1) {
    warn("LexMakeSyntax", lex_message, c_chars);
    c_chars[strlen(c_chars)-1]=0;
  }
  /* hard to tell if string is pairs with escaped characters,
     value could be zero */
  if (s_chars!=NULL) {
    /* set all arrays to null */
    for (i=0;i<LEX_N_CHARS;i++) {
      SET_LEX_TYPE(this, SPACE_TYPE, i, 0);
      if (o_type) {
	SET_LEX_TYPE(this, WORD_TYPE, i, 0);
	/* if no operator chars are specified, everything can be an operator
	   to start with, things will get eliminated later */
	SET_LEX_TYPE(this, OP_TYPE, i, (*o_chars==NULL ? i!=0 : 0));
      } else {
	/* there are no operators, so everything can be a word
	   to start with, things will get eliminated later */
	SET_LEX_TYPE(this, WORD_TYPE, i, (*o_chars==NULL ? i!=0 : 0));
	SET_LEX_TYPE(this, OP_TYPE, i, 0);
      }
      this->comm_chars[LEX_CM & i]=0;
      this->quote_chars[LEX_CM & i]=0;
      this->escape_chars[LEX_CM & i] = NO_ESCAPE;
    }
    /* if something is a word char, take it out of operators */
    if (w_chars!=NULL)
      for (;*w_chars;w_chars++) {
	SET_LEX_TYPE(this, WORD_TYPE, *w_chars, 1);
	SET_LEX_TYPE(this, OP_TYPE, *w_chars, 0);
      }
    /* set the operator characters */
    if (o_chars!=NULL)
      for (;*o_chars;o_chars++) SET_LEX_TYPE(this, OP_TYPE, *o_chars, 1);
    /* set the pairs of quote_chars */
    if (q_chars!=NULL)
      for (;*q_chars;q_chars+=2) {
	SET_LEX_TYPE(this, OP_TYPE, *q_chars, 0);
	SET_LEX_TYPE(this, WORD_TYPE, *q_chars, 0);
	this->quote_chars[LEX_CM & *q_chars] = *(q_chars+1);
      }
    /* and the pairs of comment chars */
    if (c_chars!=NULL)
      for (;*c_chars;c_chars+=2) {
	SET_LEX_TYPE(this, OP_TYPE, *c_chars, 0);
	SET_LEX_TYPE(this, WORD_TYPE, *c_chars, 0);
	this->comm_chars[LEX_CM & *c_chars] = *(c_chars+1);
      }
    /* and the pairs of escape chars */
    if (e_chars!=NULL)
      for (;*e_chars;e_chars+=2) this->escape_chars[LEX_CM & *e_chars] = *(e_chars+1);
    /* if something is a space char, take it out of word and operator */
    for (;*s_chars;s_chars++) {
      SET_LEX_TYPE(this, SPACE_TYPE, *s_chars, 1);
      SET_LEX_TYPE(this, WORD_TYPE, *s_chars, 0);
      SET_LEX_TYPE(this, OP_TYPE, *s_chars, 0);
    }
  }
  *this->op_type = o_type;
  *this->quote_break = q_break;
  *this->massage = massage;
  /* LexSetSyntax(old_lookup); */
  return old_lookup; /* return the old lookup table */
}
/**********************************************************************/
char *LexMakeSyntax(o_type, q_break,
	s_chars, w_chars, o_chars, q_chars, e_chars, c_chars, lookup)
int o_type, q_break;
char *s_chars, *w_chars, *o_chars, *q_chars, *e_chars, *c_chars, *lookup;
{
  if (defaultLex == NULL)
    defaultLex = createLexObj(0, NULL, NULL) ;

  return LexObjMakeSyntax(defaultLex,
			  o_type, q_break, TRUE,
			  s_chars, w_chars, o_chars, q_chars, e_chars, c_chars,
			  lookup) ;
}
/**********************************************************************/

/***********************************************************************
 *  LexSetSyntax() -- swap in a new lookup table and return the start
 *  address of the old one.
 */
String LexObjSetSyntax(this, lookup)
Lex    this ;  
String lookup;
{
  char	*old_lookup;
  old_lookup = (char*)this->op_type;

  if (lookup == old_lookup) 
    return lookup;

  this->op_type	      = (int*)lookup;
  this->quote_break   = this->op_type+1;
  this->massage	      = this->quote_break+1;
  this->lex_char_type = lookup + 3*sizeof(long); /* may be further along than
						  * the end of massage, but
						  * that shouldn't be a problem
						  */
  this->quote_chars   = this->lex_char_type + LEX_N_CHARS;
  this->escape_chars  = this->quote_chars   + LEX_N_CHARS;
  this->comm_chars    = this->escape_chars  + LEX_N_CHARS;

  return old_lookup;
}
/**********************************************************************/
char *LexSetSyntax(lookup)
char *lookup;
{
  if (defaultLex == NULL)
    defaultLex = createLexObj(0, NULL, NULL) ;

  return LexObjSetSyntax(defaultLex, lookup) ;
}
/**********************************************************************/

/***********************************************************************
 *  new_arg()
 *  Adds an argument into the memory structure, using memory like this:
 *          +------------------------------------------------+
 *          |   +----------------------------------------+   |
 *          |   |   +--------------------------------+   |   |
 *          |   |   |                                |   |   |
 *          |   |   |                                v   v   v
 *      I   P Q P Q P Q ...   0 0 ...   free   ...   S   S   S
 *      ^   ^ ^               ^-^         ^          ^   ^
 *      |   | |               |           |          argument strings
 *      |   | |               |           free space
 *      |   | |               nulls that terminate array of pointers
 *	|   | pointers to location in original string (if requested)
 *      |   pointers to arguments
 *      size of memory
 *          ^arg_start        ^arg_end               ^string_end
 *  arg_end points to the null that terminates the array of pointers
 *  and string_end points to the first character of the last argument
 *  string allocated.
 *  Each argument string has two characters before it that indicate
 *  if the argument was preceded by some white space, and if the
 *  argument was quoted (only if 'quote_break' is true).
 *  tokv[-1] is true if there was white space before
 *  tokv[-2] is true if the argument was quoted
 */
static int new_arg(this,
		   str, len, prev_space, quoted, is_operator, give_origins)
Lex this ;
char *str;
int len, prev_space, quoted, is_operator, give_origins;
{
  if ((len + 4*sizeof(char*) + 2) 
      > (this->string_end - (char*)this->arg_end)) {
    /*
     *  This is the out of space error!!
     */
    itf_errno = LEX_ERROR;
    IError("Too many arguments in command (lex out of memory)");
    return 0;
  }
  if (str==NULL) {
    *(this->arg_end++) = NULL;
    if (this->prev_give_origins) *(this->arg_end++) = NULL;
  } else {
    *(--this->string_end) = 0;
    this->string_end -= len;
    *(this->arg_end++) = this->string_end;
    if (give_origins) *(this->arg_end++) = str;
    (void) memcpy (this->string_end, str, len);
    *(--this->string_end) = 0;
    if (prev_space) *this->string_end |= LEX_SP_BF_TOK;
    if (is_operator) *this->string_end |= LEX_IS_OP;
    *(--this->string_end) = quoted;
    *this->arg_end = NULL;
    if (give_origins) *(this->arg_end+1) = NULL;
    this->arg_count++;
  }
  this->prev_give_origins = give_origins;
  return 1;
}
/**********************************************************************/
char	*LexObjChangeArg(this, tokp, str)
  Lex	this ;
  char	**tokp ;
  char	*str ;
{
  int	len = str ? strlen(str) : 0 ;

  if ((len + 4*sizeof(char*) + 2)
      > (this->string_end - (char*)this->arg_end)) {
    /*
     *  This is the out of space error!!
     */
    itf_errno = LEX_ERROR;
    IError("Too many arguments in command (lex out of memory)");
    return NULL;
  }

  if (str) {
    *(--this->string_end) = 0;
    this->string_end -= len;
    (void) strncpy(this->string_end, str, len);
    *(--this->string_end) = 0;
    if (LexSpBefore(*tokp))	*this->string_end |= LEX_SP_BF_TOK;
    if (LexIsOperator(*tokp))	*this->string_end |= LEX_IS_OP;
    *(--this->string_end) = LexQuotedTok(*tokp) ;
    *tokp = this->string_end + 2 ;
  } else {
    *tokp = NULL;
  }

  return *tokp ;
}
/**********************************************************************/

/**********************************************************************/
static int parse_line(this, line, only_first, give_origins)
Lex this ;
char *line;
int only_first, give_origins;
{
  int found_space=1;
  int in_quote=0;
  int had_quote;
  char *end_quote_pos, *start_quote_pos, *cur_char;
  char *escaped=NULL;
  char end_quote;
  char *arg, c;
  int found_arg, is_operator, good_arg;
  cur_char = line;
  while (*cur_char) {		/* process each argument */
    DB( printf("doing line:%s", cur_char); )
    if (SPACE_CHAR(this, *cur_char)) {
      found_space=1;
      do cur_char ++; while (SPACE_CHAR(this, *cur_char));
    }
    arg = cur_char;
    end_quote = 0;
    had_quote = 0;
    is_operator = 0;
    good_arg = 0;
    if (*arg) {
      DB( printf("Processing argument from:%s", arg); )
      if (OP_CHAR(this, *cur_char) && *this->op_type) {
	/*
	 * concatenate operator characters - operators can only be
	 * more than one character long if they are found in the table
	 */
	for (;;) {
	  is_operator = 1;
	  cur_char++;
	  c = cur_char[1]; /* use c to remember the char at cur_char[1] */
	  cur_char[1] = 0;
	  /* printf("Searching (%d) for \"%s\",
	            context %d\n", OP_CHAR(c), arg, *op_type); */
	  /* let operators have non-operator syntax in them provided
	     they are in the table */
	  if (!*cur_char /* !OP_CHAR(*this->cur_char) */
	      || !this->lex_ht 
	      || ITableLookup(this->lex_ht, arg, *this->op_type)==NULL) {
	    /* restore the cur_char to its previous state & stop looking for op */
	    cur_char[1] = c;
	    break;
	  }
	  /* restore the cur_char to its previous state & continue */
	  cur_char[1] = c;
	}
	good_arg=1;
      } else if (WORD_CHAR(this, *cur_char)
		 || this->quote_chars[LEX_CM & *cur_char] || *cur_char=='\\') {
	/*
	 *  Handle both words and quotes
	 */
	found_arg=0; in_quote=0;
	while (*cur_char && !found_arg) {
	  DB( printf("At char %c: q %d e %d eq %c\n", *cur_char, in_quote, escaped==cur_char, end_quote); )
	  if (*cur_char=='\\' && escaped!=cur_char) escaped = ++cur_char;
	  else if (in_quote && *cur_char==end_quote && cur_char!=escaped) {
	    /* coming out of a quote */
	    if (*this->quote_break) found_arg=1;
	    end_quote_pos = cur_char;
	    cur_char++; in_quote=0;
	  }
	  else if (this->quote_chars[LEX_CM & *cur_char]
		   && !in_quote && cur_char!=escaped) {
	    /* looks like we're going into a quote */
	    if (*this->quote_break && cur_char > arg) found_arg=1;
	    else {
	      start_quote_pos = cur_char;
	      in_quote=1;
	      had_quote = *cur_char;
	      end_quote = this->quote_chars[LEX_CM & *cur_char];
	      cur_char++;
	    }
	  }
	  else if (WORD_CHAR(this, *cur_char)
		   || in_quote || cur_char==escaped) cur_char++;
	  else found_arg=1;
	}
	DB( printf("Found arg, next begins at <%s>, q %d\n", cur_char, in_quote); )
	if (/* *cur_char==0 && */ in_quote) {
	  itf_errno = LEX_ERROR;
	  IError("unmatched quote: expected %c", end_quote);
	  goto error;
	}
	good_arg=1;
      } else if (this->comm_chars[LEX_CM & *cur_char]) {
	end_quote = this->comm_chars[LEX_CM & *cur_char];
	do cur_char++; while (*cur_char && (*cur_char != end_quote || *(cur_char-1)=='\\'));
	if (*cur_char==end_quote) cur_char++;
	good_arg=1;
	continue;			/* skip this comment */
      } else if (*cur_char) {
	/* warn("ParseLine", "character has unknown type: %d %c", *cur_char, *cur_char); */
	cur_char++;
	/* exit(1); */
      }
      DB( printf("length of arg is %d\n", cur_char-arg); )
      if (good_arg
	  && !new_arg(this, arg, cur_char-arg, found_space,
		      (had_quote && start_quote_pos==arg
		                && end_quote_pos==(cur_char-1)
		       ? had_quote : 0),
		      is_operator, give_origins))
	goto error;
      DB( printf("arg is:%s\n", *(arg_end-(give_origins ? 2 : 1))); )
      if (good_arg && *this->massage) {
	massage_string(this, *(this->arg_end-(give_origins ? 2 : 1)));
	DB( printf("massaged:%s\n", *(arg_end-(give_origins ? 2 : 1))); )
      }
    }
    found_space=0;
    if (only_first && this->arg_count >= only_first) break;
  }
  return 1;
error: ;
  return 0;
}
/**********************************************************************/

/*
 *  massage_string()
 *  Replace the escaped characters in a string with their escape values
 *  and remove quotes
 *  The string will be made shorter.
 *  The escape value is given in the escape_char[] array, if the value
 *  there is -1, the value of the escaped char is the character itself.
 */
static void massage_string(this, original)
Lex this ;
char *original;
{
  char *new=original;
  char end_quote=0;
  char *quote_chars=this->quote_chars;
  char *escaped=NULL;
  char *escape_chars=this->escape_chars;

  while (*original) {
    if (original == escaped) {
      *(new++) = (escape_chars[LEX_CM & *original]==NO_ESCAPE
		  ? *(original++)
		  : escape_chars[LEX_CM & *(original++)]);
    } else {
      if (*original=='\\') {
	escaped = ++original;

      } else if (end_quote && *original==end_quote) {
	/* coming out of a quote */
	end_quote=0;
	original++;

      } else if (!end_quote && quote_chars[LEX_CM& *original]) {
	/* looks like we're going into a quote */
	end_quote = quote_chars[LEX_CM & *original];
	original++;
      } else {
	*(new++) = *(original++);
      }
    }
  }
  *new=0;
}

/**********************************************************************/
char **LexObjAnalyse(this, line, tokc)
Lex this ;
String	line;
int *tokc;
{
  this->arg_end = this->arg_start = (char**)(this->mem + sizeof(long));
  *this->arg_end = NULL;
  this->arg_count = 0;
  this->string_end = this->mem + this->mem_size;
  if (parse_line(this, line, 0, 0)) {
    *tokc = this->arg_count;
    return this->arg_start ;
  } else {
    *tokc = 0 ;
    return NULL;
  }
}
/**********************************************************************/
char **LexAnalyse(syntax, line, tokc)
char *syntax, *line;
int *tokc;
{
  if (defaultLex == NULL) {
    *tokc = 0 ;
    return NULL ;
  }

  LexObjSetSyntax(defaultLex, syntax);
  return LexObjAnalyse(defaultLex, line, tokc);
}
/**********************************************************************/

/**********************************************************************/
char **LexObjAnalyseAdd(this, line, tokc)
Lex this ;
char *line;
int *tokc;
{
  char **these_args = this->arg_end;

  this->arg_count = 0;
  if (parse_line(this, line, 0, 0)) {
    *tokc = this->arg_count;
    return these_args;
  } else {
    *tokc = 0 ;
    return NULL;
  }
}
/**********************************************************************/
char **LexAnalyseAdd(syntax, line, tokc)
char *syntax;
char *line;
int *tokc;
{
  if (defaultLex == NULL) {
    *tokc = 0 ;
    return NULL ;
  }

  LexObjSetSyntax(defaultLex, syntax);
  return LexObjAnalyseAdd(defaultLex, line, tokc) ;
}
/**********************************************************************/

/**********************************************************************/
char	**LexObjAnalysePush(this, line, tokc)
  Lex	this ;
  char	*line;
  int	*tokc;
{
  char **top_args;

  if (this->stackTop + 1 >= this->maxStackSize) {
    if (this->stack == NULL)
      this->stack = (LEX_STACK *)malloc((this->stackTop+2)*sizeof(LEX_STACK)) ;
    else
      this->stack = (LEX_STACK *)realloc(this->stack,
					 (this->stackTop+2)*sizeof(LEX_STACK));
    this->maxStackSize = this->stackTop + 2 ;
  }

  ++this->stackTop ;
  this->stack[this->stackTop].arg_end	 = this->arg_end ;
  this->stack[this->stackTop].string_end = this->string_end ;
  this->stack[this->stackTop].syntax	 = this->syntax ;

  /* put the appropriate number of nils at the end */
  if (new_arg(this, NULL, 0, 0, 0, 0, 0)) {
    top_args = this->arg_end;
    this->arg_count = 0;
    if (parse_line(this, line, 0, 0))
      *tokc = this->arg_count;
    else
      *tokc = 0 ;
  } else {
    *tokc = 0 ;
    top_args = NULL ;
  }
  return top_args ;
}
/**********************************************************************/
char **LexAnalysePush(syntax, line, tokc, lex_stack)
char *syntax;
char *line;
int *tokc;
LEX_STACK *lex_stack;
{
  char **args;
  if (defaultLex == NULL) {
    *tokc = 0 ;
    return NULL ;
  }

  LexObjSetSyntax(defaultLex, syntax);
  args = LexObjAnalysePush(defaultLex, line, tokc) ;
#if 0
  memcpy(lex_stack,
	 defaultLex->stack + defaultLex->stackTop, sizeof(LEX_STACK)) ;
#else
  *lex_stack = defaultLex->stack[defaultLex->stackTop] ;
#endif

  return args ;
}
/**********************************************************************/


/**********************************************************************/
void	LexObjAnalysePop(this)
  Lex	this ;
{
  if (this->stackTop >= 0) {
    this->arg_end    = this->stack[this->stackTop].arg_end;
    this->string_end = this->stack[this->stackTop].string_end;
    this->syntax     = this->stack[this->stackTop].syntax;
    --this->stackTop ;
  }
}
/**********************************************************************/
void LexAnalysePop(lex_stack)
LEX_STACK *lex_stack;
{
  if (defaultLex == NULL)
    return ;

  while (defaultLex->stackTop >= 0
	 && memcmp(defaultLex->stack + defaultLex->stackTop,
		   lex_stack, sizeof(LEX_STACK)))
    LexObjAnalysePop(defaultLex) ;

  if (defaultLex->stackTop > 0)
    LexObjAnalysePop(defaultLex) ;
}
/**********************************************************************/


/**********************************************************************/
char	*LexSave(lex_stack)
LEX_STACK *lex_stack;
{
  char	*newMemory ;

  if (defaultLex == NULL)
    return NULL ;

  newMemory = malloc(defaultLex->mem_size) ;
  if (newMemory) {
    memcpy(newMemory, defaultLex->mem, defaultLex->mem_size) ;

    lex_stack->arg_end = defaultLex->arg_end;
    lex_stack->string_end = defaultLex->string_end;
    lex_stack->syntax = NULL;
  }
  return newMemory ;
}
/**********************************************************************/

/**********************************************************************/
void LexRestore(lex_stack, oldMemory)
LEX_STACK *lex_stack;
char *oldMemory;
{
  if (defaultLex == NULL)
    return ;

  memcpy(defaultLex->mem, oldMemory, defaultLex->mem_size) ;
  defaultLex->arg_end = lex_stack->arg_end;
  defaultLex->string_end = lex_stack->string_end;
}
/**********************************************************************/

/**********************************************************************/
char	*LexObjReconstruct(this, line, maxLen, tokc, tokv)
  Lex	this ;
  String line ;
  int	maxLen ;
  int	tokc ;
  char **tokv ;
{
  int	lineLength ;
  int	idx ;
  char	*ptr ;
  char	quoted ;

  if (maxLen <= 0)
    return NULL ;

  line[0] = '\0' ;
  lineLength = 0 ;
  for (idx = 0 ; idx < tokc && *tokv ; ++idx, ++tokv) {
    if (lineLength < maxLen && LexSpBefore(*tokv))
      line[lineLength++] = ' ' ;

    quoted = LexQuotedTok(*tokv) ;
    if (quoted && lineLength < maxLen)
      line[lineLength++] = quoted ;

    ptr = *tokv ;
    while (lineLength < maxLen && ptr && *ptr) {
      if ((   *ptr == '\\'
	   || *ptr == this->quote_chars[LEX_CM & quoted]
	   || (!quoted && this->quote_chars[LEX_CM & *ptr]))
	  && lineLength < maxLen-1)
	line[lineLength++] = '\\' ;
      line[lineLength++] = *ptr++ ;
    }

    if (quoted && lineLength < maxLen)
      line[lineLength++] = this->quote_chars[LEX_CM & quoted] ;
  }
  if (lineLength < maxLen)
    line[lineLength] = '\0' ;

  return line ;
}
/**********************************************************************/
