
/**********************************************************************
 * $Id: itf.c,v 1.29 93/04/14 10:51:10 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/config.h>

#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/param.h>
#include <sys/resource.h>

#include "itf.h"
#include "version.h"
#include "lex.h"
#include "varsub.h"
#include "itf-argf.h"
#include "loop.h"
#include "help.h"
#define DB(X)

#define EVERYTHING	1
#define ONLY_DEFAULTS	2
#define EXCEPT_DEFAULTS	3

#define MAX_COMMAND_LEN 10000
#define ERROR_BUF_SIZE 1000
#define ERROR_BUF_SAFE 200

#define ATTEMPT_COMMAND(s)	(just_do_defaults==EVERYTHING \
				 || !strcmp("default", s) \
				 || !strcmp("read", s) \
				 || !strcmp("var", s))

static int	XMode = 0 ;

PFI pre_command_function = NULL;
PFI post_command_function = NULL;
char *help_string;
extern int n_commands;
static char *prompt;
char *noclobber;
char *unset_error;
char *plus_minus_options;
static char *redirection;
static char *alias_enabled;
char *automore;
double nan_double;
static char *help_automore;
static char *echo;
static char *read_echo;
static char *options_echo;
static char *use_default_options;
static long status;
static char *print_status;
static char *command_line1;
static char *command_line2;
static char *readpath;
static char *show_function_calls;
static int command_line_len;
int auto_more_enabled = 0;
static int command_id;	/* used in IError */
static struct STACK *function_names;
static HASH_TABLE *itf_help_table;
static HASH_TABLE *lex_table;
static int itf_help_context;
static int itf_func_context;
static int op_context;	/* context for operators in the lex table */

static char simple_syntax[LEX_TABLE];

HASH_TABLE *itf_func_table;
int itf_com_context;
char command_syntax[LEX_TABLE];
char object_syntax[LEX_TABLE];
char itf_error[ERROR_BUF_SIZE];
char itf_tmp_buf[1000];
int itf_errno;
int caught_signal = 0;
enum QUIT_TYPE quit_flag;
static int just_do_defaults=EVERYTHING;
FILE *din=stdin;
FILE *dout=stdout;
char *myname;
int mypid;

static void	*itf_abort_env;

int command_read();
int command_echo();
int command_quit();
int command_exit();
int command_default();

char *itf_error_str[TOP_ITF_ERRNO+1] = {
  "",
  "Command error: ",
  "Syntax error: ",
  "Lexical error: ",
  "Unknown command: ",
  "Unknown error: ",
  "Unknown variable: ",
  "Unknown structure: ",
  "Unknown field: ",
  "Null variable name",
  "Illegal variable name: ",
  "Illegal object: ",
  "Bad object syntax: ",
  "Array index out of bounds: ",
  "Need array index for ",
  "Need structure field for ",
  "Need structre field or array index for ",
  "Variable is readonly: ",
  "Bad var type: ",
  "Internal null pointer, happenned with: ",
  "Excess in object: ",
  "Attempt to dereference null pointer: ",
  "Expected an object - found nothing ",
  "Out of memory in ",
  "Interrupt",
  "System error signal sent and caught",
  "Variable name is already used: ",
  "Redirection I/O error: ",
  "Redirection I/O error: ",
  "",
  NULL,
};

static void IInit ARGS((int max_symbols, int max_command_len,
		       int lex_space, char *name)) ;
static void	IDoInitCommands ARGS((char *argv[])) ;
static void	IInitSignals ARGS(()) ;
static void	IShowPrompt ARGS((char	*prompt)) ;
extern int	IDoCommandPipeline ARGS((char *inline, Loop *loop, int echo)) ;

int	ICatchInterrupt ARGS(());
int	ISegVioRescue ARGS(());
int	IBusErrorRescue ARGS(());
int	IFPErrorRescue ARGS(());
int	IPipeErrorRescue ARGS(());
int	IWindowChange ARGS(());

IStandardInit(argc, argv)
  int	*argc ;
  char	*argv[];
{
  XMode = initGraphics(argc, argv) ;
  IInit(2000, MAX_COMMAND_LEN, MAX_COMMAND_LEN+2000, argv[0]);

  if (XMode)
    buildGraphics() ;

  IDoInitCommands(argv) ;
  IInitSignals() ;
  IShowPrompt(prompt) ;
}

/***
 * IFunctionEnter(name)
 * Tell the interface about entering this function so that the function
 * name can be used in error messages.  Function names are kept on
 * a stack and popped off by ILeaveFunction()
 */
void IFunctionEnter(name)
char *name;
{
  if (function_names==NULL)
    function_names = TempStackNew();
  if (true(show_function_calls))
    fprintf(dout, "%*s%s\n", TempStackHeight(function_names)*2+2, ">>", name);
  TempStackPush(function_names, name);
}

void IFunctionExit(name)
char *name;
{
  char *last_name;
  last_name = TempStackPop(char *, function_names);
  if (strcmp(name, last_name))
    IErrorAbort("IFunctionExit: leaving \"%s\" but found \"%s\" on stack",
		name, last_name);
  if (true(show_function_calls))
    fprintf(dout, "%*s%s\n", TempStackHeight(function_names)*2+2, "<<", name);
}

char *IBaseName(str)
char *str;
{
  char *pos = str;
  if (str==NULL) return str;
  while ((pos=strchr(str, '/'))!=NULL) str = pos+1;
  return str;
}

static InitReadFile(va_alist)
va_dcl
{
  va_list args;
  char *fmt;
  char buf[200];
  struct stat statbuf;
  va_start(args);
  fmt = va_arg(args, char *);
  strcpy(buf, "read ");
  vsprintf(buf+5, fmt, args);
  if (stat(buf+5, &statbuf)==0) {
    if (!IDoCommandLine(buf)) {
      IReportError(stderr);
      IResetError();
      panic("IInit", "Error reading \"%s\"", buf+5);
    }
  }
}

static  void	IPrintVersion()
{
#ifndef OPERATING_SYSTEM_NAME
#define OPERATING_SYSTEM_NAME "UNIX"
#endif				/* OPERATING_SYSTEM_NAME */
  printf("  Xerion, V%s, %s %s\n",
	 version, last_change, OPERATING_SYSTEM_NAME);
  printf("  Copyright (C) 1990, 1991, 1992, 1993 University of Toronto, Toronto, Canada.\n") ;
  printf("  All rights reserved.\n") ;
  printf("  Written by: Drew van Camp, Tony Plate%s%s.\n\n", 
	 authors ? ", " : "", authors ? authors : "") ;
}

static void IInit(max_symbols, max_command_len, lex_space, name)
int max_symbols, max_command_len, lex_space;
char *name;
{
  char *space;
  char buf[200];

  /* some code depends upon all pointers being the same size - check it */
#ifndef lint
  /* this really upsets lint, and it's only for checking anyway */
  POI poi;
  assert(sizeof(poi.longint) == sizeof(poi));
  assert(sizeof(poi.pfpc) == sizeof(poi));
  assert(sizeof(poi.string) == sizeof(poi));
  assert(sizeof(poi.real_ptr) == sizeof(poi));
  assert(sizeof(poi.float_ptr) == sizeof(poi));
  assert(sizeof(poi.double_ptr) == sizeof(poi));
  assert(sizeof(poi.char_ptr) == sizeof(poi));
  assert(sizeof(poi.string_ptr) == sizeof(poi));
  assert(sizeof(poi.long_ptr) == sizeof(poi));
  assert(sizeof(poi.int_ptr) == sizeof(poi));
  assert(sizeof(poi.short_ptr) == sizeof(poi));
  assert(sizeof(poi.boolean_ptr) == sizeof(poi));
  assert(sizeof(poi.ptr) == sizeof(poi));
  assert(sizeof(poi.pptr) == sizeof(poi));
  assert(sizeof(poi.ppptr) == sizeof(poi));
#endif
  IPrintVersion() ;

  itf_func_table =
    INewTable((HASH_TABLE*)0, n_commands, n_commands, ID_INDEX);
  itf_func_context = INewTableContext(itf_func_table);
  itf_com_context = INewTableContext(itf_func_table);
  itf_help_table =
    INewTable((HASH_TABLE*)0, max_symbols, max_symbols/2, ID_INDEX);
  itf_help_context = INewTableContext(itf_help_table);
  lex_table = INewTable((HASH_TABLE*)0, 10, 10, 0);
  op_context = INewTableContext(lex_table);
  EnterCommands();	/* call the automatically constructed function */
  space = Malloc(char, lex_space);
  (void) LexInit(lex_space, space, lex_table);
  (void) LexMakeSyntax
    (1,		/* the context value for operators */
     0,			/* don't break words for quotes */
     " 	\n",		/* white space */
     "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz!@$^-+~?/#*",
     "&|;<>()[].*, =:{}%", /* operator syntax */
     "\"\"''``",	/* quote pairs */
     "0\0b\bn\nt\t",	/* escaped char values */
     "#\n",		/* comment pairs (no two-character comment sequences)*/
     object_syntax);	/* the place where the syntax table is stored */
  (void) LexMakeSyntax
    (1,			/* the context value for operators */
     0,			/* don't break words for quotes */
     " 	\n",		/* white space */
     "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz!@$^-+~?/&|;<>()[].*, =:{}%",
     "", /* operator syntax */
     "\"\"''``",	/* quote pairs */
     "0\0b\bn\nt\t",	/* escaped char values */
     "#\n",		/* comment pairs (no two-character comment sequences)*/
     simple_syntax);	/* the place where the syntax table is stored */
  (void) LexMakeSyntax
    (op_context,	/* the context value for operators */
     0,			/* don't break words for quotes */
     " 	\n",		/* white space */
#if 0
     "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz!$^*-+~[].?&()/<>",
     "%;=:, {}|<>@",	/* operator syntax */
#else
     "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz!$^*-+~[].?&/<>",
     "%;=:, {}()|<>@",	/* operator syntax */
#endif
     "\"\"''``",	/* quote pairs */
     "0\0b\bn\nt\t",	/* escaped char values */
     "#\n",		/* comment pairs (no two-character comment sequences) */
     command_syntax);	/* the place where the syntax table is stored */
  (void) ITableInsert(lex_table, ">>", op_context, NULL);
  (void) ITableInsert(lex_table, "<<", op_context, NULL);
  (void) ITableInsert(lex_table, "%<", op_context, NULL);
  (void) ITableInsert(lex_table, "|<", op_context, NULL);

  /* user generated bindings are entered from the following function */
  IBindInit(max_symbols);
  command_line1 = Malloc(char, max_command_len);
  command_line2 = Malloc(char, max_command_len);
  command_line_len = max_command_len;
  (void) ISetFeatureMask(ITF_SAVE | ITF_VISIBLE | ITF_SYSTEM);
  (void) IBindString("prompt", &prompt, P_OBJECT, 0, "");
  (void) IBindString("readpath", &readpath, P_OBJECT, 0, "");
  (void) IBindString("noclobber", &noclobber, P_OBJECT, 0, "");
  (void) IBindString("unset-error", &unset_error, P_OBJECT, 0, "");
  (void) IBindString("alias-enabled", &alias_enabled, P_OBJECT, 0, "");
  (void) IBindString("redirection", &redirection, P_OBJECT, 0, "");
  (void) IBindString("automore", &automore, P_OBJECT, 0, "");
  (void) IBindString("help-automore", &help_automore, P_OBJECT, 0, "");
  (void) IBindString("plus-minus-options", &plus_minus_options, P_OBJECT, 0, "");
  (void) IBindString("echo", &echo, P_OBJECT, 0, "");
  (void) IBindString("show-function-calls", &show_function_calls, P_OBJECT, 0, "");
  (void) IBindString("read-echo", &read_echo, P_OBJECT, 0, "");
  (void) IBindString("options-echo", &options_echo, P_OBJECT, 0, "");
  (void) IBindString("use-default-options", &use_default_options, P_OBJECT, 0, "");
  (void) IBindString("printstatus", &print_status, P_OBJECT, 0, "");
  (void) IBindLong("status", &status, P_OBJECT, 0, "");
  ISetValue("echo", "false");
  ISetValue("printstatus", "false");
  ISetValue("plus-minus-options", "false");
  /* ISetValue("read-echo", "false"); */
  ISetValue("options-echo", "true");
  ISetValue("use-default-options", "true");
  ISetValue("noclobber", "true");
  ISetValue("unset-error", "true");
  ISetValue("alias-enabled", "false");
  ISetValue("redirection", "true");
  auto_more_enabled = (isatty(1) && isatty(0));
  ISetValue("automore", auto_more_enabled ? "false" : "false");
  ISetValue("help_automore", "true") ;
  myname = IBaseName(name);
  sprintf(buf, "%s-> ", myname);
  ISetValue("prompt", buf);
  ISetValue("readpath", ".");
  INewExport("HOME");
  INewExport("TERM");
  INewExport("PAGER");
  (void) ISetFeatureMask(ITF_SAVE | ITF_VISIBLE /* | ITF_ORPHANABLE */);
  mypid = getpid();
  just_do_defaults = EVERYTHING;
  quit_flag = NORMAL;
}

static void	IDoInitCommands(argv) 
  char	*argv[] ;
{
  BindObjects();
  DoAutoInit();

  if (getenv("XERIONDIR")) {
    InitReadFile("%s/config/xerionrc", getenv("XERIONDIR"));
    if (strcmp(myname, "xerion"))
      InitReadFile("%s/config/%src", getenv("XERIONDIR"), myname);
  }
  InitReadFile("%s/.xerion", getenv("HOME"));
  InitReadFile(".xerion", getenv("HOME"));
  InitReadFile("%s/.%src", getenv("HOME"), myname);
  InitReadFile(".%src", myname);

  if (true(redirection))
    ISetValue("redirection", "falsie");	/* set to false for reading defaults */
  					/* bad spelling to detect later */

  if (strcmp(IGetValue("redirection", NULL), "falsie")==0)
    ISetValue("redirection", "true");
  IInitStream();
  just_do_defaults = EVERYTHING;

  ++argv ;
  if (*argv)
    printf("Executing command line arguments...\n");
  if (!IDoArgvCommands(argv))
    panic("IInit", "Error in command line arguments -- bye bye!", NULL);
}

static void	IInitSignals() 
{
  (void) signal(SIGINT, ICatchInterrupt);
  (void) signal(SIGPIPE, IPipeErrorRescue);
#ifndef DEBUG
  (void) signal(SIGSEGV, ISegVioRescue);
  (void) signal(SIGBUS, IBusErrorRescue);
  (void) signal(SIGFPE, IFPErrorRescue);
#endif
/*** don't need this - don't need to do anything here for winch to work
#ifdef SIGWINCH
  (void) signal(SIGWINCH, IWindowChange);
#endif
*/
}


int true(s)
char *s;
{
  if (   (s == NULL)
      || (*s == '\0')
      || ((*s == '0' || *s == '-') && *(s+1) == '\0')
      || ((*s == 'f' || *s == 'F') && (*(s+1) == '\0' || !strcmp(s+1, "alse")))
      || (!strcmp(s, "FALSE")))
    return 0 ;
  else
    return 1 ;
}
/*
 * functions to assist parsing
 */	

char *IWrongNargs(command, n)
char *command;
char *n;
{
  IError("%s: Wrong number of arguments (expected %s)", command, n);
  return itf_error;
}

char *ITooManyArgs(command, n)
char *command;
char *n;
{
  IError("%s: Too many arguments (expected %s)", command, n);
  return itf_error;
}

char *IArgError(arg, error)
char *arg, *error;
{
  IError("%s argument: %s", error, arg);
  return itf_error;
}

char *IFormatUsage(intro, command, usage)
char *intro, *command, *usage;
{
  char *buf = itf_tmp_buf;
  char *com = command;
  char *p = intro;
  char *start_line = buf;
  while (*p) *(buf++) = *(p++);
  do {
    while (*com) *(buf++) = *(com++);
    *(buf++) = ' ';
    while (*usage && *usage!='|') {
      if (*usage=='%' && (usage[1]=='s' || usage[1]=='p')) {
	if (usage[1]=='p') {
	  if (true(plus_minus_options)) *(buf++) = '+';
	  else *(buf++) = '-';
	} else if (usage[1]=='s') {
	  if (true(plus_minus_options)) buf = strcpyend(buf, "+/-");
	  else buf = strcpyend(buf, "-[no]");
	} else if (usage[1]=='y') {
	  if (true(plus_minus_options)) buf = strcpyend(buf, "+");
	  else buf = strcpyend(buf, "-");
	} else if (usage[1]=='n') {
	  if (true(plus_minus_options)) buf = strcpyend(buf, "-");
	  else buf = strcpyend(buf, "-no");
	} else if (usage[1]=='%') {
	  *(buf++) = '%';
	} else
	  *(buf++) = *(usage--); /* -- to make up for the +2 on next line */
	usage += 2;
      } else
	*(buf++) = *(usage++);
      if (buf[-1]==' ' && (buf - start_line > 60)) {
	strcpy(buf, "\n        ");
	start_line = buf+1;
	buf += strlen(buf);
      }
    }
    if (*usage=='|') {
      com = command;
      p = "\n    ";
      start_line = buf;
      while (*p) *(buf++) = *(p++);
      usage++;
    }
  } while (*usage);
  *buf = 0;
  return itf_tmp_buf;
}

char *IPrintUsage(command, usage)
char *command, *usage;
{
  return IFormatUsage("usage:\n    ", command, usage);
}

char *IInvalidNumber(name)
char *name;
{
  IError("Invalid number: \"%s\"", name);
  return itf_error;
}

void IExpectedToken(what, this, after)
char *what, *this, *after;
{
  char *quote;
  quote = (*what=='<' ? "" : "\"");
  if (this!=NULL && after!=NULL)
    IError("Expected %s%s%s after \"%s\", but found \"%s\"",
	   quote, what, quote, after, this);
  else if (this==NULL && after!=NULL)
    IError("Expected %s%s%s after \"%s\", but found nothing",
	   quote, what, quote, after);
  else if (this!=NULL && after==NULL)
    IError("Expected %s%s%s, but found \"%s\"", quote, what, quote, this);
  else if (this==NULL && after==NULL)
    IError("Expected %s%s%s, but found nothing", quote, what, quote);
}

#ifdef OBSELETE
int IExpectInteger(a)
char *a;
{
  if (a==NULL) a = "(null)";
  else if (IIsInteger(a)) return 1;
  IExpectedToken("<integer>", a, NULL);
  return 0;
}

int IExpectString(a)
char *a;
{
  if (a==NULL) a = "(null)";
  else if (LexQuotedTok(a)) return 1;
  IExpectedToken("<string>", a, NULL);
  return 0;
}

int IExpectVarName(a)
char *a;
{
  if (a==NULL) a = "(null)";
  else if (IIsLegalVarName(a)) return 1;
  IExpectedToken("<varname>", a, NULL);
  return 0;
}

int IExpectNumber(a)
char *a;
{
  if (a==NULL) a = "(null)";
  else if (IIsNumber(a)) return 1;
  IExpectedToken("<real number>", a, NULL);
  return 0;
}
#endif				/* OBSELETE */

int IExpectInteger(this, after)
char *this, *after;
{
  if (this!=NULL && IIsInteger(this)) return 1;
  IExpectedToken("<integer>", this, after);
  return 0;
}

int IExpectString(this, after)
char *this, *after;
{
  if (this!=NULL && LexQuotedTok(this)) return 1;
  IExpectedToken("<string>", this, after);
  return 0;
}

int IExpectVarName(this, after)
char *this, *after;
{
  if (this!=NULL && IIsLegalVarName(this)) return 1;
  IExpectedToken("<varname>", this, after);
  return 0;
}

int IExpectNumber(this, after)
char *this, *after;
{
  if (this!=NULL && IIsNumber(this)) return 1;
  IExpectedToken("<real number>", this, after);
  return 0;
}

void IExpected(s)
char *s;
{
  if (!itf_errno) itf_errno = COMMAND_ERROR;
  IError("Expected %s", s);
}

/***
 * should be called like IError("error is %d", x)
 * This checks for overflow in the itf_error buffer - if
 * less than ERROR_BUF_SAFE chars remain, it doesn't put any more in
 */
/*VARARGS0*/
void IError(va_alist)
va_dcl
{
  va_list ap;
  va_start(ap);
  IErrorV(&ap);
  va_end(ap);
}

void Error(va_alist)
va_dcl
{
  va_list ap;
  va_start(ap);
  IErrorV(&ap);
  va_end(ap);
}

ErrorV(ap)
va_list *ap;
{
  IErrorV(ap);
}

void IErrorAbort(va_alist)
va_dcl
{
  va_list ap;
  va_start(ap);
  IErrorV(&ap);
  va_end(ap);
  IAbort();
}

/***
 * ErrorAbort is the same as IErrorAbort
 */
void ErrorAbort(va_alist)
va_dcl
{
  va_list ap;
  va_start(ap);
  IErrorV(&ap);
  va_end(ap);
  IAbort();
}

/***
 * should be called like IErrorV(&ap)
 * This checks for overflow in the itf_error buffer - if
 * less than ERROR_BUF_SAFE chars remain, it doesn't put any more in
 */
void IErrorV(ap)
va_list *ap;
{
  char *fmt, *str;
  TBL_ITEM *item = ITableLookupById(itf_func_table, command_id);
  char *last_command;
  last_command = item==NULL?"unknown":item->name;
  str = itf_error;
  if (itf_errno<=0 || itf_errno>=TOP_ITF_ERRNO)
    itf_errno = COMMAND_ERROR;
  if (*itf_error) {
    str += strlen(itf_error);
    *str++ = '\n';
    /* strcat(str, "\nThis led to error: "); */
    /* str += strlen(str); */
  }
  fmt = va_arg(*ap, char*);
  if (str-itf_error > ERROR_BUF_SIZE-ERROR_BUF_SAFE) {
    fprintf(stderr, "%s\n%s: (in command \"%s\"):",
	    "IError: out of buffer space for saving up error messages!",
	    itf_error_str[itf_errno], last_command);
    vfprintf(stderr, fmt, *ap);
    fputc('\n', stderr);
  } else {
    strcpy(str, itf_error_str[itf_errno]);
    str += strlen(str);
    if (itf_errno==COMMAND_ERROR) {
      sprintf(str, "%s: ", last_command);
      str += strlen(str);
    }
    if (function_names && function_names->top) {
      sprintf(str, "%s: ", TempStackTop(char *, function_names));
      str += strlen(str);
    }
    vsprintf(str, fmt, *ap);
  }
  if (!itf_errno)
    itf_errno = COMMAND_ERROR;
}

/***
 * should be called as:
 * IOutOfMemory("function-name", NULL)
 * or
 * IOutOfMemory("function-name", "format", arg1, arg2, ...)
 * VARARGS0
 */
char *IOutOfMemory(va_alist)
va_dcl
{
  va_list ap;
  char *fmt, *name, *p;
  va_start(ap);
  name = va_arg(ap, char*);
  fmt = va_arg(ap, char*);
  itf_errno = OUT_OF_MEMORY;
  IError("\"%s\" ", name);
  if (fmt) {
    p = itf_error + strlen(itf_error);
    vsprintf(p, fmt, ap);
  }
  va_end(ap);
  return itf_error;
}

int IHaveCulprit()
{
  return !(itf_error[0]==0);
}

/***
 * Prints out a help line doing the following
 * - indent 4 spaces unless the line is all upper case
 * - change '%p' to '-'
 * - change '%s' to '+/-' or '-[no]' depending on value of 'plus_minus_options'
 */
static void PrintHelpLine(out, line)
FILE *out;
char *line;
{
  int all_caps, i;
  all_caps = 1; /* a "header" line is all capitals, and is not indented */
  if (line[0]!=0) {
    for (i=0;line[i];i++)
      if (!isupper(line[i])&&!isspace(line[i])) {all_caps=0;break;}
    if (!all_caps) fputs("    ", out);
    else if (line[0]) fputc('\n', out); /* extra nl before heading */
    while (*line) {
      if (*line!='%') putc(*(line++), out);
      else {
	if (line[1]=='s') {
	  if (true(plus_minus_options)) fputs("+/-", out);
	  else fputs("-[no]", out);
	} else if (line[1]=='p') {
	  if (true(plus_minus_options)) fputc('+', out);
	  else fputc('-', out);
	} else if (line[1]=='y') {
	  if (true(plus_minus_options)) fputc('+', out);
	  else fputc('-', out);
	} else if (line[1]=='n') {
	  if (true(plus_minus_options)) fputc('-', out);
	  else fputs("-no", out);
	} else if (line[1]=='%') {
	  fputc('%', out);
	} else
	  putc(*(line--), out); /* -- to make up for +2 on next line */
	line+=2;
      }
    }
  }
  fputc('\n', out);
}

void IHelp(va_alist)
va_dcl
{
  va_list ap;
  int help_type;
  char *name, *usage, *synopsis, *long_help, **extraHelp;
  va_start(ap);
  help_type	= va_arg(ap, int);
  name		= va_arg(ap, char*);
  usage		= va_arg(ap, char*);
  synopsis	= va_arg(ap, char*);
  long_help	= va_arg(ap, char*);
  if (help_type==SHORT_HELP) {
    help_string = synopsis;
  } else {
    struct TBL_ITEM *item;
    fprintf(dout, "NAME\n    %s - %s\n\n", name, synopsis);
    if (usage!=NULL && *usage!=0) {
      fputs(IFormatUsage("USAGE\n    ", name, usage), dout);
      fputc('\n', dout);
    }
    /* try to find the command in the args stuff, and print out usage */
    /* arg_table may not be set yet, so don't call if not */
    if (arg_table!=NULL)
      item = ITableLookup(arg_table, name, arg_table_context);
    else
      item = NULL;
    if (item!=NULL) {
      fputs("USAGE\n", dout);
      PrintArgUsage((struct ARG_DESC*)item->data.ptr, dout, 0);
      fputc('\n', dout);
    }
    
    if (long_help!=NULL) {
      fprintf(dout, "\nDESCRIPTION\n");
      while (long_help!=NULL) {
	PrintHelpLine(dout, long_help);
	long_help	= va_arg(ap, char*);
      }
      fputc('\n', dout);
    }
    if (item!=NULL) {
      fputs("OPTIONS AND ARGUMENTS\n", dout);
      PrintArgHelp((struct ARG_DESC*)item->data.ptr, dout, 0);
    }
    fprintf(dout, "\n");
    extraHelp = IGetHelpStrings(name) ;
    if (extraHelp != NULL) {
      fprintf(dout, "\nEXTENSIONS\n");
      while (*extraHelp != NULL)
	PrintHelpLine(dout, *extraHelp++);
      fputc('\n', dout);
    }
  }
}

IResetError()
{
  itf_errno = 0;
  itf_error[0] = 0;
  return itf_errno;
}

/***
 * Swaps the abort env with env
 *
 * Should be used as:
 * jmp_buf env;
 * void	   *old_env;
 * ...
 * if (setjmp(env))
 *   handle jump
 * else
 *   old_env = ISetAbortEnv(env);
 * ...
 * ISetAbortEnv(old_env);
 * return;
 */
void *ISetAbortEnv(env)
void *env;
{
  void *old;
  old = itf_abort_env;
  itf_abort_env = env;
  return old;
}

/***
 * Abort to the last place that set 'itf_abort_env'
 */
void IAbort()
{
  longjmp(itf_abort_env, 1);
}

/***
 * Abort to the last place that set 'itf_abort_env'
 */
void Abort()
{
  longjmp(itf_abort_env, 1);
}

/***
 * Catch an Interrupt
 */
int	ICatchInterrupt()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGINT, ICatchInterrupt);
#endif
  if (!itf_errno) {
    itf_errno = INTERRUPT;
    caught_signal = SIGINT;
    fprintf(stderr, "\n");
  }
  longjmp(itf_abort_env, 1);
}

static int seg_vio_count=0;
static int bus_error_count=0;
static int fp_error_count=0;
/***
 * rescue from a segmentation violation
 */
int	ISegVioRescue()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGSEGV, ISegVioRescue);
#endif
  fprintf(stderr, "Segmentation violation caught - aborted\n");
  seg_vio_count++;
  itf_errno = ERROR_SIGNAL;
  caught_signal = SIGSEGV;
  IErrorAbort("");
}

int	IBusErrorRescue()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGBUS, IBusErrorRescue);
#endif
  fprintf(stderr, "Bus error caught - aborted\n");
  bus_error_count++;
  itf_errno = ERROR_SIGNAL;
  caught_signal = SIGBUS;
  IErrorAbort("");
}

int	IFPErrorRescue()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGFPE, IFPErrorRescue);
#endif
  fprintf(stderr, "Floating point error caught - aborted\n");
  fp_error_count++;
  itf_errno = ERROR_SIGNAL;
  caught_signal = SIGFPE;
  IErrorAbort("");
}

int	IPipeErrorRescue()
{
#ifdef SYSV_SIGNALS
  (void) signal(SIGPIPE, IPipeErrorRescue);
#endif
  if (errno!=EPIPE) {
    /* fprintf(stderr, "Pipe error caught but errno not set - continuing\n"); */
    return;
  }
  /* fprintf(stderr, "Pipe error caught - aborted\n"); */
  itf_errno = ERROR_SIGNAL;
  caught_signal = SIGPIPE;
  IErrorAbort("");
}

int	IWindowChange()
{
  fprintf(stderr, "Got a window change signal, but don't know what to do with it!\n");
  caught_signal = SIGWINCH;
}

IReportError(s)
FILE *s;
{
  if (itf_errno<0 || itf_errno>=TOP_ITF_ERRNO)
    warn("IReportError", "errno %d out of range", itf_errno);
  if (true(print_status)) {
    fprintf(dout, "Status: %d:", itf_errno);
    if (itf_errno==ERROR_REPORTED) printf("\n");
  }
  if (itf_errno!=ERROR_REPORTED && itf_error[0]!='\0')
    fprintf(s, "%s\n", itf_error);
}

static int *line_numbers_ok_p = NULL;
IMuckUpLineNumbers()
{
  if (line_numbers_ok_p!=NULL)
    *line_numbers_ok_p = 0;
}

static char* GetInputLine(line, line_len, in, line_no)
char *line;
int line_len;
FILE *in;
int *line_no;
{
  char *p = line;
  char *end = line+line_len;
  char *ret;
  /* int cont = 1; */
  while (1) {
    line_len = end-p;
    ret = fgets(p, line_len, in);
    if (line_no!=NULL) (*line_no)++;
    if (ret==NULL) return (p==line?NULL:line);
    while (*p) p++;
    if ((p-1)>line && p[-2]=='\\') {
      p[-2] = p[-1];
      p[-1] = '\0';
      p--;
      /* cont = 1; */
    } else
      break;
  }
  return ret;
}
/*********************************************************************
 *	Name:		IDoSingleCommand
 *	Description:	does a single command (pipe, read from stream)
 *	Parameters:
 *	  FILE	*this_instream, *this_outstream - input and output
 *			streams
 *	Return Value:
 *	  int	IDoSingleCommand - 1 on success
 *********************************************************************/
int	IDoSingleCommand(this_instream, this_outstream)
  FILE	*this_instream, *this_outstream;
{
  static Loop	*loop ;
  int	idx, tokc, numCommands;
  char	**tokv, **command;
  char	*line;

  if (loop == NULL)
    loop = createLoop() ;

  IResetError();

  dout = this_outstream;
  din  = this_instream;

  if (GetInputLine(command_line1, command_line_len, din, NULL)==NULL) {
    fputs("\n", dout) ;
    exit(0) ;
  }

  IDoCommandPipeline(command_line1, loop, true(echo)) ;

  if (quit_flag == QUIT || quit_flag == EXIT) {
    if (quit_flag == EXIT)
      quit_flag = NORMAL ; 
    return 0 ;

  } else {
    if (itf_errno || true(print_status))
      IReportError(stderr) ;

    IShowPrompt(prompt) ;
    return 1 ;
  }
}
/********************************************************************/

static int	nonGraphicsLoop(this_instream, this_outstream)
  FILE	*this_instream, *this_outstream;
{
  while(IDoSingleCommand(this_instream, this_outstream) != 0)
    ;
  return 0 ;
}

static void	IShowPrompt(prompt)
  char	*prompt ;
{
  if (*prompt) {
    fputs(prompt, dout);
    fflush(dout);
  }
}

int	ICommandLoop(this_instream, this_outstream, new_prompt)
  FILE	*this_instream, *this_outstream;
  char	*new_prompt;
{
  jmp_buf	env;
  int		*old_env;

  if (quit_flag == QUIT) 	/* 'quit' means QUIT! */
    return 1;
  if (new_prompt != NULL) 
    (void) ISetValue("prompt", new_prompt);

  if (setjmp(env)) {
    IReportError(stderr);
  } else {
    old_env = ISetAbortEnv(env);
  }  

  if (XMode != 0) {
    graphicsLoop(this_instream, this_outstream) ;
  } else {
    nonGraphicsLoop(this_instream, this_outstream) ;
  }

  (void) ISetAbortEnv(old_env);

  return 0 ;
}

int IDoArgvCommands(argv)
char *argv[];
{
  char *p=command_line1;
  Loop *loop;

  for (*p = '\0' ; *argv ; *(p++) = ' ') {
    strcpy(p, *argv);
    p += strlen(*argv) ;
    ++argv ;
  }
  *p = '\0' ;

  loop = createLoop() ;
  IDoCommandPipeline(command_line1, loop, FALSE) ;
  destroyLoop(loop) ;

  if (itf_errno) {
    IReportError(stderr);
    IResetError();
    return 0;
  } else
    return 1;
}

int IDoCommandLine(line)
char *line;
{
  Loop 		*loop ;
  Boolean	error ;

  if (line == NULL) 
    return 1 ;

  loop	= createLoop() ;
  error	= IDoCommandPipeline(line, loop, FALSE) ;
  destroyLoop(loop) ;

  return error ? 0 : 1 ;
}

#ifdef OBSELETE
#define MAXARGS 10
int IDoCommandList(va_alist)
va_dcl
{
  char *tokv[MAXARGS];
  int tokc=0;
  va_list pvar;
  va_start(pvar);
  while ((tokv[tokc++]=va_arg(pvar, char*))!=NULL)
    if (tokc>=MAXARGS) panic("IDoCommandList", "too many arguments");
  tokc--;
  va_end(pvar);
  return IDoCommand(tokc, tokv);
}
#endif				/* OBSELETE */

char *ISwapWords(p)
char *p;
{
  char buf[100];
  if (!strchr(p, '-')) return p;
  strcpy(buf, strchr(p, '-')+1);
  *strchr(p, '-') = 0;
  strcat(buf, "-");
  strcat(buf, p);
  strcpy(p, buf);
  return buf;
}

/*
 * Finds a command - the command might have "syllables" in
 * a different order to the given name, this will rotate
 * the syllables until it finds the command or will return NULL
 */
TBL_ITEM *IFindCommand(name, id)
char *name;
int *id;
{
  int dash_count = 0;
  char *p;
  TBL_ITEM *function;
  function=ITableLookup(itf_func_table, name, itf_com_context);
  if (function==NULL) {
    for (p=name;*p;p++) if (*p=='-') dash_count++;
    if (dash_count>0)
      for (;dash_count>=0;dash_count--) {
	ISwapWords(name);
	if (function==NULL)
	  function=ITableLookup(itf_func_table, name, itf_com_context);
      }
#if 1
    /* if still no luck, look for a script */
    if (function == NULL) {
      /* try to open the file and see if it is a xerion script */
      if (IFindFileInPath(name, readpath, 0)) {
	FILE	*fp ;

	if (fp = IOpenFileOrAbort(name, "r", NULL)) {
	  char	buffer[BUFSIZ] ;
	  fgets(buffer, BUFSIZ, fp) ;

	  ICloseFile(fp, NULL) ;

	  /* IT IS A SCRIPT!!! */
	  if (strncmp(buffer, "#!xerion", 8) == 0)
	    function=ITableLookup(itf_func_table, "exec", itf_com_context);
	}
      }
    }
#endif

  }
  if (function!=NULL && id!=NULL) *id = function->id;
  return function;
}


/********************************************************************/
static char	*nextSplit(line) 
  String	line ;
{
  int		in_quote = 0 ;
  Boolean	escape_pending = FALSE ;

  if (strchr(line, ';') == NULL)	/* most times return here */
    return NULL ;

  while (*line && *line != '#') {
    Boolean	escaped = escape_pending ;
    escape_pending = FALSE ;

    if ((*line == ';' || *line == '#') && !(escaped || in_quote))
      break ;

    if (!escaped) {
      switch (*line) {
      case '\\':		/* escape character - set escape pending */
	escape_pending = TRUE ;
	break ;

      case '\'':		/* single or double quote character */
      case '"':	
	if (!in_quote) 
	  in_quote = *line ;
	else if (*line == in_quote) 
	  in_quote = 0 ;
	break ;
	    
      default:			/* everything else */
	break ;
      }
    } 
    ++line ;
  }
  
  return *line == ';' ? line : NULL ;
}
/*********************************************************************
 * Executes a pipeline of commands (commands separated by ";")
 ********************************************************************/
int		IDoCommandPipeline(inline, loop, echo)
  char		*inline ;
  Loop		*loop ;
  int		echo ;
{
  int		tokc ;
  char		**tokv, *command, *split, oldc ;
  char		pipeline[MAX_COMMAND_LEN], buffer[MAX_COMMAND_LEN] ;
  LEX_STACK	stack ;

  if (true(alias_enabled)) {
    aliasSubstitute(inline, pipeline, MAX_COMMAND_LEN) ;
    command = pipeline ;
  } else {
    command = inline ;
  }

  do {
    split = nextSplit(command) ;
    if (split) {
      oldc = *split ;
      *split = '\0' ;
    }

    if (loop && (inLoop(loop) || isLoop(loop, command))) {
      addLoopCommand(loop, command, echo) ;

    } else if (command = VarSubstitute(command,buffer,MAX_COMMAND_LEN,echo)) {
      tokv = LexAnalysePush(command_syntax, command, &tokc, &stack) ;
      if (tokc && tokv && ATTEMPT_COMMAND(tokv[0])
	  && commandSubstitute(tokc, tokv, echo)) {
	if (echo) {
	  LexObjReconstruct(defaultLex, buffer, MAX_COMMAND_LEN, tokc, tokv) ;
	  fputc('+',  stderr);
	  fputs(buffer,  stderr);
	  fputc('\n', stderr);
	  fflush(stderr);
	}

	if (pre_command_function)
	  pre_command_function();
	IDoCommand(tokc, tokv);
	if (post_command_function)
	  post_command_function();
      }
      LexAnalysePop(&stack);
    }
    if (itf_errno || quit_flag != NORMAL) {
      if (loop) resetLoop(loop) ;
      break ;
    }
    if (split) {
      *split = oldc ;
      command = split + 1 ;
    }
  } while (split) ;

  return itf_errno ;
}
/********************************************************************/

/*
 * Executes a command given a tokv vector
 */
int IDoCommand(tokc, tokv)
int tokc;
char *tokv[];
{
  TBL_ITEM *function;
  char *p, **pp, **optv=NULL, *options=NULL;
  int optc=0;
  jmp_buf env;
  int *old_env;
  SignalHandler old_signal_handler;
  int old_command_id = command_id;
  struct STACK *old_function_names = function_names;
  struct STREAM *old_instream, *old_outstream;
  struct STREAM *old_command_stdin = command_stdin;
  struct STREAM *old_command_stdout = command_stdout;
  struct TEMP_ALLOC temp_alloc_rec;
  /* remember whether to redirect - value might change as result of command */
  int redirect = true(redirection);
  Boolean	success = TRUE;
  if (tokc<1) return 0;
  errno = 0;
  /* pre-process just the command name converting underscores to dashes */
  for(pp=tokv;pp==tokv;pp++)
    for (p= *pp;*p;p++) if (*p=='_') *p='-';
  function = IFindCommand(tokv[0], &command_id);
  command_stdin = current_instream;	/* don't have to remember these coz */
  command_stdout = current_outstream;	/* they're set before each command */
  function_names = NULL; /* stack is allocated if needed */
  if (function!=NULL) {
    if (true(use_default_options)) {
      strcpy(itf_tmp_buf, function->name);
      strcat(itf_tmp_buf, "-options");
      if (IIsObjectName(itf_tmp_buf)) {
	options = IGetValue(itf_tmp_buf, NULL);
	if (itf_errno || options==NULL) {
	  itf_errno = COMMAND_ERROR;
	  IError("failed to get default options");
	  goto finish;
	}
	/* fprintf(dout, "Options are: \"%s\"\n", options); */
	optv = LexAnalyseAdd(command_syntax, options, &optc);
	/*
	for (i=0;i<tokc;i++)
	  fprintf(dout, "tokv[%d] 0x%X = \"%s\"\n", i, tokv+i, tokv[i]);
	for (i=0;i<optc;i++)
	  fprintf(dout, "optv[%d] 0x%X = \"%s\"\n", i, optv+i, optv[i]);
	  */
      }
    }
    IBeginTempAlloc(&temp_alloc_rec);
    if (optc) { /* insert the extra options after the command */
      int i;
      char **tokv2;
      tokv2 = ITempCalloc(char*, tokc+optc+1);
      tokv2[0] = tokv[0];
      for (i=0;i<optc;i++) tokv2[i+1] = optv[i];
      for (i=1;i<tokc;i++) tokv2[optc+i] = tokv[i];
      tokc += optc;
      tokv = tokv2;
    }
    if (/* true(echo) || */ (optc&&true(options_echo))) {
      char *p = IComposeTokvLine(tokv, 1);
      if (p!=NULL) {
	fputs(p, dout);
	fputc('\n', dout);
	fflush(dout);
      }
    }
    if (!redirect || IDoRedirection(&tokc, tokv, &old_instream, &old_outstream)) {
      if (setjmp(env)==0) {
	old_signal_handler = (SignalHandler)signal(SIGINT, 
						   (SignalHandler)ICatchInterrupt);
	old_env = ISetAbortEnv(env);
	success = (function->data.pfi)(tokc, tokv);	/* call the command */
      } 
      (void) ISetAbortEnv(old_env);
      (void) signal(SIGINT, old_signal_handler);
      if (redirect) IEndRedirection(old_instream, old_outstream);
      
    }
    IFreeTempAllocs(&temp_alloc_rec);
  } else {
    itf_errno = UNKNOWN_COMMAND;
    IError("%s", tokv[0]);
  }
 finish:
  function_names = old_function_names;
  command_stdin = old_command_stdin;
  command_stdout = old_command_stdout;
  command_id = old_command_id;
  if (!success && itf_errno == 0)
    itf_errno = ERROR_REPORTED;
  return itf_errno;
}

char *IComposeTokvLine(tokv, use_quotes)
char *tokv[];
int use_quotes;
{
  char *p = command_line1;
  char *q;
  int on_first_token=1;
  while (*tokv) {
    if (LexSpBefore(*tokv) && !on_first_token) *p++ = ' ';
    if (use_quotes && LexQuotedTok(*tokv)) *p++ = LexQuotedTok(*tokv);
    for (q = *tokv;*q;q++) *p++ = *q;
    if (use_quotes && LexQuotedTok(*tokv)) *p++ = LexQuotedTok(*tokv);
    if (p-command_line1 > command_line_len) {
      itf_errno = LEX_ERROR;
      IError("Command too long");
      return NULL;
    }
    on_first_token=0;
    tokv++;
  }
  *p = 0;
  return command_line1;
}

int IEnterCommand(name, function)
char *name;
PFI function;
{
  TBL_ITEM *item;
  char *p=name;
  for (;*p;p++) if (*p=='_') *p='-';
  item = ITableInsert(itf_func_table, name, itf_com_context, function);
  if (item == NULL && hash_errno==HT_DUPLICATE) {
    /* fprintf(dout, "IEnterCommand: entering new command \"%s\"", name); */
    item = ITableLookup(itf_func_table, name, itf_com_context);
    item->data.pfi = function;
  }
  if (item == NULL && hash_errno==HT_OVERFLOW)
    panic("IEnterCommand", "table overflow on: %s", name);
  return 0;
}

int IEnterFunction(name, function)
char *name;
PFI function;
{
  TBL_ITEM *item;
  char *p=name;
  for (;*p;p++) if (*p=='_') *p='-';
  item = ITableInsert(itf_func_table, name, itf_func_context, function);
  if (item == NULL && hash_errno==HT_DUPLICATE)
    panic("IEnterFunction", "function is already in table: %s", name);
  if (item == NULL && hash_errno==HT_OVERFLOW)
    panic("IEnterFunction", "table overflow on: %s", name);
  return 0;
}

PFI IGetFunction(name)
char *name;
{
  TBL_ITEM *item;
  char *p=name;
  for (;*p;p++) if (*p=='_') *p='-';
  item = ITableLookup(itf_func_table, name, itf_func_context);
  if (item!=NULL)
    return item->data.pfi;
  else
    return NULL;
}

int	IEnterHelpStrings(in_name, string)
  char	*in_name;
  char	**string;
{
  TBL_ITEM	*item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);

  item = ITableInsert(itf_help_table, name, itf_help_context, string);
  if (item == NULL && hash_errno==HT_DUPLICATE) {
    item = ITableLookup(itf_help_table, name, itf_help_context);
    item->data.ptr = (char *)string;
  }
  if (item == NULL && hash_errno==HT_OVERFLOW)
    panic("IEnterHelpString", "table overflow on: %s", name);
  return 0;
}

char	**IGetHelpStrings(in_name)
  char	*in_name;
{
  TBL_ITEM	*item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);

  item = ITableLookup(itf_help_table, name, itf_help_context);
  if (item)
    return (char **)item->data.ptr;
  else
    return NULL;
}

command_sh(tokc, tokv)
int tokc;
char *tokv[];
{
  int echo_command = 0;
  int	returnStatus ;
  int	fildes ;
  char *command;
  int r;
  if (GiveHelp(tokc)) {
    IUsage("[-echo] <command...>");
    ISynopsis("run a shell command");
    IHelp
      (IHelpArgs,
       "The shell  command  is  executed as typed, except   that any escaped",
       "characters are converted to the character they stand for, e.g. '\\n'",
       "is converted to the newline character.",
       "",
       "Quote characters are retained in the line sent to the  shell, except",
       "if the interface sees only one token in the  shell command, in which",
       "case the   quotes are  omitted.   This  allows the quoting  of shell",
       "commands in   order to get   the shell  and   not the interface   to",
       "interpret redirection and other meta-characters.",
       "NOTE ABOUT REDIRECTION",
       "Redirection does  not work  smoothly with  the   'sh' command.  More",
       "specifically, any redirection  that  is interpreted by  the  command",
       "interface  will not  have  any effect on  the  output  of the  shell",
       "command.  Thus a command like \"sh cat foo > tmp\" will result  in the",
       "contents of \"foo\" being displayed on the terminal and in \"tmp\" being",
       "an empty file.  To get shell commands redirected, enclose  the shell",
       "command in quotes, e.g., \"sh  'cat foo  >  tmp'\"  This way the shell",
       "will do the redirection.",
       "OPTIONS",
       "-echo : echo the command just before sending it to the  shell.  What",
       "  is printed is  exactly what is  sent to the  shell (except for the",
       "  newline that is added for printing.",
       "SIGNALS",
       "The process  is given a different  parent process group, if the last",
       "character in the last argument is a '&'.  This will prevent  it from",
       "receiving  keyboard interrupt   signals,  which could  result in the",
       "unintended killing of a backgrounded process.",
       "",
       NULL);
    return 1;
  }
  tokv++;
  tokc--;
  if (*tokv && !strcmp("-echo", *tokv)) {
    echo_command=1;
    tokv++;
    tokc--;
  }
  if (tokc<1) {
    IError("no shell command specified");
    return 0;
  }
  command = IComposeTokvLine(tokv, tokc>1);
  if (command==NULL) return 0;
  if (echo_command) fprintf(dout, "%s\n", command);
  if (strlen(tokv[tokc-1])>0 && tokv[tokc-1][strlen(tokv[tokc-1])-1]=='&') {
    /***
     * this fork is here because we want to detach the child from our
     * pgroup so that backgrounded processes do not inherit signals
     * REALLY UGLY!
     */
    r = fork();
    if (r<0) IErrorAbort("fork failed");

    /* make stdout filedes same as dout */
    if (fileno(dout) != 1) {
      fildes = dup(1) ;
      close(1) ;
      dup(fileno(dout)) ;
    }

    if (r==0) {
      setpgrp(getpid(), getpid());
      system(command);
      exit(1);
    }
    returnStatus = 1 ;
  } else {
    /* make stdout filedes same as dout */
    if (fileno(dout) != 1) {
      fildes = dup(1) ;
      close(1) ;
      dup(fileno(dout)) ;
    }

    returnStatus = !system(command) ;
  }

  /* restore stdout filedes */
  if (fileno(dout) != 1) {
    close(1) ;
    dup(fildes) ;
    close(fildes) ;
  }
  
  return returnStatus ;
}

command_echo(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("[-n] <arg1> <arg2> ...");
    ISynopsis("echo arguments");
    IHelp
      (IHelpArgs,
       "The specified arguments are written to the shell's  standard output,",
       "separated  by spaces,  and terminated  with a newline unless  the -n",
       "option is specified.",
       NULL);
    return 1;
  }

  if (*++tokv) {
    Boolean newline ;
    if (*tokv && !strcmp(*tokv, "-n")) {
      ++tokv, --tokc ;
      newline = FALSE ;
    } else {
      newline = TRUE ;
    }
      
    while (*tokv) {
      fprintf(dout, "%s", tokv[0]);
      if (*++tokv) fputc(' ', dout);
    }
    if (newline)
      fputc('\n', dout);
  }
  return 1;
}

command_quit(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage(" ");
    ISynopsis("quit the simulator");
    IHelp(IHelpArgs,
	  NULL);
    return 1;
  }
  quit_flag = QUIT;
  return 1;
}

command_exit(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage(" ");
    ISynopsis("stop reading the current input file");
    IHelp
      (IHelpArgs,
       "If the simulator is reading a file, it will  stop reading it. If the",
       "simulator is at the top level (i.e. prompting), it will quit.",
       "",
       NULL);
    return 1;
  }
  quit_flag = EXIT;
  return 1;
}

command_args(tokc, tokv)
int tokc;
char *tokv[];
{
  int i=0;
  if (GiveHelp(tokc)) {
    IUsage("[ <arg1> [ <arg2> ...] ]");
    ISynopsis("print its arguments, on separate lines");
    IHelp(IHelpArgs,
	  NULL);
    return 1;
  }
  
  fprintf(dout, "Command had %d arguments\n", tokc);
  while (*tokv) {
    if (LexQuotedTok(*tokv))
      fprintf(dout, "tokv[%d] = %c%s%c%s\n", i,
	      LexQuotedTok(*tokv), tokv[0], LexQuotedTok(*tokv),
	      (LexIsOperator(*tokv) ? " (operator)" : ""));
    else
      fprintf(dout, "tokv[%d] = %s%s\n", i,
	      tokv[0], (LexIsOperator(*tokv) ? " (operator)" : ""));
    tokv++;
    i++;
  }
  return 1;
}

static int strcmp_p(a, b)
const char **a, **b;
{
  return strcmp(*a, *b);
}

static int max(a, b)
int a, b;
{
  return a>b ? a : b;
}

/*
 * The "synopsis" strings from all commands are kept in this array
 */
static char **help_array;
static int help_array_size=0;
BuildSynopsisArray()
{
  TBL_ITEM *function;
  int id, i;
  char *help_tokv[3];

  help_tokv[0] = NULL;
  help_tokv[1] = "help";
  help_tokv[2] = NULL;
  /***
   * Build and sort the array of help strings, first loop to count them,
   * second to sort them
   */
  id = 0;
  while ((function=ITableNextByContext(itf_func_table, &id, itf_com_context))!=NULL) {
    help_tokv[0] = function->name;
    help_string = NULL;
    (void) (function->data.pfi)(SHORT_HELP, help_tokv);
    if (itf_errno) {
      warn("help", "command \"%s\" gave error code %d!!", function->name, itf_errno);
      warn("help", "All commands MUST have a help section");
      IReportError(stderr);
      IResetError();
    }
    if (help_string!=NULL)
      help_array_size++;
    else {
      warn("help", "No help provided for the command \"%s\"", function->name);
      warn("help", "All commands MUST have a help section");
    }
  }
  help_array = (char**)calloc((unsigned)help_array_size, sizeof(char*));
  id = 0;
  i = 0;
  while ((function=ITableNextByContext(itf_func_table, &id, itf_com_context))!=NULL) {
    help_tokv[0] = function->name;
    help_string = NULL;
    (void) (function->data.pfi)(SHORT_HELP, help_tokv);
    if (help_string!=NULL) {
      help_array[i] = (char*)
	malloc((unsigned)strlen(help_string)+6+max(16, strlen(function->name)));
      sprintf(help_array[i], "%-16s - %s\n", function->name, help_string);
      i++;
    }
  }
  qsort((char*)help_array, help_array_size, sizeof(char*), strcmp_p);
}


command_help(tokc, tokv)
int tokc;
char *tokv[];
{
  TBL_ITEM *function;
  char *keyword=0;
  char *help_tokv[3];
  int i;
  if (GiveHelp(tokc)) {
    IUsage("[[-k] command-name]");
    ISynopsis("provide help for commands or keywords");
    IHelp
      (IHelpArgs,
       "\"Help\" can be used in three ways:",
       "help",
       "  Without any arguments,  a  listing  of  all  commands,  with their",
       "  synopsis lines, is provided.\n",
       "help <command-name>",
       "  With a command name as an argument, detailed help for that command",
       "  is printed.\n",
       "help -k <keyword> ",
       "  With the -k option, a list of all commands  with <keyword> (can be",
       "  an arbitrary string) in their synopsis is printed.",
       NULL);
    return 1;
  }

  help_tokv[0] = NULL;
  help_tokv[1] = "help";
  help_tokv[2] = NULL;
  if (tokc>1 && !strcmp("-k", tokv[1])) {
    keyword = tokv[2];
    if (keyword==NULL) {
      IExpectedToken("<keyword>", tokv[2], tokv[1]);
      return 0;
    }
  }
  if (tokc<2) {
    /*
     * print short help on each command
     */
    if (help_array_size==0) BuildSynopsisArray();
    for (i=0;i<help_array_size;i++)
      fputs(help_array[i], dout);
  } else if (keyword) {
    if (help_array_size==0) BuildSynopsisArray();
    for (i=0;i<help_array_size;i++)
      if (substr(keyword, help_array[i]))
	fputs(help_array[i], dout);
  } else {
    /*
     * print long help on a particular command
     */
    function = IFindCommand(tokv[1], NULL);
    if ((function!=NULL)) {
      help_string = NULL;
      help_tokv[0] = tokv[1];
      (function->data.pfi)(LONG_HELP, help_tokv);
      if (itf_errno) {
	warn("help", "command \"%s\" gave error code %d!!", function->name, itf_errno);
	IReportError(stderr);
	IResetError();
      }
      if (help_string!=NULL)
	fprintf(dout, "%s - %s\n", tokv[1], help_string);
    } else {
      itf_errno = UNKNOWN_COMMAND;
      IError("%s", tokv[1]);
      return 0;
    }
  }
  return 1;
}

/* ARGSUSED */
command_hello(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("");
    ISynopsis("print out the string \"Hello!\"");
    IHelp(IHelpArgs,
	  NULL);
    return 1;
  }
  return fprintf(dout, "Hello!\n");
}

/***
 * find the first file in the path, path components separated by comma,
 * colon, or space, file could end in `.Z'
 */
char *IFindFileInPath(name, path, Z_suffix)
char *name;
char *path;
int Z_suffix;
{
  static char pathbuf[MAXPATHLEN];
  struct stat statbuf;
  char *path_ptr = path;
  char *cur_comp;
  char *r;
  extern int errno;
  if (name==NULL || *name==0)
    return NULL;
  if (*name=='/')
    return name;
  while (*path_ptr) {
    while (*path_ptr==' ')
      path_ptr++;
    r = pathbuf;
    cur_comp = path_ptr;
    while (*cur_comp && *cur_comp!=' ' && *cur_comp!=',' && *cur_comp!=':')
      cur_comp++;				/* put the initial path */
    while (path_ptr<cur_comp && *path_ptr)
      *r++ = *path_ptr++;
    if (r>pathbuf && *(r-1)!='/')
      *r++ = '/';	/* add / at end if needed */
    *r = 0;
    if (!strcmp(pathbuf, "./"))
      r = pathbuf;
    path_ptr = name;
    while (*path_ptr)
      *r++ = *path_ptr++;	/* and the name */
    *r = 0;
    path_ptr = cur_comp;
    if (*path_ptr)
      path_ptr++;
    /* printf("Checking \"%s\"\n", pathbuf); */
    if (stat(pathbuf, &statbuf)==0)
      return pathbuf;
    if (   Z_suffix && errno==ENOENT
	&& !(r[-1]=='Z' && r[-2]=='.')) {/* try adding .Z */
      r[0] = '.';
      r[1] = 'Z';
      r[2] = '\0';
      if (stat(pathbuf, &statbuf)==0)
	return pathbuf;
    }
    if (errno!=ENOENT) {
      IError("System error: %s \"%s\"\n", SysError(errno), pathbuf);
      break;
    }
    errno = 0;
  }
  return NULL;
}

/***
 * IOpenFileOrAbort: Finds a file in readpath, and opens it.
 * The file can end in `.Z', in which case a zcat pipe is openned.
 * A record is kept of whether a file or a pipe was openned, for
 * use by ICloseFile.
 *
 * *** NOTE: streams openned with this function must be closed
 * with ICloseFile, as it is necessary that pipe streams be
 * closed by pclose, and file streams by fclose.
 */
typedef enum IOpenType {NOTOPEN, AS_FILE, AS_PIPE};
static enum IOpenType *IOpennedAs;
static int IMaxOpen = 0;

FILE *IOpenFileOrAbort(name, mode, full_namep)
char *name;
char *mode;
char **full_namep;
{
  char *full_name;
  int len, i, filedes;
  enum IOpenType openType;
  FILE *file;

  if (!strcmp(mode, "r"))
    full_name = IFindFileInPath(name, readpath, 1);
  else
    full_name = name;
  if (full_name==NULL || itf_errno)
    ErrorAbort("cannot find file \"%s\"", name);
  if (full_namep!=NULL)
    *full_namep = full_name;

  len = strlen(full_name);
  if (!strcmp(mode, "r") && !strcmp(full_name+len-2, ".Z")) {
    char buf[MAXPATHLEN];
    sprintf(buf, "zcat %s", full_name);
    file = (FILE *)popen(buf, "r");
    openType = AS_PIPE;
  } else {
    file = fopen(full_name, mode);
    openType = AS_FILE;
  }
  if (file==NULL)
    ErrorAbort("cannot open file \"%s\": %s", full_name, SysError(errno));

  filedes = fileno(file);
  if (filedes >= IMaxOpen) {
    if (IMaxOpen==0)
      IOpennedAs = CallocOrAbort(enum IOpenType, filedes+1);
    else
      IOpennedAs = ReallocOrAbort(enum IOpenType, IOpennedAs, filedes+1);
    for (i=IMaxOpen; i<filedes+1; i++)
      IOpennedAs[i] = NOTOPEN;
    IMaxOpen = filedes+1;
  }
#ifdef DEBUG
  if (IOpennedAs[filedes]!=NOTOPEN) {
    fprintf(stderr, "Warning: IOpenFileOrAbort: fileno %d (%s) not closed with ICloseFile\n", filedes, IOpennedAs[filedes]==AS_PIPE ? "pipe" : IOpennedAs[filedes]==AS_FILE ? "file" : "unknown type");
  }
#endif
  IOpennedAs[filedes] = openType;
  return file;
}

int ICloseFile(file, wasPipe)
  FILE *file ;
  int *wasPipe ;
{
  int status;
  int filedes = fileno(file);
  if (filedes>=IMaxOpen) {
#ifdef DEBUG
    fprintf(stderr, "Warning: ICloseFile: fileno %d not openned with IOpenFileOrAbort\n", filedes);
#endif
    if (wasPipe!=NULL)
      *wasPipe = 0;
    return fclose(file);
  }
  if (IOpennedAs[filedes]==AS_PIPE) {
    if (wasPipe!=NULL)
      *wasPipe = 1;
    status = pclose(file);
  } else if (IOpennedAs[filedes]==AS_FILE) {
    if (wasPipe!=NULL)
      *wasPipe = 0;
    status = fclose(file);
  } else if (IOpennedAs[filedes]==NOTOPEN) {
    if (wasPipe!=NULL)
      *wasPipe = 0;
    status = fclose(file);
#ifdef DEBUG
    fprintf(stderr, "Warning: ICloseFile: fileno %d not openned with IOpenFileOrAbort\n", filedes);
#endif
  } else {
    if (wasPipe!=NULL)
      *wasPipe = 0;
    status = fclose(file);
#ifdef DEBUG
    fprintf(stderr, "Warning: ICloseFile: fileno %d record has bad value %d\n",
	    filedes, IOpennedAs[filedes]);
#endif
  }
  IOpennedAs[filedes] = NOTOPEN;
  return status;
}

int command_read(tokc, tokv)
int tokc;
char *tokv[];
{
  static int read_level = 0 ;
  extern int delayDisplayResets() ;
  extern int catchupOnDisplayResets() ;
  char linebuf[MAX_COMMAND_LEN];
  char *line;
  int pipe=0;
  int line_no=0;
  int *old_line_numbers_ok_p = line_numbers_ok_p;
  int line_numbers_ok = 1;
  int read_status;
  FILE *old_din = din;
  int numCommands,idx,tokc2,toknum;
  char **tokv2, **command;
  char *file_name, *tmp;
  Loop *loop ;
  LEX_STACK stack;
  struct STREAM read_stream;
  struct STREAM *old_instream = current_instream;
  int len;
  Boolean	quiet, noBreak ;
  if (GiveHelp(tokc)) {
    IUsage("[ -quiet ] [-noBreak] <filename> <filename> ...");
    ISynopsis("read a file of commands");
    IHelp
      (IHelpArgs,
       "The file is searched for in the  directories  listed in the variable",
       "\"readpath\" (unless  the filename  begins  with a '/').  The  current",
       "directory will not be searched  unless  \".\" is in \"readpath\".    The",
       "directories in \"readpath\" should be separated by a  comma, colon, or",
       "space.",
       "",
       "The \"-quiet\"  option suppresses messages about what  file  is  being",
       "read, but does not suppress warning or error messages.",
       "",
       "The \"-noBreak\" option causes read to  continue reading the file even",
       "if a command error occurs while reading the file.",
       "",
       NULL);
    return 1;
  }

  if (tokc<2) {
    IExpected("<file-name>");
    --read_level ;
    return 0;
  }

  quiet   = FALSE ;
  noBreak = FALSE ;
  while (tokc > 1 && tokv[1][0] == '-') {
    if (strncmp("-quiet", tokv[1], strlen(tokv[1])) == 0) {
      quiet = TRUE ;
    } else if (strncmp("-noBreak", tokv[1], strlen(tokv[1])) == 0) {
      noBreak = TRUE ;
    } else {
      IErrorAbort("Unknown option \"%s\"", tokv[1]) ;
    }
    ++tokv, --tokc ;
  }

  if ((++read_level) == 1 && XMode) delayDisplayResets() ;

  loop = createLoop() ;
  for (toknum = 1 ; toknum < tokc && *++tokv != NULL ; ++toknum) {
    file_name = NULL;
    tmp = IFindFileInPath(*tokv, readpath, 0);
    if (tmp!=NULL) {
      file_name = ITempCalloc(char, strlen(tmp)+1);
      strcpy(file_name, tmp);
    } else {
      /* look for the equivalent .Z file */
      tmp = ITempCalloc(char, strlen(*tokv)+3);
      strcpy(tmp, *tokv);
      strcat(tmp, ".Z");
      tmp = IFindFileInPath(tmp, readpath, 0);
      if (tmp!=NULL) {
	file_name = ITempCalloc(char, strlen(tmp)+1);
	strcpy(file_name, tmp);
      }
    }
    if (file_name==NULL || (din=fopen(file_name, "r")) == 0) {
      if (XMode) catchupOnDisplayResets() ;
      IError("could not open file: \"%s\"", (file_name==NULL?*tokv:file_name));
      din = old_din;
      destroyLoop(loop) ;
      --read_level ;
      return 0;
    }
    read_stream.file = din;
    read_stream.desc = file_name;
    read_stream.name = file_name;
    read_stream.machine = NULL;
    read_stream.type = STREAM_FILE;
    read_stream.status = STREAM_OPEN;
    read_stream.port = read_stream.real_port = read_stream.search = 0;
    read_stream.permanent = 0;
    read_stream.next = NULL;
    read_stream.mode = "r";
    /***
     * check for a .Z suffix - close the file & pipe through zcat
     * the previous open was useful to check that the file was there
     */
    len = strlen(file_name);
    if (len>1 && file_name[len-1]=='Z' && file_name[len-2]=='.') {
      pipe = 1 ;
      status = fclose(din);
      strcpy(linebuf, "zcat ");
      strcat(linebuf, file_name);
      din = (FILE *)popen(linebuf, "r");
      read_stream.file = din;
      read_stream.desc = linebuf;
      read_stream.type = STREAM_PORT;
    }
    current_instream = &read_stream;
    if (!quiet)
      fprintf(dout, "Reading file \"%s\".\n", file_name);
    while (!feof(din)) {
      if (GetInputLine(linebuf, MAX_COMMAND_LEN, din, &line_no)==0) break;
      if (just_do_defaults==ONLY_DEFAULTS) {
	char *end;	/* if the line starts with a number, don't do it */
	strtod(linebuf, &end);
	if (end > linebuf)
	  continue;
      }
      read_status = 0;
      if (IDoCommandPipeline(linebuf, loop, true(echo) && true(read_echo))) {
	IReportError(dout);
	IResetError();

	if (!noBreak) {
	  itf_errno = COMMAND_ERROR;
	  IError("Error reading file \"%s\" %s line %d", 
		 file_name, line_numbers_ok ? "at" : "somewhere after",
		 line_no);
	  break;
	}
      }
      if (quit_flag==QUIT) break;
      if (quit_flag==EXIT) {quit_flag = NORMAL; break; };
    }
    if (pipe==0) {
      fclose(din);
    } else {
      pclose(din);
    }
    din = old_din;
    line_numbers_ok_p = old_line_numbers_ok_p;
    current_instream = old_instream;
    if (itf_errno) break;
  }
  destroyLoop(loop) ;
  if ((--read_level) == 0 && XMode) catchupOnDisplayResets() ;
  return 1;
}

/* ARGSUSED */
int command_segv(tokc, tokv)
int tokc;
char *tokv[];
{
  int *p;
  if (GiveHelp(tokc)) {
    IUsage(" ");
    ISynopsis("cause a segmentation violation");
    IHelp(IHelpArgs,
	  NULL);
    return 1;
  }
  
  p = NULL;
  *p = 0;
  return 1;
}

int command_ls(tokc, tokv)
int tokc;
char *tokv[];
{
  int	idx ;
  char	buffer[BUFSIZ] ;
  int	returnStatus ;
  int	fildes ;

  if (GiveHelp(tokc)) {
    IUsage("<ls args> name ...");
    ISynopsis("list contents of directory");
    IHelp(IHelpArgs,
	  "\"ls\" is a simple  front end  to the UNIX  \"ls\" command. It takes the",
	  "same arguments and behaves in the same way.",
	  "SEE ALSO",
	  "cd, pwd, read",
	  NULL);
    return 1;
  }
  
  buffer[0] = '\0' ;
  for (idx = 0 ; idx < tokc ; ++idx) {
    strcat(buffer, tokv[idx]) ;
    strcat(buffer, " ") ;
  }

  /* make stdout filedes same as dout */
  if (fileno(dout) != 1) {
    fildes = dup(1) ;
    close(1) ;
    dup(fileno(dout)) ;
  }

  returnStatus = !system(buffer) ;

  /* restore stdout filedes */
  if (fileno(dout) != 1) {
    close(1) ;
    dup(fildes) ;
    close(fildes) ;
  }
  
  return returnStatus ;
}


int command_cd(tokc, tokv)
int tokc;
char *tokv[];
{
  int 	returnVal ;
  char	*ptr ;

  if (GiveHelp(tokc)) {
    IUsage("[ name ]");
    ISynopsis("change current working directory");
    IHelp(IHelpArgs,
	  "\"cd\" sets the current working directory to \"name\".",
	  "SEE ALSO",
	  "ls, pwd, read",
	  NULL);
    return 1;
  }

  if (tokc == 1)
    ptr = IGetValue("HOME", "%s") ;
  else if (tokc == 2)
    ptr = tokv[1] ;
  else
    IErrorAbort("Too many arguments") ;

  returnVal = chdir(ptr) ;

  if (returnVal == -1) {
    switch(errno) {
    case ENOTDIR:
      IError("A component of the path name is not a directory.") ;
      break ;
    case ENOENT:
      IError("Directory does not exist.") ;
      break ;
    case EACCES:
      IError("Cannot read directory.") ;
      break ;
    case EFAULT:
      IError("Bad path.") ;
      break ;
    case ELOOP:
      IError("Too many symbolic links.") ;
      break ;
    case ENAMETOOLONG:
      IError("Name too long.") ;
      break ;
    default:
      IError("Unknown error.") ;
      break ;
    }
    return 0 ;
  } else 
    return 1 ;
}


int command_pwd(tokc, tokv)
int tokc;
char *tokv[];
{
  char	*ptr, path[MAXPATHLEN] ;

  if (GiveHelp(tokc)) {
    IUsage(" ");
    ISynopsis("print current working directory");
    IHelp(IHelpArgs,
	  "SEE ALSO",
	  "ls, cd, read",
	  NULL);
    return 1;
  }

  if (tokc != 1)
    IErrorAbort("Too many arguments") ;

  ptr = getwd(path) ;
  
  fprintf(dout, "%s\n", path) ;

  return ptr ? 1 : 0 ;
}


/***
 * Find the n'th argument after the one that matches arg
 */
char *IFindArg(tokc, tokv, arg, n)
char *tokv[];
char *arg;
int n;
{
  int i;
  if (tokv==NULL || arg==NULL) return NULL;
  assert(n>=0);
  for (i=0;i<tokc;i++)
    if (!strcmp(arg, tokv[i]))
      if (i+n<tokc) return tokv[i+n];
      else return NULL;
  return NULL;
}

/***
 * Find the pos of the n'th argument after the one that matches arg
 * Return tokc if can't find it or if n'th arg is beyond tokc
 */
int IFindArgPos(tokc, tokv, arg, n)
int tokc;
char *tokv[];
char *arg;
int n;
{
  int i;
  if (tokv==NULL || arg==NULL) return tokc;
  assert(n>=0);
  for (i=0;i<tokc;i++)
    if (!strcmp(arg, tokv[i]))
      if (i+n<tokc) return i+n;
      else return tokc;
  return tokc;
}

char *INumberth(n)
int n;
{
  static char buf[20];
  switch (n) {
  case 1: return "1'st";
  case 2: return "2'nd";
  case 3: return "3'rd";
  default:
    sprintf(buf, "%d'th", n);
    return buf;
  }
}

InputMatches(file, str)
FILE *file;
char *str;
{
  int c;
  if (str==NULL) return 1;
  if (file==NULL) return 0;
  while (*str!=NULL) {
    c=fgetc(file);
    if (c!=*str) return 0;
    str++;
  }
  return 1;
}

char *SysError(n)
int n;
{
  static char str[30];
  extern char *sys_errlist[];
  extern int sys_nerr;
  if (n>0 || n<sys_nerr)
    return sys_errlist[n];
  else {
    sprintf(str, "No error message for %d", n);
    return str;
  }
}

int	command_for(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("var in list ; do ; commandList ; done") ;
    ISynopsis("loop through a command list");
    IHelp
      (IHelpArgs,
       "The \"for\" construct allows a list of commands to be repeated several",
       "times. The construct iteratively sets var to the values in list, and",
       "then executes the commands in commandList. The loop terminates after",
       "var has been assigned the last value in list.",
       "",
       "EXAMPLE",
       "	xerion-> var int idx",
       "	xerion-> for idx in 1 2 3 4 ; do",
       "	xerion->   echo -n \"$idx \"",
       "	xerion-> done",
       "	1 2 3 4",
       "if, while, sh, same",
       NULL);
    return 1;
  }

  /* THIS COMMAND SHOULD NEVER BE CALLED EXCEPT FOR HELP */
  IErrorAbort("How the hell did I get called?") ;
  return 0 ;
}

int	command_while(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("command ; do ; commandList ; done") ;
    ISynopsis("loop through a command list");
    IHelp
      (IHelpArgs,
       "The  \"while\"  construct  allows  a  list of commands to be  repeated",
       "several times. The construct executes command, and then executes the",
       "commands  in commandList. The loop terminates when command returns a",
       "0 exit status (encounters an error), or sets itf_errno.",
       "",
       "EXAMPLE",
       "	xerion-> var int idx",
       "	xerion-> set idx = 1",
       "	xerion-> while sh test $idx -lt 5 ; do",
       "	xerion->   echo -n \"$idx \"",
       "	xerion->   set idx = `sh expr $idx + 1`",
       "	xerion-> done",
       "	1 2 3 4",
       "NOTES",
       "The while loop can  be negated by preceding  the test command with a",
       "\"!\", or using \"until\" instead of \"while\"",
       "",
       "SEE ALSO",
       "if, for, sh, same",
       NULL);
    return 1;
  }

  /* THIS COMMAND SHOULD NEVER BE CALLED EXCEPT FOR HELP */
  IErrorAbort("How the hell did I get called?") ;
  return 0 ;
}

int	command_if(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("command ; then ; list ; [ elif command ; then ; list ] ... [ else ; list ; ] fi") ;
    ISynopsis("conditionally do a command list");
    IHelp
      (IHelpArgs,
       "The \"if\" construct allows a  list of commands  to  be  conditionally",
       "executed.   The  construct executes command.  If  command  completes",
       "successfully it then executes the commands in the \"if\" command list.",
       "Otherwise,  the  command  following \"elif\" is  executed and,  if  it",
       "completes  successfully,  the  list  following  the next  \"then\"  is",
       "executed.  Failing  that,  it  executes  the commands  in the \"else\"",
       "command list (if  it is present).  It is possible to negate the test",
       "by preceding command with '!'.",
       "",
       "EXAMPLE",
       "	xerion-> var int tmp",
       "	xerion-> set tmp = `calc ${currentNet.error} < 0.5`",
       "	xerion-> if same \"$tmp\" \"1\" ; then",
       "	xerion->   echo \"\\terror less than 0.5\"",
       "	xerion-> else",
       "	xerion->   echo \"\\terror greater than 0.5\"",
       "	xerion-> fi",
       "		error less than 0.5",
       "SEE ALSO",
       "while, for, calc, sh, same",
       NULL);
    return 1;
  }

  /* THIS COMMAND SHOULD NEVER BE CALLED EXCEPT FOR HELP */
  IErrorAbort("How the hell did I get called?") ;
  return 0 ;
}

int command_prompt(tokc, tokv)
int tokc;
char *tokv[];
{
  char	response[BUFSIZ] ;
  int	length ;

  IUsage("<var> <prompt>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("prompt the user for a value");
    IHelp
      (IHelpArgs,
       "The prompt command  will print <prompt> to standard output, and then",
       "read a line from standard input. The line read will be stored in the",
       "variable  <var>.  If nothing is read  (i.e.  the  user  simply  hits",
       "<return>), the value of <var> is unchanged.",
       "",
       "If <var> does not exist,  or the line read  is not a valid value for",
       "its type,  then an error message  will be printed, and the value (if",
       "any) of <var> will be unchanged.",
       "",
       NULL);
    return 1;
  }

  if (tokc != 3)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  if (!IIsObjectName(tokv[1]))
    IErrorAbort("%s is not a variable.", tokv[1]) ;
  
  fprintf(stdout, tokv[2]) ;
  fflush(stdout) ;
  fgets(response, BUFSIZ, stdin) ;

  /* strip trailing new line */
  length = strlen(response) ;
  if (response[length-1] == '\n')
    response[--length] = '\0' ;

  if (length)
    return ISetValue(tokv[1], response) ;
  else
    return 1 ;
}


/* ************************************************* */
/* Here is a good random number generator
   and the auxiliary routines to print the execution time */

#define SHUFFLE 256		/* size of random array */
#define MULT 25173
#define MOD ((long int) 65536)
#define INCR 13849
#define FMOD ((double) 65536.0)

static int initialized = FALSE ;
static long int	randSeed = 696969;
static double	random_array[SHUFFLE];	/* random variables */

/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	double ISimpleRandReal(void) - returns a random number between 0 and 1
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables:
 *	double randSeed 	    - The rng seed
 * Returns: None
 * Calls: None
 * Description:
 *	This routine returns the random number generator between 0 and 1
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
static double ISimpleRandReal()
{
  randSeed = (MULT * randSeed + INCR) % MOD;
  return (double) randSeed / FMOD;
}
/***********************************************************************/

/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	initialize_rng() - to initialize the random number generator
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables: None
 * Returns: None
 * Calls:
 *	ISimpleRandReal()
 *	IRandReal()
 * Description:
 *	This routine initializes the random number generator
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
static void initialize_rng()
{
  int n;
  double x;
  
  for (n = 0; n < SHUFFLE; ++n)
    random_array[n] = ISimpleRandReal();
  for (n = 0; n < 1000; ++n)	/* warm up random generator */
    x = IRandReal();
}


/***********************************************************************
 * Author: Lester Ingber, Bruce Rosen (copyright) (c)
 * Date	 5 Nov 92
 * Procedure:
 *	double IRandReal(void) - Shuffles random numbers in random_array[]
 * Parameters: None
 * Inputs:
 * Outputs:
 * Global Variables: None
 * Returns: None
 * Calls:
 *	ISimpleRandReal()
 * Description:
 *	This routine initializes the random number generator
 * Local Variables: None
 * Portability:
 * Other:
 ***********************************************************************/
double IRandReal()
{
  double rranf;
  int kranf;
  if (!initialized) {
    initialized = TRUE ;
    initialize_rng() ;
  }
  kranf = (int) (ISimpleRandReal() * SHUFFLE) % SHUFFLE;
  rranf = random_array[kranf];
  random_array[kranf] = ISimpleRandReal();
  
  return rranf;
}
/***********************************************************************/
void	ISeed(seed)
  int	seed ;
{
  if (!initialized) {
    initialized = TRUE ;
    initialize_rng() ;
  }
  randSeed = seed ;
  initialize_rng() ;
}
/***********************************************************************/

/***********************************************************************/
typedef struct Positional {
  char	*tokc ;
  char	**tokv ;
  char	*line ;
} Positional ;
static Positional	*positional = NULL ;
static int		execLevel = -1 ;
static int		maxExecLevel = 0 ;
/***********************************************************************/
static void	createAndSet(type, var, value)
  char		*type ;
  char		*var ;
  char		*value ;
{
  if (IMakeScalar(type, var) == NULL)
    IErrorAbort("couldn't make parameter %s", var) ;

  if (ISetValue(var, value) == NULL)
    IErrorAbort("couldn't set parameter %s", var) ;
}
/***********************************************************************/
static void	setPositionals(positional)
  Positional	*positional ;
{
  int	tokc = atoi(positional->tokc) ;
  int	idx ;

  createAndSet("int", "tokc", positional->tokc) ;
  IMakeVector("String", "tokv", tokc) ;
  for (idx = 0 ; idx < tokc ; ++idx) {
    char	buffer[32] ;
    sprintf(buffer, "tokv[%d]", idx) ;
    if (ISetValue(buffer, positional->tokv[idx]) == NULL)
      IErrorAbort("couldn't set parameter %s", buffer) ;
  }
  createAndSet("String", "*", positional->line) ;
}
/***********************************************************************/
static void	unsetPositionals(positional)
  Positional	*positional ;
{
  IDeleteObject("tokv") ;
  IDeleteObject("tokc") ;
  IDeleteObject("*") ;
}
/***********************************************************************/
static void	pushPositionals(tokc, tokv)
  int tokc;
  char *tokv[];
{
  int	idx ;
  char	*ptr, buffer[BUFSIZ] ;

  if (++execLevel >= maxExecLevel) {
    ++maxExecLevel ;
    if (positional)
      positional = (Positional *)realloc(positional, 
					 maxExecLevel*sizeof(Positional)) ;
    else
      positional = (Positional *) calloc(maxExecLevel, sizeof(Positional)) ;
  }
  
  /* make tokc */
  positional[execLevel].tokc = strdup(itoa(tokc)) ;

  /* make tokv[*] */
  positional[execLevel].tokv = (char **)calloc(tokc, sizeof(char *)) ;
  for (idx = 0 ; idx < tokc ; ++idx)
    positional[execLevel].tokv[idx] = strdup(tokv[idx]) ;

  /* create $* */
  LexObjReconstruct(defaultLex, buffer, BUFSIZ, tokc, tokv) ;
  positional[execLevel].line = strdup(buffer) ;

  setPositionals(&positional[execLevel]) ;
}
/***********************************************************************/
static void	popPositionals()
{
  int	idx ;
  int	tokc ;

  unsetPositionals(&positional[execLevel]) ;

  tokc = atoi(positional[execLevel].tokc) ;
  for (idx = 0 ; idx < tokc ; ++idx)
    free(positional[execLevel].tokv[idx]) ;
  free(positional[execLevel].tokv) ;
  free(positional[execLevel].line) ;
  free(positional[execLevel].tokc) ;

  if (--execLevel >= 0)
    setPositionals(&positional[execLevel]) ;
}
/***********************************************************************/
int command_exec(tokc, tokv)
int tokc;
char *tokv[];
{
  int	idx ;
  char	buffer[BUFSIZ] ;

  IUsage("file [args]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("execute a xerion script file");
    IHelp
      (IHelpArgs,
       "The  \"exec\"  command  sources  and executes  a  xerion shell script,",
       "passing  in its arguments as positional  parameters.   Xerion  shell",
       "scripts must contain the exact string \"#!xerion\" as the  first  line",
       "of the  file. When a script is executed the following parameters are",
       "automatically created and set:",
       "",
       "  $tokc - the number of  arguments (positional parameters) passed to",
       "          the script.",
       "  $tokv - a vector containing the positional parameters.",
       "  $*    - all of the positional parameters.",
       "",
       "NOTES",
       "Shell scripts are found by searching the \"readpath\" variable.",
       "",
       "The \"exec\"  command may be  ommited;  that is, a shell script may be",
       "executed by simply entering its name at the command line.",
       "",
       "EXAMPLE",
       "To execute the following shell script (stored in file \"tmp\"):",
       "",
       "#!xerion",
       "echo ",
       "echo \"I'm in the file \\\"${tokv[0]}\\\".\"",
       "echo \"I have $(calc $tokc - 1) arguments.\"",
       "echo \"The command line was: $*.\"",
       "",
       "type either of the following two commands.",
       "",
       "bp-> exec tmp a b c",
       "I'm in the file \"tmp\".",
       "I have 3 arguments.",
       "The command line was:  tmp a b c.",
       "bp-> tmp a b c",
       "I'm in the file \"tmp\".",
       "I have 3 arguments.",
       "The command line was:  tmp a b c.",
       "",
       NULL);
    return 1;
  }

  if (strcmp(tokv[0], "exec") == 0)
    --tokc, ++tokv ;

  pushPositionals(tokc, tokv) ;

  /* do the read command */
  sprintf(buffer, "read -quiet \"%s\"", tokv[0]) ;
  IDoCommandLine(buffer) ;

  popPositionals() ;
  
  return 1 ;
}
/***********************************************************************/
int command_same(tokc, tokv)
int tokc;
char *tokv[];
{
  IUsage("string1 string2") ;
  if (GiveHelp(tokc)) {
    ISynopsis("compare two strings");
    IHelp
      (IHelpArgs,
       "This command compares two strings.  It completes successfully if the",
       "two are  identical,  and fails  if  they  are  different. It is most",
       "useful in an \"if\", or \"while\" test.",
       "",
       "EXAMPLE",
       "var String name",
       "...",
       "if same $name \"Output\" ; do",
       "  echo \"Group has name $name.\"",
       "done",
       NULL);
    return 1;
  }

  if (tokc != 3)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  return !strcmp(tokv[1], tokv[2]) ;
}
/***********************************************************************/

/***********************************************************************/
double	ITime(timeType, includeChildren)
  ITimeType	timeType ;
  int		includeChildren ;
{
  struct tms	timeBuf ;
  double	seconds ;

  times(&timeBuf) ;
  switch (timeType) {
  case ITimeUser:
    seconds = timeBuf.tms_utime ;
    if (includeChildren)
      seconds += timeBuf.tms_cutime ;
    break ;
  case ITimeSystem:
    seconds = timeBuf.tms_stime ;
    if (includeChildren)
      seconds += timeBuf.tms_cstime ;
    break ;
  case ITimeBoth:
  default:
    seconds = timeBuf.tms_utime + timeBuf.tms_stime ;
    if (includeChildren)
      seconds += timeBuf.tms_cutime + timeBuf.tms_cstime ;
    break ;
  }
  return seconds/HZ ;
}
/***********************************************************************/
int command_time(tokc, tokv)
int tokc;
char *tokv[];
{
  char		*name ;
  ITimeType	timeType = ITimeBoth ;
  Boolean	children = FALSE ;

  IUsage("[-u] [-s] [-c]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("print process times");
    IHelp
      (IHelpArgs,
       "This command prints the time (in seconds) used by  the process since",
       "startup.  The value printed is a floating point  number, accurate to",
       "the  system dependent parameter  1/HZ seconds  defined in  the  file",
       "param.h in /usr/include/sys.  \"Time\" can be used, along with \"calc\",",
       "to time specific commands.",
       "",
       "The  \"-u\" option causes \"time\" to print only the time used executing",
       "instructions in the user space. The \"-s\" option causes  it  to print",
       "only the time  used by  the  system on  behalf of  the  process.  By",
       "default the sum of the two is printed.",
       "",
       "The \"-c\" option causes the  appropriate times used by any terminated",
       "child processes to be also included.",
       "EXAMPLE",
       "",
       "  xerion-> var Real time",
       "  xerion-> set time = $(time)",
       "  xerion-> minimize 1000 > minimize.out",
       "  xerion-> set time = $(calc $(time) - ${time})",
       "  xerion-> echo ${time}",
       NULL);
    return 1;
  }

  name = *tokv ;
  for (++tokv, --tokc ; tokc ; ++tokv, --tokc) {
    if (strcmp(*tokv, "-c") == 0) {
      children = TRUE ;
    } else if (strcmp(*tokv, "-u") == 0) {
      timeType = ITimeUser ;
    } else if (strcmp(*tokv, "-s") == 0) {
      timeType = ITimeSystem ;
    } else if (*tokv[0] == '-') {
      IErrorAbort("Invalid option: %s", *tokv) ;
    } else {
      break ;
    }
  }

  if (tokc)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  fprintf(dout, "%g\n", ITime(timeType, children)) ;
  return 1 ;
}
/***********************************************************************/
