/*
 * $Id: hash-table.c,v 1.2 92/11/30 11:39:49 drew Exp $
 * 
 */

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appears in all copies and that 
 * both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of University of Toronto 
 * not be used in advertising or publicity pertaining to distribution 
 * of the software without specific, written prior permission.  
 * University of Toronto makes no representations about the suitability 
 * of this software for any purpose. It is provided "as is" without 
 * express or implied warranty. 
 *
 * UNIVERSITY OF TORONTO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS, IN NO EVENT SHALL UNIVERSITY OF TORONTO BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER 
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF 
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 **********************************************************************/

#include <xerion/useful.h>
#include "hash-table.h"
#ifndef NULL
#define NULL (0L)
#endif

/*
 *  hash-table.c -	implements a general hash table
 *
 *  items in the table have four fields: name, context, id and value
 *  the context and id fields are integers, the name is a string, and
 *  the value field is an array of 8 integers
 *  items can be retrieved on any of the three first fields
 *  name and context together form a unique key, as does id by itself
 */

/*
 * TODO: make insert look for id's of deleted items when it fills up
 * maybe keep a count of live items to assist this
 */

#define MAX_STRING (1024)
int hash_errno;
char *hash_error[10] = {
  "everything is ok, there's no error!",
  "name not found in table",
  "name is ambiguous",
  "name is already in table",
  "id out of range",
  "out of memory",
  "name is null or zero length",
  "hash table overflow",
  "not configured for performing this task",
  NULL};

HASH_TABLE *INewTable(ht, max_items, hash_bins, index_types)
HASH_TABLE *ht;
int max_items;	/* int - maximum number of items (used for size of id_index & alpha_index) */
int hash_bins;	/* int - size of the hash table */
int index_types;	/* can be ID_INDEX or ALPHA_INDEX */
{
  hash_errno = 0;
  if (hash_bins<1) hash_bins=1;
  if (ht==NULL)
    if((ht = (HASH_TABLE*)malloc(sizeof(HASH_TABLE)))==NULL)
      goto insuff_mem;
  ht->id_vec = NULL;
  ht->alpha_vec = NULL;
  ht->hash_bins = hash_bins;
  if ((ht->hash_vec=Calloc(TBL_ITEM*, hash_bins))==NULL)
    goto insuff_mem;
  if ((index_types&ID_INDEX)
      && ((ht->id_vec=Calloc(TBL_ITEM*, max_items))==NULL))
    goto insuff_mem;
  ht->max_items = max_items;
  ht->free_id = 1;
  ht->free_context = 1;
  ht->free_item = NULL;
  ht->new_item = NULL;
  return ht;

 insuff_mem:
  ErrorAbort("INewTable: out of memory");
  hash_errno = HT_NO_MEMORY;
  return NULL;
}

TBL_ITEM *ITableLookup(ht, name, context)
HASH_TABLE *ht;
const char *name;
int context;
{
  int hv;
  TBL_ITEM *i, *prev_i;
  hash_errno = 0;
  if (name==NULL || name[0]==0) {
    hash_errno = HT_BAD_NAME;
    return NULL;
  }
  /*
   * look with the hash table
   */
  hv = hash(name, ht->hash_bins, context);
  i = ht->hash_vec[hv];
  prev_i = 0;
  while (i!=NULL) {
    if ((context==i->context) && !strcmp(name, i->name)) {
      /* put the found item at the start of the list */
      if (prev_i!=0) {
	prev_i->next = prev_i->next->next;
	i->next = ht->hash_vec[hv];
	ht->hash_vec[hv] = i;
      }
      return i;
    }
    prev_i = i;
    i = i->next;
  }
  hash_errno = HT_NOT_FOUND;
  return NULL;
}

int ht_lookup_id(ht, name, context)
HASH_TABLE *ht;
char *name;
int context;
{
  int hv;
  TBL_ITEM *i, *prev_i;
  hash_errno = 0;
  if (name==NULL || name[0]==0) {
    hash_errno = HT_BAD_NAME;
    return NULL;
  }
  /*
   * look with the hash table
   */
  hv = hash(name, ht->hash_bins, context);
  i = ht->hash_vec[hv];
  prev_i = 0;
  while (i!=NULL) {
    if ((context==i->context) && !strcmp(name, i->name)) {
      /* put the found item at the start of the list */
      if (prev_i!=0) {
	prev_i->next = prev_i->next->next;
	i->next = ht->hash_vec[hv];
	ht->hash_vec[hv] = i;
      }
      return i->id;
    }
    prev_i = i;
    i = i->next;
  }
  hash_errno = HT_NOT_FOUND;
  return 0;
}

int INewTableContext(ht)
HASH_TABLE *ht;
{
  return ht->free_context++;
}

TBL_ITEM *ITableLookupById(ht, id)
HASH_TABLE *ht;
int id;
{
  int i;
  TBL_ITEM *j;
  hash_errno = 0;
  if (id<1 || id>=ht->free_id) {
    hash_errno = HT_BAD_ID;
    return NULL;
  }
  /*
   * use the id vector to return it instantly
   */
  if (ht->id_vec!=NULL) return ht->id_vec[id-1];
  /*
   * or, if that doesn't exits, laboriously traverse
   * the hash table
   */
  i = 0;
  while (i<ht->hash_bins) {
    j = ht->hash_vec[i];
    while (j!=NULL) if (j->id==id) return j; else j=j->next;
    i++;
  }
  ErrorAbort("ITableLookupById : couldn't find id %d", id);
  hash_errno = HT_NOT_FOUND;
  return NULL;
}

#define EXTEND 16

TBL_ITEM *ITableInsert_(ht, name, context, info)
HASH_TABLE *ht;
const char *name;
int context;
long info;
{
  int len, hv, found;
  TBL_ITEM *i;

  hash_errno = 0;
  /* check if the string is unreasonable small or large */
  if (((len=strlen(name))==0) || len >= MAX_STRING) {
    hash_errno = HT_BAD_NAME;
    return NULL;
  }
  len++;
  if (ht->free_id>ht->max_items) {
    /* reallocate hash, id, & alpha vector instead of returning NULL */
    TBL_ITEM **vec;
    int i;
    if (ht->id_vec!=NULL) {
      vec = Realloc(TBL_ITEM*, ht->id_vec, ht->max_items+EXTEND);
      if (vec==NULL) goto out_of_mem;
      for (i=0;i<EXTEND;i++) vec[ht->max_items+i] = NULL;
      ht->id_vec = vec;
    }
    if (ht->alpha_vec!=NULL) {
      vec = Realloc(TBL_ITEM*, ht->alpha_vec, ht->max_items+EXTEND);
      if (vec==NULL) goto out_of_mem;
      for (i=0;i<EXTEND;i++) vec[ht->max_items+i] = NULL;
      ht->alpha_vec = vec;
    }
    ht->max_items += EXTEND;
    goto OK;

  out_of_mem:
    hash_errno = HT_OVERFLOW;
    return NULL;
  OK:;
  }

  hv = hash(name, ht->hash_bins, context);
  i = ht->hash_vec[hv];
  found=0;
  while (i!=NULL && !found) {
    if ((context==i->context) && !strcmp(name, i->name))
      found=1;
    else
      i = i->next;
  }
  if (found) {
    hash_errno = HT_DUPLICATE;
    return NULL;
  }

  /* get a new item */
  if (ht->new_item!=NULL)
    i = (ht->new_item)(len);
  else
    i = Calloc(TBL_ITEM, 1);
  if (i==NULL) {
    hash_errno = HT_NO_MEMORY;
    return NULL;
  }
  if ((i->name=Calloc(char, len))==NULL) {
    /* can't be bothered to delete the item, leave it around as trash */
    hash_errno = HT_NO_MEMORY;
    return NULL;
  }
  i->id = ht->free_id++;
  i->context = context;
  i->data.token = info;
  (void) strncpy(i->name, name, len);
  if (ht->id_vec!=NULL) ht->id_vec[(i->id)-1] = i;
  i->next = ht->hash_vec[hv];
  ht->hash_vec[hv] = i;
  return i;
}

int ITableDelete(ht, name, context)
HASH_TABLE *ht;
const char *name;
int context;
{
  int hv, found;
  TBL_ITEM *item, *p;

  hv = hash(name, ht->hash_bins, context);
  item = ht->hash_vec[hv];
  found=0;
  p=NULL;
  while (item!=NULL && !found) {
    if ((context==item->context) && !strcmp(name, item->name))
      found=1;
    else {
      p=item;
      item = item->next;
    }
  }
  if (!found) return 0;
  else {
    if (p==NULL)
      ht->hash_vec[hv] = ht->hash_vec[hv]->next;
    else
      p->next = p->next->next;
    if (ht->id_vec!=NULL) ht->id_vec[item->id-1]=NULL;
    /* modify alpha_vec here */
    if (ht->free_item==NULL) {
      free(item->name);
      free((char*)item);
    } else
      (ht->free_item)(item);
  }
  return 1;
}

ITablePrint(ht)
HASH_TABLE *ht;
{
  int i;
  TBL_ITEM *j;
  printf("Hash table, %d locations\n", ht->hash_bins);
  for (i=0;i<ht->hash_bins;i++) {
    j = ht->hash_vec[i];
    printf("VEC[%d] ", i);
    while (j!=NULL) {
      printf("(%s, %d, %d) ", j->name, j->context, j->id);
      j=j->next;
    }
    printf("\n");
  }
}

TBL_ITEM *ITableNext(ht, id)
HASH_TABLE *ht;
int *id;
{
  if (ht->id_vec == 0) {
    Error("ITableNext : no id vector, cannot do");
    return 0;
  }
  while (++(*id) < ht->free_id) /* skip holes (made by delete) */
    if (ht->id_vec[(*id)-1]!=NULL) return ht->id_vec[(*id)-1];
  *id = 0;
  return 0;
}

TBL_ITEM *ITableNextByContext(ht, id, context)
HASH_TABLE *ht;
int *id, context;
{
  hash_errno = 0;
  if (ht->id_vec == 0) {
    Error("ITableNextByContext : no id vector, cannot do");
    return NULL;
  }
  while (++(*id) < ht->free_id) /* skip holes (made by delete) */
    if (   ht->id_vec[(*id)-1]!=NULL
	&& (context==0 || ht->id_vec[(*id)-1]->context==context))
      return ht->id_vec[(*id)-1];
  *id = 0;
  return NULL;
}

int hash(str, max_hash, context)
const char *str;
int max_hash, context;
{
  register int h;
  register int n;
  h = context;
  while ((n = *(str++))!=0) {
    h = h*11 + (n-'a');
  }
  h %= max_hash;
  if (h<0) h = -h;
  return(h);
}

int ITableDeleteByContext(ht, context)
HASH_TABLE *ht;
{
  int i, c;
  TBL_ITEM *prev, *cur, *next;
  c = 0;
  hash_errno = 0;
  for (i=0;i<ht->hash_bins;i++) {
    cur = prev = ht->hash_vec[i];
    while (cur!=NULL) {
      next = cur->next;
      if (cur->context == context) {
	c++;
	/* adjust the pointers */
	if (cur==prev)
	  ht->hash_vec[i] = cur->next;
	else
	  prev->next = cur->next;
	/* adjust the storage vectors */
	if (ht->id_vec!=NULL)
	  ht->id_vec[cur->id-1]=NULL;
	/* modify alpha vec here */
	/* free the item */
	if (ht->free_item==NULL) {
	  free(cur->name);
	  free((char*)cur);
	} else
	  (ht->free_item)((char*)cur);
      } else {
	prev = cur;
      }
      cur = next;
    }
  }
  return c;
}
