
/**********************************************************************
 * $Id: bind-print.c,v 1.9 93/03/29 14:15:38 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include "itf.h"
#include "lex.h"
#include "alias.h"

#define SWAP	1
#define NO_SWAP	0

char default_format_code[ITF_VAR_TYPES] =
  {  0, 's', 'g', 'g', 'g', 'd', 'd', 'd', 'd', 't',  0,  0};
char default_format_width[ITF_VAR_TYPES] =
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0};
char default_format_prec[ITF_VAR_TYPES] =
  {  0,  0,  6,  6,  6,  0,  0,  0,  0,  0,  0,  0};
char use_format_prec[ITF_VAR_TYPES] =
  {  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0};
char *legal_format_codes[ITF_VAR_TYPES] =
  { "", "sdxX", "gfeEb", "gfeEb", "gfeEb", "dxXct", "dxXct", 
      "dxXct", "dxXct", "dt", "", ""};
char *itf_type_names[ITF_VAR_TYPES] = 
  { "no type", "String", "Real", "float", "double", "long", 
      "int", "short", "char", "Boolean", "psuedo-field", "struct" };

static char format_buf[100];
char 	*show_array_on_line = "true";
int 	name_width      = 20;
int 	show_indent_inc = 2;
#define NAMEBUF 256

#ifndef _NO_LIBPW
static int	getRange() ;
#endif

/* ARGSUSED */
int command_fpPrecision(tokc, tokv)
int tokc;
char *tokv[];
{
  int real_precision = -1;
  int float_precision = -1;
  int double_precision = -1;
  struct ARG_DESC *argd;
  argd = StartArgs(tokv[0]);
  Args(argd, "[<real-precision>%n]", &real_precision);
  Args(argd, "[<float-precision>%n]", &float_precision);
  Args(argd, "[<double-precision>%n]", &double_precision);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("set the precision for printing floating point values");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "If no arguments are given,  the current precisions  are printed out.",
       "If one or more arguments  are given, the  precision are set to those",
       "values.",
       NULL);
    return 1 ;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (real_precision>-1) {
    default_format_prec[ITF_REAL] = real_precision;
    default_format_prec[ITF_FLOAT] = (float_precision==-1
				      ? real_precision : float_precision);
    default_format_prec[ITF_DOUBLE] = (double_precision==-1
				       ? real_precision : double_precision);
  }
  fprintf(dout, "Precision:\nReal\t%d\nfloat\t%d\ndouble\t%d\n",
	  default_format_prec[ITF_REAL],
	  default_format_prec[ITF_FLOAT],
	  default_format_prec[ITF_DOUBLE]);
  return 1 ;
}

char *IMakeFormatStr(var_type, width, precision, code)
  int	var_type ;
  int	width ;
  int	precision ;
  int	code ;
{
  if (!strchr(legal_format_codes[var_type], code)) {
    /* illegal - use default */
    warn("IMakeFormatStr", "illegal format char %d for type %d", code, var_type);
    code = default_format_code[var_type];
    width = default_format_width[var_type];
    precision = default_format_prec[var_type];
  }
  if (code) {
    char *ptr=format_buf;

    if (code=='x' || code=='X') {
      *(ptr++) = '0' ;
      *(ptr++) = 'x' ;
    }

    *(ptr++) = '%' ;
    if (width) {
      strcpy(ptr, itoa(width));
      ptr = format_buf + strlen(format_buf) ;
    }
    if (use_format_prec[var_type] && precision >= 0) {
      *(ptr++) = '.' ;
      strcpy(ptr, itoa(precision));
      ptr = format_buf + strlen(format_buf) ;
    }

    *(ptr++) = code;
    *ptr = '\0' ;
  }
  return format_buf;
}

int IPrintPoi(buffer, len, poi, var_type, format_width, format_prec, format_code)
char *buffer;
int len;
POI *poi;
int var_type;
int format_width, format_prec, format_code;
{
  char *format;
#ifdef lint
  len = 0; len = len; /* just to use len */
#endif
  if (var_type < ITF_VAR_TYPES) {
    if (!strchr(legal_format_codes[var_type], format_code)) {
      /* illegal - use default */
      warn("IPrintPoi", "illegal format char %d for type %d", format_code, var_type);
      format_code = default_format_code[var_type];
      format_width = default_format_width[var_type];
      format_prec = default_format_prec[var_type];
    }
    format = IMakeFormatStr(var_type, format_width, format_prec, format_code);
  }
  if (format_code=='t') {
    /* true/false */
    int t;
    switch (var_type) {
    case ITF_STRING:	t = true(*(poi->string_ptr)); break;
    case ITF_LONG:	t = (*(poi->long_ptr) != 0); break;
    case ITF_INT:	t = (*(poi->int_ptr) != 0); break;
    case ITF_SHORT:	t = (*(poi->short_ptr) != 0); break;
    case ITF_CHAR:	t = (*(poi->char_ptr) != 0); break;
    case ITF_BOOLEAN:	t = (*(poi->boolean_ptr) != 0); break;
    default:
      warn("IPrintPoi", "bad variable type for Boolean %d", var_type);
      return 0;
    }
    *strchr(format, 't') = 's';
    sprintf(buffer, format, (t ? "true" : "false"));
  } else {
    switch (var_type) {
    case ITF_STRING:
      if ((*poi->string_ptr)==NULL)
	buffer[0]=0;
      else if (format_code=='s') {
	if (strcmp(format, "%s"))
	  sprintf(buffer, format, *(poi->string_ptr));
	else
	  strcpy(buffer, *(poi->string_ptr));
      } else if (strchr(legal_format_codes[ITF_DOUBLE], format_code)!=NULL)
	sprintf(buffer, format, atof(*(poi->string_ptr)));
      else if (strchr(legal_format_codes[ITF_LONG], format_code)!=NULL)
	sprintf(buffer, format, atoi(*(poi->string_ptr)));
      break;
    case ITF_REAL:	sprintf(buffer, format, *(poi->real_ptr)); break;
    case ITF_FLOAT:	sprintf(buffer, format, *(poi->float_ptr)); break;
    case ITF_DOUBLE:	sprintf(buffer, format, *(poi->double_ptr)); break;
    case ITF_LONG:	sprintf(buffer, format, *(poi->long_ptr)); break;
    case ITF_INT:	sprintf(buffer, format, *(poi->int_ptr)); break;
    case ITF_SHORT:	sprintf(buffer, format, *(poi->short_ptr)); break;
    case ITF_CHAR:	sprintf(buffer, format, *(poi->char_ptr)); break;
    case ITF_BOOLEAN:	sprintf(buffer, format, *(poi->boolean_ptr)); break;
    default:
      warn("IPrintPoi", "bad variable type %d", var_type);
      return 0;
    }
  }
  return 1;
}

static void	IPrintExtraHelp(name)
  char		*name ;
{
  char		**extraHelp ;

  extraHelp = IGetHelpStrings(name) ;
  if (extraHelp != NULL) {
    fprintf(dout, "%s:\n", name) ;
    while (*extraHelp != NULL)
      fprintf(dout, "%s\n", *extraHelp++);
  }
}
/***
 * IPrintWhatis
 * Looks for the following things
 * (1) variable name (could be a structure)
 * (2) type name (simple or structure, if structure, print fields)
 * (3) struct-type field (a `complex' type) (might have struct-type missing)
 * (4) command
 * In the first three cases the name could be a compound object
 */
/* ARGSUSED */
int	IPrintWhatis(in_name, verbose)
  char	*in_name;
  int	verbose;
{
  TBL_ITEM *item;
  BINDING *binding;
  char *name = itf_name_buffer;
  char *aliasExpansion = NULL;
  int found=0;
  int type=0;
  int struct_type;
  int take_address;
  struct ADDRESS_ENV env;
  if (!in_name || !*in_name) {
    itf_value_buffer[0]=0;
    itf_errno = NULL_VARNAME;
  }
  MASSAGE_VAR_NAME(in_name, name);
  /* fprintf(dout, "whatis: \"%s\"\n", name); */
  /***
   * Object (variable or structure)
   */
  if (IParseObject(name, &env, &take_address)) {
    found++;
    IPrintExtraHelp(name) ;
    fprintf(dout, "%s is an object of type %s%s\n", name,
	    IGetTypeName(env.binding->var_type), IArrayCounter(env.binding));
  }
  /* reset the error */
  IResetError();
  /***
   * Type or object
   */
  if ((item = IParseFieldOrObject(name)) != NULL) {
    found++;
    if (item->context==itf_type_context) {
      IPrintExtraHelp(name) ;
      if (IIsStructType(item->data.binding->var_type)) {
	int id;
	fprintf(dout, "\"%s\" is a structure type with fields:\n", name);
	type = item->data.binding->var_type;
	id = 0;
	while ((item=ITableNextByContext(itf_table, &id, type))!=NULL) {
	  binding = item->data.binding;
	  fprintf(dout, "%12s\t%s%s\n", IGetTypeName(binding->var_type),
		  item->name, IArrayCounter(binding));
	}
      } else
	fprintf(dout, "\"%s\" is a simple type\n", name);
    } else if (item->context==itf_obj_context) {
      /* this is taken care of in the first case
      binding = item->data.binding;
      type = binding->var_type;
      fprintf(dout, "\"%s\" is an object of type \"%s\"%s\n", name,
              IGetTypeName(type), IIsArray(binding) ? " (array)" : "");
      */
    } else {
      binding = item->data.binding;
      type = binding->var_type;
      IPrintExtraHelp(name) ;
      fprintf(dout, "\"%s\" is a field of type \"%s\"%s%s in the structure \"%s\"\n",
	      binding->tbl_entry->name, IGetTypeName(type),
	      IIsArray(binding) ? " array" : "",
	      IArrayCounter(binding), IGetTypeName(item->context));
    }
  }
  /* reset the error */
  IResetError();
  /***
   * Field - search in all structure contexts
   */
  for (struct_type=ITF_VAR_TYPES;
       struct_type<itf_top_struct_type;
       struct_type++) {
    if ((item=ITableLookup(itf_table, name, struct_type))!=NULL) {
      binding = item->data.binding;
      type = binding->var_type;
      IPrintExtraHelp(name) ;
      fprintf(dout, "\"%s\" is a field of type \"%s\"%s%s in the structure \"%s\"\n",
	      name, IGetTypeName(type), IIsArray(binding) ? " array" : "",
	      IArrayCounter(binding), IGetTypeName(struct_type));
      found++;
    }
  }
  /***
   * Command
   */
  if (aliasExpansion = getAlias(name)) {
    fprintf(dout, "\"%s\" is an alias for \"%s\"\n", name, aliasExpansion);
    found++;
  } else if (((item=ITableLookup(itf_func_table,name,itf_com_context))!=NULL)){
    char *tokv[2];
    found++;
    tokv[0] = name;
    tokv[1] = 0;
    (void) (item->data.pfi)(SHORT_HELP, tokv);
    fprintf(dout, "\"%s\" is a command to %s\n", name, help_string);
  }
  if (!found) {
    IError("\"%s\" is not an variable, type, field, alias or command", name);
    return 0 ;
  } else
    return 1 ;
}

int command_whatis(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("<object1> [ <object2> ... ]");
    ISynopsis("print what certain objects are");
    IHelp(IHelpArgs,
	  "Print out details of variables, types, fields, and structures.",
	  NULL);
    return 1;
  }
  while (*++tokv)
    IPrintWhatis(*tokv, 0);
  return 1 ;
}

int command_format(tokc, tokv)
int tokc;
char *tokv[];
{
  char *format;
  int width, precision, code;
  if (GiveHelp(tokc)) {
    IUsage("<object> \"[%[<width>][.<precision>]<print-code>]\"");
    ISynopsis("set the print format for a variable or structure field");
    IHelp
      (IHelpArgs,
       "The  format is a subset of  those useable  with printf().  The width",
       "and precision are optional  (but if  the  precision is specified  it",
       "must be preceded by a period).  The print  code should  be one of d,",
       "i, u, x, X, f, e, E, g, G, c, or s.  One  additional conversion type",
       "is available, it is t, which will print as true or false.",
       "",
       "Some examples are (characters after '#' are comments):\n",
       "format UnitRec.output \"%f\" # field \"output\" of UnitRec prints as a float",
       "format NetRec.error \"%.6g\" # error prints with 6 digits precision.",
       NULL);
    return 1;
  }

  if (tokc<2 || tokc>3) {
    IExpected("one or two arguments");
    return 0;
  }
  if (tokc==3) {
    if (!IParseFormat(tokv[2], &width, &precision, &code)) {
      IError("bad format: \"%s\"", tokv[2]);
      return 0;
    }
    ISetFormat(tokv[1], width, precision, code);
  } else {
    if (format = IGetFormat(tokv[1]))
      fprintf(dout, "The format of \"%s\" is \"%s\"\n", tokv[1], format);
  }
  return 1;
}

int command_default(tokc, tokv)
int tokc;
char *tokv[];
{
  char *val;
  if (GiveHelp(tokc)) {
    IUsage("<object> <default-value>");
    ISynopsis("specify the default value of a structure field");
    IHelp(IHelpArgs,
	  NULL);
    return 1;
  }
  
  if (tokc<2 || tokc>3) {
    IExpected("one or two arguments");
    return 0;
  }
  if (tokc==3) {
    IAddDefault(tokv[1], tokv[2]);
  } else {
    if (val = IGetDefault(tokv[1]))
      fprintf(dout, "The default for %s is \"%s\"\n", tokv[1], val);
  }
  return 1;
}

char *IGetValue(in_name, format)
char *in_name;
char *format;
{
  char *name = itf_name_buffer;
  int take_address;
  struct ADDRESS_ENV env;
  if (!in_name || !*in_name) {
    itf_value_buffer[0]=0;
    itf_errno = NULL_VARNAME;
    return itf_value_buffer;
  }
  MASSAGE_VAR_NAME(in_name, name);
  if (   IParseObject(name, &env, &take_address)
      && IEnvIsSimpleSetError(&env))
    return IGetSimpleValue(&env, take_address, format, 0);
  if (true(unset_error)) {
    if (!IHaveCulprit()) 
      IError("%s", name);
  } else
    IResetError();
  itf_value_buffer[0]=0;
  return itf_value_buffer;
}

/***
 * IGetValueCompositeName(va_alist)
 * Builds a variable name by concatenating a list of strings,
 * and returns the value, if the variable is defined.  If the
 * variable is undefined, it returns NULL.  Any itf_errors
 * occurring are cleared.
 */
/*VARARGS*/
char *IGetValueCompositeName(va_alist)
va_dcl
{
  va_list args;
  char *p, *pos;
  char *name, *value;
  int len = 0;
  /* calculate the total length of the name */
  va_start(args);
  for (;;) {
    p = va_arg(args, char *);
    if (p==NULL)
      break;
    len += strlen(p);
  }
  va_end(args);
  len++;
  name = ITempCalloc(char, len);
  /* and put all the components into name */
  name[0] = '\0';
  pos = name;
  va_start(args);
  for (;;) {
    p = va_arg(args, char *);
    if (p==NULL)
      break;
    pos = strcatend(pos, p);
  }
  va_end(args);
  value = IGetValue(name, NULL);
  if (itf_errno) {
    IResetError();
    return NULL;
  }
  return value;
}

/***
 * IGetObjectPtr - return a pointer to the named structure
 */
generic *IGetObjectPtr(object_type_name, in_name)
char *object_type_name, *in_name;
{
  int object_type, size, take_address;
  char *name = itf_name_buffer;
  struct ADDRESS_ENV env;
  if (!in_name || !*in_name) {
    itf_value_buffer[0]=0;
    itf_errno = NULL_VARNAME;
    return NULL;
  }
  object_type = IGetType(object_type_name, &size);
  MASSAGE_VAR_NAME(in_name, name);
  if (!IParseObject(name, &env, &take_address)) return NULL;
  while (IEnvIsPointer(&env)) IMoveToPointer(&env);
  if (IEnvIsArray(&env)) {
    itf_errno = NEED_ARRAY_INDEX;
    IError("\"%s\"", name);
    return NULL;
  }
  /* no need for this restriction!
  if (!IEnvIsStruct(&env)) {
    itf_errno = OBJECT_SYNTAX;
    IError("\"%s\" is not a structre", name);
    return NULL;
  }
  */
  if (IEnvIsNull(&env)) {
    return NULL;
  }
  if (env.ilevel!=1) {
    panic("IGetObjectPtr", "ilevel!=1 (%d)", env.ilevel);
  }
  if (env.binding->var_type != object_type) {
    itf_errno = BAD_VARTYPE;
    IError("\"%s\" is not of type \"%s\"", name, object_type_name);
    return NULL;
  }
  return env.poi.ptr;
}

/***
 * IDeleteObject - deletes all record of an object, and returns a
 * pointer to it (so it can be freed).  Refuses to do so for
 * a component.
 */
generic *IDeleteObject(in_name)
char *in_name;
{
  int tokc;
  char *name = itf_name_buffer;
  generic *ptr;
  TBL_ITEM *item;
  LEX_STACK stack;
  if (!in_name || !*in_name) {
    itf_value_buffer[0]=0;
    itf_errno = NULL_VARNAME;
    return NULL;
  }
  MASSAGE_VAR_NAME(in_name, name);
  (void) LexAnalysePush(object_syntax, name, &tokc, &stack);
  if (tokc!=1) {
    itf_errno = OBJECT_SYNTAX;
    IError("IDeleteObject: cannot delete an object member: \"%s\"", name);
    return NULL;
  }
  item = ITableLookup(itf_table, name, itf_obj_context);
  if (!item) {
    itf_errno = UNKNOWN_VAR;
    IError("\"%s\"", name);
    return NULL;
  }
  if (item->data.binding->ilevel > 1) {
    itf_errno = BAD_VARTYPE;
    IError("IDeleteObject: object indirection level > 1: \"%s\"", name);
    return NULL;
  }
  if (item->data.binding->ilevel == 1)
    ptr = item->data.binding->value.poi.ptr;
  else
    ptr = NULL;
  if (item->data.binding->default_val!=NULL)
		free(item->data.binding->default_val);
  free(item->data.binding);
  ITableDelete(itf_table, name, itf_obj_context);
  return ptr;
}

/*
 * put the value of something into a string
 *
 * don't damage the binding inside address
 * this function assumes that the poi in address is a pointer to the object
 */
char *IGetSimpleValue(env, take_address, format, quote)
struct ADDRESS_ENV *env;
int take_address;
char *format;
int quote; /* true if strings should be quoted */
{
  char *value=itf_value_buffer;
  *value=0;
  assert(NULL!=env->binding);
  if (take_address) {
    if (env->poi.ptr!=NULL)
      sprintf(value=itf_value_buffer, "0x%x", env->poi.ptr);
    else value = "0x0000";
  } else if ((env->binding->object_prop & IS_STRUCT)
	     || (env->ilevel!=1)) {
    itf_errno=NEED_FIELD_OR_INDEX;
    IError("\"%s\"", env->binding->tbl_entry->name);
  } else {
    int code, width, prec;
    if (format==NULL) {
      code = env->binding->format_code;
      width = env->binding->format_width;
      prec = env->binding->format_prec;
    } else {
      (void) IParseFormat(format, &width, &prec, &code);
    }
    if (env->poi.ptr==NULL)
      value = "<NULL>";
    else {
      if (env->binding->var_type==ITF_STRING && quote) *(value++)='"';
      if (!IPrintPoi(value, VALUE_BUFFER, &(env->poi),
		     env->binding->var_type, width, prec, code)) {
	itf_errno = BAD_VARTYPE;
	IError("binding name \"%s\" type %d",
		env->binding->tbl_entry->name, env->binding->var_type);
      }
      if (env->binding->var_type==ITF_STRING && quote) strcat(value--, "\"");
    }
  }
  return value;
}

int INPercent(str)
char *str;
{
  int i = 0;
  int prev = 0;
  if (str==NULL) return 0;
  for (;*str;str++) {
    /* don't count double %'s */
    if (*str=='%') {
      i++;
      if (prev) {
	i--;
	prev = 0;
      } else
	prev = 1;
    } else
      prev = 0;
  }
  return i;
}

char *IPrintRealFormatted(r, format)
double r;
char *format;
{
  int width, prec, code;
  if (format==NULL) {
    width = 0;
    prec = 0;
    code = 'g';
    format = "%g";
  } else if (!IParseFormat(format, &width, &prec, &code) || INPercent(format)>1){
    IError("bad format \"%s\"", format);
    return NULL;
  }
  if (strchr(legal_format_codes[ITF_REAL], code)) {
    sprintf(itf_value_buffer, format, r);
    return itf_value_buffer;
  } else {
    IError("bad format \"%s\" for Real", format);
    return NULL;
  }
}

char *IPrintStringFormatted(s, format)
char *s, *format;
{
  int width, prec, code;
  int i;
  double d;
  if (format==NULL) {
    width = 0;
    prec = 0;
    code = 's';
    format = "%s";
  } else if (!IParseFormat(format, &width, &prec, &code) || INPercent(format)>1){
    IError("bad format \"%s\"", format);
    return NULL;
  }
  if (strchr(legal_format_codes[ITF_DOUBLE], code)) {
    d = atof(s);
    sprintf(itf_value_buffer, format, d);
  } else if (strchr(legal_format_codes[ITF_LONG], code)) {
    i = atoi(s);
    sprintf(itf_value_buffer, format, i);
  } else if (strchr(legal_format_codes[ITF_STRING], code)) {
    sprintf(itf_value_buffer, format, s);
  } else {
    IError("bad format \"%s\" for string", format);
    return NULL;
  }
  return itf_value_buffer;
}

static int	printValues(name, format, 
			    value_printed, print_space, leading_space)
  char		*name ;
  char		*format ;
  int		value_printed ;
  int		print_space ;
  int		leading_space ;
{
  char		*value ;
  char		preString[132], postString[132] ;
  int		upper, lower ;

#ifdef _NO_LIBPW
  value = IGetValue(name, format);
  if (itf_errno && itf_errno!=ERROR_REPORTED) {
    if (value_printed)
      fputc('\n', dout);
    IReportError(dout);
    IResetError();
    itf_errno = ERROR_REPORTED;
    value_printed = 0;
  } else {
    if (   (value_printed && print_space)
	|| (!value_printed && leading_space))
      fputc(' ', dout);
    fputs(value, dout);
    value_printed++;
  }
#else
  if (getRange(name, preString, postString, &lower, &upper) != 0) {
    char	newName[BUFSIZ] ;
    while (lower <= upper) {
      sprintf(newName, "%s[%d]%s", preString, lower++, postString) ;
      value_printed = printValues(newName, format, 
				  value_printed, print_space, leading_space) ;
    }
  } else {
    value = IGetValue(name, format);
    if (itf_errno && itf_errno!=ERROR_REPORTED) {
      if (value_printed)
	fputc('\n', dout);
      IReportError(dout);
      IResetError();
      itf_errno = ERROR_REPORTED;
      value_printed = 0;
    } else {
      if (   (value_printed && print_space)
	  || (!value_printed && leading_space))
	fputc(' ', dout);
      fputs(value, dout);
      value_printed++;
    }
  }
#endif
  return value_printed ;
}

int	command_print(tokc, tokv)
  int	tokc;
  char	*tokv[];
{
  char *s, *format, *name;
  int value_printed=0;
  int print_space=1;
  int print_nl=1;
  int leading_space = 0;
  if (GiveHelp(tokc)) {
    IUsage("[-/+s] [-ls] [-n] <varname> [format] <varname> [format] ...");
    ISynopsis("print out the values of the given simple objects");
    IHelp
      (IHelpArgs,
       "The format  string is optional.    The  available  formats  are very",
       "similar to those of printf().  More description of formats is in the",
       "help for the command \"format\".",
       "",
       "Structures and  arrays  cannot  be  printed as a  whole.  Individual",
       "fields  in a structure, or array  elements  may  be  printed.  Array",
       "indices may contain a range (e.g. [0-3], [*]) as used by  the \"show\"",
       "command.",
       "",
       "If the option '-s' ('+s') is given, no (one) intervening spaces will",
       "be  printed.  '-ls' implies a  leading space, and  '-n' means do not",
       "print a new line.",
       "",
       "Use `show' to see the values of complex (struct and array) objects.",
       "SEE ALSO",
       "show, format",
       NULL);
    return 1;
  }

  if (tokc>1) {
    ++tokv;
    while (*tokv) {
      if (LexQuotedTok(*tokv) || IIsNumber(*tokv)) {
	/* either a number (eg. "4"), or a string constant */
	name = tokv[0];
	/* let format strings be quoted */
	if (tokv[1] && /* !LexQuotedTok(tokv[1]) && */ tokv[1][0]=='%')
	  format = *++tokv;
	else format = "%s";
	if (s=IPrintStringFormatted(name, format)) {
	  if (   (value_printed && print_space)
	      || (!value_printed && leading_space))
	    fputc(' ', dout);
	  fputs(s, dout);
	  value_printed++;
	} else {
	  if (value_printed) fputc('\n', dout);
	  IReportError(dout);
	  IResetError();
	  itf_errno = ERROR_REPORTED;
	  value_printed=0;
	}
      } else if (!strcmp(*tokv, "-s")) {
	print_space=0;
      } else if (!strcmp(*tokv, "-ls")) {
	leading_space = 1;
      } else if (!strcmp(*tokv, "-n")) {
	print_nl=0;
      } else if (!strcmp(*tokv, "+s")) {
	print_space=1;
      } else if (!strcmp(*tokv, ", ")) {
	fputc('\n', dout);
      } else {
	name = *tokv;
	if (tokv[1] && /* !LexQuotedTok(tokv[1]) && */ tokv[1][0]=='%')
	  format = *++tokv;
	else format = NULL;
	value_printed = printValues(name, format, 
				    value_printed, print_space, leading_space);
      }
      tokv++;
    }
    if (value_printed && print_nl) fputc('\n', dout);
  }
  return 1;
}

int command_set(tokc, tokv)
int tokc;
char *tokv[];
{
  char *val, *name;
  int i, vector;
  Boolean	errorOccured = FALSE;
  if (GiveHelp(tokc)) {
    IUsage("<varname1>=<value2> <varname2>=<value2> ...");
    ISynopsis("set the value of a variable");
    IHelp
      (IHelpArgs,
       "set               : prints out the values of system variables",
       "set VAR = value   : sets scalar var to have this value",
       "set VAR = {value1, value2, ...} : sets vector var to these values",
       "set VAR   : sets string var to \"true\"",
       "unset VAR : sets string var to \"false\"",
	  NULL);
    return 1;
  }
  
  if (tokc==1)
    IShowVarValues(dout, ITF_SYSTEM, 0);
  else {
    tokv++;
    while (*tokv) {
      name = *tokv;
      vector = 0;
      if (tokv[1] && !strcmp(tokv[1], "=") && !LexQuotedTok(tokv[1])) {
	val = tokv[2];
	if (!val) {
	  itf_errno = SYNTAX_ERROR;
	  IExpected("value after \"=\"");
	  return 0;
	}
	if (!LexQuotedTok(tokv[2]) && !strcmp("{", tokv[2]))
	  vector = 1;
	tokv+=3;
      } else {
	val = "true";
	++tokv;
      }
      if (vector) {
	char	vectorName[BUFSIZ] ;
	i = 0;
	while (*tokv && (LexQuotedTok(*tokv)||strcmp("}", *tokv))) {
	  if (itf_errno == 0) {
	    sprintf(vectorName, "%s[%d]", name, i++) ;
	    ISetValue(vectorName, *tokv) ;
	  }
	  tokv++;
	  /* optional comma */
	  if (*tokv && !LexQuotedTok(*tokv) && !strcmp(",", *tokv)) tokv++;
	}
	if (*tokv==NULL) { /* didn't find "}" */
	  itf_errno = SYNTAX_ERROR;
	  IExpected("}");
	  return 0;
	}
	tokv++;
	if (itf_errno) {
	  errorOccured = TRUE;
	  IReportError(dout);
	  IResetError();
	}
      } else {
	if (!IIsObjectName(name) && IIsLegalVarName(name)
	    && IAutoCreateVar(name)) {
	  (void) IBindString(name, NULL, OBJECT, 0, NULL);
	  ISetSystem(name, 1);
	}
	(void) ISetValue(name, val);
      }
      if (itf_errno) {
	errorOccured = TRUE;
	IReportError(dout);
	IResetError();
      }
    }
  }
  if (errorOccured)
    return 0;
  else
    return 1;
}

int command_unset(tokc, tokv)
int tokc;
char *tokv[];
{
  Boolean	errorOccured = FALSE;
  if (GiveHelp(tokc)) {
    IUsage("<varname1> <varname2> ...");
    ISynopsis("set the value of variables to \"false\"");
    IHelp
      (IHelpArgs,
       "set VAR   : sets string var to \"true\"",
       "unset VAR : sets string var to \"false\"",
	  NULL);
    return 1;
  }
  
  while (*++tokv) {
    (void) ISetValue(tokv[0], "false");
    if (itf_errno) {
      errorOccured = TRUE;
      IReportError(dout);
      IResetError();
    }
  }
  if (errorOccured)
    return 0;
  else
    return 1;
}

int	IAutoCreateVar(in_name)
char	*in_name;
{
  char *name = itf_name_buffer;
  TBL_ITEM *item;
  char *suffix;
  MASSAGE_VAR_NAME(in_name, name);
  if (strlen(name)>8 && !strcmp("-options", name+strlen(name)-8)) {
    name[strlen(name)-8] = '\0';
    suffix = name + strlen(name) - 7;
  } else if (strlen(name)>9 && !strcmp("-automore", name+strlen(name)-9)) {
    name[strlen(name)-9] = '\0';
    suffix = name + strlen(name) - 8;
  } else
    return 0;
  if (ITableLookup(itf_func_table, name, itf_com_context)!=NULL)
    return 1;
  else {
    if ((item=IFindCommand(name, NULL))!=NULL)
      IError("try \"set %s-%s = ...\"\nThe command name must be correctly ordered", suffix, item->name);
    else
      IError("\"%s\" is not a command", name);
  }
  return 0;
}

/*
 * IParseFieldOrObject returns a binding for a field type or
 * for a variable (or structure)
 * This is the way to get at the features of objects and types.
 */
TBL_ITEM *IParseFieldOrObject(name)
char *name;
{
  LEX_STACK stack;
  TBL_ITEM *item;
  char **tokv;
  int tokc;
  tokv = LexAnalysePush(object_syntax, name, &tokc, &stack);
  if (!tokv) {
    itf_errno = NULL_VARNAME;
    goto return_null;
  }
  if ((item=ITableLookup(itf_table, tokv[0], itf_obj_context))!=NULL) {
    /***
     * First token is an object
     */
    if (tokc>1) {
      itf_errno = OBJECT_SYNTAX;
      IError("Fields must be specified with the structure type");
      goto return_null;
    }
  } else if ((item=ITableLookup(itf_table, tokv[0], itf_type_context))!=NULL) {
    /***
     * First token is a type - can allow fields
     */
    if (item->data.binding->var_type>=ITF_VAR_TYPES) {
      if (tokc==3 && !strcmp(".", tokv[1])) {
	if (!(item = ITableLookup(itf_table, tokv[2], item->data.binding->var_type))) {
	  itf_errno = UNKNOWN_FIELD;
	  IError("\"%s\"", tokv[2]);
	  goto return_null;
	}
      } else if (tokc!=1) {
	itf_errno = OBJECT_SYNTAX;
	IError("expected \".\"<fieldname>, found \"%s\"", tokv[1]);
	goto return_null;
      }
    } else if (tokc!=1) {
      itf_errno = OBJECT_EXCESS;
      IError("%s", tokv[2]);
      goto return_null;
    }
  } else {
    itf_errno = (tokc>1?UNKNOWN_STRUCT:UNKNOWN_VAR);
    IError("\"%s\"", tokv[0]);
    goto return_null;
  }
  LexAnalysePop(&stack);
  return item;

 return_null:
  LexAnalysePop(&stack);
  return NULL;
}

/***
 * Set the format in a field object binding
 * itf_errno is set if there are any problems
 */
void	ISetFormat(in_name, format_width, format_prec, format_code)
char	*in_name;
int	format_width;
int	format_prec;
int	format_code;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL) {
    if (item->context==itf_type_context) {
      IError("types cannot have formats");
      return;
    }
    binding = item->data.binding;
    if (IIsStruct(binding)) {
      IError("structures cannot have any print format");
      return;
    }
    if (binding->var_type<0 || binding->var_type>=ITF_VAR_TYPES) {
      warn("ISetFormat", "var_type out of range %d", binding->var_type);
      return;
    }
    if (!strchr(legal_format_codes[binding->var_type], format_code)) {
      IError("bad format code '%c'", format_code);
      return;
    }
    binding->format_width = format_width;
    binding->format_prec = format_prec;
    binding->format_code = format_code;
  } else {
    if (!itf_errno)
      IError("unknown object or type \"%s\"", name);
    return;
  }
}

char *IGetFormat(in_name)
char *in_name;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL) {
    if (item->context==itf_type_context) {
      IError("types cannot have formats");
      return 0;
    }
    binding = item->data.binding;
    return IMakeFormatStr(binding->var_type, binding->format_width,
			  binding->format_prec, binding->format_code);
  } else {
    if (!itf_errno)
      IError("unknown object or type \"%s\"", name);
    return NULL;
  }
}

int IParseFormat(format, width, precision, code)
char *format;
int *width, *precision, *code;
{
  char	c;
#if 1
  char	*next;
#endif

  /* find the first single '%' in the format string */
  for (;;) {
    format = strchr(format, '%');
    if (format==NULL || !(format[0]=='%' && format[1]=='%')) break;
    format += 2;
  }
  *width = *code = 0;
  *precision = -1;
  if (format==NULL) return 0;
#if 0
  if (   sscanf(format, "%%%d.%d%c", width, precision, &c)==3
      || sscanf(format, "%%%d%c", width, &c)==2
      || sscanf(format, "%%.%d%c", precision, &c)==2
      || sscanf(format, "%%%c", &c)==1) {
    *code = c;
    return 1;
  } else
    return 0;
#else
  if (*format != '%')
    return 0 ;

  /* skip %, then look for width */
  *width = strtol(++format, &next, 0) ;

  if (next != format)		/* got a width */
    format = next ;		/* skip ahead  */

  if (*format == '.') {		/* try for a precision */
    *precision = strtol(++format, &next, 0) ;
    if (next == format)	{	/* precision failed */
      *precision = -1 ;
      return 0 ;
    }
    format = next ;
  } 

  if (*format == '\0')
    return 0 ;

  *code = *format ;

  return 1 ;
#endif
}

int command_export(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("<var1> <var2> ...");
    ISynopsis("mark a variable to be exported into the environment");
    IHelp
      (IHelpArgs,
       "The  variables given  are put in  the environment (and are passed to",
       "subprograms).  If the variable does  not exist in the simulator, but",
       "does  already exist in the  environment,  then it is created  in the",
       "simulator and its  value is copied from  the environment.  Each time",
       "the value   is  changed in  the simulator,   the copy of it   in the",
       "environment is also changed.",
       "SEE ALSO",
       "var, set, show",
       NULL);
    return 1;
  }
  for (tokv++;*tokv;tokv++) 
    if (INewExport(*tokv) == 0)
      return 0;
  return 1;
}

int command_visible(tokc, tokv)
int tokc;
char *tokv[];
{
  int on = 0;
  IUsage("[+/-] <struct-name>.<field-name>");
  if (GiveHelp(tokc)) {
    ISynopsis("set a structure field to be visible or invisible");
    IHelp
      (IHelpArgs,
       "This command  sets  the value  of  the visibility flag for fields of",
       "structures.  If the visibility flag is off for a particular field in",
       "a structure type, then when object  of that structure  type is shown",
       "using \"show\", that field will not appear.",
       "",
       "This is the same flag that is turned off  by having the \"invisible:\"",
       "flag in a comment after a structure field in a header file.",
       "EXAMPLE",
       "visible NetExtensionRec.epsilon      # make epsilon visible",
       "visible + NetExtensionRec.epsilon    # make epsilon visible",
       "visible - NetExtensionRec.epsilon    # make epsilon invisible",
       NULL);
    return 1;
  }
  if (tokc<1 || tokc>3
      || (tokc==3&&strcmp(tokv[1], "-")&&strcmp(tokv[1], "+")))
    IErrorAbort(IPrintUsage(tokv[0], usage));
  tokv++;
  if (tokc == 2 || (tokc==3 && *tokv[0] != '-'))
    on = 1;
  else
    on = 0 ;

  if (tokc==3) tokv++;
  return ISetVisible(tokv[0], on);
}

int	INewExport(name)
char	*name;
{
  int new;
  if (!IIsLegalVarName(name)) {
    itf_errno = ILLEGAL_VARNAME;
    IError("%s", name);
    return 0;
  }
  if (!IIsObjectName(name)) {
    (void) IBindString(name, NULL, OBJECT, 0, NULL);
    if (getenv(name)) ISetValue(name, getenv(name));
    new = 1;
  } else
    new = 0;
  ISetExport(name, 1);
  if (!new) IExport(name, IGetValue(name, NULL));
  return 1;
}

int command_reset(tokc, tokv)
int tokc;
char *tokv[];
{
  if (GiveHelp(tokc)) {
    IUsage("<structure-var>");
    ISynopsis("reset the fields of a structure to their defaults");
    IHelp(IHelpArgs, NULL);
    return 1;
  }
  
  while (*++tokv)
    if (IResetObject(*tokv) == 0)
      return 0 ;
  return 1 ;
}

int	IResetObject(in_name)
char	*in_name;
{
  ADDRESS_ENV env;
  int take_address;
  int r;
  char *name = itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  r = IParseObject(name, &env, &take_address);
  if (!r) return 0;
  ISetEnvDefaults(&env);
  return 1;
}

/***
 * IParseIndex
 * returns a status code, and the index value in a parameter
 */
int IParseIndex(index_valuep, tokvp)
int *index_valuep;
char ***tokvp;
{
  int index_value;
  char *value, *culprit;
  ADDRESS_ENV index_env;
  if (!**tokvp || strcmp(**tokvp, "[")) {
    itf_errno = OBJECT_SYNTAX;
    IExpectedToken("[", **tokvp, NULL);
    return 0;
  }
  (*tokvp)++;
  culprit = **tokvp;
  if (IIsInteger(**tokvp)) {
    index_value = atoi(**tokvp);
    (*tokvp)++;
  } else {
    if (!IParseObjectTokv(&index_env, tokvp)) {
      sprintf(itf_tmp_buf, " while parsing array index beginning \"%s\"", culprit);
      strcat(itf_error, itf_tmp_buf); /* append this to the culprit */
      return 0;
    }
    if (!(value=IGetSimpleValue(&index_env, 0, "%d", 0))) {
      sprintf(itf_tmp_buf, " in array index beginning \"%s\"", culprit);
      strcat(itf_error, itf_tmp_buf); /* append this to the culprit */
      return 0;
    }
    if (!IIsInteger(value)) {
      itf_errno = OBJECT_SYNTAX;
      IError("non-integer \"%s\" used as array index, var name begins \"%s\"",
	     value, culprit);
      return 0;
    }
    index_value = atoi(value);
  }
  if (!**tokvp || strcmp(**tokvp, "]")) {
    itf_errno = OBJECT_SYNTAX;
    IExpectedToken("]", **tokvp, NULL);
    return 0;
  }
  (*tokvp)++;
  *index_valuep = index_value;
  return 1;
}

/***
 * returns the id of the field (it's id in the hash table - 0 for error
 */
int IParseField(parent, field_id, tokvp)
ADDRESS_ENV *parent;
int *field_id;
char ***tokvp;
{
  TBL_ITEM *item;
  if (!**tokvp || strcmp(**tokvp, ".")) {
    itf_errno = OBJECT_SYNTAX;
    IExpectedToken(".", **tokvp, NULL);
    return 0;
  }
  (*tokvp)++;
  if (!**tokvp) {
    itf_errno = OBJECT_SYNTAX;
    IExpectedToken("<field-name>", **tokvp, ".");
    return 0;
  }
  item = ITableLookup(itf_table, **tokvp, parent->binding->var_type);
  if (item==NULL) {
    itf_errno = UNKNOWN_FIELD;
    IError("\"%s\"", **tokvp);
    return 0;
  }
  (*tokvp)++;
  *field_id = item->id;
  return 1;
}

/***
 * IEnvIsArray - the object indicated by env is an array
 */
int IEnvIsArray(env)
ADDRESS_ENV *env;
{
  int r =
    (   env->ilevel>=1
     && (env->binding->itype[env->ilevel] & ENV_IS_ARRAY)
     && !env->have_index[env->ilevel]);
  return r;
}

/***
 * IEnvIsPointer - the object indicated by env is a pointer to some object
 */
int IEnvIsPointer(env)
ADDRESS_ENV *env;
{
  int r =
    (env->ilevel>1
     && (   (env->binding->itype[env->ilevel] & ENV_IS_POINTER)
	 || ((env->binding->itype[env->ilevel] & ENV_IS_ARRAY)
	     && env->have_index[env->ilevel])));
  return r;
}

/***
 * IEnvIsStruct - the object indicated by env is a structure
 */
int IEnvIsStruct(env)
ADDRESS_ENV *env;
{
  int r = (env->binding->object_prop & IS_STRUCT);
  return r!=0;
}

/***
 * IMoveToElement - env did indicate an array - move to an element
 * doesn't change indirection level
 * The parent environment is necessary in case the array bounds are
 * another field in the structure.
 * Returns 1 for a successful move.
 */
int IMoveToElement(this, index_value)
ADDRESS_ENV *this;
int index_value;
{
  if (IEnvIsNull(this)) return 0;
  if (!IWithinBounds(this, index_value)) return 0;
  if (this->ilevel>1) {
    /* Array of pointers */
    if (this->poi.ptr==NULL) {
      itf_errno = NULL_POINTER;
      IError("%s", this->name);
      return 0;
    }
    this->poi.pptr += index_value;
  } else {
    /* Array of objects. */
    assert(this->binding->element_size >= 0);
    this->poi.ptr += index_value * (this->binding->element_size);
  }
  this->have_index[this->ilevel]=1;
  return 1;
}

/***
 * IMoveToField - env did indicate a structure - move to one of it's fields
 * swap -
 *   parent env is assumed to be trash, and field is the structure
 *   set parent to be the old field env, and field to be the new field
 * noswap -
 *   old content of field env is assumed to be trash,
 *   and parent is the structure.
 *   keep parent as it is, set field to be the new field
 * returns 1 for successful move
 */
int IMoveToField(parent, field, field_id, swap)
ADDRESS_ENV *parent;
ADDRESS_ENV *field;
int field_id;
int swap;
{
  TBL_ITEM *item;
  int i;
  item = ITableLookupById(itf_table, field_id);
  if (item==NULL) panic("IMoveToField", "bad field id %d", field_id);
  if (swap==SWAP) {
    parent->binding	= field->binding;
    parent->poi.ptr	= field->poi.ptr;
    for (i=0;i<=MAX_INDIRECTION;i++)
      parent->have_index[i]	= field->have_index[i];
    parent->ilevel	= field->ilevel;
    parent->name	= field->name;
    parent->parent_binding = NULL;
    parent->parent_poi.ptr = NULL;
  }
  if (IEnvIsNull(parent)) return 0;
  field->parent_binding	= parent->binding;
  field->parent_poi.ptr	= parent->poi.ptr;
  field->binding	= item->data.binding;
  /* move to the field within the structure */
  field->poi.ptr	= parent->poi.ptr + field->binding->value.poi.longint;
  field->name		= field->binding->tbl_entry->name;
  field->ilevel	= field->binding->ilevel;
  for (i=0;i<=MAX_INDIRECTION;i++)
    field->have_index[i] = 0;
  if (field->binding->object_prop & IS_UNION) {
    panic("Union fields NYI");
  }
  return 1;
}

/***
 * IMoveToPointer - env indicates a pointer - dereference that pointer
 */
int IMoveToPointer(this)
ADDRESS_ENV *this;
{
  while (IEnvIsPointer(this)) {
    if (IEnvIsNull(this)) return 0;
    this->poi.ptr = *(this->poi.pptr);
    this->ilevel--;
  }
  return 1;
}

/***
 * IEnvIsNull - returns 0 if non-null
 */
int IEnvIsNull(this)
ADDRESS_ENV *this;
{
  if (this->poi.ptr==NULL) {
    itf_errno = NULL_POINTER;
    IError("\"%s\" (ilevel=%d)", this->name, this->ilevel);
    return 1;
  } else
    return 0;
}

/***
 * IGetArrayBound - get the upper bound on the array
 */
int IGetArrayBound(this)
ADDRESS_ENV *this;
{
  TBL_ITEM *item;
  char *val;
  struct ADDRESS_ENV bound;
  long int bound_value;
  if (this->binding->itype[this->ilevel] & FIXED_ARRAY) {
    bound_value = this->binding->array_count[this->ilevel];
  } else {
    /*
     * variable size array - find the counter - check it is the right type
     */
    item = ITableLookupById(itf_table, this->binding->array_count[this->ilevel]);
    if (!item) {
      warn("IGetArrayBound", "can't find counter %d",
	     this->binding->array_count[this->ilevel]);
      return -1;
    }
    bound.binding = item->data.binding;
    bound.ilevel = bound.binding->ilevel;
    bound.name = bound.binding->tbl_entry->name;
    if ((  (this->binding->object_prop&IS_FIELD)
	 !=(bound.binding->object_prop&IS_FIELD))
	|| !(bound.binding->object_prop&IS_COUNTER) || (bound.ilevel>1)) {
      panic("WithinBounds", "type mismatch for counter %s 0x%X %s 0x%X %d",
	     this->binding->tbl_entry->name, this->binding->object_prop,
	     bound.binding->tbl_entry->name, bound.binding->object_prop,
	     bound.ilevel);
      return -1;
    }
    assert(bound.binding->object_prop&IS_SIMPLE);
    if (this->binding->object_prop&IS_FIELD) {
      assert(bound.ilevel==1);
      assert(this->parent_binding->object_prop&IS_STRUCT);
      bound.poi.ptr = this->parent_poi.ptr + bound.binding->value.poi.longint;
    } else {
      assert(bound.ilevel<=1);
      if (bound.ilevel==0)
	bound.poi.pptr = &(bound.binding->value.poi.ptr);
      else
	bound.poi.ptr = bound.binding->value.poi.ptr;
    }
    val = IGetSimpleValue(&bound, 0, "%d", 0);
    if (!val) {
      warn("IGetArrayBound", "couldn't get simple value for %s", bound.name);
      return -1;
    }
    bound_value = atol(val);
  }
  return bound_value;
}

/***
 * IWithinBounds - this indicates an array - is the index val OK?
 */
int IWithinBounds(this, index_value)
ADDRESS_ENV *this;
int index_value;
{
  int bound_value;
  bound_value = IGetArrayBound(this);
  if (bound_value<0) return 0;
  if (index_value>=0 && index_value<bound_value)
    return 1;
  else {
    itf_errno = ARRAY_BOUNDS;
    IError("index=%d in array \"%s\", bound=%d",
	    index_value, this->name, bound_value);
    return 0;
  }
}

int		ISetEnvFromItem(env, item)
ADDRESS_ENV	*env;
TBL_ITEM	*item;
{
  int i;
  env->binding = item->data.binding;
  env->poi.ptr = env->binding->value.poi.ptr;
  env->ilevel = env->binding->ilevel;
  for (i=0;i<=MAX_INDIRECTION;i++)
    env->have_index[i] = 0;
  env->name = item->name;
  env->parent_binding = NULL;
  env->parent_poi.ptr = NULL;
  return 1;
}

int		IIncIndirection(this)
ADDRESS_ENV	*this;
{
  if (this->ilevel>0) return 0;
  switch (this->binding->var_type) {
  case ITF_STRING:
  case ITF_REAL:
  case ITF_LONG:
    this->poi.pptr = &(this->binding->value.poi.ptr);
    break;
  default:
    panic("IIncIndirection", "bad object type %d for ilevel=0", this->binding->var_type);
  }
  this->ilevel=1;
  return this->ilevel;
}

/***
 * IParseObjectTokv - parse some object that has been split into tokens
 */
int IParseObjectTokv(result, tokvp)
ADDRESS_ENV *result;
char ***tokvp;
{
  TBL_ITEM *item;
  struct ADDRESS_ENV *this;

  if (!**tokvp) {
    itf_errno = EMPTY_OBJECT;
    return 0;
  }

  /***
   * lookup the first token in the table of vars
   */
  item = ITableLookup(itf_table, **tokvp, itf_obj_context);
  if (!item) {
    itf_errno = UNKNOWN_VAR;
    IError("\"%s\"", **tokvp);
    return 0;
  }

  this = result;
  ISetEnvFromItem(this, item);
  (*tokvp)++;

  return IContParseObjectTokv(this, tokvp);
}

int IContParseObjectTokv(this, tokvp)
ADDRESS_ENV *this;
char ***tokvp;
{
  struct ADDRESS_ENV e1, *parent;
  int index_value, field_id;
  parent = &e1;

  if (this->ilevel<1) { /* value is stored within POI */
    IIncIndirection(this);
  } else {
    while (**tokvp && this->ilevel>=1) {
      /***
       * Try for array indices and pointer refs
       */
      while (IEnvIsPointer(this) || (**tokvp && IEnvIsArray(this))) {
	if (IEnvIsPointer(this)) {
	  /***
	   * Only automatic following of pointers at the moment TODO: fix?
	   */
	  if (!IMoveToPointer(this)) return 0;
	} else {
	  /***
	   *     ARRAY    look for array index: "[" "index-expression" "]"
	   */
	  if (!IParseIndex(&index_value, tokvp)) return 0;
	  if (!IMoveToElement(this, index_value)) return 0;
	}
      }
      if (this->ilevel>1 || !IEnvIsStruct(this))
	break; /* can't have any fields */
      if (**tokvp && IEnvIsStruct(this)) {
	/***
	 *     STRUCTURE look for field: "." "field-name"
	 */
	if (!IParseField(this, &field_id, tokvp)) return 0;
	if (!IMoveToField(parent, this, field_id, SWAP)) return 0;
      }
    }
  }
  return 1;
}

int IParseObject(name, env, take_address)
char *name;
struct ADDRESS_ENV *env;
int *take_address;
{
  int tokc, r;
  char **tokv;
  LEX_STACK stack;
  r = 1;
  tokv = LexAnalysePush(object_syntax, name, &tokc, &stack);
  if (tokv && *tokv) {
    if (!strcmp("&", tokv[0])) {
      if (take_address!=NULL) *take_address = 1;
      tokv++;
      if (!tokv || !*tokv) {
	itf_errno = NULL_VARNAME;
	IError("");
	r = 0;
      }
    } else
      if (take_address!=NULL) *take_address = 0;
    if (r) r = IParseObjectTokv(env, &tokv);
    if (*tokv && !itf_errno && r) {
      itf_errno = OBJECT_EXCESS;
      IError("%s", *tokv);
      r = 0;
    }
  } else if (!tokv || !*tokv) {
    itf_errno = NULL_VARNAME;
    IError("");
    r = 0;
  }
  LexAnalysePop(&stack);
  return r;
}

int IContParseObject(name, env, take_address)
char *name;
struct ADDRESS_ENV *env;
int *take_address;
{
  int tokc, r;
  char **tokv;
  LEX_STACK stack;
  r = 1;
  tokv = LexAnalysePush(object_syntax, name, &tokc, &stack);
  if (tokv && *tokv) {
    if (!strcmp("&", tokv[0])) {
      if (take_address!=NULL) *take_address = 1;
      tokv++;
      if (!tokv || !*tokv) {
	itf_errno = NULL_VARNAME;
	IError("");
	r = 0;
      }
    } else
      if (take_address!=NULL) *take_address = 0;
    r = IContParseObjectTokv(env, &tokv);
    if (*tokv && !itf_errno && r) {
      itf_errno = OBJECT_EXCESS;
      IError("%s", *tokv);
      r = 0;
    }
  } else {
    itf_errno = NULL_VARNAME;
    IError("");
    r = 0;
  }
  LexAnalysePop(&stack);
  return r;
}

char	**IListStructFields(in_name,invisible)
  char	*in_name ;
  int	invisible;
{
  String	name = itf_name_buffer;
  String	*list ;
  int type=0;
  ADDRESS_ENV	env, fieldEnv ;
  int		take_address ;
  TBL_ITEM 	*item ;
  int 		idx, fieldId ;

  MASSAGE_VAR_NAME(in_name, name) ;
  if (item = IParseFieldOrObject(name)) {
    if (!IIsStructType(item->data.binding->var_type))
      return NULL ;
  } else
    return NULL ;

  type = item->data.binding->var_type ;

  /* count the number of fields and allocate memory */
  fieldId = 0 ;
  idx = 0 ;
  while ((item = ITableNextByContext(itf_table, &fieldId, type)) != NULL){
    if (invisible || item->data.binding->features&ITF_VISIBLE)
      ++idx ;
  }
  list = (String *)calloc(idx + 1, sizeof(String)) ;


  fieldId = 0 ;
  idx=0;
  while ((item = ITableNextByContext(itf_table, &fieldId, type)) != NULL)
    if (invisible || item->data.binding->features&ITF_VISIBLE)
      list[idx++] = strdup(item->name) ;
  list[idx] = NULL ;

  return list ;
}

/***
 * IShowObject(name, style)
 * Display an object - either simple, struct, or array
 */
void IShowObject(in_name, style)
char *in_name;
enum SHOW_STYLE style;
{
  ADDRESS_ENV env;
  int take_address;
  char *name = itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (!IParseObject(name, &env, &take_address)) return;
  IShowObjectEnv(name, &env, style, 0);
}

/***
 * IShowObjectEnv(name, env, style, indent)
 */
int	IShowObjectEnv(name, env, style, indent)
char	*name;
ADDRESS_ENV *env;
enum SHOW_STYLE style;
int indent;
{
  if (!IMoveToPointer(env) && style==SHOW_SAVE) return 0;
  if (IEnvIsSimple(env) || (style==SHOW_BRIEF && !IEnvIsSimpleArray(env))) {
    if (style==SHOW_BRIEF) style = SHOW_DISPLAY;
    IShowSimpleObject(name, env, style, indent);
  } else if (IEnvIsArray(env))
    IShowArrayObject(name, env, style, indent, 0);
  else if (IEnvIsStruct(env))
    IShowStructObject(name, env, style, indent);
  return 1;
}

/***
 * IShowSimpleObject
 */
int	IShowSimpleObject(name, env, style, indent)
char *name;
ADDRESS_ENV *env;
enum SHOW_STYLE style;
int indent;
{

  if (!IMoveToPointer(env) && style==SHOW_SAVE) return 0;

  if (IEnvIsArray(env)) {
    assert(style!=SHOW_SAVE);
    fprintf(dout, "%*s%*s = array %s of %s", indent, "", name_width, name,
	    IArrayCounter(env->binding),
	    (IEnvIsStruct(env) ? "struct " : ""),
	    IGetTypeName(env->binding->var_type));
    fprintf(dout, "%s\n", IGetSimpleValue(env, 1, NULL, 0));
  } else if (IEnvIsStruct(env)) {
    assert(style!=SHOW_SAVE);
    fprintf(dout, "%*s%*s = struct %s %s\n", indent, "", name_width, name,
	    IGetTypeName(env->binding->var_type),
	    IGetSimpleValue(env, 1, NULL, 0));
  } else {
    if (style==SHOW_SAVE) {
      /***
       * Try to stop it from printing <NULL> when style==SHOW_SAVE - it
       * really mucks things back up on re-reading back in
       */
      if (!IEnvIsNull(env)) {
	if (env->binding->var_type==ITF_STRING) { /* escape any quotes */
	  fprintf(dout, "set %s = ", name);
	  IPutQuotedString(IGetSimpleValue(env, 0, NULL, 0), '"', dout);
	  fputc('\n', dout);
	} else
	  fprintf(dout, "set %s = %s\n", name, IGetSimpleValue(env, 0, NULL, 0));
      }
    } else
      fprintf(dout, "%*s%*s = %s\n", indent, "", name_width, name,
	      IGetSimpleValue(env, 0, NULL, 1));
  }
  return 1;
}

int	IPutQuotedString(str, quote, out)
char *str;
int  quote;
FILE *out;
{
  fputc(quote, out);
  if (str!=NULL)
    for (;*str;str++) {
      if (*str==quote) fputc('\\', out);
      if (*str=='\\') fputc('\\', out);
      if (*str=='\n') fputc('\\', out);
      fputc(*str, out);
    }
  fputc(quote, out);
  return 1;
}

/***
 * IShowArrayObject
 */
int	IShowArrayObject(name, env, style, indent, subarray)
char *name;
ADDRESS_ENV *env;
enum SHOW_STYLE style;
int indent, subarray;
{
  ADDRESS_ENV e, *elt_env;
  int show_on_line = 1;
  int bound, j;
  int subarray_newline, subarray_indent;
  char elt_name[NAMEBUF];
  elt_env = &e;
  bound = IGetArrayBound(env);
  if (bound<0) {
    warn("IShowArrayObject", "couldn't get bound");
    return 0;
  }
  if (IEnvIsNull(env)) {
    if (style==SHOW_DISPLAY)
      IShowSimpleObject(name, env, style, indent);
    IResetError();
    return 0;
  }
  if (IEnvIsSimpleArray(env) && true(show_array_on_line) && style!=SHOW_SAVE) {
    show_on_line = 1;
    if (subarray)
      fprintf(dout, "%*s{", indent, "");
    else
      fprintf(dout, "%*s%*s = {", indent, "", name_width, name);
  } else
    show_on_line = 0;
  if (strlen(name) > name_width) {
    subarray_indent = indent+4;
    subarray_newline = 1;
  } else {
    subarray_indent = indent+4+name_width;
    subarray_newline = 0;
  }
  for (j=0;j<bound;j++) {
    COPY_STRUCT(env, elt_env);
    if (!IMoveToElement(elt_env, j)) {
      if (show_on_line)
	fprintf(dout, "<NULL>");
      else if (style==SHOW_DISPLAY)
	fprintf(dout, "%*s%*s[%d] = <NULL>\n", indent, "", name_width, name, j);
      IResetError();
      continue;
    }
    if (show_on_line) {
      if (IEnvIsSimple(elt_env)) {
	if (j>0) fputc(' ', dout);
	fprintf(dout, "%s", IGetSimpleValue(elt_env, 0, NULL, 1));
      } else if (IMoveToPointer(elt_env) && IEnvIsSimpleArray(elt_env)) {
	if (j>0 || subarray_newline) fprintf(dout, "\n");
	IShowArrayObject(name, elt_env, style,
			 j==0 && !subarray_newline ? 0 : subarray_indent, 1);
      } else {
	fprintf(dout, " ?");
      }
    } else {
      sprintf(elt_name, "%s[%d]", name, j);
      IShowObjectEnv(elt_name, elt_env, style, indent);
    }
  }
  if (show_on_line) {
    fprintf(dout, "}");
    if (!subarray)
      fprintf(dout, "\n");
  }
  return 1;
}

/***
 * IShowStructObject
 */
int	IShowStructObject(name, env, style, indent)
char *name;
ADDRESS_ENV *env;
enum SHOW_STYLE style;
int indent;
{
  ADDRESS_ENV e, *field_env;
  TBL_ITEM *item;
  int field_id=0;
  char field_name_buf[NAMEBUF];
  char *field_name = field_name_buf;
  enum SHOW_STYLE field_style;
  field_env = &e;
  assert(!IEnvIsArray(env));
  if (IEnvIsNull(env)) {
    if (style==SHOW_DISPLAY)
      fprintf(dout, "%*s%*s = (struct %s) <NULL>\n", indent, "", name_width, name,
	      IGetTypeName(env->binding->var_type));
    IResetError();
    return 0;
  }
  field_style=style;
  if (style==SHOW_DISPLAY) {
    fprintf(dout, "%*s%s = {\n", indent, "", name);
    field_style=SHOW_BRIEF;
  }

  while ((item=ITableNextByContext(itf_table, &field_id,
				   env->binding->var_type))!=NULL) {
    if (!IMoveToField(env, field_env, item->id, NO_SWAP))
      panic("IShowStructObject", "couldn't move to field");
    if (style==SHOW_SAVE) {
      if (!(field_env->binding->features&ITF_SAVE)) continue;
      sprintf(field_name, "%s.%s", name, item->name);
    } else {
      if (!(field_env->binding->features&ITF_VISIBLE)) continue;
      field_name = item->name;
    }
    IShowObjectEnv(field_name, field_env, field_style, indent+show_indent_inc);
  }
  if (style==SHOW_DISPLAY)
    fprintf(dout, "%*s}\n", indent, "");
  return 1;
}

#ifndef _NO_LIBPW
/***********************************************************************
 *	Name:		getIntValue
 *	Description:	returns an integer value associated with the string
 *			passed in.
 *	Parameters:	
 *		char	*string - the string to convert (e.g. "1", or
 *				"currentNet.numGroups")
 *	Return Value:	the integer value of the string.
 ***********************************************************************/
static int	getIntValue(string)
  char		*string ;
{
  char	*value ;

  if (IIsInteger(string)) {
    value = string ;
  } else {
    int		takeAddress;
    ADDRESS_ENV	indexEnv ;
    if (!IParseObject(string, &indexEnv, &takeAddress)) {
      itf_errno = OBJECT_SYNTAX;
      IErrorAbort("Unknown variable: \"%s\"", string);
    }
    value = IGetSimpleValue(&indexEnv, 0, "%d", 0) ;
    if (value == NULL) {
      itf_errno = OBJECT_SYNTAX;
      IErrorAbort("Variable is not a simple object: \"%s\"", string);
    } else if (!IIsInteger(value)) {
      itf_errno = OBJECT_SYNTAX;
      IErrorAbort("Variable is not an integer: \"%s\" has value \"%s\"", 
		  string, value) ;
    }
  }
  return atoi(value);
}
/**********************************************************************/

/***********************************************************************
 *	Name:		counterName
 *	Description:	returns the name of the counter for a given array
 *	Parameters:	
 *		char	*arrayName - the FULL name of the array
 *	Return Value:	
 *		char	*counterName - the FULL name of the counter.
 ***********************************************************************/
static char	*counterName(arrayName)
  char		*arrayName ;
{
  String	counterString ;
  char		baseName[BUFSIZ], fullName[BUFSIZ] ;
  int		takeAddress;
  ADDRESS_ENV	env ;

  if (!IParseObject(arrayName, &env, &takeAddress)) {
    itf_errno = OBJECT_SYNTAX;
    IErrorAbort("Unknown variable: \"%s\"", arrayName);
  }
  if (!IIsArray(env.binding)) {
    itf_errno = OBJECT_SYNTAX;
    IErrorAbort("Variable is not an array: \"%s\"", arrayName);
  }

  counterString = IArrayCounter(env.binding) ;
  if (counterString[0] == '[')
    ++counterString ;
  if (counterString[strlen(counterString) - 1] == ']')
    counterString[strlen(counterString) - 1] = '\0' ;

  strcpy(baseName, arrayName) ;
  arrayName = strrchr(baseName, '.') ;
  if (arrayName == NULL || IIsInteger(counterString)) {
    sprintf(fullName, "%s", counterString) ;
  } else {
    *arrayName = '\0' ;
    sprintf(fullName, "%s.%s", baseName, counterString) ;
  }
  return fullName ;
}
/**********************************************************************/
static int	getMaxIndex(arrayName)
  char		*arrayName ;
{
  return getIntValue(counterName(arrayName)) - 1 ;
}
/**********************************************************************/

/***********************************************************************
 *	Name:		getRangeStr
 *	Description:	returns a whole bunch of strings describing the
 *			ranges of an array subscript
 *	Parameters:	
 *	Return Value:	1 if a range is found, 0 if not.
 ***********************************************************************/
static int	getRangeStr(name, preString, postString, 
				lowerString, upperString)
  char		name[] ;
  char		preString[] ;
  char		postString[] ;
  char		lowerString[] ;
  char		upperString[] ;
{
  char		tmpName[BUFSIZ] ;

  static char	*compiledRange1, *compiledRange2, *compiledRange3 ;
  static char	*compiledStar1,  *compiledStar2,  *compiledStar3 ;

  /* find the first "[" "<range>" "]" that exists in the name,
   * where <range> is "a-b" or "*". */
  if (compiledRange1 == NULL) 
    compiledRange1 = regcmp("^(.*\\[[^]-]+-[^]-]+\\])$0",
			    ".*\\[[^]-]+-[^]-]+\\].*",
			    NULL) ;
  if (compiledRange2 == NULL) 
    compiledRange2 = regcmp("^(.*\\[[^]-]+-[^]-]+\\])$0",
			    ".*\\[\\*\\].*",
			    NULL) ;
  if (compiledRange3 == NULL) 
    compiledRange3 = regcmp("^(.*)$0",
			    "\\[([^]-]+)$1-([^]-]+)$2\\]",
			    NULL) ;
  if (compiledStar1 == NULL) 
    compiledStar1 = regcmp("^(.*\\[\\*\\])$0",
			     ".*\\[\\*\\].*",
			   NULL) ;
  if (compiledStar2 == NULL) 
    compiledStar2 = regcmp("^(.*\\[\\*\\])$0",
			    ".*\\[[^]-]+-[^]-]+\\].*",
			   NULL) ;
  if (compiledStar3 == NULL) 
    compiledStar3 = regcmp("^(.*)$0",
			   "\\[\\*\\]",
			   NULL) ;
  preString[0]   = '\0' ;
  postString[0]  = '\0' ;
  lowerString[0] = '\0' ;
  upperString[0] = '\0' ;
  strcpy(tmpName, name) ;
  while (regex(compiledRange1, tmpName, preString)
	 || regex(compiledRange2, tmpName, preString)
	 || regex(compiledStar1,  tmpName, preString)
	 || regex(compiledStar2,  tmpName, preString)) {
    strcpy(tmpName, preString) ;
  }

  if (regex(compiledRange3, tmpName, preString, lowerString, upperString)) {
    char	*postPtr ;
    if (strcmp(upperString, "$") == 0)
      strcpy(upperString, counterName(preString)) ;
    postPtr = name + strlen(preString) ;
    postPtr = strchr(postPtr, ']') + 1 ;
    strcpy(postString, postPtr) ;
    return 1 ;
  } else if (regex(compiledStar3, tmpName, preString)) {
    char	*postPtr ;
    strcpy(lowerString, "0") ;
    strcpy(upperString, counterName(preString)) ;
    postPtr = name + strlen(preString) ;
    postPtr = strchr(postPtr, ']') + 1 ;
    strcpy(postString, postPtr) ;
    return 1 ;
  } else {
    return 0 ;
  }
}
/**********************************************************************/

/***********************************************************************
 *	Name:		getRange
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
#define MIN(x, y)	((x) < (y) ? (x) : (y))
static int	getRange(name, preString, postString, lower, upper)
  char		name[] ;
  char		preString[] ;
  char		postString[] ;
  int		*lower ;
  int		*upper ;
{
  char		lowerString[BUFSIZ] ;
  char		upperString[BUFSIZ] ;
  int		max ;

  /* find the first "[" "<range>" "]" that exists in the name,
   * where <range> is "a,b" or "*". */

  getRangeStr(name, preString, postString, lowerString, upperString) ;

  if (lowerString[0] == '\0' || upperString[0] == '\0')
    return 0 ;

  *lower = getIntValue(lowerString) ;
  if (*lower < 0)
    return 0 ;

  *upper = getIntValue(upperString) ;
  max    = getMaxIndex(preString) ;
  *upper = MIN(*upper, max) ;
  if (*upper < 0)
    return 0 ;

  return 1 ;
}
#undef MIN
/**********************************************************************/

/***********************************************************************
 *	Name:		IShowObjects
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void 	IShowObjects(name, style)
  char		*name ;
  SHOW_STYLE	style ;
{
  char		preString[132], postString[132] ;
  int		upper, lower ;

  if (getRange(name, preString, postString, &lower, &upper) != 0) {
    char	newName[BUFSIZ] ;
    while (lower <= upper) {
      sprintf(newName, "%s[%d]%s", preString, lower++, postString) ;
      IShowObjects(newName, style) ;
    }
  } else {
    IShowObject(name, style);
    if (itf_errno)
      IAbort() ;
  }
}
/**********************************************************************/
#endif				/* _NO_LIBPW */

int command_show(tokc, tokv)
int tokc;
char *tokv[];
{
  enum SHOW_STYLE style = SHOW_DISPLAY;
  if (GiveHelp(tokc)) {
    IUsage("[-set] [object] ...");
    ISynopsis("show the values of struct or array objects");
    IHelp
      (IHelpArgs,
       "Show the values in simple objects, structures, or array objects. The",
       "indices for an array may contain a range.",
       "EXAMPLES",
       "To show the first 4 elements  of an array, type",
       "",
       "\txerion-> show array[0-3]",
       "",
       "To show all elements of an array, type",
       "",
       "\txerion-> show array[*]",
       "SEE ALSO",
       "print, set",
       NULL);
    return 1;
  }
  
  while (*++tokv && **tokv=='-') {
    if (!strcmp("-set", tokv[0])) style = SHOW_SAVE;
    else {
      IError("bad option \"%s\"", tokv[0]);
      return 0;
    }
  }

  if (*tokv)
    for (;*tokv;tokv++)
#ifdef _NO_LIBPW
      IShowObject(*tokv, style);
#else
      IShowObjects(*tokv, style);
#endif
  else {
    int id=0;
    TBL_ITEM *item;
    if (style!=SHOW_SAVE) style=SHOW_BRIEF;
    while (item=ITableNextByContext(itf_table, &id, itf_obj_context)) {
      if (style==SHOW_SAVE&&!(item->data.binding->features&ITF_SAVE)) continue;
      IShowObject(item->name, style);
    }
  }
  return 1;
}

/***
 * Set the defaults of a structure variable, given the address env
 */
void ISetEnvDefaults(env)
ADDRESS_ENV *env;
{
  ADDRESS_ENV *child, e;
  child = &e;
  while (IEnvIsPointer(env)) if (!IMoveToPointer(env)) return;
  if (IEnvIsNull(env)) return;
  /*
  fprintf(dout, "-->%12s\t%s%s\t%s 0x%x\n", IGetTypeName(env->binding->var_type),
	  env->name, IArrayCounter(env->binding),
	  IEnvIsStruct(env) ? "Structure" : "Simple", env->poi.ptr);
  */
  if (IEnvIsArray(env)) {
    int element, bound;
    bound = IGetArrayBound(env);
    if (bound<0) {
      warn("ISetEnvDefaults", "couldn't get bound");
      return;
    }
    for (element=0;element<bound;element++) {
      COPY_STRUCT(env, child);
      if (IMoveToElement(child, element))
	ISetEnvDefaults(child);
    }
  } else if (IEnvIsStruct(env)) {
    int id;
    TBL_ITEM *item;
    id = 0;
    while ((item=ITableNextByContext(itf_table, &id, env->binding->var_type))!=NULL) {
      /*
      fprintf(dout, "%12s\t%s%s\t%s\n", IGetTypeName(item->data.binding->var_type),
	      item->name, IArrayCounter(item->data.binding),
	      IIsStruct(item->data.binding) ? "Structure" : "Simple");
      */
      if (IMoveToField(env, child, item->id, NO_SWAP))
	ISetEnvDefaults(child);
      else
	IResetError();
    }
  } else if (IEnvIsSimple(env)) {
    if (!(env->binding->features & ITF_READONLY)
	&& env->binding->default_val!=NULL) {
      /* fprintf(dout, "setting simple field \"%s\" to default \"%s\"\n", env->name, env->binding->default_val); */
      ISetSimpleValue(env, env->binding->default_val);
    }
  } else {
    panic("ISetEnvDefaults", "don't know what to do!");
  }
  IResetError();
}

/***
 * Set the defaults of a structure variable, given the variable binding
 */
void ISetBindingDefaults(binding)
BINDING *binding;
{
  ADDRESS_ENV env;
  int i;
  env.binding = binding;
  env.parent_binding = NULL;
  env.poi.ptr = binding->value.poi.ptr;
  env.parent_poi.ptr = NULL;
  env.name = binding->tbl_entry->name;
  env.ilevel = binding->ilevel;
  for (i=0;i<=MAX_INDIRECTION;i++)
    env.have_index[i] = 0;
  /* fprintf(dout, "ISetBindingDefaults: %s\n", binding->tbl_entry->name); */
  ISetEnvDefaults(&env);
  IResetError();
}

/***
 * Set the defaults of a single structure, given its type name
 */
void ISetDefaults(object_type_name, struct_p)
char *object_type_name;
char *struct_p;
{
  TBL_ITEM *item;
  char *name = itf_name_buffer;
  MASSAGE_VAR_NAME(object_type_name, name);
  item = ITableLookup(itf_table, name, itf_type_context);
  if (item==NULL) IError("ISetDefaults: bad type \"%s\"", object_type_name);
  if (!(item->data.binding->object_prop & HAS_DEFAULT)) return;
  if (IBindStruct("-@tmp", object_type_name, struct_p, P_OBJECT, NULL)==NULL) {
    IError("ISetDefaults: couldn't form binding");
    return;
  }
  /*
   * don't need the following because IBindStruct sets the defaults
  item = ITableLookup(itf_table, "-@tmp", itf_obj_context);
  if (item==NULL) {
    warn("ISetDefaults", "couldn't find object \"-@tmp\"");
    return;
  }
  ISetBindingDefaults(item->data.binding);
  */
  IResetError();
}

/***
 * Set the defaults of a vector structure, given its type name
 */
void ISetVectorDefaults(object_type_name, struct_p, n)
char *object_type_name;
char *struct_p;
int  n;
{
  TBL_ITEM *item;
  char *name = itf_name_buffer;
  char count[10];
  sprintf(count, "%d", n);
  MASSAGE_VAR_NAME(object_type_name, name);
  item = ITableLookup(itf_table, name, itf_type_context);
  if (item==NULL) IError("ISetDefaults: bad type \"%s\"", object_type_name);
  if (!(item->data.binding->object_prop & HAS_DEFAULT)) return;
  if (IBindStruct("-@tmp", object_type_name, struct_p, P_OBJECT|P_FIXED, count)
      ==NULL) {
    IError("ISetDefaults: couldn't form binding");
    return;
  }
  /*
   * don't need the following because IBindStruct sets the defaults
  item = ITableLookup(itf_table, "-@tmp", itf_obj_context);
  if (item==NULL) {
    warn("ISetDefaults", "couldn't find object \"-@tmp\"");
    return;
  }
  ISetBindingDefaults(item->data.binding);
  */
  IResetError();
}

int ISetDefault(in_name, value)
char *in_name;
char *value;
{
  return IAddDefault(in_name, value);
}

static int MarkHasDefault(this)
TBL_ITEM *this;
{
  TBL_ITEM *item;
  int id = 0;
  if (this->data.binding->object_prop & HAS_DEFAULT) return 1;
  this->data.binding->object_prop |= HAS_DEFAULT;
  if (this->context==itf_type_context || this->context==itf_obj_context)
    return 1; /* top level already */
  /***
   * Now mark all the structures that the parent of this type is a field in.
   * The structure that this field is in has this_type = this->context
   * We should mark all items in the table that have
   *   item->data.binding->var_type = this_type, and call recursively
   */
  while ((item=ITableNextByContext(itf_table, &id, 0))!=NULL) {
    if (   item->data.binding->var_type == this->context
	&& !(item->data.binding->object_prop & HAS_DEFAULT)) {
      MarkHasDefault(item);
    }
  }
  return 1;
}

int IAddDefault(in_name, value)
char *in_name;
char *value;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL) {
    if (item->context==itf_type_context) {
      IError("Basic types cannot have defaults");
      return 0;
    }
    binding = item->data.binding;
    /*
    if (!IIsField(binding)) {
      IError("Only fields of structures can have defaults");
      return 0;
    }
    */
    if (IIsStruct(binding)) {
      IError("Only simple fields can have defaults");
      return 0;
    }
    if (IIsCounter(binding)) {
      IError("Counter fields cannot have defaults");
      return 0;
    }
    if (binding->var_type<0 || binding->var_type>=ITF_VAR_TYPES) {
      warn("ISetFormat", "var_type out of range %d", binding->var_type);
      return 0;
    }
    IAssignString(&(binding->default_val), value);
    MarkHasDefault(item);
  } else {
    if (!itf_errno)
      IError("unknown object or type \"%s\"", name);
    return 0;
  }
  return 1;
}

char *IGetDefault(in_name)
char *in_name;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL) {
    if (item->context==itf_type_context) {
      IError("Types do not have defaults");
      return 0;
    }
    binding = item->data.binding;
    if (!IIsField(binding)) {
      IError("Only fields of structures do have defaults");
      return 0;
    }
    if (IIsStruct(binding)) {
      IError("Only simple fields do have defaults");
      return 0;
    }
    if (IIsCounter(binding)) {
      IError("Counter fields do not have defaults");
      return 0;
    }
    if (binding->var_type<0 || binding->var_type>=ITF_VAR_TYPES) {
      warn("ISetFormat", "var_type out of range %d", binding->var_type);
      return 0;
    }
    return binding->default_val!=NULL ? binding->default_val : "";
  } else {
    if (!itf_errno)
      IError("unknown object or type \"%s\"", name);
    return 0;
  }
}


/***
 * Recalculate the address of the field for the new structure address
 */
int IRecalcFieldEnv(env, struct_addr)
ADDRESS_ENV *env;
generic *struct_addr;
{
  if (struct_addr==NULL) {
    itf_errno = NULL_POINTER;
    return 0;
  } else {
    env->poi.ptr = struct_addr + env->binding->value.poi.longint;
    env->parent_poi.ptr = struct_addr;
    return 1;
  }
  /* NOTREACHED */
}

/***
 * Set up an ADDRESS_ENV for a field in a structure
 */
int ISetupStructFieldEnv(env, struct_type, item)
ADDRESS_ENV *env;
int struct_type;
TBL_ITEM *item; /* the table entry for the field type */
{
  int i;
  if (!IIsStructType(struct_type)) {
    itf_errno = BAD_VARTYPE;
    IError("%s", IGetTypeName(struct_type));
    return 0;
  }
  if (item==NULL) {
    itf_errno = UNKNOWN_FIELD;
    IError("%s", "null field item");
    return 0;
  }
  env->binding = item->data.binding;
  env->poi.ptr = NULL;
  for (i=0;i<=MAX_INDIRECTION;i++)
    env->have_index[i] = 0;
  env->ilevel = env->binding->ilevel;
  env->name = item->name;
  env->parent_poi.ptr = NULL;
  env->parent_binding = NULL;
  if (!IEnvIsSimple(env)) {
    itf_errno = BAD_VARTYPE;
    IError("%s", env->name);
    return 0;
  }
  return 1;
}

/***
 * Create an env for a pointer, given a type name
 */
int ICreateEnv(env, type_name, ptr)
ADDRESS_ENV *env;
char *type_name;
generic *ptr;
{
  TBL_ITEM *item;
  item = ITableLookup(itf_table, type_name, itf_type_context);
  if (item==NULL) {
    itf_errno = BAD_VARTYPE;
    IError("%s", type_name);
    return 0;
  }
  ISetEnvFromItem(env, item);
  env->poi.ptr = ptr;
  env->ilevel = 1;
  return 1;
}
