
/**********************************************************************
 * $Id: bind-enter.c,v 1.7 93/03/29 14:15:36 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include "itf.h"

#define DB(X) 

/*
 *	VAR_PACK
 *
 *  A few functions to let us do things in a nice way with variables.
 *  They associate an arbitrary name with a
 *  C variable of a simple type, (including strings), so that the C
 *  variable can be easily changed interactively.
 *  The String space allocation is handled by this program, so
 *  String pointers should not be changed except using ISetValue().
 *
 *  A separate table is used to store the names, it is called 'itf_table'.
 *
 */

extern  int	EnterBindings() ;

int itf_obj_context; /* used for bound variables and structures */
int itf_type_context; /* used for struct types */
HASH_TABLE *itf_table;
static HASH_TABLE table;
static long feature_mask;
static int anon_name_count[ITF_VAR_TYPES];
static char* anon_names[ITF_VAR_TYPES];
int itf_top_struct_type = 0;
char itf_value_buffer[VALUE_BUFFER];
char itf_name_buffer[MAX_VAR_NAME+10];

static Boolean atob ARGS((char *value));
/*
 *  IBindInit()
 */
void IBindInit(max_vars)
int max_vars;
{
  int i;
  TBL_ITEM *item;
  itf_table = &table;
  (void) INewTable(&table, max_vars, max_vars/2, ID_INDEX);
  /* get the table context for variables (a static var) */
  itf_obj_context = INewTableContext(&table);
  itf_type_context = INewTableContext(&table);
  /* chew up some contexts, so that the contexts for structures
     will be distinct from types */
  while (INewTableContext(&table) <= ITF_VAR_TYPES);
  feature_mask = 0;
  for (i=0;i<ITF_VAR_TYPES;i++) {
    anon_name_count[i]=0;
    anon_names[i]=NULL;
  }
  anon_names[ITF_STRING] = "String%d";
  anon_names[ITF_REAL] = "Real%d";
  anon_names[ITF_FLOAT] = "float%d";
  anon_names[ITF_DOUBLE] = "double%d";
  anon_names[ITF_LONG] = "long%d";
  anon_names[ITF_INT] = "int%d";
  anon_names[ITF_SHORT] = "short%d";
  anon_names[ITF_CHAR] = "char%d";

  item = ITableInsert(&table, "String", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_STRING;
  item->data.binding->element_size = sizeof(String);

  item = ITableInsert(&table, "Real", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_REAL;
  item->data.binding->element_size = sizeof(Real);

  item = ITableInsert(&table, "float", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_FLOAT;
  item->data.binding->element_size = sizeof(float);

  item = ITableInsert(&table, "double", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_DOUBLE;
  item->data.binding->element_size = sizeof(double);

  item = ITableInsert(&table, "long", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_LONG;
  item->data.binding->element_size = sizeof(long);

  item = ITableInsert(&table, "int", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_INT;
  item->data.binding->element_size = sizeof(int);

  item = ITableInsert(&table, "short", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_SHORT;
  item->data.binding->element_size = sizeof(short);

  item = ITableInsert(&table, "char", itf_type_context, 0);
  assert(item!=NULL);
  item->data.binding = Calloc(BINDING, 1);
  assert(item->data.binding!=NULL);
  item->data.binding->var_type = ITF_CHAR;
  item->data.binding->element_size = sizeof(char);

  IEnterType("STREAM", sizeof(struct STREAM), IS_STRUCT);
  IEnterType("TRACE", sizeof(struct TRACE), IS_STRUCT);

  EnterBindings();
}

static char *massage(in_name)
char *in_name;
{
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  return name;
}

long ISetFeatureMask(m)
long m;
{
  long old;
  old = feature_mask;
  feature_mask = m;
  return old;
}

/*
 *  char *IBindPoi(name, type, poi, mode, elt_size, context, count)
 *  Create an entry in the table for the variable to be bound.
 *  'poi' is a pointer to the C variable.
 *  If the mode is OBJECT, the value is kept entirely internally,
 *  and 'ptr' is ignored.
 *  If the ptr is NULL, space is allocated
 *  If there was a variable previously bound with this name, it is
 *  unbound, and its binding is left hanging.
 *  A pointer to the binding of the variable is returned.
 */

BINDING *IBindPoi(in_name, type, poi, mode, prop, elt_size, context, count)
char *in_name;
int type;
POI poi;
int mode, prop;
int elt_size;
int context;
int count;
{
  int i, count_level=0;
  TBL_ITEM *item;
  BINDING *binding, *old_binding;
  char *name=itf_name_buffer;
  /*
   * manufacture a name if there is not already one
   */
  if (!in_name || !*in_name) {
    in_name = itf_value_buffer;
    if (type >= ITF_VAR_TYPES || anon_names[type]==NULL) {
      warn("IBindPoi", "can't give anon name to type %d\n", type);
      return NULL;
    }
    /* make an anomymous name */
    do
      sprintf(itf_value_buffer, anon_names[type], ++(anon_name_count[type]));
    while (IIsObjectName(itf_value_buffer));
  }
  if (strlen(in_name) > MAX_VAR_NAME) {
    warn("IBindPoi", "name too long: %s", in_name);
    return NULL;
  }
  MASSAGE_VAR_NAME(in_name, name);

  /*
   * look up any old variable of this name
   */
  item = ITableLookup(&table, name, context);

  if (item!=NULL) {
    old_binding = item->data.binding;
    if ((old_binding->features & ITF_ORPHANABLE) || !strncmp(name, "-@", 2)) {
      old_binding->tbl_entry = NULL;
      old_binding->reference_count--;
      if (old_binding->reference_count<=0 && strncmp(name, "-@", 2))
	warn("IBindPoi", "\"%s\" just got orphanned!", name);
    } else {
      IError("name \"%s\" is in use", name);
      goto undo;
    }
  } else {
    item = ITableInsert(&table, name, context, 0);
    old_binding = NULL;
  }

  if (item==NULL) {
    warn("IBindPoi", "could not insert \"%s\" - %s", name, hash_error[hash_errno]);
    goto undo;
  } else {
    if ((binding = Calloc(BINDING, 1))==NULL) goto no_mem;
    item->data.ptr = (char*)binding;
    binding->var_type = type;
    binding->features = feature_mask;
    binding->func_to_call = NULL;
    for (i=0;i<=MAX_INDIRECTION;i++)
      binding->itype[i]=AUTO_POINTER;
    if (mode&OBJECT) binding->ilevel=0;
    else if (mode&P_OBJECT) binding->ilevel=1;
    else if (mode&PP_OBJECT) binding->ilevel=2;
    else if (mode&PPP_OBJECT) binding->ilevel=3;

    if (mode&(P_COUNT|P_FIXED)) count_level = 1;
    else if (mode&(PP_COUNT|PP_FIXED)) count_level = 2;
    else if (mode&(PPP_COUNT|PPP_FIXED)) count_level = 3;
    if (count_level) {
      if (count_level>binding->ilevel)
	warn("IBindPoi", "count level for \"%s\" too high", name);
      binding->array_count[count_level] = count;
      binding->itype[count_level] = (mode&(P_COUNT|PP_COUNT|PPP_COUNT)
				     ? COUNTED_ARRAY : FIXED_ARRAY);
    }
    if (type>=ITF_VAR_TYPES) binding->object_prop = IS_STRUCT;
    else binding->object_prop = IS_SIMPLE;
    if (mode&ARRAY_PS) binding->object_prop |= IS_ARRAY;
    binding->object_prop |= (prop & OBJECT_IS_PROP);
    binding->element_size = elt_size;
    binding->tbl_entry = item;
    binding->reference_count = 1;
    /* set the format from the defaults */
    if (type < ITF_VAR_TYPES) {
      binding->format_code = default_format_code[type];
      binding->format_width = default_format_width[type];
      binding->format_prec = default_format_prec[type];
    } else {
      /* structure */
      binding->format_code = binding->format_width = binding->format_prec = 0;
    }
    if (prop & IS_COUNTER) binding->features |= ITF_READONLY;
    if (prop & IS_FIELD) {
      binding->value.poi.longint = poi.longint; /* the field offset */
    } else {
      if ((poi.ptr==NULL) && !(mode & OBJECT)) {
	warn("IBindPoi", "null ptr and !OBJECT for %s", name);
	return 0;
      }
      if (binding->ilevel==0) {
	assert(poi.ptr==NULL);
	/*
	 * Allocate space if needed, for OBJECT, and attach
	 * But can't allocate space for a variable sized array
	 * TODO: this!, and make a string an array of chars
	 */
	if (type==ITF_REAL)
	  binding->value.real = 0.0;
	else if (type==ITF_LONG)
	  binding->value.poi.longint = 0;
	else if (type == ITF_STRING) {
	  if (!(binding->value.poi.ptr = (generic *)calloc(16, 1)))
	    goto no_mem;
	} else
	  panic("IBindPoi: ilevel==0 & bad type", NULL);
      } else if (binding->ilevel==1) {
	if (type==ITF_STRING)
	  if (!(*(poi.string_ptr)=(String)calloc(16, 1)))
	    goto no_mem;
      }
      if (binding->ilevel>0) {
	assert(poi.ptr!=NULL);
	binding->value.poi.ptr = poi.ptr;
      }
    }
  }
  return binding;
 no_mem:
  warn("IBindPoi", "Out of memory, could not create variable: \"%s\"", name);
 undo:
  if (old_binding!=NULL) {
    item->data.ptr = (char*)old_binding;
    old_binding->tbl_entry = item;
  }
  return NULL;
}

int IGetType(in_name, size)
char *in_name;
int *size;
{
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  item = ITableLookup(&table, name, itf_type_context);
  if ((item!=NULL)) {
    if (size!=NULL) *size = item->data.binding->element_size;
    return item->data.binding->var_type;
  } else {
    if (size!=NULL) *size = 0;
    return 0;
  }
}

/***
 * IGetTypeName - given a type id, find the name.  This
 * is a linear search, but not through many items.
 */
char *IGetTypeName(type)
int type;
{
  TBL_ITEM *item;
  int id=0;
  if (type <= 0 || type >= itf_top_struct_type) {
    warn("IGetTypeName", "bad type %d", type);
    return 0;
  }
  if (type<ITF_VAR_TYPES)
    return itf_type_names[type];
  else
    while ((item=ITableNextByContext(itf_table, &id, itf_type_context))!=NULL)
      if (item->data.binding->var_type==type) return item->name;
  return NULL;
}

/***
 * IGetObjectTypeName - given an object name, find the type name.
 */
char *IGetObjectTypeName(in_name)
char *in_name;
{
  TBL_ITEM *item;
  struct ADDRESS_ENV env;
  int take_address;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (IParseObject(name, &env, &take_address))
    return IGetTypeName(env.binding->var_type) ;
  else
    return NULL;
}

int IIsArray(binding)
BINDING *binding;
{
  return (binding->object_prop & IS_ARRAY) != 0;
}

int IIsStruct(binding)
BINDING *binding;
{
  return (binding->object_prop & IS_STRUCT) != 0;
}

int IIsField(binding)
BINDING *binding;
{
  return (binding->object_prop & IS_FIELD) != 0;
}

int IIsCounter(binding)
BINDING *binding;
{
  return (binding->object_prop & IS_COUNTER) != 0;
}

char *IArrayCounter(binding)
BINDING *binding;
{
  int level;
  if (!IIsArray(binding)) return "";
  for (level=binding->ilevel;level>=0;level--)
    if (binding->itype[level] & ENV_IS_ARRAY) break;
  if (binding->itype[level]&FIXED_ARRAY)
    sprintf(itf_value_buffer, "[%d]", binding->array_count[level]);
  else {
    TBL_ITEM *item;
    item = ITableLookupById(itf_table, binding->array_count[level]);
    if (item!=NULL)
      sprintf(itf_value_buffer, "[%s]", item->name);
    else
      sprintf(itf_value_buffer, "[unknown counter:%d]", binding->array_count[level]);
  }
  return itf_value_buffer;
}

int IIsStructType(type)
int type;
{
  if (type <= 0 || type >= itf_top_struct_type) {
    warn("IIsStructType", "bad type %d", type);
    return 0;
  }
  if (type<ITF_VAR_TYPES) return 0;
  else return 1;
}

int IEnterType(in_name, size, prop)
char *in_name;
int size;
int prop;
{
  TBL_ITEM *item;
  BINDING *binding;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  item = ITableInsert(&table, name, itf_type_context, 0);
  binding = Calloc(BINDING, 1);
  /* printf("Entering structure type: \"%s\"\n", name); */
  if (item==NULL || binding==NULL) {
    warn("IEnterType", "can't enter type: %s", name);
    return 0;
  } else {
    item->data.binding = binding;
    binding->var_type = INewTableContext(&table);
    if (INewTableContext(&table) != binding->var_type+1
	|| binding->var_type<=ITF_VAR_TYPES)
      panic("IEnterType: context problems!", NULL);
    binding->element_size = size;
    itf_top_struct_type = binding->var_type+1;
    binding->object_prop = prop;
    return binding->var_type;
  }
}

/*
 * work out what the counter should be,
 * return 0 for errors
 */
static int IResolveCounter(mode, in_name, context)
short mode;
int context;
char *in_name;
{
  char *name = itf_name_buffer;
  TBL_ITEM *item;
  if (!(mode & ARRAY_PS))
    return 1;
  if (mode & FIXED_PS) {
    if (!IIsInteger(in_name)) {
      warn("IResolveCounter", "%s is not a number", in_name);
      return 0;
    }
    return atoi(in_name);
  } else if (mode & COUNT_PS) {
    MASSAGE_VAR_NAME(in_name, name);
    item = ITableLookup(&table, name, context);
    if (item==NULL) {
      warn("IResolveCounter", "%s is not a variable", name);
      return 0;
    }
    if (   !(item->data.binding->object_prop & IS_COUNTER)
	|| !(item->data.binding->object_prop & IS_SIMPLE)) {
      warn("IResolveCounter", "%s is not marked as a counter or is not a simple type", name);
      return 0;
    }
    return item->id;
  }

  warn("IResolveCounter", "bad mode %X for name \"%s\"", mode, name);
  return 0;
}

int IEnterField(struct_type_name, field_type_name, name, offset, mode, prop, counter_name)
char *name;
char *struct_type_name, *field_type_name;
int offset;
int mode, prop;
char *counter_name;
{
  int size;
  int struct_type, field_type, counter;
  POI poi;
  struct_type = IGetType(massage(struct_type_name), &size); /* size not important here */
  field_type = IGetType(massage(field_type_name), &size); /* is important here */
  prop |= IS_FIELD;
  if (!struct_type || !field_type) {
    warn("IEnterField", "unknown field type \"%s\" for \"%s\" in structure \"%s\"",
	   field_type_name, name, struct_type_name);
    return 0;
  }
  if (!name && !*name) {
    warn("IEnterField: no name!");
    return 0;
  }
  /* look up the counter field if there is one */
  if ((counter=IResolveCounter(mode, counter_name, struct_type))==0) {
    warn("IEnterField", "counter \"%s\" for field \"%s\" is not in struct \"%s\"",
	   counter_name, name, struct_type_name);
    return 0;
  }
  poi.longint = offset;
  if (IBindPoi(name, field_type, poi, mode, prop, size, struct_type, counter)!=NULL)
    return 1;
  else
    return 0;
}

int IEnterFieldCounter(struct_type_name, name, mode, prop, counter_name)
char *name;
char *struct_type_name;
int mode, prop;
char *counter_name;
{
  int size, count_level=0;
  int struct_type, counter;
  POI poi;
  BINDING *binding;
  TBL_ITEM *item;
  struct_type = IGetType(massage(struct_type_name), &size); /* size not important here */
  if (!struct_type) {
    warn("IEnterFieldCounter", "unknown structure type \"%s\"",
	 struct_type_name);
    return 0;
  }
  if (!name && !*name) {
    warn("IEnterField: no name!");
    return 0;
  }
  /* look up the counter field if there is one */
  if ((counter=IResolveCounter(mode, counter_name, struct_type))==0) {
    warn("IEnterField", "counter \"%s\" for field \"%s\" is not in struct \"%s\"",
	   counter_name, name, struct_type_name);
    return 0;
  }

  item = ITableLookup(&table, name, struct_type);

  /* check that item exists */
  if (item==NULL) {
    warn("IEnterFieldCounter", "could not find field \"%s\" in struct \"%s\"",
	 name, struct_type_name);
    return 0;
  }

  binding = item->data.binding;

  /* alter those two fields in binding */
  if (mode&(P_COUNT|P_FIXED)) count_level = 1;
  else if (mode&(PP_COUNT|PP_FIXED)) count_level = 2;
  else if (mode&(PPP_COUNT|PPP_FIXED)) count_level = 3;
  if (count_level) {
    if (count_level>binding->ilevel)
      warn("IBindPoi", "count level for \"%s\" too high", name);
    binding->array_count[count_level] = counter;
    binding->itype[count_level] = (mode&(P_COUNT|PP_COUNT|PPP_COUNT)
				   ? COUNTED_ARRAY : FIXED_ARRAY);
  }
  return 1;
}

char *IBindStruct(name, typename, struct_ptr, mode, count)
char *name, *typename;
generic *struct_ptr;
int mode;
char *count;
{
  BINDING *binding;
  POI poi;
  int struct_context, size;
  struct_context = IGetType(massage(typename), &size);
  if (!struct_context) {
    warn("IBindStruct", "unknown structure type: %s", typename);
    return NULL;
  }
  if (!name || !*name) { /* generate a new name */
    int i=0;
    do {
      i++;
      sprintf(itf_value_buffer, "%s%d", i);
    } while (IIsObjectName(itf_value_buffer));
    name = itf_value_buffer;
  }
  poi.ptr = struct_ptr;
  binding = IBindPoi(name, ITF_STRUCT, poi, mode, IS_STRUCT,
			 size, itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) {
    binding->var_type = struct_context;
    binding->features |= ITF_READONLY;
    ISetBindingDefaults(binding);
    return binding->tbl_entry->name;
  } else {
    /* try and work out why the binding failed - counter not found,
       or insert failed */
    return NULL;
  }
}

char *IBindReal(name, real_ptr, mode, prop, count)
char *name;
Real *real_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.real_ptr = real_ptr;
  binding = IBindPoi(name, ITF_REAL, poi, mode, prop,
			 sizeof(Real), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindDouble(name, double_ptr, mode, prop, count)
char *name;
double *double_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.double_ptr = double_ptr;
  binding = IBindPoi(name, ITF_DOUBLE, poi, mode, prop,
			 sizeof(double), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindFloat(name, float_ptr, mode, prop, count)
char *name;
float *float_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.float_ptr = float_ptr;
  binding = IBindPoi(name, ITF_FLOAT, poi, mode, prop,
			 sizeof(float), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindLong(name, long_ptr, mode, prop, count)
char *name;
long *long_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.long_ptr = long_ptr;
  binding = IBindPoi(name, ITF_LONG, poi, mode, prop,
			 sizeof(long), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindInt(name, int_ptr, mode, prop, count)
char *name;
int *int_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.int_ptr = int_ptr;
  binding = IBindPoi(name, ITF_INT, poi, mode, prop,
			 sizeof(int), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindShort(name, short_ptr, mode, prop, count)
char *name;
short *short_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.short_ptr = short_ptr;
  binding = IBindPoi(name, ITF_SHORT, poi, mode, prop,
			 sizeof(short), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindChar(name, char_ptr, mode, prop, count)
char *name;
char *char_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.char_ptr = char_ptr;
  binding = IBindPoi(name, ITF_CHAR, poi, mode, prop,
			 sizeof(char), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

char *IBindString(name, string_ptr, mode, prop, count)
char *name;
String *string_ptr;
int mode, prop;
char *count;
{
  BINDING *binding;
  POI poi;
  poi.string_ptr = string_ptr;
  binding = IBindPoi(name, ITF_STRING, poi, mode, prop,
			 sizeof(String), itf_obj_context,
			 IResolveCounter(mode, count, itf_obj_context));
  if (binding) return binding->tbl_entry->name;
  else return NULL;
}

/*
 * ISetFunction(name, function)
 */
PFPC ISetFunction(in_name, function)
char *in_name;
PFPC function;
{
  TBL_ITEM *item;
  PFPC old;
  char *name = itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  item = ITableLookup(&table, name, itf_obj_context);
  if (!(item!=NULL)) {
    warn("ICheckFunction", "%s not found", name);
    return NULL;
  }
  old = item->data.binding->func_to_call;
  item->data.binding->func_to_call = function;
  return old;
}

/*
 * IDoFeatures(name, action, feature)
 *
 * private function that does the feature manipulation
 */
static long IDoFeatures(in_name, action, feature)
char *in_name;
enum ITF_FEATURE_ACTION action;
long feature;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (   (item = IParseFieldOrObject(name))!=NULL
      && item->context!=itf_type_context) {
    binding = item->data.binding;
    switch (action) {
    case SET_FEATURE:
      binding->features |= feature;
      break;
    case UNSET_FEATURE:
      binding->features &= ~feature;
      break;
    case ENQUIRE:
    default:
      return feature & binding->features;
    }
    return 1;
  } else
    return 0;
}

/*
 * functions to set and get features
 */
long IGetFeatures(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, 0L);
}

int ISetFeatures(name, feat, on)
char *name;
long feat;
int on;
{
  if (feat & ITF_RESERVED_FEATURES) {
    warn("ISetFeatures", "attempt to set reserved feature");
    return 0;
  }
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), feat);
}

int ISetReadonly(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_READONLY);
}


int IIsReadonly(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, ITF_READONLY);
}


int ISetVisible(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_VISIBLE);
}

int ISetSystem(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_SYSTEM);
}

int IIsExport(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, ITF_EXPORT);
}

int ISetExport(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_EXPORT);
}

int IIsSimple(in_name)
char *in_name;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (   (item = IParseFieldOrObject(name))!=NULL
      && item->context!=itf_type_context) {
    binding = item->data.binding;
    return binding->object_prop & IS_SIMPLE ;
  } else
    return 0;
}

int IIsStructure(in_name)
char *in_name;
{
  BINDING *binding;
  TBL_ITEM *item;
  ADDRESS_ENV env;
  int take_address ;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL)
    return IIsStruct(item->data.binding) ;
  else
    return 0;
}

int IIsArrayObject(in_name)
char *in_name;
{
  BINDING *binding;
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if ((item = IParseFieldOrObject(name))!=NULL)
    return IIsArray(item->data.binding) ;
  else
    return 0;
}

int IArrayObjectBound(in_name)
char *in_name;
{
  TBL_ITEM *item;
  struct ADDRESS_ENV env;
  int take_address;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (IParseObject(name, &env, &take_address) && IMoveToPointer(&env))
    return IGetArrayBound(&env);
  else
    return -1;
}

int IIsNULL(in_name)
char *in_name;
{
  TBL_ITEM *item;
  struct ADDRESS_ENV env;
  int take_address;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (IParseObject(name, &env, &take_address) 
      && IMoveToPointer(&env) && IEnvIsNull(&env)) {
    IResetError() ;
    return 1;
 } else
    return 0;
}

int IIsVisible(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, ITF_VISIBLE);
}

int ISetSave(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_SAVE);
}


int IIsSave(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, ITF_SAVE);
}


int ISetMark(name, on)
char *name;
int on;
{
  return IDoFeatures(name, (on ? SET_FEATURE : UNSET_FEATURE), ITF_MARK);
}


int IIsMarked(name)
char *name;
{
  return IDoFeatures(name, ENQUIRE, ITF_MARK);
}

/*
 *	ISetValue(name, value)
 *  Set the value of an bound variable, from a String description of
 *  the value.  Any errors will result in no change to the variable.
 *  If the variable is not in the table, an error message will be printed.
 */
int ISetValue(in_name, value)
char *in_name, *value;
{
  struct ADDRESS_ENV env;
  int take_address;
  char *name=itf_name_buffer;
  if (!in_name || !*in_name) {
    itf_value_buffer[0]=0;
    itf_errno = NULL_VARNAME;
    return 0;
  }
  MASSAGE_VAR_NAME(in_name, name);
  if (   IParseObject(name, &env, &take_address)
      && IEnvIsSimpleSetError(&env))
    return ISetSimpleValue(&env, value);
  if (!IHaveCulprit()) IError("%s", name);
  itf_value_buffer[0]=0;
  return 0;
}

int IAssignString(strp, value)
char **strp;
char *value;
{
#if 0
  int new_size, old_size;
  char *str = *strp;
  new_size = strlen(value)+1;
  if (str!=NULL) old_size = strlen(str)+1;
  if (str == NULL || old_size < new_size
      || (new_size < old_size/2 && new_size > 16)) {
    str = Malloc(char, new_size);
    if (str == NULL) {
      warn("IAssignString", "string too long");
      return 0;
    } else {
      if (*strp!=NULL) free(*strp);
      *strp = str;
    }
  }
  (void) strncpy(str, value, new_size);
  return 1;
#else
  int	new_size, old_size;
  char *str = *strp;

  new_size = strlen(value)+1;
  old_size = str ? strlen(str)+1 : 0;

  if (old_size < new_size || (new_size < old_size/2 && new_size > 16)) {
    str = Malloc(char, new_size);
    if (str == NULL) {
      warn("IAssignString", "string too long");
      return 0;
    } 
    if (*strp)
      free(*strp);
    *strp = str;
  }
  (void) strcpy(str, value);
  return 1;
#endif
}

/***
 * Put something in the environment
 * This gratuitously tosses away memory - bad!
 */
int IExport(name, value)
char *name;
char *value;
{
  if ((getenv(name)==NULL || strcmp(getenv(name), value)) && ! strchr(name, '-')) {
    String	string = (String)malloc(strlen(name) + strlen(value) + 2) ;
    sprintf(string, "%s=%s", name, value) ;
    return putenv(string) ;
  } else
    return 0 ;
}

int ISetSimpleValue(env, value)
struct ADDRESS_ENV *env;
char *value;
{
  if (value==NULL) value="";
  assert(env->binding!=NULL);
  if (IEnvIsNull(env)) return 0;
  if (env->binding->features & ITF_READONLY) {
    itf_errno = READONLY_VAR;
    IError("%s", env->binding->tbl_entry->name);
    return 0;
  } else if (  (env->binding->object_prop & IS_STRUCT)
	     ||(env->ilevel!=1)) {
    itf_errno=NEED_FIELD_OR_INDEX;
    IError("%s", env->binding->tbl_entry->name);
    return 0;
  } else {
    if ((env->binding->func_to_call!=NULL)
	&& (NULL==
	    (value=(env->binding->func_to_call)(env->binding, value))))
      return 0;
    switch (env->binding->var_type) {
    case ITF_STRING :
      if (!IAssignString(env->poi.string_ptr, value)) return 0;
      if (env->binding->features&ITF_EXPORT) IExport(env->name, value);
      break;
    case ITF_REAL:
      *(env->poi.real_ptr) = atof(value);
      break;
    case ITF_FLOAT:
      *(env->poi.float_ptr) = atof(value);
      break;
    case ITF_DOUBLE:
      *(env->poi.double_ptr) = atof(value);
      break;
    case ITF_LONG:
      *(env->poi.long_ptr) = atol(value);
      break;
    case ITF_INT:
      *(env->poi.int_ptr) = atol(value);
      break;
    case ITF_SHORT:
      *(env->poi.short_ptr) = atol(value);
      break;
    case ITF_CHAR:
      *(env->poi.char_ptr) = atol(value);
      break;
    case ITF_BOOLEAN:
      *(env->poi.boolean_ptr) = atob(value);
      break;
    default:
      panic("ISetSimpleValue: bad var_type", NULL);
    }
    return 1;
  }
}

int IEnvIsSimpleSetError(env)
struct ADDRESS_ENV *env;
{
  if (env==NULL) {
    itf_errno = INTERNAL_NULL;
    return 0;
  } else if (IEnvIsArray(env)) {
    itf_errno = NEED_ARRAY_INDEX;
    IError("%s", env->binding->tbl_entry->name);
    return 0;
  } else if (IEnvIsStruct(env)) {
    itf_errno = NEED_STRUCT_FIELD;
    IError("%s", env->binding->tbl_entry->name);
    return 0;
  } else if (env->ilevel>1) {
    itf_errno = NEED_ARRAY_INDEX;
    IError("%s", env->binding->tbl_entry->name);
    return 0;
  }
  return 1;
}

int IEnvIsSimple(env)
struct ADDRESS_ENV *env;
{
  int r;
  if (env==NULL)
    r = 0;
  else if (env->ilevel>1)
    r = 0;
  else if ((  env->binding->itype[env->ilevel]
	    & ENV_IS_ARRAY)
	   && !env->have_index[env->ilevel])
    r = 0;
  else if (env->binding->object_prop & IS_STRUCT)
    r = 0;
  else
    r = 1;
  return r;
}

int IEnvIsSimpleArray(env)
struct ADDRESS_ENV *env;
{
  int r;
  if (env==NULL)
    r = 0;
  else if (env->binding->object_prop & IS_STRUCT)
    r = 0;
  else if ((  env->binding->itype[env->ilevel]
	    & ENV_IS_ARRAY)
	   && !env->have_index[env->ilevel])
    r = 1;
  else
    r = 0;
  return r;
}

char *IGetName(in_name)
char *in_name;
{
  TBL_ITEM *item;
  char *name =itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  item = ITableLookup(&table, name, itf_obj_context);
  if (!(item!=NULL))
    return NULL;
  else
    return item->name;
}

int IIsObjectName(in_name)
char *in_name;
{
  TBL_ITEM *item;
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  item = ITableLookup(&table, name, itf_obj_context);
  if ((item!=NULL))
    return 1;
  else
    return 0;
}

int ICheckExistSvar(in_name)
char *in_name;
{
  char *name=itf_name_buffer;
  MASSAGE_VAR_NAME(in_name, name);
  if (IIsObjectName(name)) return 1;
  else {
    POI poi;
    poi.string_ptr = NULL;
    if (IBindPoi(name, ITF_STRING, poi, 0, 0, sizeof(String), itf_obj_context, 0))
      return 1;
    else
      return 0;
  }
}

/*
 *  IShowVarValues(file, mask, all)
 *  show the values of all the variables in the table whose
 *  features have some bit in common with the mask.
 */
void IShowVarValues(f, mask, all)
FILE *f;
long mask;
int all;
{
  int id=0;
  int i;
  TBL_ITEM *item;
  struct ADDRESS_ENV env;
  char *value, *quote;
  while ((item=ITableNextByContext(&table, &id, itf_obj_context))!=NULL) {
    env.binding = item->data.binding;
    env.ilevel = env.binding->ilevel;
    env.parent_binding = NULL;
    env.name = item->name;
    for (i=0;i<=MAX_INDIRECTION;i++) env.have_index[i] = 0;
    if (env.ilevel>0)
      env.poi.ptr = env.binding->value.poi.ptr;
    else {
      env.poi.pptr = &env.binding->value.poi.ptr;
      env.ilevel = 1;
    }
    if (all || (env.binding->features & mask) != 0)
      if (IMoveToPointer(&env) && IEnvIsSimple(&env)
	  && (value = IGetSimpleValue(&env, 0, NULL, 0))!=NULL) {
	for (i=0;isalnum(value[i])||value[i]=='-';i++) ;
	if (value[i] || value[0]=='\0') quote = "\""; else quote = "";
	fprintf(f, "set %s = %s%s%s\n", item->name, quote, value, quote);
      } else
	IResetError();
  }
  return;
}

/***********************************************************************
 *	Name:		listNetParams
 *	Description:	returns an array of the network learning
 *			parameters.
 *	Parameters:	NONE
 *	Return Value:	
 *		char	**listNetParams - a STATIC array of the names.
 ***********************************************************************/
static char	**paramList = NULL ;
static int	numParams = 0 ;
/**********************************************************************/
static int	compare(s1, s2)
  const void	*s1 ;
  const void	*s2 ;
{
  return strcmp(*(char **)s1, *(char **)s2) ;
}
/**********************************************************************/
int	IAddNetParam(name) 
  char	*name ;
{
  if (paramList == NULL)
    paramList = (char **)malloc(2*sizeof(char *)) ;
  else
    paramList = (char **)realloc(paramList, (numParams+2)*sizeof(char *)) ;

  paramList[numParams++] = strdup(name) ;
  paramList[numParams] = NULL ;

  qsort((void *)paramList, numParams, sizeof(*paramList), compare) ;

  return 1 ;
}
/**********************************************************************/
char	**IListNetParams() {
  if (paramList == NULL) {
    paramList = (char **)malloc(sizeof(char *)) ;
    paramList[numParams++] = NULL ;
  }
  return paramList ;
}
/**********************************************************************/
static char	**minimizeParamList = NULL ;
static int	numMinimizeParams = 0 ;
/**********************************************************************/
int	IAddMinimizeParam(name) 
  char	*name ;
{
  if (minimizeParamList == NULL)
    minimizeParamList = (char **)malloc(2*sizeof(char *)) ;
  else
    minimizeParamList = (char **)realloc(minimizeParamList, 
					 (numMinimizeParams+2)*sizeof(char *));

  minimizeParamList[numMinimizeParams++] = strdup(name) ;
  minimizeParamList[numMinimizeParams] = NULL ;

  qsort((void *)minimizeParamList, numMinimizeParams, 
	sizeof(*minimizeParamList), compare) ;

  return 1 ;
}
/**********************************************************************/
char	**IListMinimizeParams() {
  if (minimizeParamList == NULL) {
    minimizeParamList = (char **)malloc(sizeof(char *)) ;
    minimizeParamList[numMinimizeParams++] = NULL ;
  }
  return minimizeParamList ;
}
/**********************************************************************/
  
#if 0
char *itoa(i)
int i;
{
  sprintf(itf_value_buffer, "%d", i);
  return itf_value_buffer;
}
#else
#define BASE	10
char *itoa(i)
int i;
{
  char *ptr = itf_value_buffer ;
  int	divisor ;

  if (i < 0) {
    *(ptr++) == '-' ;
    i = -i ;
  }

  for (divisor = BASE ; i / divisor > 0 ; divisor *= BASE) 
    ;

  while(divisor > 1) {
    divisor /= BASE ;
    (*ptr++) = '0' + i / divisor ;
    i = i % divisor ;
  }

  *(ptr) = '\0' ;

  return itf_value_buffer;
}
#endif

static Boolean atob(value)
char *value;
{
  return atoi(value)>0 || *value=='T' || *value=='t' || *value=='+';
}

#ifdef OBSELETE
char *ltoa(i)
long int i;
{
  sprintf(itf_value_buffer, "%d", i);
  return itf_value_buffer;
}

char *ftoa(f)
float f;
{
  (void) sprintf(itf_value_buffer, "%g", f);
  return itf_value_buffer;
}

char *dtoa(f)
double f;
{
  (void) sprintf(itf_value_buffer, "%lg", f);
  return itf_value_buffer;
}

char *btoa(b)
int b;
{
  if (b)
    strcpy(itf_value_buffer, "true");
  else
    strcpy(itf_value_buffer, "false");
  return itf_value_buffer;
}

double atod(value)
char *value;
{
  return atof(value);
}
#endif				/* OBSELETE */
