
/**********************************************************************
 * $Id: background.c,v 1.2 92/11/30 11:39:37 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#include <errno.h>

#include "itf.h"

#ifndef MIN
#define MIN(x,y)	((x) < (y) ? (x) : (y))
#endif

extern int	errno ;

static char	*createScriptFile() ;

/*********************************************************************
 *	Name:		command_background
 *	Description:	backgrounds xerion, and continues to read 
 *			commands from a file
 *	Parameters:
 *	  int	tokc - the number of command line tokens
 *	  char	*tokv[] - the vector of command line tokens
 *	Return Value:
 *	  int	command_background - 1 on success, 0 on error
 *********************************************************************/
int	command_background(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  extern char	*sys_errlist[] ;

  struct stat	statbuf ;
  SignalHandler handler ;
  FILE	*out_file = NULL ;
  FILE	*in_file = NULL ;

  char	*in_name ;
  int	out_d,in_d ;

  IUsage("[infile] [outfile]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("reconnects stdin and stdout to files") ;
    IHelp
      (IHelpArgs,
       "Disconnects stdin and stdout from the  terminal and connects them to",
       "files.  This allows  the process to be  stopped and backgrounded and",
       "the user can logout.",
       "",
       "The command can be used in the following ways:",
       "",
       "  background infile outfile",
       "  background infile > outfile",
       "  background > outfile         (reads input from terminal until EOF)",
       "  background outfile           (reads input from terminal until EOF)",
       "",
       "USE WITH FEP",
       "If you are running the simulator under fep (a front end command line",
       "processor), you need to  exit from fep to  run the simulator  in the",
       "background.  Do  this by typing \"fep-exit\",  which will cause fep to",
       "exit.",
       "",
       "The signals SIGTERM  and SIGHUP are  set to  be ignored, unless  the",
       "action specified for them is  not the default action, in  which case",
       "they are unchanged.  This is done because fep sends these signals in",
       "an effort to kill the program when \"fep-exit\" is typed.",
       NULL) ;
    return 1 ;
  }

  /* check if we have output redirection - if we do, use that for outfile */
  if (current_outstream != command_stdout)
    out_file = current_outstream->file ;

  /* check if we have input redirection - if we do, use that for infile */
  if (current_instream != command_stdin)
    in_file = current_instream->file ;

  if ((tokc + (out_file != NULL) + (in_file != NULL)) > 3)
    IErrorAbort("Too many arguments!") ;


  /* If we don't already have an outfile, look for one as the LAST argument */
  if (out_file == NULL) {
    if (tokc < 2) 
      IErrorAbort("Expecting an outfile name") ;

    if (stat(tokv[tokc-1], &statbuf) == 0)
      IErrorAbort("\"%s\" already exists - can't use as outfile",
		  tokv[tokc-1]) ;

    out_file = fopen(tokv[tokc-1], "w") ;
    if (out_file == NULL)
      IErrorAbort("Cannot open \"%s\" for writing: %s",
		  tokv[tokc-1], sys_errlist[errno]) ;
    --tokc ;
  }
  
  /***
   * If we don't already have an input, look for one as the last argument,
   * or read from stdin and store in in a temporary file.
   */
  if (in_file == NULL) {
    if (tokc > 1) {
      in_name = tokv[1] ;
      --tokc ;
    } else {
      in_name = createScriptFile() ;
      if (in_name == NULL) {
	fclose(out_file) ;
	IErrorAbort("Cannot generate temporary file name") ;
      }
    }

    in_file = fopen(in_name, "r") ;
    if (in_file==NULL) {
      fclose(out_file) ;
      IErrorAbort("cannot open \"%s\" for reading: %s",
		  in_name, sys_errlist[errno]) ;
    }
  }

  if (tokc > 1) {
    fclose(out_file) ;
    fclose(in_file) ;
    IErrorAbort("Too many arguments") ;
  }

  in_d  = fileno(in_file) ;
  out_d = fileno(out_file) ;

  if (dup2(in_d, 0) == -1)
    warn("background", 
	 "cannot join infile and stdin: %s", sys_errlist[errno]) ;
  
  if (dup2(out_d,1) == -1)
    warn("background",
	 "cannot join outfile and stdout: %s", sys_errlist[errno]) ;

  handler = (SignalHandler)signal(SIGHUP, SIG_IGN) ;
  if (handler != (SignalHandler)SIG_DFL) 
    signal(SIGHUP,handler) ;
  else 
    fprintf(stderr, "Ignoring signal SIGHUP\n") ;

  handler = (SignalHandler)signal(SIGTERM, SIG_IGN) ;
  if (handler != (SignalHandler)SIG_DFL) 
    signal(SIGTERM, handler) ;
  else 
    fprintf(stderr,"Ignoring signal SIGTERM\n") ;

  fprintf(stderr, "Now type \"fep-exit\" if you are using fep, or else type CTRL-Z and \"bg\"\n") ;

  if (dup2(1, 2) == -1)
    warn("background",
	 "cannot join stdout and stderr: %s", sys_errlist[errno]) ;

  destroyGraphics() ;

  ISetValue("echo", "true") ;
  return 1 ;
}
/********************************************************************/


/*******************************************************************
 *	Name:		createScriptFile
 *	Description:	reads commands from stdin and writes them to
 *			a temporary file. Terminated by EOF
 *	Parameters:	NONE
 *	Return Value:
 *	  static char	*createScriptFile - name of the script file,
 *				NULL if couldn't create the file.
 *******************************************************************/
static char	*createScriptFile()
{
  static char	namebuf[L_tmpnam] ;

  struct stat	statbuf ;
  FILE 		*tmp ;
  int		c, size ;

  /* IErrorAbort("expecting an infile name") ;*/
  if (tmpnam(namebuf) == NULL)
    return NULL ;
  tmp = fopen(namebuf, "w") ;
  if (tmp == NULL)
    return NULL ;

  fprintf(stderr, "Type the commands to be executed, end with CTRL-D\n") ;

  for (size = 0 ; (c = getc(din)) != EOF ; ++size)
    putc(c, tmp) ;
  fclose(tmp) ;

  /*
   * now we have to wait for the file to be available for reading
   * this takes time on an nfs system
   */
  c = 0 ;
  do {
    sleep(1) ;
    (void) stat(namebuf, &statbuf) ;
    if (statbuf.st_size < size)
      fprintf(stderr, "Waiting for \"%s\" to appear\n", namebuf) ;
  } while (++c < 100 && statbuf.st_size < size) ;

  return namebuf ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		command_nice
 *	Description:	changes the nice value for the simulator
 *	Parameters:
 *	  int	tokc - the number of command line tokens
 *	  char	*tokv[] - the vector of command line tokens
 *	Return Value:
 *	  int command_nice - 1 on success, 0 on failure
 *******************************************************************/
int command_nice(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	oldNice ;
  int	increment = 0 ;
  struct ARG_DESC	*argd ;

  argd = StartArgs(tokv[0]) ;
  Args(argd, "[<value>%n %?]", &increment,
       "increment the nice number for the shell by value.") ;
  EndArgs(argd) ;

  if (GiveHelp(tokc)) {
    ISynopsis("set the nice value of the simulator") ;
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "\"Nice\" increments the nice value  of the current  process. It is not",
       "possible to  decrement the nice  value.  If  no arguments are given,",
       "the nice value is not changed, just printed out.",
       NULL) ;
    return 1 ;
  }
  (void) ParseArgs(argd, tokc, tokv, 0) ;

  /* reset errno, since a -1 return may be an actual priority */
  errno = 0 ;
  oldNice = getpriority(PRIO_PROCESS, 0) ;
  if (oldNice == -1 && errno != 0)
    IErrorAbort("Failed to read current nice value: %s", SysError(errno)) ;

  if (increment != 0) {
    int	newNice = MIN(20, oldNice + increment) ;
    if (setpriority(PRIO_PROCESS, 0, newNice) == -1 && errno != 0)
      IErrorAbort("Cannot set nice value to %d: %s",
		  newNice, SysError(errno)) ;
    fprintf(dout, "Old nice value %d, new value %d\n", oldNice, newNice) ;
  } else
    fprintf(dout, "Current nice value %d\n", oldNice) ;
  return 1 ;
}
/******************************************************************/
