/*
 * $Id: arg-store.c,v 1.3 93/02/23 13:17:35 drew Exp $
 * 
 */

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appears in all copies and that 
 * both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of University of Toronto 
 * not be used in advertising or publicity pertaining to distribution 
 * of the software without specific, written prior permission.  
 * University of Toronto makes no representations about the suitability 
 * of this software for any purpose. It is provided "as is" without 
 * express or implied warranty. 
 *
 * UNIVERSITY OF TORONTO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS, IN NO EVENT SHALL UNIVERSITY OF TORONTO BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER 
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF 
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 **********************************************************************/

#include <xerion/useful.h>
#include "arg-desc.h"
#include "arg-actual.h"
#include "arg-store.h"

extern long strtol ARGS((const char*, char**, int));

/*
**  ARGtype -- argument translation routines.
**
**	Each of these converts a parameter value to the internal form,
**	including validity checking.  Their parameters and return values
**	all behave identically.
**
**	Parameters:
**		arg_simple -- the argument descriptor for this parameter.
**		arg -- a pointer to the string input value.
**		flags - COPYSTR
**			-- the value will be destroyed later,
**			and so should be copied if it will be retained
**			(as for a string).
**			OPTIONAL
**			-- no error messages are given on failure
**			CHECK
**			-- nothing at all is changed, and TRUE is
**			returned if the value can be converted
**			
**
**	Returns:
**		TRUE -- if the conversion was successful.  The actual
**			value should be stored in the location indicated
**			by arg_simple->store.
**		FALSE -- if the conversion failed.  If optional==FALSE,
**			an error message should be posted with Error()
**
**	Side Effects:
**		The value should be stored through arg_simple->store.
*/

enum ARG_TYPE arg_type_by_char[128];
ArgStoreFunction arg_store_function_by_type[NUMBER_ARG_TYPES];
char* arg_type_name[NUMBER_ARG_TYPES];
char arg_type_char[NUMBER_ARG_TYPES];
int is_string_type[NUMBER_ARG_TYPES];
int arg_types_initialized = 0;

autoinit_argtypes()
{
  int i;
  arg_types_initialized = 1;
  for (i=0;i<128;i++)
    arg_type_by_char[i] = ARG_BAD_TYPE;
  for (i=0;i<NUMBER_ARG_TYPES;i++) {
    arg_store_function_by_type[i] = NULL;
    arg_type_name[i] = "bad type";
    arg_type_char[i] = '\0';
  }
  arg_type_by_char['i'] = ARG_INTEGER;
  arg_type_by_char['f'] = ARG_FLOAT;
  arg_type_by_char['r'] = ARG_REAL;
  arg_type_by_char['D'] = ARG_DOUBLE;
  arg_type_by_char['s'] = ARG_STRING;
  arg_type_by_char['c'] = ARG_CHAR;
  arg_type_by_char['B'] = ARG_BOOLEAN;
  arg_type_by_char['n'] = ARG_NATURAL;
  arg_type_by_char['h'] = ARG_SHORT;
  arg_type_by_char['l'] = ARG_LONG;
  arg_type_by_char['F'] = ARG_FORMAT;
  arg_type_by_char['A'] = ARG_AREA_FORMAT;
  arg_type_by_char['R'] = ARG_RANGE_STRING;
  arg_type_by_char['v'] = ARG_VAR_NAME;
  arg_type_by_char['m'] = ARG_MEMBER_NAME;
  arg_type_by_char['I'] = ARG_FILE_IN;
  arg_type_by_char['O'] = ARG_FILE_OUT;
  for (i=0;i<128;i++)
    if (arg_type_by_char[i]!=ARG_BAD_TYPE)
      arg_type_char[(int) arg_type_by_char[i]] = i;

  arg_type_name[(int)ARG_LITERAL] = "literal";
  arg_type_name[(int)ARG_INTEGER] = "integer";
  arg_type_name[(int)ARG_FLOAT] = "float";
  arg_type_name[(int)ARG_REAL] = "Real";
  arg_type_name[(int)ARG_DOUBLE] = "double";
  arg_type_name[(int)ARG_STRING] = "String";
  arg_type_name[(int)ARG_CHAR] = "char";
  arg_type_name[(int)ARG_BOOLEAN] = "Boolean";
  arg_type_name[(int)ARG_NATURAL] = "natural";
  arg_type_name[(int)ARG_SHORT] = "short";
  arg_type_name[(int)ARG_LONG] = "long";
  arg_type_name[(int)ARG_FORMAT] = "format";
  arg_type_name[(int)ARG_AREA_FORMAT] = "area format";
  arg_type_name[(int)ARG_RANGE_STRING] = "range string";
  arg_type_name[(int)ARG_VAR_NAME] = "var name";
  arg_type_name[(int)ARG_MEMBER_NAME] = "member name";
  arg_type_name[(int)ARG_FILE_IN] = "input file";
  arg_type_name[(int)ARG_FILE_OUT] = "output file";

  arg_store_function_by_type[(int)ARG_INTEGER] = argInteger;
  arg_store_function_by_type[(int)ARG_FLOAT] = argFloat;
  arg_store_function_by_type[(int)ARG_REAL] = argReal;
  arg_store_function_by_type[(int)ARG_DOUBLE] = argDouble;
  arg_store_function_by_type[(int)ARG_STRING] = argString;
  arg_store_function_by_type[(int)ARG_CHAR] = argChar;
  arg_store_function_by_type[(int)ARG_BOOLEAN] = argBoolean;
  arg_store_function_by_type[(int)ARG_NATURAL] = argNatural;
  arg_store_function_by_type[(int)ARG_SHORT] = argShort;
  arg_store_function_by_type[(int)ARG_LONG] = argLong;
  arg_store_function_by_type[(int)ARG_FORMAT] = argFormatStr;
  arg_store_function_by_type[(int)ARG_AREA_FORMAT] = argAreaFormatStr;
  arg_store_function_by_type[(int)ARG_RANGE_STRING] = argRangeStr;
  arg_store_function_by_type[(int)ARG_VAR_NAME] = argVarName;
  arg_store_function_by_type[(int)ARG_MEMBER_NAME] = argMemberName;
  arg_store_function_by_type[(int)ARG_FILE_IN] = argFileIn;
  arg_store_function_by_type[(int)ARG_FILE_OUT] = argFileOut;

  is_string_type[(int)ARG_INTEGER] = 0;
  is_string_type[(int)ARG_FLOAT] = 0;
  is_string_type[(int)ARG_REAL] = 0;
  is_string_type[(int)ARG_DOUBLE] = 0;
  is_string_type[(int)ARG_STRING] = 1;
  is_string_type[(int)ARG_CHAR] = 0;
  is_string_type[(int)ARG_BOOLEAN] = 0;
  is_string_type[(int)ARG_NATURAL] = 0;
  is_string_type[(int)ARG_SHORT] = 0;
  is_string_type[(int)ARG_LONG] = 0;
  is_string_type[(int)ARG_FORMAT] = 1;
  is_string_type[(int)ARG_AREA_FORMAT] = 1;
  is_string_type[(int)ARG_RANGE_STRING] = 1;
  is_string_type[(int)ARG_VAR_NAME] = 1;
  is_string_type[(int)ARG_MEMBER_NAME] = 1;
  is_string_type[(int)ARG_FILE_IN] = 0;
  is_string_type[(int)ARG_FILE_OUT] = 0;

}


/***
 * Macro to add a value into a simple arg - three cases have to be
 * handled - ordinary scalar, vector of String types,
 * and vector of other types.
 */
#define ADD_VALUE(ARG_SIMPLE, TYPE, VAL) \
  if (!ARG_SIMPLE->vector) {\
    memcpy((char*)(ARG_SIMPLE->store), (char*)&VAL, sizeof(TYPE));\
  } else if (is_string_type[(int)ARG_SIMPLE->type]) {\
    TYPE **argv = *(TYPE ***) ARG_SIMPLE->store;\
    TYPE *cp;\
    int n;\
    if (argv!=NULL) {\
      n = ARG_SIMPLE->vector_len;\
      argv = ITempRealloc(TYPE *, argv, n+2);\
    } else {\
      n = ARG_SIMPLE->vector_len = 0;\
      argv = ITempCalloc(TYPE *, 2);\
    }\
    cp = ITempCalloc(TYPE, 1);\
    memcpy((char*)cp, (char*)&(VAL), sizeof(TYPE));\
    argv[n] = cp;\
    argv[n+1] = NULL;\
    *(TYPE ***) ARG_SIMPLE->store = argv;\
    ARG_SIMPLE->vector_len++;\
    if (ARG_SIMPLE->store_len!=NULL)\
      *ARG_SIMPLE->store_len = ARG_SIMPLE->vector_len;\
  } else {\
    TYPE *argv = *(TYPE **) ARG_SIMPLE->store;\
    int n;\
    if (argv!=NULL) {\
      n = ARG_SIMPLE->vector_len;\
      argv = ITempRealloc(TYPE , argv, n+2);\
    } else {\
      n = ARG_SIMPLE->vector_len = 0;\
      argv = ITempCalloc(TYPE , 2);\
    }\
    memcpy((char*)(argv+n), (char*)&(VAL), sizeof(TYPE));\
    memset((char*)(argv+n+1), '\0', sizeof(TYPE));\
    *(TYPE **) ARG_SIMPLE->store = argv;\
    ARG_SIMPLE->vector_len++;\
    if (ARG_SIMPLE->store_len!=NULL)\
      *ARG_SIMPLE->store_len = ARG_SIMPLE->vector_len;\
  }\


ArgStoreFunction arg_store_function[] =
{
  NULL,
  NULL,
  argInteger,
  argFloat,
  argReal,
  argDouble,
  argString,
  argChar,
  argBoolean,
  argNatural,
  argShort,
  argLong,
  argFormatStr,
  argAreaFormatStr,
  argRangeStr,
  argVarName,
  argMemberName
};

Boolean
StoreArg(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  if (   (int)(arg_simple->type) < 0
      || (int)(arg_simple->type) > (int) ARG_MEMBER_NAME
      || arg_store_function[(int)arg_simple->type] == NULL)
    ArgErrorAbort(global, "bad type %d in simple arg\n");
  return
    arg_store_function[(int)arg_simple->type](arg_simple, arg, flags, global);
}

Boolean
argChar(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  int status = FALSE;
  char c;
  
  if (strlen(arg) == 2 && arg[0]=='^') {
    c = arg[1] ^ '@';
    status = TRUE;
  } else if (strlen(arg) > 1 && arg[0]=='\\') {
    c = (int) strtol(&arg[1], &argp, 8);
    if (*argp == '\0')
      status = TRUE;
  } else if (strlen(arg) == 1) {
    c = *arg;
    status = TRUE;
  }
  
  if (status == TRUE) {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, char, c);
  } else if (!(flags&(OPTIONAL|ONLYCHECK))) {
    ArgError(global, "invalid character argument '%s' for <%s>", arg, arg_simple->prompt);
    return FALSE;
  }
  return status;
}

Boolean
argString(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *cp;
  
  if (flags&ONLYCHECK)
    return TRUE;

  if (flags&COPYSTR) {
    int i;
    i = strlen(arg) + 1;
    cp = ITempCalloc(char, i);
    strcpy(cp, arg);
  } else {
    cp = arg;
  }

  if (arg_simple->vector) {
    char **argv = *(char ***) arg_simple->store;
    int n;
    if (argv==NULL) {
      argv = ITempCalloc(char *, 2);
      arg_simple->vector_len = n = 0;
    } else {
      n = arg_simple->vector_len;
      argv = ITempRealloc(char *, argv, n+2);
    }
    argv[n] = cp;
    argv[n+1] = NULL;
    *(char ***) arg_simple->store = argv;
    arg_simple->vector_len++;
    if (arg_simple->store_len!=NULL)
      *arg_simple->store_len = arg_simple->vector_len;
  } else
    *(char **) arg_simple->store = cp;

  return (TRUE);
}

Boolean
argRangeStr(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  int is_range=1;
  char *p = arg;
  /* go through a simple fsm to recognize a range */
  if (*p=='\0')
    is_range=0;
  else if (*p=='X')
    p++;
  else {
    while (isdigit(*p)) p++;
    if (*p=='-') p++;
    while (isdigit(*p)) p++;
    if (*p=='+') p++;
    while (isdigit(*p)) p++;
  }
  if (*p!='\0')
    is_range = 0;
  if (!is_range) {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid range argument \"%s\" for <%s>", arg, arg_simple->prompt);
    return FALSE;
  }
  return argString(arg_simple, arg, flags, global);
}

Boolean
argFormatStr(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  int count=0;
  char *p = arg;
  char prev = '\0';
  /* go through the string & count the active %'s */
  for (;*p;p++) {
    if (*p=='%')
      if (prev=='%') {
	count--;
	prev = '\0';
	continue;
      } else if (prev!='\\')
	count++;
    prev = *p;
  }
  if (count!=1) {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid print format argument \"%s\" for <%s>",
	       arg, arg_simple->prompt);
    return FALSE;
  }
  return argString(arg_simple, arg, flags, global);
}

Boolean
argMemberName(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  if (arg[0]=='.' || (arg[0]=='-' && arg[1]=='>'))
    return argString(arg_simple, arg, flags, global);
  else {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid field name argument \"%s\" for <%s>",
	       arg, arg_simple->prompt);
    return FALSE;
  }
}

Boolean
argVarName(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  if (IsLegalVarName(arg))
    return argString(arg_simple, arg, flags, global);
  else {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid field name argument \"%s\" for <%s>",
	       arg, arg_simple->prompt);
    return FALSE;
  }
}

Boolean
argAreaFormatStr(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *p = arg;
  int good = 0;
  if (isdigit(*p)) {
    while (isdigit(*p)) p++;
    if (*p=='x') {
      p++;
      if (isdigit(*p)) {
	while (isdigit(*p)) p++;
	if (*p=='j') p++;
	if (*p=='\0')
	  good = 1;
      }
    }
  }
  if (good)
    return argString(arg_simple, arg, flags, global);
  else {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid area format argument \"%s\" for <%s>",
	       arg, arg_simple->prompt);
    return FALSE;
  }
}

/*ARGSUSED*/
Boolean
argInteger(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  int value;
  
  value = strtol(arg, &argp, 0);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid integer argument \"%s\" for <%s>", arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, int, value);
    /* *(int *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argNatural(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  int value;
  
  value = strtol(arg, &argp, 0);
  if (*argp != '\0' || value<0) {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid natural number argument \"%s\" for <%s>",
		  arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, int, value);
    /* *(int *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argShort(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  short value;
  
  value = strtol(arg, &argp, 0);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid integer argument \"%s\" for <%s>", arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, short, value);
    /* *(short *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argLong(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  long value;
  
  value = strtol(arg, &argp, 0);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid integer argument \"%s\" for <%s>", arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, long, value);
    /* *(long *) arg_simple->store = value; */
    return (TRUE);
  }
}

typedef struct booltab
{
	char	*bname;		/* string to match against */
	char	bneedmatch;	/* number of characters that must match */
	Boolean	bval;		/* value to use */
} booltab ;

static struct booltab	BoolTab[] =
{
	"yes",		1,	TRUE,
	"no",		1,	FALSE,
	"true",		1,	TRUE,
	"false",	1,	FALSE,
	"on",		2,	TRUE,
	"off",		3,	FALSE,
	NULL
};

/*ARGSUSED*/
Boolean
argBoolean(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  struct booltab *b;
  char *cp;
  int l;
  
  /* convert input to lower case */
  for (l = 0, cp = arg; *cp != '\0'; l++, cp++) {
    if (isupper(*cp))
      *cp = tolower(*cp);
  }
  
  /* search for a match in the table */
  for (b = BoolTab; b->bname != NULL; b++) {
    /* if too short, don't even bother trying */
    if (l < b->bneedmatch)
      continue;
    
    if (memcmp(arg, b->bname, l) == 0) {
      /* got a match */
      if (!(flags&ONLYCHECK))
	ADD_VALUE(arg_simple, Boolean, b->bval);
      /* *(Boolean *) arg_simple->store = b->bval; */
      return (TRUE);
    }
  }
  if (!(flags&(OPTIONAL|ONLYCHECK)))
    ArgError(global, "invalid Boolean argument \"%s\" for <%s>", arg, arg_simple->prompt);
  return (FALSE);
}

/*ARGSUSED*/
Boolean
argDouble(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  double value;

  value = strtod(arg, &argp);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid floating point argument \"%s\" for <%s>",
		  arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, double, value);
    /* *(double *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argReal(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  Real value;

  value = strtod(arg, &argp);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid floating point argument \"%s\" for <%s>",
		  arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, Real, value);
    /* *(Real *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argFloat(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  char *argp;
  float value;

  value = strtod(arg, &argp);
  if (*argp != '\0') {
    if (!(flags&(OPTIONAL|ONLYCHECK)))
      ArgError(global, "invalid floating point argument \"%s\" for <%s>",
		  arg, arg_simple->prompt);
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, float, value);
    /* *(float *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argFileIn(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  FILE *file;
  extern int errno;
  extern int sys_nerr;
  extern char *sys_errlist[];

  file = fopen(arg, "r");
  if (file==NULL) {
    if (!(flags&(OPTIONAL|ONLYCHECK))) {
      ArgError(global, "cannot open file \"%s\" for <%s> for reading: %s",
	       arg, arg_simple->prompt,
	       errno<sys_nerr ? sys_errlist[errno] : "unknown error");
    }
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, FILE*, file)
    else
      fclose(file);
    /* *(int *) arg_simple->store = value; */
    return (TRUE);
  }
}

/*ARGSUSED*/
Boolean
argFileOut(arg_simple, arg, flags, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
int flags;
struct GLOBAL *global;
{
  FILE *file;
  extern int errno;
  extern int sys_nerr;
  extern char *sys_errlist[];

  file = fopen(arg, "w");
  if (file==NULL) {
    if (!(flags&(OPTIONAL|ONLYCHECK))) {
      ArgError(global, "cannot open file \"%s\" for <%s> for writing: %s",
	       arg, arg_simple->prompt,
	       errno<sys_nerr ? sys_errlist[errno] : "unknown error");
    }
    return (FALSE);
  } else {
    if (!(flags&ONLYCHECK))
      ADD_VALUE(arg_simple, FILE*, file)
    else
      fclose(file);
    /* *(int *) arg_simple->store = value; */
    return (TRUE);
  }
}
