/*
 * $Id: arg-print.c,v 1.3 92/11/30 11:39:31 drew Exp $
 * 
 */

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appears in all copies and that 
 * both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of University of Toronto 
 * not be used in advertising or publicity pertaining to distribution 
 * of the software without specific, written prior permission.  
 * University of Toronto makes no representations about the suitability 
 * of this software for any purpose. It is provided "as is" without 
 * express or implied warranty. 
 *
 * UNIVERSITY OF TORONTO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS, IN NO EVENT SHALL UNIVERSITY OF TORONTO BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER 
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF 
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 **********************************************************************/

#include <xerion/useful.h>
#include "argf.h"
#include "arg-desc.h"
#include "arg-actual.h"
#include "arg-store.h"
#include "arg-print.h"

/***
 * Functions in this file print structures out as ascii text,
 * controlled by the PRINT_ flags.
 * The ascii text is put in a buffer, and all of these functions
 * assume that buf[-1] is accessible.  When they are called from
 * outside they should be called as follows:
 *
 * char buffer[SIZE];
 * buffer[0] = '\n';
 * ... = SprintArgList(buffer+1, arg_list, flags, indent);
 *
 * i.e. so that buf[-1] is legal and contains '\n'
 */

#define strappendchar(buf, c) (*(buf) = (c), *(++buf) = '\0')

static char *Indent ARGS((char *, int, int));
static char *NewLineIndent ARGS((char *, int, int));
static char *NewLine ARGS((char *, int));
static char *EraseWhiteSpace ARGS((char *, int));
static int AdvanceToColumn ARGS((int, int, FILE *));
static int help_colon_pos = 30;
static int line_length = 79;
static int tabstop = 8;

void PrintArgDesc(arg_desc, flags, indent, file)
struct ARG_DESC *arg_desc;
int flags;
int indent;
FILE *file;
{
  char buffer[10000];
  char *end;
  buffer[0] = '\n';	/* so that indent works properly */
  fprintf(file, "Arguments for \"%s\"\n",arg_desc->name);
  end = SprintArgDesc(buffer+1, arg_desc, flags, indent);
  NewLine(end, flags);
  fputs(buffer+1, file);
}

void PrintArgHelp(arg_desc, file, print_caption)
struct ARG_DESC *arg_desc;
FILE *file;
int print_caption;
{
  char buffer[10000];
  char *p;
  int flags = (PRINT_OBJECTS|PRINT_ONEPERLINE|PRINT_HELP
	       |PRINT_TYPES|PRINT_OPTION_SIGN);
  int indent = 1;
  int linepos, i, after_colon = 0;
  char *next_space;
  buffer[0] = '\n';	/* so that indent works properly */
  if (print_caption)
    fprintf(file, "Arguments and options for \"%s\"\n", arg_desc->name);
  p = SprintArgDesc(buffer+1, arg_desc, flags, indent);
  NewLine(p, flags);
  p = buffer+1;
  linepos = 0;
  while (*p) {
    if (*p==':' && p[-1]==' ' && p[1]==' ') {
      /* want to advance to column <help_colon_pos> */
      linepos = AdvanceToColumn(linepos, help_colon_pos, file);
      after_colon = 1;
    }
    if (*p==' ') {
      /* maybe we want to wrap the line */
      next_space = p+1;
      while (*next_space!='\0' && !isspace(*next_space))
	next_space++;
      if ((next_space-p)+linepos > line_length) {
	fputc('\n', file);
	linepos = 0;
	if (after_colon)
	  linepos = AdvanceToColumn(linepos, help_colon_pos+2, file);
      } else
	fputc(*p, file);
    } else
      fputc(*p, file);
    if (*p=='\n') {
      linepos = 0;
      after_colon = 0;
    } else
      linepos++;
    p++;
  }
}

static int AdvanceToColumn(current, column, file)
int current;
int column;
FILE *file;
{
  int next_tab_stop, i;
  if (current > column) {
    fputc('\n', file);
    current = 0;
  }
  while (current < column) {
    next_tab_stop = (current/tabstop+1)*tabstop;
    if (next_tab_stop<=column) {
      fputc('\t', file);
      current = next_tab_stop;
    } else {
      for (i=current;i<column;i++) {
	fputc(' ', file);
	current++;
      }
    }
  }
  return current;
}

void PrintArgUsage(arg_desc, file, print_usage_label)
struct ARG_DESC *arg_desc;
FILE *file;
int print_usage_label;
{
  char buffer[10000];
  char *p;
  int flags = (PRINT_OBJECTS|PRINT_BRACKETS|PRINT_OPTION_SIGN|PRINT_GROUPPERLINE);
  int indent = 1;
  int linepos = 0;
  char *next_line_start;
  char *next_line_end;
  int next_line_len;
  buffer[0] = '\n';	/* so that indent works properly */
  p = buffer+1;
  if (print_usage_label)
    p = strcpyend(p, "Usage: ");
  else
    p = strcpyend(p, "    ");
  (void) sprintf(p, "%s ", arg_desc->name);
  p += strlen(p);
  p = SprintArgDesc(p, arg_desc, flags, indent);
  NewLine(p, flags);
  p = buffer+1;
  linepos = 0;
  while (*p==' ') {
    fputc(*p, file);
    linepos++;
    p++;
  }
  while (*p) {
    linepos++;
    if (*p=='\n') {
      /* maybe we don't want to put out a newline, instead just a space */
      next_line_start = p+1;
      while (*next_line_start && *next_line_start==' ')
	next_line_start++;
      for (next_line_end=next_line_start+1;
	   *next_line_end && *next_line_end!='\n';
	   next_line_end++) ;
      if (*next_line_start)
	next_line_len = next_line_end - next_line_start;
      else
	next_line_len = 0;
      if (next_line_len>0 && (next_line_len + linepos + 10< line_length)) {
	fputc(' ', file);
      } else {
	if (next_line_len>0)
	  fputs("\n      ", file);
	else
	  fputc('\n', file);
	linepos = 0;
      }
    } else
      fputc(*p, file);
    if (*p==' ')
      while (*p && *p==' ')
	p++;
    else
      p++;
  }
}

void PrintArgValues(arg_desc, file)
struct ARG_DESC *arg_desc;
FILE *file;
{
  PrintArgDesc(arg_desc,
	       PRINT_OBJECTS|PRINT_ONEPERLINE|PRINT_TYPES
	       |PRINT_ACTUALS|PRINT_NOT_INDENT|PRINT_ACTUAL_STATUS,
	       1, file);
}

void PrintArgTree(arg_desc, file)
struct ARG_DESC *arg_desc;
FILE *file;
{
  PrintArgDesc(arg_desc, PRINT_OBJECTS|PRINT_BRACKETS|PRINT_STRUCTS
	       |PRINT_ACTUALS|PRINT_OPTION_SIGN|PRINT_SEQS, 0, file);
}

/***
 * WhiteSpace - makes sure there is some whitespace in the
 * buffer
 */
static char *WhiteSpace(buf)
char *buf;
{
  if (!isspace(buf[-1]))
    strappendchar(buf, ' ');
  return buf;
}

/***
 * Indent(buf, indent, flags)
 * 'indent' spaces are put into buf, if it is positioned at
 * the start of a line (detected by checking if buf[-1]=='\n')
 */
static char *Indent(buf, indent, flags)
char *buf;
int indent;
int flags;
{
  int i;
  if (buf[-1]!='\n')
    return buf;
  else {
    if (!(flags&(PRINT_NOT_INDENT|PRINT_NOT_NEWLINE))) {
      for (i=0;i<(indent*4);i++)
	*buf++ = ' ';
    }
    *buf = 0;
    return buf;
  }
}

char *InitBuf(buf)
char *buf;
{
  buf[0] = '\n';
  buf[1] = '\0';
  return buf+1;
}

/***
 * NewLineIndent(buf, indent, flags)
 * a NEWLINE + 'indent' spaces are put into buf
 */
static char *NewLineIndent(buf, indent, flags)
char *buf;
int indent;
{
  int i;
  buf = EraseWhiteSpace(buf, flags);
  if (!(flags&PRINT_NOT_NEWLINE)) {
    if (buf[-1]!='\n')
      *buf++ = '\n';
    if (!(flags&PRINT_NOT_INDENT)) {
      for (i=0;i<(indent*4);i++)
	*buf++ = ' ';
    }
  } else if (buf[-1]!=' ' && buf[-1]!='\n') {
    *buf++ = ' ';
  }
  *buf = 0;
  return buf;
}

/***
 * NewLineIndent(buf)
 * a NEWLINE is put into buf
 */
static char *NewLine(buf, flags)
char *buf;
{
  int i;
  buf = EraseWhiteSpace(buf, flags);
  if (!(flags&PRINT_NOT_NEWLINE)) {
    if (buf[-1]!='\n')
      *buf++ = '\n';
  } else if (buf[-1]!=' ' && buf[-1]!='\n') {
    *buf++ = ' ';
  }
  *buf = 0;
  return buf;
}

/***
 * EraseWhiteSpace(buf, flags)
 * move backwards over white space
 */
static char *EraseWhiteSpace(buf, flags)
char *buf;
{
  int i;
  while (buf[-1]==' ')
    buf--;
  *buf = 0;
  return buf;
}

char * SprintArgDesc(buf, arg_desc, flags, indent)
char *buf;
struct ARG_DESC *arg_desc;
int flags;
int indent;
{
  int i;
  if (flags&PRINT_STRUCTS)
    indent += 1;
  for (i=0;i<arg_desc->n_lists;i++) {
    if (flags&PRINT_STRUCTS) {
      buf = NewLineIndent(buf, indent-1, flags);
      sprintf(buf, "ARG_DESC list %d\n", i);
      buf += strlen(buf);
    }
    buf = SprintArgList(buf, arg_desc->arg_list[i], flags, indent);
    if (flags&PRINT_GROUPPERLINE)
      buf = NewLine(buf, flags);
  }
  return buf;
}

char * SprintArgList(buf, arg_list, flags, indent)
char *buf;
struct ARG_LIST *arg_list;
int flags;
int indent;
{
  if (arg_list==NULL)
    return buf;
  if (flags&PRINT_STRUCTS) {
    buf = NewLineIndent(buf, indent, flags);
    sprintf(buf, "ARG_LIST %d parent=%d\n", arg_list->id,
	    (arg_list->parent!=NULL ? arg_list->parent->id : 0));
    buf += strlen(buf);
    indent++;
  }
  buf = SprintArgGroup(buf, arg_list->group, flags, indent);
  if (arg_list->next!=NULL) {
    if (!(flags&INSIDE_GROUP))
      buf = NewLine(buf, flags);
    buf = SprintArgJoin(buf, arg_list->join, flags, indent);
    if (flags&PRINT_STRUCTS)
      indent--;
    buf = SprintArgList(buf, arg_list->next, flags, indent);
  }

  return EraseWhiteSpace(buf, flags);
}

char * SprintArgOption(buf, arg_option, flags, indent)
char *buf;
struct ARG_OPTION *arg_option;
int flags;
int indent;
{
  char *pre_indent, *post_indent=NULL;
  if (flags&PRINT_STRUCTS) {
    buf = NewLineIndent(buf, indent, flags);
    buf = strcpyend(buf, "ARG_OPTION\n");
    indent++;
  }
  pre_indent = buf;
  buf = Indent(buf,indent, flags);
  post_indent = buf;
  if (flags&PRINT_OPTION_SIGN) {
    if (arg_option->is_signed)
      if (use_plus_minus_options)
	buf = strcpyend(buf, "+/-");
      else
	buf = strcpyend(buf, "-[no]");
    else
      if (use_plus_minus_options)
	buf = strcpyend(buf, "+");
      else
	buf = strcpyend(buf, "-");
  } else {
    if (flags&PRINT_OPTION_DASH)
      strappendchar(buf, '-');
    if (flags&PRINT_SPECIFIERS) {
      if (arg_option->no_prefix)
	strappendchar(buf, 'F');
      if (arg_option->is_signed)
	strappendchar(buf, 'S');
      else
	strappendchar(buf, 'P');
    }
  }
  if (flags&PRINT_OPTIONS)
    buf = strcpyend(buf, arg_option->name);
  if (flags&PRINT_TYPES) {
    buf = WhiteSpace(buf);
    strappendchar(buf, '(');
    buf = strcpyend(buf, arg_option->is_signed ? "signed" : "presence");
    strappendchar(buf, ')');
  }
  if (flags&PRINT_ACTUALS) {
    strappendchar(buf, '=');
    if (arg_option->is_signed) {
      if (arg_option->actual&OPTION_ON)
	buf = strcpyend(buf, true_string);
      else if (arg_option->actual&OPTION_OFF)
	buf = strcpyend(buf, false_string);
      else {
	buf = strcpyend(buf, "=unset(default=");
	if (*arg_option->sign==TRUE)
	  buf = strcpyend(buf, true_string);
	else if (*arg_option->sign==FALSE)
	  buf = strcpyend(buf, false_string);
	else if (*arg_option->sign==UNDEFINED)
	  buf = strcpyend(buf, undefined_string);
	else
	  buf = strcpyend(buf, bad_boolean_string);
	buf = strcpyend(buf, ")");
      }
    } else {
      if (arg_option->actual&OPTION_PRESENT)
	buf = strcpyend(buf, "present");
      else
	buf = strcpyend(buf, "missing");
    }
    if (arg_option->presence!=NULL) {
      sprintf(buf, " presence=%d", *arg_option->presence);
      buf += strlen(buf);
    }
    if (arg_option->neg_presence!=NULL) {
      sprintf(buf, " neg_presence=%d", *arg_option->neg_presence);
      buf += strlen(buf);
    }
  }
  if (flags&PRINT_OPTIONS)
    strappendchar(buf, ' ');
  if (arg_option->sign!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(buf, "%S ");
    if (flags&PRINT_ADDRESSES) {
      sprintf(buf, "0x%x ", arg_option->sign);
      buf += strlen(buf);
    }
  }
  if (arg_option->neg_sign!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(buf, "%!S ");
    if (flags&PRINT_ADDRESSES) {
      sprintf(buf, "0x%x ", arg_option->neg_sign);
      buf += strlen(buf);
    }
  }
  if (arg_option->presence!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(buf, "%P ");
    if (flags&PRINT_ADDRESSES) {
      sprintf(buf, "0x%x ", arg_option->presence);
      buf += strlen(buf);
    }
  }
  if (arg_option->neg_presence!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(buf, "%!P ");
    if (flags&PRINT_ADDRESSES) {
      sprintf(buf, "0x%x ", arg_option->neg_presence);
      buf += strlen(buf);
    }
  }
  if (arg_option->help!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(buf, "%? ");
    if (flags&PRINT_ADDRESSES) {
      sprintf(buf+strlen(buf), "0x%x ", arg_option->help);
      buf += strlen(buf);
    }
    if (flags&PRINT_HELP) {
      sprintf(buf, ": %s\n", arg_option->help);
      buf += strlen(buf);
    }
  }

  if (buf==post_indent) {
    buf = pre_indent;
    *buf = '\0';
  } else {
    if (flags&PRINT_ONEPERLINE)
      buf = NewLine(buf, flags);
  }
  
  return EraseWhiteSpace(buf, flags);
}

char * SprintArgGroup(buf, arg_group, flags, indent)
char *buf;
struct ARG_GROUP *arg_group;
int flags;
int indent;
{
  char *pre_indent, *post_indent=NULL;
  if (flags&PRINT_STRUCTS) {
    buf = NewLineIndent(buf, indent, flags);
    buf = strcpyend(buf, "ARG_GROUP");
    if ((flags&PRINT_ACTUALS) && arg_group->presence!=NULL) {
      sprintf(buf," presence=%d", *arg_group->presence);
      buf += strlen(buf);
    }
    if ((flags&PRINT_ACTUALS) && arg_group->neg_presence!=NULL) {
      sprintf(buf," neg_presence=%d", *arg_group->neg_presence);
      buf += strlen(buf);
    }
    strappendchar(buf, '\n');
    indent++;
  }
  pre_indent = buf;
  if (flags&PRINT_PARENS) {
    buf = Indent(buf, indent, flags);
    buf = strcpyend(buf, "( ");
  }
  if (arg_group->tag==ARG_GROUP_SIMPLE) {
    buf = SprintArgSimple(buf, arg_group->u.simple.arg_simple, flags, indent);
  } else if (arg_group->tag==ARG_GROUP_OPTION) {
    if (arg_group->u.option.optional && (flags&PRINT_BRACKETS)) {
      buf = Indent(buf, indent, flags);
      buf = WhiteSpace(buf);
      strappendchar(buf, '[');
    }
    if (arg_group->u.option.option!=NULL)
      buf = SprintArgOption(buf, arg_group->u.option.option, flags, indent);
    if (arg_group->u.option.arg_list!=NULL)
      buf = SprintArgList(buf, arg_group->u.option.arg_list,flags|INSIDE_GROUP,
		    indent+(1-arg_group->u.option.optional));
    if (arg_group->u.option.optional && (flags&PRINT_BRACKETS)) {
      /* buf = WhiteSpace(buf); */
      strappendchar(buf, ']');
    }
  }

  if (arg_group->presence!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(WhiteSpace(buf), "%P");
    if (flags&PRINT_ADDRESSES) {
      sprintf(WhiteSpace(buf), "0x%x", arg_group->presence);
      buf += strlen(buf);
    }
    if (flags&PRINT_ACTUALS) {
      buf = strcatend(WhiteSpace(buf), "presence=");
      buf = strcatend(buf, BooleanString(*arg_group->presence));
    }
  }
  if (arg_group->neg_presence!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(WhiteSpace(buf), "%!P");
    if (flags&PRINT_ADDRESSES) {
      sprintf(WhiteSpace(buf), "0x%x", arg_group->neg_presence);
      buf += strlen(buf);
    }
    if (flags&PRINT_ACTUALS) {
      buf = strcatend(WhiteSpace(buf), "neg_presence=");
      buf = strcatend(buf, BooleanString(*arg_group->neg_presence));
    }
  }

  if (arg_group->help!=NULL) {
    if (flags&PRINT_PERCENTS)
      buf = strcatend(WhiteSpace(buf), "%?");
    if (flags&PRINT_ADDRESSES) {
      sprintf(WhiteSpace(buf), "0x%x", arg_group->help);
      buf += strlen(buf);
    }
    if (flags&PRINT_HELP) {
      sprintf(WhiteSpace(buf), ": %s", arg_group->help);
      buf += strlen(buf);
    }
  }

  if (flags&PRINT_PARENS)
    buf = strcatend(WhiteSpace(buf), ")");

  if ((flags&PRINT_ONEPERLINE) && arg_group->tag==ARG_GROUP_SIMPLE)
    buf = NewLine(buf, flags);

  if (buf==post_indent) {
    buf = pre_indent;
    *buf = '\0';
  }

  return EraseWhiteSpace(buf, flags);
}

char * SprintArgSimple(buf, arg_simple, flags, indent)
char *buf;
struct ARG_SIMPLE *arg_simple;
int flags;
int indent;
{
  char *pre_indent,*post_indent;
  if (flags&PRINT_STRUCTS) {
    buf = NewLineIndent(buf, indent, flags);
    (void) sprintf(buf, "ARG_SIMPLE %d\n", arg_simple->id);
    buf += strlen(buf);
    indent++;
  }
  pre_indent = buf;
  buf = Indent(buf, indent, flags);
  post_indent = buf;
  if (arg_simple->type==ARG_LITERAL) {
    if (flags&PRINT_LITERALS) {
      sprintf(WhiteSpace(buf), "'%s'", arg_simple->prompt);
      buf += strlen(buf);
    }
  } else {
    if (flags&PRINT_SIMPLES) {
      if (buf[-1]!='[')
	buf = WhiteSpace(buf);
      sprintf(buf, "<%s>", arg_simple->prompt);
      buf += strlen(buf);
      if (arg_simple->vector)
	buf = strcatend(buf, "...");
    }
    if (flags&PRINT_TYPES) {
      buf = WhiteSpace(buf);
      strappendchar(buf, '(');
      buf = strcpyend(buf, arg_type_name[(int) arg_simple->type]);
      strappendchar(buf, ')');
    }
    if (flags&PRINT_SPECIFIERS) {
      strappendchar(buf, '%');
      if (arg_simple->verify!=NULL)
	strappendchar(buf, 'V');
      if (arg_type_char[(int)arg_simple->type]=='\0')
	ErrorAbort("bad type specifier %d", arg_simple->type);
      else
	strappendchar(buf, arg_type_char[(int) arg_simple->type]);
    }
    if (arg_simple->verify && (flags&PRINT_ADDRESSES)) {
      sprintf(WhiteSpace(buf), "0x%x", arg_simple->verify);
      buf += strlen(buf);
    }
    if (flags&PRINT_ADDRESSES) {
      sprintf(WhiteSpace(buf), "0x%x", arg_simple->store);
      buf += strlen(buf);
    }
    if (flags&PRINT_ACTUALS) {
      buf = strcpyend(buf, " = ");
      if (arg_simple->vector) {
	generic **ptr;
	strappendchar(buf, '{');
	if (arg_simple->store!=NULL && *(generic***)arg_simple->store!=NULL)
	  for (ptr = *(generic***)arg_simple->store; *ptr!=NULL; ptr++) {
	    buf = SprintArgValue(buf, arg_simple->type, (generic*)ptr, flags);
	    if (ptr[1]!=NULL)
	      strappendchar(buf, ' ');
	  }
	strappendchar(buf, '}');
	if (arg_simple->store_len) {
	  sprintf(buf, " (len=%d)", *arg_simple->store_len);
	  buf += strlen(buf);
	}
      } else {
	buf = SprintArgValue(buf, arg_simple->type, arg_simple->store, flags);
      }
    }
    if (flags&PRINT_ACTUAL_STATUS) {
      if (arg_simple->actual&FOUND_ACTUAL) {
	buf = strcpyend(WhiteSpace(buf), "(found value)");
      } else if (arg_simple->actual&ACTUAL_CLEARED) {
	buf = strcpyend(WhiteSpace(buf), "(found value but list cleared)");
      } else {
	buf = strcpyend(WhiteSpace(buf), "(original value)");
      }
    }
  }
  if (buf==post_indent) {
    buf = pre_indent;
    *buf = '\0';
  }
  return EraseWhiteSpace(buf, flags);
}

char *SprintArgValue(buf, type, ptr, flags)
char *buf;
enum ARG_TYPE type;
generic *ptr;
int flags;
{
  switch (type) {
  case ARG_FLOAT:
    (void) sprintf(buf, "%g", *(float *)ptr);
    break;
  case ARG_REAL:
    (void) sprintf(buf, "%g", *(Real *)ptr);
    break;
  case ARG_DOUBLE:
    (void) sprintf(buf, "%g", *(double *)ptr);
    break;
  case ARG_BOOLEAN:
    (void) sprintf(buf, "%s", BooleanString(*(Boolean *)ptr));
    break;
  case ARG_INTEGER:
  case ARG_NATURAL:
    (void) sprintf(buf, "%d", *(int *)ptr);
    break;
  case ARG_SHORT:
    (void) sprintf(buf, "%d", *(short *)ptr);
    break;
  case ARG_LONG:
    (void) sprintf(buf, "%ld", *(long *)ptr);
    break;
  case ARG_CHAR:
    if (isprint(*(char *)ptr))
      (void) sprintf(buf, "'%c'", *(char *)ptr);
    else
      (void) sprintf(buf, "'\\%03o'", *(char *)ptr);
    break;
  case ARG_STRING:
  case ARG_FORMAT:
  case ARG_AREA_FORMAT:
  case ARG_RANGE_STRING:
  case ARG_VAR_NAME:
  case ARG_MEMBER_NAME:
    (void) sprintf(buf, "\"%s\"", *(char **)ptr);
    break;
  case ARG_FILE_IN:
    (void) sprintf(buf, "0x%x", *(FILE **)ptr);
    break;
  case ARG_FILE_OUT:
    (void) sprintf(buf, "0x%x", *(FILE **)ptr);
    break;
  default:
    assert(0);
  }
  return buf + strlen(buf);
}

/*ARGSUSED*/
char * SprintArgJoin(buf, join, flags, indent)
char *buf;
enum ARG_JOIN join;
int flags;
int indent;
{
  char *pre_indent,*post_indent;
  pre_indent = buf;
  buf = Indent(buf, indent, flags);
  buf = WhiteSpace(buf);
  post_indent = buf;
  switch (join) {
  case ARG_OR:
    buf = strcpyend(buf, "|");
    break;
  case ARG_AND:
    buf = strcpyend(buf, "&");
    break;
  case ARG_SEQ:
    if (flags&PRINT_SEQS)
      buf = strcpyend(buf, "+");
    break;
  case ARG_XOR:
    buf = strcpyend(buf, "^");
    break;
  }
  if (buf==post_indent) {
    buf = pre_indent;
    *buf = '\0';
  }
  return EraseWhiteSpace(buf, flags);
}

char *FlagPrefix(flag)
int flag;
{
  if (flag) {
    if (use_plus_minus_options)
      return "+";
    else
      return "-";
  } else {
    if (use_plus_minus_options)
      return "-";
    else
      return "-no";
  }
}
