/*
 * $Id: arg-desc.c,v 1.3 92/11/30 11:39:26 drew Exp $
 * 
 */

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appears in all copies and that 
 * both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of University of Toronto 
 * not be used in advertising or publicity pertaining to distribution 
 * of the software without specific, written prior permission.  
 * University of Toronto makes no representations about the suitability 
 * of this software for any purpose. It is provided "as is" without 
 * express or implied warranty. 
 *
 * UNIVERSITY OF TORONTO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS, IN NO EVENT SHALL UNIVERSITY OF TORONTO BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER 
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF 
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 **********************************************************************/

#include <xerion/useful.h>
#include "argf.h"
#include "arg-desc.h"
#include "arg-print.h"
#include "arg-actual.h"
#include "arg-store.h"

/***
 * This file parses argument descriptions, and builds the structures
 * corresponding to them.
 */

struct GLOBALS {
  va_list	ap;
  generic	***ap_loc;	/* where the varargs got stored */
  int		*ap_should_point_to_null;
  int		i;		/* next vararg to use */
  int		N;		/* total number of varargs */
  char		*spec;		/* the original line */
};

#define SkipSpace(str) while (isspace(*(str))) (str)++;

static int ArgList
  ARGS((char*, char**, struct ARG_LIST**, struct GLOBALS *, struct ARG_LIST *));
static int ArgJoin
  ARGS((char*, char**, enum ARG_JOIN*, enum ARG_JOIN));
static int ArgGroup
  ARGS((char*, char**, struct ARG_GROUP**, struct GLOBALS *, struct ARG_LIST *));
static int ArgOption
  ARGS((char*, char**, struct ARG_OPTION**, struct GLOBALS *));
static int ArgSimple
  ARGS((char*, char**, struct ARG_SIMPLE**, struct GLOBALS *));
static int ArgStore
  ARGS((char*, char**, struct ARG_SIMPLE*, struct GLOBALS *));

extern struct ARG_DESC *FindOldArgDesc ARGS((char*));

struct ARG_DESC *StartArgs(name)
char *name;
{
  struct ARG_DESC *arg_desc;
  if (!arg_types_initialized)
    autoinit_argtypes();
  if (!IsAsciiString(name))
    fprintf(stderr,"Warning: argument to StartArgs() is not an ascii string\n");
  arg_desc = FindOldArgDesc(name);
  if (arg_desc==NULL) {
    arg_desc = CallocOrAbort(struct ARG_DESC, 1);
    arg_desc->arg_list = NULL;
    arg_desc->filling_aps = 0;
    arg_desc->name = CallocOrAbort(char, strlen(name)+1);
    arg_desc->fill_list = -1;
    arg_desc->n_lists = 0;
    arg_desc->option_names = NULL;
    strcpy(arg_desc->name,name);
  } else {
    arg_desc->filling_aps = 1;
    arg_desc->fill_list = 0;
  }
  arg_desc->magic_number = ARG_DESC_MAGIC_START;
  arg_desc->complete = 0;
  return arg_desc;
}

void Args(va_alist)
va_dcl
{
  struct ARG_DESC *arg_desc;
  struct ARG_LIST *arg_list;
  char *arg_spec;
  int i;
  va_list ap;
  va_start(ap);
  arg_desc = va_arg(ap, struct ARG_DESC*);
  arg_spec = va_arg(ap, char*);
  if (arg_desc->magic_number!=ARG_DESC_MAGIC_START)
    if (arg_desc->magic_number==ARG_DESC_MAGIC_END)
      ArgErrorAbort("Args: bad magic number for argd, EndArgs() already called?");
    else
      ArgErrorAbort("Args: bad magic number for argd");
  if (arg_desc->filling_aps) {
    /***
     * fill in the arg pointer slots with the var args passed to use
     */
    if (arg_desc->fill_list<0 || arg_desc->fill_list>=arg_desc->n_lists)
      ErrorAbort("Args: no more fill lists for spec \"%s\"",arg_spec);
    arg_list = arg_desc->arg_list[arg_desc->fill_list];
    for (i=0;i<arg_list->n_aps;i++) {
      *(arg_list->ap_loc[i]) = va_arg(ap, generic*);
      if (   *(generic**)(arg_list->ap_loc[i])==NULL
	  || *(generic**)(arg_list->ap_loc[i])==(generic*)1) {
	Error("Args: argument %d (after arg spec) is not a pointer", i+1);
	ErrorAbort("Command \"%s\", arg specification: \"%s\"",
		    arg_desc->name, arg_spec);
      }
      if (   arg_list->ap_should_point_to_null[i]
	  && **(generic***)(arg_list->ap_loc[i])!=NULL) {
	Error("Args: vector pointer argument %d (after arg spec) should point to NULL", i+1);
	ErrorAbort("Command \"%s\", arg specification: \"%s\"",
		    arg_desc->name, arg_spec);
      }
    }
    arg_desc->fill_list++;
  } else {
    /***
     * parse the string and build the arg tree
     */
    struct GLOBALS globals;
    char *end;
    globals.ap = ap;
    globals.N = 0;
    globals.i = 0;
    globals.spec = arg_spec;
    end = arg_spec;
    /* count the number of % parameters */
    while (end=strchr(end,'%')) {
      globals.N++;
      end++;
    }
    globals.N++; /* allow for a NULL at the end */
    globals.ap_loc = CallocOrAbort(generic **, globals.N);
    globals.ap_should_point_to_null = CallocOrAbort(int, globals.N);
    for (i=0;i<globals.N;i++) {
      globals.ap_loc[i] = NULL;
      globals.ap_should_point_to_null[i] = 0;
    }
    if (!ArgList(arg_spec, &end, &arg_list, &globals, NULL))
      ErrorAbort("Args: Failed to parse \"%s\"",arg_spec);
    SkipSpace(end);
    if (*end!='\0')
      ErrorAbort("Args: extra junk \"%s\" at end of \"%s\"", end, arg_spec);
    if (globals.i >= globals.N)
      ErrorAbort("Args: used too many aps in parsing \"%s\"", arg_spec);
    arg_list->n_aps = globals.i;
    arg_list->ap_loc = globals.ap_loc;
    arg_list->ap_should_point_to_null = globals.ap_should_point_to_null;
    if (arg_desc->arg_list==NULL) {
      arg_desc->arg_list = CallocOrAbort(struct ARG_LIST*, 2);
      arg_desc->arg_list[0] = arg_list;
      arg_desc->n_lists = 1;
    } else {
      arg_desc->arg_list
	= ReallocOrAbort(struct ARG_LIST*, arg_desc->arg_list,
			 arg_desc->n_lists+2);
      arg_desc->arg_list[arg_desc->n_lists] = arg_list;
      arg_desc->n_lists++;
    }
    arg_desc->arg_list[arg_desc->n_lists] = NULL;
    for (i=0;i<arg_list->n_aps;i++) {
      if (   *(generic**)(arg_list->ap_loc[i])==NULL
	  || *(generic**)(arg_list->ap_loc[i])==(generic*)1) {
	Error("Args: argument %d (after arg spec) is not a pointer", i+1);
	ErrorAbort("Command \"%s\", arg specification: \"%s\"",
		   arg_desc->name, arg_spec);
      }
    }
  }
}

void EndArgs(arg_desc)
struct ARG_DESC *arg_desc;
{
  if (arg_desc->magic_number!=ARG_DESC_MAGIC_START)
    if (arg_desc->magic_number==ARG_DESC_MAGIC_END)
      ArgErrorAbort("EndArgs: bad magic number for argd, EndArgs() already called?");
    else
      ArgErrorAbort("EndArgs: bad magic number for argd");

  if (arg_desc->filling_aps && arg_desc->fill_list!=arg_desc->n_lists)
    ErrorAbort("EndArgs: was filling aps, but fill_list!=n_lists");

  if (arg_desc->option_names==NULL) {
    /***
     * Build a list of the option names
     */
#define ENDARGSBUFSIZ 1000
    char buffer[ENDARGSBUFSIZ];/* big enough to hold all option names! */
    char *end, *start;
    char **name;
    int n = 0;		/* count the number of options */
    end = SprintArgDesc(InitBuf(buffer), arg_desc,
			PRINT_OPTIONS|PRINT_NOT_INDENT|PRINT_ONEPERLINE, 0);
    assert(end < buffer+ENDARGSBUFSIZ);
    end = strchr(buffer+1, '\n');
    while (end!=NULL) {
      end++;
      n++;
      end = strchr(end, '\n');
    }
    name = arg_desc->option_names = CallocOrAbort(char *, n+1);
    start = buffer+1;
    while (1) {
      end = strchr(start, '\n');
      if (end==NULL)
	break;
      assert(end-start > 0);
      *name = CallocOrAbort(char, end-start+1);
      strncpy_null(*name, start, end-start+1);
      start = end+1;
      name++;
    }
    *name = NULL;
  }
  arg_desc->filling_aps = 0;
  arg_desc->complete = 1;
  arg_desc->magic_number = ARG_DESC_MAGIC_END;
  StoreArgDesc(arg_desc);
}

/***
 * ArgList
 * - Parses:
ARGLIST -> [ ARGGROUP ARGMORE ]
ARGLIST -> ARGGROUP ARGNEXT
ARGLIST -> NULL

ARGNEXT	-> ARGJOIN ARGLIST			- default JOIN is AND
ARGNEXT	-> NULL

ARGMORE	-> ARGJOIN ARGLIST			- default JOIN is AND
ARGMORE	-> NULL
 */
static int ArgList(str, end_out, retobj, globals, parent)
char *str;
char **end_out;
struct ARG_LIST **retobj;
struct GLOBALS *globals;
struct ARG_LIST *parent;
{
  struct ARG_LIST *arg_list;
  struct ARG_LIST *top_arg_list = NULL;
  struct ARG_LIST *tail_arg_list;
  struct ARG_GROUP *arg_group;
  static int arg_list_id = 0;
  enum ARG_JOIN join;
  char *end;
  int found_join = 0;

  while (1) {
    SkipSpace(str);
    arg_list = CallocOrAbort(struct ARG_LIST, 1);
    arg_list->id = ++arg_list_id;
    if (ArgGroup(str, &end, &arg_group, globals, arg_list)) {
      str = end;
      arg_list->group = arg_group;
      arg_list->next = NULL;
      arg_list->join = ARG_NOJOIN;
      arg_list->parent = parent;
      found_join = ArgJoin(str, &end, &join, ARG_SEQ);
      str = end;
      if (top_arg_list!=NULL)
	tail_arg_list->next = arg_list;
      else
	top_arg_list = arg_list;
      arg_list->join = join;
      tail_arg_list = arg_list;
    } else {
      free(arg_list);
      arg_list_id--;
      break;
    }
  }
  if (found_join)
    ErrorAbort("expected arg, found \"%s\"\nIn: \"%s\"",
		str, globals->spec);

  if (top_arg_list==NULL)
    return 0;
  else {
    *end_out = str;
    *retobj = top_arg_list;
    return 1;
  }
}

/***
 * ArgGroup
 * - Parses:
ARGGROUP-> ( ARGBODY PRESENCE HELP )	- () is optional

ARGBODY	-> SIMPLEARG
ARGBODY	-> OPTION ARGLIST
ARGBODY	-> [ OPTION ARGLIST ]
ARGBODY -> [ ARGLIST ]

PRESENCE-> NULL
PRESENCE-> % [!] P

HELP	-> NULL
HELP	-> %?
 */

static int ArgGroup(str, end_out, retobj, globals, parent)
char *str;
char **end_out;
struct ARG_GROUP **retobj;
struct GLOBALS *globals;
struct ARG_LIST *parent;
{
  struct ARG_GROUP *arg_group;
  struct ARG_OPTION *arg_option = NULL;
  struct ARG_SIMPLE *arg_simple = NULL;
  struct ARG_LIST *arg_list = NULL;
  int looking_for_paren = 0;	/* around the ARGGROUP */
  int looking_for_bracket = 0;	/* around the ARGLIST */
  int have_option = 0;
  int have_arg_list = 0;
  int have_arg_simple = 0;
  int optional = 0;
  int stop;
  char *marker;

  SkipSpace(str);

  if (*str=='\0')
    return 0;

  if (*str=='(') {
    looking_for_paren = 1;
    str++;
  }

  SkipSpace(str);
  if (*str=='[') {
    marker = ++str;
    optional = 1;
  }
  have_option = ArgOption(str, &str, &arg_option, globals);
  if (have_option || optional)
    have_arg_list = ArgList(str, &str, &arg_list, globals, parent);
  else
    have_arg_simple = ArgSimple(str, &str, &arg_simple, globals);

  if (optional && !have_option && !have_arg_list)
    ErrorAbort("Args: expected arg list after \"%s\", but found \"%s\"\nIn: \"%s\"",
		marker, str, globals->spec);

  if (optional) {
    SkipSpace(str);
    if (*str!=']')
      ErrorAbort("Args: expected closing \"]\", but found \"%s\"\nIn: \"%s\"", str, globals->spec);
    str++;
  }

  if (!have_option && !have_arg_simple && !have_arg_list) {
    if (looking_for_paren)
      ErrorAbort("Args: cannot parse \"%s\"\nIn: \"%s\"", str, globals->spec);
    else
      return 0;
  }

  arg_group = CallocOrAbort(struct ARG_GROUP, 1);
  if (optional || have_option) {
    arg_group->tag = ARG_GROUP_OPTION;
    arg_group->u.option.option = arg_option;
    arg_group->u.option.arg_list = arg_list;
    arg_group->u.option.optional = optional;
  } else {
    arg_group->tag = ARG_GROUP_SIMPLE;
    arg_group->u.simple.arg_simple = arg_simple;
  }
  arg_group->presence = NULL;
  arg_group->neg_presence = NULL;
  arg_group->help = NULL;

  str = str;
  SkipSpace(str);

  stop = 0;
  while (*str=='%' && !stop) {	/* find %P and %? */
    if (str[1]=='P' && arg_group->presence==NULL) {
      arg_group->presence = va_arg(globals->ap, int*);
      globals->ap_loc[globals->i++] = (generic**) &arg_group->presence;
      str += 2;
    } else if (str[1]=='!' && str[2]=='P' && arg_group->neg_presence==NULL) {
      arg_group->neg_presence = va_arg(globals->ap, int*);
      globals->ap_loc[globals->i++] = (generic**) &arg_group->neg_presence;
      str += 3;
    } else if (str[1]=='?' && arg_group->help==NULL) {
      arg_group->help = va_arg(globals->ap, char*);
      globals->ap_loc[globals->i++] = (generic**) &arg_group->help;
      str += 2;
    } else {
      stop = 1;
      break;
    }
    if (!stop)
      SkipSpace(str);
  }

  SkipSpace(str);
  if (looking_for_paren) {
    if (*str!=')')
      ErrorAbort
	("Args: expected closing \")\" for arg group, but found \"%s\"\nIn: \"%s\"", str, globals->spec);
    str++;	/* skip the ) */
  }
  
  *end_out = str;
  *retobj = arg_group;

  return 1;
    
}

/***
 * ArgSimple
 * - Parses:
SIMPLEARG	-> <description> ARGVECT {ARGSTORE}
SIMPLEARG	-> "String"
SIMPLEARG	-> 'String'

ARGVECT -> NULL
ARGVECT	-> ...
 */

static int ArgSimple(str, end_out, retobj, globals)
char *str;
char **end_out;
struct ARG_SIMPLE **retobj;
struct GLOBALS *globals;
{
  char *end, *start;
  struct ARG_SIMPLE *arg_simple;
  static int arg_simple_id = 1;
  start = str;
  SkipSpace(str);
  if (*str=='<') {
    end = str;
    while (*end && *end!='>') end++;			/* find the <  > */
    if (*end=='\0')
      ErrorAbort("Args: no terminating '>' for string \"%s\"\nIn: \"%s\"",
		  str, globals->spec);

    arg_simple = CallocOrAbort(struct ARG_SIMPLE, 1);	/* allocate */
    arg_simple->prompt = CallocOrAbort(char, end-str);
    arg_simple->id = arg_simple_id++;
    strncpy_null(arg_simple->prompt, str+1, end-str);

    str = end+1;
    SkipSpace(str);					/* look for a "..." */
    if (str[0]=='.' && str[1]=='.' && str[2]=='.') {
      arg_simple->vector = 1;
      str += 3;
    } else
      arg_simple->vector = 0;

    if (!ArgStore(str, &end, arg_simple, globals))
      ErrorAbort
	("Args: expected arg store spec (something like \"%i\"), but found \"%s\"\nIn: \"%s\"",
	 str, globals->spec);
    str = end;
  } else if (*str=='\'' || *str=='"') {
    end = str+1;
    while (*end && *end!=*str) end++;
    if (*end=='\0')
      ErrorAbort("Args: no terminating %c for string \"%s\"\nIn: \"%s\"",
		  *str, str, globals->spec);
    arg_simple = CallocOrAbort(struct ARG_SIMPLE, 1);
    arg_simple->id = arg_simple_id++;
    arg_simple->prompt = CallocOrAbort(char, end-str);
    strncpy_null(arg_simple->prompt, str+1, end-str);
    arg_simple->vector = 0;
    arg_simple->type = ARG_LITERAL;
    arg_simple->verify = NULL;
    arg_simple->store = NULL;
    end++;
    str = end;
  } else
    return 0;

  *end_out = end;
  *retobj = arg_simple;
  return 1;

}

/***
 * ArgStore
 * - Parses: {ARGSTORE} (at least one)
ARGSTORE	-> % ARGTYPE

ARGTYPE	-> V				- verify function provided
ARGTYPE	-> L				- length of vector

ARGTYPE	-> i				- integer
ARGTYPE	-> f				- float
ARGTYPE	-> r				- Real
ARGTYPE	-> D				- double
ARGTYPE	-> s				- String
ARGTYPE	-> c				- char
ARGTYPE	-> B				- Boolean
ARGTYPE	-> n				- natural number (integer >= 0)
ARGTYPE	-> h				- short
ARGTYPE	-> l				- long
ARGTYPE	-> F				- format (as in %9.3g etc.)
ARGTYPE	-> A				- area format
ARGTYPE	-> R				- range string
ARGTYPE	-> v				- variable name
ARGTYPE	-> m				- member (field) name
 */

static int ArgStore(str, end_out, arg_simple, globals)
char *str;
char **end_out;
struct ARG_SIMPLE *arg_simple;
struct GLOBALS *globals;
{
  char *end;
  end = str;
  SkipSpace(end);
  if (*end!='%') return 0;
  str = end;
  while (*str=='%') {
    str++;
    if (*str=='V') {
      arg_simple->verify = va_arg(globals->ap, verifyfunc);
      globals->ap_loc[globals->i] = (generic**) &arg_simple->verify;
    } else if (*str=='L') {
      if (!arg_simple->vector)
	ErrorAbort("Args: <%s> is not a vector argument - cannot use '%%L'\nIn: \"%s\"",
		   arg_simple->prompt, globals->spec);
      arg_simple->store_len = va_arg(globals->ap, int*);
      globals->ap_loc[globals->i] = (generic**) &arg_simple->store_len;
    } else if (arg_type_by_char[*str]!=ARG_BAD_TYPE) {
      arg_simple->type = arg_type_by_char[*str];
      arg_simple->store = va_arg(globals->ap, generic*);
      globals->ap_loc[globals->i] = &arg_simple->store;
      if (arg_simple->vector) {
	globals->ap_should_point_to_null[globals->i] = 1;
	if (*(char**)arg_simple->store!=NULL)
	  ErrorAbort("Args: pointer for arg list <%s>... contains non-null value\nIn: \"%s\"",
		     arg_simple->prompt, globals->spec);
      }
    } else {
      /* must be a %P or %? that gets parsed elsewhere */
      break;
    }
    globals->i++;
    end = ++str;
    SkipSpace(str);
  }
  *end_out = end;
  return 1;
}

/***
 * ArgJoin
 * - Parses:
ARGJOIN -> ^ XOR				- one but not both
ARGJOIN -> -> SEQ				- both in this sequence
ARGJOIN -> & AND				- both in any sequence
ARGJOIN -> | OR 				- one or both (or neither)
ARGJOIN -> NULL (default, OR or AND)
 */
static int ArgJoin(str, end_out, retobj, default_join)
char *str;
char **end_out;
enum ARG_JOIN *retobj;
enum ARG_JOIN default_join;
{
  char *end;
  end = str;
  SkipSpace(end);
  if (*end=='^') {
    end++;
    *retobj = ARG_XOR;
  } else if (*end=='+') {
    end++;
    *retobj = ARG_SEQ;
  } else if (*end=='&') {
    end++;
    *retobj = ARG_AND;
  } else if (*end=='|') {
    end++;
    *retobj = ARG_OR;
  } else {
    *retobj = default_join;
    return 0;
  }
  *end_out = end;
  return 1;
}

/***
 * ArgOption
 * - Parses:
OPTION	-> '-' OPTFLAGS OPTTYPE OPTNAME OPTSTORE HELP
 */
static int ArgOption(str, end_out, retobj, globals)
char *str;
char **end_out;
struct ARG_OPTION **retobj;
struct GLOBALS *globals;
{
  struct ARG_OPTION *arg_option;
  char *mark = NULL;
  int stop = 0;
  char *start = str;
  SkipSpace(str);
  if (*str!='-') {					/* not an option */
    return 0;
  } else {						/* is an option */
    str++;						/* skip the - */
    arg_option = CallocOrAbort(struct ARG_OPTION, 1);
    arg_option->no_prefix = 0;
    while (isalpha(*str) && !(*str=='S' || *str=='P')) {/* flags */
      switch (*str) {
      case 'F':
	arg_option->no_prefix = 1;
	break;
      default:
	ErrorAbort("Args: bad flag %s in option \"%s\"\nIn: \"%s\"", pchar(*str), start, globals->spec);
      }
      str++;
    }
    if (*str=='S')					/* type */
      arg_option->is_signed = 1;
    else if (*str=='P')
      arg_option->is_signed = 0;
    else
      ErrorAbort("Args: can't find type of option ('B' or 'P') in \"%s\"\nIn: \"%s\"", start, globals->spec);
    str++;

    SkipSpace(str);

    mark = str;
    while (isalnum(*str) || *str=='-') str++;			/* name */
    arg_option->name = CallocOrAbort(char, 1+str-mark);
    strncpy_null(arg_option->name, mark, 1+str-mark);

    SkipSpace(str);
    while (*str=='%' && !stop) {
      if (str[1]=='!') {
	switch (str[2]) {
	case 'S' :
	  arg_option->neg_sign = va_arg(globals->ap, int*);
	  globals->ap_loc[globals->i++] = (generic**) &arg_option->neg_sign;
	  break;
	case 'P' :
	  arg_option->neg_presence = va_arg(globals->ap, int*);
	  globals->ap_loc[globals->i++] = (generic**)&arg_option->neg_presence;
	  break;
	default:
	  stop = 1;
	  break;
	}
	if (!stop) {
	  str += 3;
	  SkipSpace(str);
	}
      } else {	/* not negated */
	switch (str[1]) {
	case 'S' :
	  arg_option->sign = va_arg(globals->ap, int*);
	  globals->ap_loc[globals->i++] = (generic**) &arg_option->sign;
	  break;
	case 'P' :
	  arg_option->presence = va_arg(globals->ap, int*);
	  globals->ap_loc[globals->i++] = (generic**) &arg_option->presence;
	  break;
	case '?' :
	  arg_option->help = va_arg(globals->ap, char*);
	  globals->ap_loc[globals->i++] = (generic**) &arg_option->help;
	  break;
	default:
	  stop = 1;
	  break;
	}
	if (!stop) {
	  str += 2;
	  SkipSpace(str);
	}
      }
    }
    *end_out = str;
    *retobj = arg_option;
    return 1;
  }
}
