/*
 * $Id: arg-actual.c,v 1.2 92/11/30 11:39:23 drew Exp $
 * 
 */

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appears in all copies and that 
 * both the copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of University of Toronto 
 * not be used in advertising or publicity pertaining to distribution 
 * of the software without specific, written prior permission.  
 * University of Toronto makes no representations about the suitability 
 * of this software for any purpose. It is provided "as is" without 
 * express or implied warranty. 
 *
 * UNIVERSITY OF TORONTO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS 
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS, IN NO EVENT SHALL UNIVERSITY OF TORONTO BE LIABLE FOR ANY 
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER 
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF 
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 **********************************************************************/

#include <xerion/useful.h>
#include "argf.h"
#include "arg-desc.h"
#include "arg-print.h"
#include "arg-actual.h"
#include "arg-store.h"

/***
 * parseact.c - parse the actual arguments
 * To fill an arglist (A):
 *   find an unused arg_list (B) that an actual (X) fits in
 *   put it there, if (X) is an option, then clear out the arglist (B)
 *   fill arglist (B)
 *   fill the parent arglist (C), if it is not = (A)
 *   mark (A) as used
 *   check if (A) is satisfied (all non-optional args filled),
 *     if not print error message
 *
 * For the filling of vector arguments, we keep record of where
 * we used the last actual argument, then we can add to a vector arg
 * if we put the last actual in the same place
 */

int use_plus_minus_options = 0;

enum SET_MODE {
  SET_ON,
  SET_OFF};

enum OPTION_FILL_RESULT {
  NOT_FILLED,
  FILLED_FIRST_TIME,
  FILLED_AGAIN};

enum FIND_GROUP_RESULT {
  NOT_FOUND_NOT_FULL,
  NOT_FOUND_EMPTY,
  NOT_FOUND_FULL,
  FOUND_GROUP_OPTION_OR_ARGLIST,
  FOUND_GROUP_SIMPLE};

static int IsOption ARGS((char *));

static char **ParseArgsDesc ARGS((struct ARG_DESC*, char**, int, int, int));

static struct ARG_LIST *FindActualInArgDesc
  ARGS((struct ARG_DESC*, char*, struct GLOBAL *));
static struct ARG_LIST *FindActualInArgList
  ARGS((struct ARG_LIST*, char*, struct GLOBAL *));
static enum FIND_GROUP_RESULT FindActualInArgGroup
  ARGS((struct ARG_GROUP*, char*, struct GLOBAL*));

static char **FillArgList ARGS((struct ARG_LIST*, char**, struct GLOBAL*));
static int FillArgSimple
  ARGS((struct ARG_SIMPLE*, char*, struct GLOBAL*));
static enum OPTION_FILL_RESULT FillArgOption
  ARGS((struct ARG_OPTION*, char*, struct GLOBAL*));

static void CheckArgListSatisfied
  ARGS((struct ARG_LIST*, char**, struct GLOBAL*));
static void CheckArgGroupSatisfied
  ARGS((struct ARG_GROUP*, char**, struct GLOBAL*));

static void SAFArgDesc ARGS((struct ARG_DESC*, int, enum SET_MODE));
static void SAFArgList ARGS((struct ARG_LIST*, int, enum SET_MODE));
static void SAFArgGroup ARGS((struct ARG_GROUP*, int, enum SET_MODE));
static void SAFArgSimple ARGS((struct ARG_SIMPLE*, int, enum SET_MODE));
static void SAFArgOption ARGS((struct ARG_OPTION*, int, enum SET_MODE));

  /* Sun's do not have case insensitive string comparison */
#ifdef sun
#ifndef MAX(x,y)
#define MAX(x,y)	((x) > (y) ? (x) : (y))
#endif

static char	*downcase(s1, s2, n)
  char		*s1 ;
  const char	*s2 ;
  int		n ;
{
  char	*ptr = s1 ;
  if (n < 0)
    n = strlen(s2) + 1 ;

  strncpy(s1, s2, n) ;

  while (n-- && *ptr) {
    if (isupper((int)*ptr))
      *ptr = tolower((int)*ptr) ;
    ++ptr ;
  }
  return s1 ;
}

static int	strcasecmp(s1, s2)
  const char	*s1 ;
  const char	*s2 ;
{
  static char	*internalS1, *internalS2 ;
  static int	length ;

  int	length1 = strlen(s1) ;
  int	length2 = strlen(s2) ;
  if (length1 >= length || length2 >= length) {
    length = MAX(length1, length2) + 1 ;
    if (internalS1 == NULL)
      internalS1 = (char *)malloc(length) ;
    else
      internalS1 = (char *)realloc(internalS1, length) ;

    if (internalS2 == NULL)
      internalS2 = (char *)malloc(length) ;
    else
      internalS2 = (char *)realloc(internalS2, length) ;
  }
  
  downcase(internalS1, s1, -1) ;
  downcase(internalS2, s2, -1) ;
  return strcmp(internalS1, internalS2) ;
}  

static int	strncasecmp(s1, s2, n)
  const char	*s1 ;
  const char	*s2 ;
  int		n ;
{
  static char	*internalS1, *internalS2 ;
  static int	length ;

  if (n >= length) {
    length = n + 1 ;
    if (internalS1 == NULL)
      internalS1 = (char *)malloc(length) ;
    else
      internalS1 = (char *)realloc(internalS1, length) ;

    if (internalS2 == NULL)
      internalS2 = (char *)malloc(length) ;
    else
      internalS2 = (char *)realloc(internalS2, length) ;
  }
  
  downcase(internalS1, s1, n) ;
  downcase(internalS2, s2, n) ;
  return strncmp(internalS1, internalS2, n) ;
}

#endif

char **ParseArgs(arg_desc, argc, argv, extra)
struct ARG_DESC *arg_desc;
int argc;
char *argv[];
int extra;
{
  struct ARG_DESC *my_arg_desc;
  int arg_help = 0;
  int arg_values = 0;
  int arg_usage = 0;
  int arg_debug = 0;
  int arg_tree = 0;
  if (argc<0)
    return argv;
  if (arg_desc->magic_number!=ARG_DESC_MAGIC_END)
    ArgErrorAbort
      ("ParseArgs: bad magic number for argd \"%s\".\nWrong argument order or has not been processed by EndArgs()?",
       arg_desc->name);
  my_arg_desc = StartArgs("argargs");
  use_plus_minus_options = GetPlusMinusOptionSwitch();
  Args(my_arg_desc, "[-FSargplusminus %S]", &use_plus_minus_options);
  Args(my_arg_desc, "[-FPargtree %P]", &arg_tree);
  Args(my_arg_desc, "[-FPargdebug %P]", &arg_debug);
  Args(my_arg_desc, "[-FPargusage %P]", &arg_usage);
  Args(my_arg_desc, "[-FPargvalues %P]", &arg_values);
  Args(my_arg_desc, "[-FParghelp %P]", &arg_help);
  EndArgs(my_arg_desc);
  argv = ParseArgsDesc(my_arg_desc, argv+1, 1, 1, 0);
  if (arg_debug) {
    fprintf(dout, "Argument debugging turned on, internal argument flags:\n");
    fprintf(dout, "  arg_help=%s arg_values=%s arg_usage=%s arg_debug=%s\n",
	    BooleanString(arg_help), BooleanString(arg_values),
	    BooleanString(arg_usage), BooleanString(arg_debug));
    fprintf(dout, "  arg_tree=%s use_plus_minus_options=%s\n",
	    BooleanString(arg_tree), BooleanString(use_plus_minus_options));
  }
  if (arg_help)
    PrintArgHelp(arg_desc, dout, 1);
  else if (arg_usage)
    PrintArgUsage(arg_desc, dout, 1);
  else {
    argv = ParseArgsDesc(arg_desc, argv, (extra&EXTRA_ARGS)!=0,
			 (extra&EXTRA_OPTIONS)!=0, arg_debug);
    if (!(arg_values || arg_tree || arg_debug))
      return argv;				/* normal return */
    else {
      if (arg_values)
	PrintArgValues(arg_desc, dout);
      if (arg_tree)
	PrintArgTree(arg_desc, dout);
    }
  }
  if (!(extra&RETURN_ON_ERROR)) {
    Abort();
  } else
    return NULL;	/* indication to stop! */
}

static char **ParseArgsDesc(arg_desc, argv, extra_args, extra_opts, debug)
struct ARG_DESC *arg_desc;
char *argv[];
int extra_args;
int extra_opts;
int debug;
{
  int argc = 0;
  int i;
  struct ARG_LIST *arg_list;
  struct GLOBAL globals;
  char **extra;
  for (i=0;argv[i]!=0;i++) ;
  extra = ITempCalloc(char*, i+1);
  i = 0;
  globals.arg_desc = arg_desc;
  globals.last_simple_id = 0;
  globals.debug = debug;
  globals.usage_on_error = 1;
  SAFArgDesc(arg_desc, ALL_ACTUAL_FLAGS, SET_OFF);
  while (*argv) {
    arg_list = FindActualInArgDesc(arg_desc, *argv, &globals);
    if (arg_list)
      argv = FillArgList(arg_list, argv+1, &globals);
    else if (IsOption(*argv))
      if (extra_opts)
	extra[i++] = *argv++;
      else
	ArgErrorAbort(&globals, "option \"%s\" not understood",*argv);
    else if (extra_args)
	extra[i++] = *argv++;
      else
	ArgErrorAbort(&globals, "argument \"%s\" not understood",*argv);
  }
  extra[i] = NULL;
  if (arg_desc->arg_list!=NULL)
    for (i=0; i<arg_desc->n_lists; i++)
      CheckArgListSatisfied(arg_desc->arg_list[i], extra, &globals);
  return extra;
}

/***
 * this function does two things - try to fill up the arg_list
 * and also checks that it is satisfied
 */
static char **FillArgList(arg_list, argv, global)
struct ARG_LIST* arg_list;
char ** argv;
struct GLOBAL *global;
{
  struct ARG_LIST *sub_arg_list;
  if (*argv!=NULL) {
    sub_arg_list = FindActualInArgList(arg_list, *argv, global);
    if (sub_arg_list==NULL) {
      CheckArgListSatisfied(arg_list, argv, global);
      return argv;
    }
    argv++;
  }
  if (*argv!=NULL)
    argv = FillArgList(sub_arg_list, argv, global);
  if (*argv!=NULL && sub_arg_list!=arg_list)
    argv = FillArgList(arg_list, argv, global);
  if (*argv!=NULL && arg_list->parent!=NULL)
    argv = FillArgList(arg_list->parent, argv, global);
  CheckArgListSatisfied(arg_list, argv, global);
  return argv;
}

static void CheckArgListSatisfied(arg_list, argv, global)
struct ARG_LIST *arg_list;
char **argv;
struct GLOBAL *global;
{
  if (arg_list->actual&SATISFIED_ARGS)
    return;
  if (arg_list->join==ARG_XOR && (arg_list->next->actual&PARTIALLY_FILLED))
    CheckArgListSatisfied(arg_list->next, argv, global);
  else {
    /***
     * the following stuff is an attempt to do args that are `joined'
     * with OR, AND, SEQ or XOR, but it doesn't work (or maybe the
     * join values are not being read properly).
     */
    if (   (arg_list->group->actual&PARTIALLY_FILLED)
	|| arg_list->join==ARG_SEQ || arg_list->join==ARG_AND
	|| arg_list->join==ARG_XOR || arg_list->join==ARG_NOJOIN)
      CheckArgGroupSatisfied(arg_list->group, argv, global);
    if (   arg_list->next!=NULL
	&& (   (arg_list->next->actual&PARTIALLY_FILLED)
	    || arg_list->join==ARG_SEQ || arg_list->join==ARG_AND))
      CheckArgListSatisfied(arg_list->next, argv, global);
    if (   arg_list->join==ARG_OR
	&& !(arg_list->group->actual&SATISFIED_ARGS)
	&& !(arg_list->next->actual&SATISFIED_ARGS)) {
      char g_buf[1000];
      char l_buf[1000];
      (void) SprintArgGroup(InitBuf(g_buf), arg_list->group,
			    PRINT_BRIEF_LINE, 0);
      (void) SprintArgList(InitBuf(l_buf), arg_list->next,
			   PRINT_BRIEF_LINE, 0);
      ArgErrorAbort(global, "expected argument to put in \"%s\" or \"%s\"",
		  g_buf+1, l_buf+1);
    }
  }
  arg_list->actual |= SATISFIED_ARGS;
}

static void CheckArgGroupSatisfied(arg_group, argv, global)
struct ARG_GROUP *arg_group;
char **argv;
struct GLOBAL *global;
{
  if (arg_group->actual&SATISFIED_ARGS)
    return;
  if (arg_group->tag==ARG_GROUP_SIMPLE) {
    if (!(arg_group->u.simple.arg_simple->actual&FOUND_ACTUAL)) {
      char g_buf[1000];
      (void) SprintArgGroup(InitBuf(g_buf), arg_group, PRINT_BRIEF_LINE, 0);
      if (g_buf[1]=='\'') /* is ARG_LITERAL */
	ArgErrorAbort(global, "Expected argument to match %s", g_buf+1);
      else
	ArgErrorAbort(global, "Expected argument to put in \"%s\"", g_buf+1);
    }
    if (arg_group->presence)
      *(arg_group->presence) = 1;
    if (arg_group->neg_presence)
      *(arg_group->neg_presence) = 0;
  } else {
    struct ARG_LIST *list = arg_group->u.option.arg_list;
    struct ARG_OPTION *option = arg_group->u.option.option;
    assert(arg_group->tag==ARG_GROUP_OPTION);
    /***
     * the group can be satisfied by it being optional
     * & having nothing, or by having an option & the arg_list
     * being satisfied
     */
    if (   arg_group->u.option.optional
	&& (option==NULL || !(option->actual&FOUND_ACTUAL))
	&& (list==NULL || !(list->actual&PARTIALLY_FILLED))) {
      /* it's ok - optional & have nothing */
    } else {
      if (option!=NULL && !(option->actual&FOUND_ACTUAL)) {
	char o_buf[1000];
	(void) SprintArgOption(InitBuf(o_buf), option, PRINT_BRIEF_LINE, 0);
	ArgErrorAbort(global, "Expected option \"%s\"", o_buf+1);
      }
      if (list!=NULL) {
	CheckArgListSatisfied(list, argv, global);
	assert(list->actual&SATISFIED_ARGS);
      }
      if (arg_group->presence)
	*(arg_group->presence) = 1;
      if (arg_group->neg_presence)
	*(arg_group->neg_presence) = 0;
    }
  }
  arg_group->actual |= SATISFIED_ARGS;
}

/***
 * this fills an argument
 */
static int FillArgSimple(arg_simple, arg, global)
struct ARG_SIMPLE *arg_simple;
char *arg;
struct GLOBAL *global;
{
  /***
   * we only try to fill if either
   * (1) the formal arg is unfilled, or
   * (2) the formal arg is a vector type, and the last actual was used here
   */
  if (!(   !(arg_simple->actual&FOUND_ACTUAL)
	|| (arg_simple->vector && global->last_simple_id==arg_simple->id))) {
    return 0;
  } else {
    /*
     * to fill:
     * (1) check if it is of the right type,
     * (2) check if it passes the verify function
     * (3) store the argument
     */
    int right_type = 0;
    int passes_verify = 1;
    if (arg_simple->type==ARG_LITERAL) {	/* literal should just match */
      if (arg_simple->verify==NULL)
	right_type = !strcmp(arg, arg_simple->prompt);
      else
	right_type = 1;				/* unless a verify function */
    } else {					/* check the type */
      right_type = arg_store_function_by_type[(int) arg_simple->type]
	(arg_simple, arg, ONLYCHECK, global);
    }
    if (global->debug)
      fprintf(dout, "Argument \"%s\" %s type check with %c%s%c (%s)\n",
	      arg, (right_type ? "passed" : "failed"),
	      (arg_simple->type==ARG_LITERAL ? '\'' : '<'),
	      arg_simple->prompt,
	      (arg_simple->type==ARG_LITERAL ? '\'' : '>'),
	      arg_type_name[(int) arg_simple->type]);
    if (!right_type)				/* not the right type */
      return 0;
    if (arg_simple->verify!=NULL) {		/* use verify function */
      passes_verify = arg_simple->verify
	(arg, arg_simple->prompt, global->debug);
      if (global->debug)
	fprintf(dout, "Argument \"%s\" %s verify check with %c%s%c (%s)\n",
		arg, (passes_verify ? "passed" : "failed"),
		(arg_simple->type==ARG_LITERAL ? '"' : '<'),
		arg_simple->prompt,
		(arg_simple->type==ARG_LITERAL ? '"' : '>'),
		arg_type_name[(int) arg_simple->type]);
    }
    if (!passes_verify)				/* didn't pass verify */
      return 0;
    if (arg_simple->type!=ARG_LITERAL) {	/* don't store literals */
      (void) arg_store_function_by_type[(int)arg_simple->type]
	(arg_simple, arg, COPYSTR, global);
    }
    arg_simple->actual |= FOUND_ACTUAL;
    global->last_simple_id = arg_simple->id;
    return 1;
  }
}

static enum OPTION_FILL_RESULT FillArgOption(arg_option, arg, global)
struct ARG_OPTION *arg_option;
char *arg;
struct GLOBAL *global;
{
  enum OPTION_FILL_RESULT retval;
  int pos_match = 0;
  int pos_perfect_match = 0;
  int other_pos_match = 0;
  int other_pos_perfect_match = 0;
  int other_pos_match_n;
  int neg_match = 0;
  int neg_perfect_match = 0;
  int other_neg_match = 0;
  int other_neg_perfect_match = 0;
  int other_neg_match_n;
  int i, len, match;
  char *p, *pos_arg, *neg_arg;
  char *pos_flag, *neg_flag, *ambiguity;
  char **name = global->arg_desc->option_names;
  char debug_intro[100];
  if (!IsOption(arg))
    return NOT_FILLED;
  assert(*arg=='-' || *arg=='+');
  debug_intro[0] = '\0';

  /* strip off the '-' or '+' and look for a prefix match */
  p = arg+1;
  pos_arg = p;
  len = strlen(p);
  match = !strncasecmp(p, arg_option->name, len);
  pos_perfect_match = match && (len == strlen(arg_option->name));
  pos_match = (match && (!arg_option->no_prefix || pos_perfect_match));
  if (global->debug && match) {
    sprintf(debug_intro, "Matching \"%s\" with target option \"%s\"\n",
	    arg, arg_option->name);
    if (pos_perfect_match)
      fprintf(dout, "%s  Perfect match of \"[%c]%s\" with \"%s\"\n",
	      debug_intro, *arg, p, arg_option->name);
    else if (match)
      fprintf(dout, "%s  Matched prefix of \"[%c]%s\" with \"%s\"%s\n",
	      debug_intro, *arg, p, arg_option->name, !pos_match ?
	      ", but partial matches are not accepted for this option" : "");
  }

  /* strip off the "no" and look for a prefix match */
  if (arg[0]=='-' && p[0]=='n' && p[1]=='o' && len>2) {
    p += 2;
    len -= 2;
    neg_arg = p;
    match = !strncasecmp(p, arg_option->name, len);
    neg_perfect_match = match && (len == strlen(arg_option->name));
    neg_match = (match && (!arg_option->no_prefix || neg_perfect_match));
    if (global->debug && match) {
      if (debug_intro[0]=='\0')
	sprintf(debug_intro, "Matching \"%s\" with target option \"%s\"\n",
		arg, arg_option->name);
      else
	debug_intro[0]='\0';
      if (neg_perfect_match)
	fprintf(dout, "%s  Perfect match of \"[-no]%s\" with \"%s\"\n",
		debug_intro, p, arg_option->name);
      else if (match)
	fprintf(dout, "%s  Matched prefix of \"[-no]%s\" with \"%s\"%s\n",
		debug_intro, p, arg_option->name, !neg_match ?
		", but partial matches are not accepted for this option" : "");
    }
  } else
    neg_arg = NULL;

  if (!pos_match && !neg_match)
    return NOT_FILLED;

  /* look for ambiguities */
  if (pos_match || neg_match) {
    len = strlen(pos_arg);
    for (i=0;name[i]!=NULL;i++) {
      if (   strcasecmp(arg_option->name, name[i])
	  && !strncasecmp(pos_arg, name[i], len)
	  && (!pos_perfect_match || len==strlen(name[i]))) {
	other_pos_match_n = i;
	other_pos_match = 1;
	other_pos_perfect_match = (len == strlen(name[i]));
	if (global->debug)
	  fprintf(dout, "  %s of \"[%c]%s\" with \"%s\"\n",
		  (other_pos_perfect_match
		   ? "Perfect match" : "Matched prefix"),
		   *arg, pos_arg, name[i]);
	break;
      }
    }

    if (neg_arg) {
      len = strlen(neg_arg);
      for (i=0;name[i]!=NULL;i++) {
	if (   strcasecmp(arg_option->name, name[i])
	    && !strncasecmp(neg_arg, name[i], len)
	    && (!neg_perfect_match || len==strlen(name[i]))) {
	  other_neg_match_n = i;
	  other_neg_match = 1;
	  other_neg_perfect_match = (len == strlen(name[i]));
	  if (global->debug)
	    fprintf(dout, "  %s of \"[-no]%s\" with \"%s\"\n",
		    (other_neg_perfect_match
		     ? "Perfect match" : "Matched prefix"),
		    neg_arg, name[i]);
	  break;
	}
      }
    }
  }

  assert(!(pos_perfect_match && neg_perfect_match));
  if (  pos_perfect_match + neg_perfect_match
      + other_pos_perfect_match + other_neg_perfect_match > 1) {
    ArgError(global, "Command options are inherently ambiguous\n\
The source should be changed, ambiguous options are: ");
    if (pos_perfect_match)
      ArgError(global, "%s", arg);
    if (neg_perfect_match)
      ArgError(global, "-no%s", arg+1);
    if (other_pos_perfect_match)
      ArgError(global, "%c%s", *arg, name[other_pos_match_n]);
    if (other_neg_perfect_match)
      ArgError(global, "-no%s", name[other_neg_match_n]);
    Abort();
  }
  /***
   * perfect matches override, so if we matched other perfectly,
   * but not this one, return 
   */
  if (   (other_neg_perfect_match || other_pos_perfect_match)
      && !(pos_perfect_match || neg_perfect_match)) {
    if (global->debug)
      fprintf(dout, "  Perfect match with other option overrides partial match with target option\n");
    return NOT_FILLED;
  }
  /***
   * and if we matched ours perfectly, then ignore others
   */
  if (pos_perfect_match || neg_perfect_match) {
    if (global->debug && (other_pos_match || other_neg_match))
      fprintf(dout, "  Perfect match with target option overrides partial match with other option\n");
    other_pos_match = other_neg_match = 0;
  }
  if (pos_perfect_match) {
    if (global->debug && neg_match)
      fprintf(dout, "  Perfect match with target option overrides partial match with [-no]target\n");
    neg_match = 0;
  }
  if (neg_perfect_match) {
    if (global->debug && pos_match)
      fprintf(dout, "  Perfect match with [-no]target option overrides partial match with target\n");
    pos_match = 0;
  }
  /***
   * now, check for any remaining ambiguity
   */
  pos_flag = (use_plus_minus_options ? "+" : "-");
  neg_flag = (use_plus_minus_options ? "-" : "-no");
  ambiguity = "Ambigous option \"%s\", could mean \"%s%s\" or \"%s%s\"";
  if (pos_match && (neg_match || other_neg_match))
    ArgErrorAbort(global, ambiguity, arg, neg_flag, arg_option->name, neg_flag,
		(neg_match ? arg_option->name : name[other_neg_match_n]));
  if (neg_match && (pos_match || other_pos_match))
    ArgErrorAbort(global, ambiguity, arg, neg_flag, arg_option->name, pos_flag,
		(pos_match ? arg_option->name : name[other_neg_match_n]));
  if (pos_match && other_pos_match)
    ArgErrorAbort(global, ambiguity, arg, pos_flag, arg_option->name, pos_flag,
		name[other_pos_match_n]);
  if (neg_match && other_neg_match)
    ArgErrorAbort(global, ambiguity, arg, neg_flag, arg_option->name, neg_flag,
		name[other_neg_match_n]);

  /* should be no ambiguity now */
  assert(pos_match + neg_match + other_pos_match + other_neg_match == 1);
  
  /* convert to a real pos/neg - taking into account use_plus_minus_options */
  if (use_plus_minus_options && *arg=='-') {
    pos_match = 0;
    neg_match = 1;
  }
  if (neg_match && !arg_option->is_signed)
    ArgErrorAbort(global, "option \"%s%s\" is not signed - cannot use \"%s%s\"",
		pos_flag, arg_option->name, neg_flag, arg_option->name);

  if (arg_option->actual && FOUND_ACTUAL)
    retval = FILLED_AGAIN;
  else
    retval = FILLED_FIRST_TIME;
  arg_option->actual =
    (FOUND_ACTUAL | (pos_match
		     ? (arg_option->is_signed ? OPTION_ON : OPTION_PRESENT)
		     : OPTION_OFF));
  if (arg_option->presence!=NULL) {
    *(arg_option->presence) = 1;
    if (global->debug)
      fprintf(dout, "  Presence of \"%s\" set to true\n", arg_option->name);
  }
  if (arg_option->neg_presence!=NULL) {
    *(arg_option->neg_presence) = 0;
    if (global->debug)
      fprintf(dout, "  Neg presence of \"%s\" set to false\n",
	      arg_option->name);
  }
  if (arg_option->sign!=NULL) {
    *(arg_option->sign) = pos_match;
    if (global->debug)
      fprintf(dout, "  Sign of \"%s\" set to %s\n", arg_option->name,
	      BooleanString(pos_match));
  }
  if (arg_option->neg_sign!=NULL) {
    *(arg_option->neg_sign) = !pos_match;
    if (global->debug)
      fprintf(dout, "  Neg sign of \"%s\" set to %s\n", arg_option->name,
	      BooleanString(!pos_match));
  }
  global->last_simple_id = 0;	/* not any simple_id */
  return retval;
}

static struct ARG_LIST *FindActualInArgDesc(arg_desc, arg, global)
struct ARG_DESC* arg_desc;
char *arg;
struct GLOBAL *global;
{
  int i;
  struct ARG_LIST *arg_list;
  for (i=0; i<arg_desc->n_lists; i++) {
    arg_list = FindActualInArgList(arg_desc->arg_list[i], arg, global);
    if (arg_list!=NULL)
      return arg_list;
  }
  return NULL;
}

static struct ARG_LIST *FindActualInArgList(arg_list, arg, global)
struct ARG_LIST* arg_list;
char *arg;
struct GLOBAL *global;
{
  enum FIND_GROUP_RESULT find_group_result = NOT_FOUND_EMPTY;
  int no_do_group = 0;
  struct ARG_LIST *next;

  next = arg_list->next;
  /***
   * have to do some sequencing stuff here
   * we try to fill the group first, (unless join=XOR && arglist is partial)
   *
   * retval from FindActualInArgGroup
   * 			/ join	OR/AND		XOR		SEQ
   * NOT_FOUND_NOT_FULL		Try next	Ret NULL	Ret NULL
   * NOT_FOUND_EMPTY		Try next	Try next	Ret NULL
   * NOT_FOUND_FULL		Try next	Ret NULL	Try next
   * FOUND_GROUP_OPTION_OR_ARG_LIST -- Ret group_arg_list (arg_list if NULL) --
   * FOUND_GROUP_SIMPLE_ARG	    -----     Ret arg_list    -----
   */
  no_do_group = (   arg_list->group==NULL
		 || (   next!=NULL && arg_list->join==ARG_XOR
		     && (next->actual & PARTIALLY_FILLED)));
  if (!no_do_group)
    find_group_result = FindActualInArgGroup(arg_list->group, arg, global);
  else
    find_group_result = NOT_FOUND_EMPTY;

  if (   find_group_result==FOUND_GROUP_SIMPLE
      || find_group_result==FOUND_GROUP_OPTION_OR_ARGLIST)
    arg_list->actual |= PARTIALLY_FILLED;
  
  if (find_group_result==FOUND_GROUP_SIMPLE)
    return arg_list;

  if (find_group_result==FOUND_GROUP_OPTION_OR_ARGLIST)
    if (arg_list->group->u.option.arg_list!=NULL)
      return arg_list->group->u.option.arg_list;
    else
      return arg_list;

  if (next==NULL)
    return NULL;

  if (   (arg_list->join==ARG_AND || arg_list->join==ARG_OR)
      || (arg_list->join==ARG_XOR && find_group_result==NOT_FOUND_EMPTY)
      || (arg_list->join==ARG_SEQ && find_group_result==NOT_FOUND_FULL))
      return FindActualInArgList(arg_list->next, arg, global);
  else
    return NULL;
}

static enum FIND_GROUP_RESULT FindActualInArgGroup(arg_group, arg, global)
struct ARG_GROUP* arg_group;
char *arg;
struct GLOBAL *global;
{
  if (arg_group->tag==ARG_GROUP_SIMPLE) {
    int complete_but_space =
      (   arg_group->u.simple.arg_simple!=NULL
       && (arg_group->u.simple.arg_simple->actual & FOUND_ACTUAL)
       && arg_group->u.simple.arg_simple->vector
       && global->last_simple_id==arg_group->u.simple.arg_simple->id);
    
    if (arg_group->u.simple.arg_simple==NULL
	|| (   (arg_group->u.simple.arg_simple->actual & FOUND_ACTUAL)
	    && !complete_but_space))
      return NOT_FOUND_FULL;
    else if (   !IsOption(arg)
	     && FillArgSimple(arg_group->u.simple.arg_simple, arg, global)) {
      arg_group->actual |= PARTIALLY_FILLED;
      return FOUND_GROUP_SIMPLE;
    } else {
      if (complete_but_space)
	return NOT_FOUND_FULL;
      else
	return NOT_FOUND_EMPTY;
    }
  } else {
    enum OPTION_FILL_RESULT opt_fill_res;
    struct ARG_LIST *arg_list;
    assert( arg_group->tag==ARG_GROUP_OPTION );
    /***
     * if arg is an option && option is not NULL, try to fill the option
     * otherwise, try to fill the arglist if it is ready
     */
    if (IsOption(arg) && arg_group->u.option.option!=NULL) {
      opt_fill_res = FillArgOption(arg_group->u.option.option, arg, global);
      if (opt_fill_res==FILLED_AGAIN && arg_group->u.option.arg_list!=NULL) {
	SAFArgList(arg_group->u.option.arg_list, ACTUAL_CLEARED, SET_ON);
	SAFArgList(arg_group->u.option.arg_list, ACTUAL_FLAGS, SET_OFF);
	arg_group->actual |= ACTUAL_CLEARED;	/* set this */
	arg_group->actual &= ~ACTUAL_FLAGS;	/* clear these */
	if (global->debug) {
	  char buf[1000];
	  (void) SprintArgGroup(InitBuf(buf), arg_group, PRINT_BRIEF_LINE, 0);
	  fprintf(dout,"  Option \"%s\" found again, clearing sublist \"%s\"\n",
		  arg_group->u.option.option->name, buf+1);
	}
      }
      if (opt_fill_res==FILLED_FIRST_TIME || opt_fill_res==FILLED_AGAIN) {
	arg_group->actual = PARTIALLY_FILLED;
	return FOUND_GROUP_OPTION_OR_ARGLIST;
      }
    }

    /***
     * return if the option is not filled
     */
    if (   arg_group->u.option.option!=NULL
	&& !(arg_group->u.option.option->actual & FOUND_ACTUAL)) {
      if (   arg_group->u.option.arg_list!=NULL
	  && !(arg_group->u.option.arg_list->actual & PARTIALLY_FILLED))
	return NOT_FOUND_NOT_FULL;
      else
	return NOT_FOUND_EMPTY;
    }

    /***
     * if there is an option, it is full.
     * now, try to fill the arg_list
     */
    if (   arg_group->u.option.arg_list==NULL
	|| (arg_group->u.option.arg_list->actual&FILLED_AND_CLOSED))
      return NOT_FOUND_FULL;

    arg_list = FindActualInArgList(arg_group->u.option.arg_list, arg, global);

    if (arg_list==NULL) {	/* didn't find */
      if (arg_group->u.option.arg_list->actual & SATISFIED_ARGS) {
	arg_group->u.option.arg_list->actual |= FILLED_AND_CLOSED;
	return NOT_FOUND_FULL;
      } else {
	return NOT_FOUND_NOT_FULL;
      }
    } else {
      return FOUND_GROUP_OPTION_OR_ARGLIST;
    }
  }
}


/***
 * SAF* functions "Set All Flags"
 */
static void SAFArgDesc(arg_desc, flags, mode)
struct ARG_DESC *arg_desc;
int flags;
enum SET_MODE mode;
{
  int i;
  for (i=0; i<arg_desc->n_lists; i++)
    SAFArgList(arg_desc->arg_list[i], flags, mode);
}

static void SAFArgList(arg_list, flags, mode)
struct ARG_LIST *arg_list;
int flags;
enum SET_MODE mode;
{
  if (arg_list->group!=NULL)
    SAFArgGroup(arg_list->group, flags, mode);
  if (arg_list->next!=NULL)
    SAFArgList(arg_list->next, flags, mode);
  if (mode==SET_ON)
    arg_list->actual |= flags;
  else if (mode==SET_OFF)
    arg_list->actual &= ~flags;
  else
    assert(0);
}

static void SAFArgGroup(arg_group, flags, mode)
struct ARG_GROUP *arg_group;
int flags;
enum SET_MODE mode;
{
  if (arg_group->tag==ARG_GROUP_OPTION) {
    if (arg_group->u.option.option!=NULL)
      SAFArgOption(arg_group->u.option.option, flags, mode);
    if (arg_group->u.option.arg_list!=NULL)
      SAFArgList(arg_group->u.option.arg_list, flags, mode);
  } else if (arg_group->tag==ARG_GROUP_SIMPLE) {
    if (arg_group->u.simple.arg_simple!=NULL)
      SAFArgSimple(arg_group->u.simple.arg_simple, flags, mode);
  }
  if (mode==SET_ON)
    arg_group->actual |= flags;
  else if (mode==SET_OFF)
    arg_group->actual &= ~flags;
  else
    assert(0);
}

static void SAFArgSimple(simple_arg, flags, mode)
struct ARG_SIMPLE *simple_arg;
int flags;
enum SET_MODE mode;
{
  if (mode==SET_ON)
    simple_arg->actual |= flags;
  else if (mode==SET_OFF)
    simple_arg->actual &= ~flags;
  else
    assert(0);
}

static void SAFArgOption(arg_option, flags, mode)
struct ARG_OPTION *arg_option;
int flags;
enum SET_MODE mode;
{
  if (mode==SET_ON)
    arg_option->actual |= flags;
  else if (mode==SET_OFF)
    arg_option->actual &= ~flags;
  else
    assert(0);
}

static int IsOption(arg)
char *arg;
{
  if ((*arg=='-' || *arg=='+') && isalpha(arg[1])) return 1;
  else return 0;
}

void ArgErrorAbort(va_alist)
va_dcl
{
  va_list args;
  struct GLOBAL *global;
  va_start(args);
  global = va_arg(args, struct GLOBAL *);
  ErrorV(&args);
  if (global->usage_on_error) {
    PrintArgUsage(global->arg_desc, stderr, 1);
  }
  Abort();
}

void ArgError(va_alist)
va_dcl
{
  va_list args;
  struct GLOBAL *global;
  va_start(args);
  global = va_arg(args, struct GLOBAL *);
  ErrorV(&args);
}

int ArgNElements(ptr)
char **ptr;
{
  if (ptr==NULL)
    return 0;
  else {
    int i = 0;
    for (; *ptr!=NULL; ptr++)
      i++;
    return i;
  }
}
