
/**********************************************************************
 * $Id: alias.c,v 1.5 93/03/08 12:13:17 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <stdio.h>
#include "itf.h"
#include "lex.h"
#include "alias.h"

#define IS_SEPARATOR(s)	(!LexQuotedTok(s) && strcmp(s, ";") == 0)
#define IS_QUOTE(c)	((c) == '\'' || (c) == '"' || (c) == '`')
#define MAX_ALIASES	512

static HASH_TABLE	*aliasTable ;
static int		aliasContext ;

/*********************************************************************
 *	Private function
 *********************************************************************/
static void	createAliasTable ARGS((int maxItems)) ;
static char	*insertAlias  ARGS((const char *name, const char *value)) ;
static void	deleteAlias   ARGS((const char *name)) ;
static char	**listAliases ARGS(()) ;
/********************************************************************/


/*********************************************************************
 *	Name:		command_alias
 *	Description:	command to create aliases for other commands
 *	Parameters:
 *	  int	tokc - 
 *	  char	**tokv - 
 *	Return Value:
 *	  int	command_alias - 1 on success, 0 on failure
 *********************************************************************/
int	command_alias(tokc, tokv)
  int	tokc ;
  char	**tokv ;
{
  char		*commandName, *aliasName ;

  IUsage("[<name> [ <def>]]");
  if (GiveHelp(tokc)) {
    ISynopsis("create an alias for a command");
    IHelp
      (IHelpArgs,
       "Assign <def> to the alias <name>.  <def> is a single (quoted) string",
       "that may contain  escaped  variable substitutions.    <name>  is not",
       "allowed to  be alias  or unalias.  If  <def>  is omitted,  the alias",
       "<name> is  displayed along  with  its current  definition.   If both",
       "<name> and <def> are omitted, all aliases are displayed.",
       "EXAMPLES",
       "To create an alias for the command to print a variable named \"foo\":",
       "",
       "\txerion-> alias p 'print $foo'",
       "SEE ALSO",
       "unalias",
       NULL);
    return 1;
  }
  commandName = *tokv ;
  ++tokv, --tokc ;

  if (tokc == 0) {
    char	**list = listAliases() ;
    while (*list != NULL)
      fprintf(dout, "%s\n", *list++) ;
  } else if (tokc == 1) {
    String	 value ;
    if (!tokv || !*tokv) {
      itf_value_buffer[0]=0;
      itf_errno = NULL_ALIASNAME;
      IErrorAbort("") ;
    }
    value = getAlias(*tokv) ;
    if (value != NULL)
      fprintf(dout, "%s\t%s\n", *tokv, value) ;

  } else if (tokc == 2) {
    if (strcmp(tokv[0], "alias") == 0 || strcmp(tokv[0], "unalias") == 0) {
      itf_errno = ILLEGAL_ALIASNAME ;
      IErrorAbort("%s", tokv[0]) ;
      return 0 ;
    }
    aliasName = insertAlias(tokv[0], tokv[1]) ;
    if (aliasName == NULL)
      IAbort() ;
  } else {
    IErrorAbort(IPrintUsage(commandName, usage)) ;
    return 0 ;
  }
  return 1 ;
}
/********************************************************************/
  

/*********************************************************************
 *	Name:		command_unalias
 *	Description:	command to delete aliases for other commands
 *	Parameters:
 *	  int	tokc - 
 *	  char	**tokv - 
 *	Return Value:
 *	  int	command_unalias - 1 on success, 0 on failure
 *********************************************************************/
int	command_unalias(tokc, tokv)
  int	tokc ;
  char	**tokv ;
{
  char	*commandName ;

  IUsage("[<name> ...]");
  if (GiveHelp(tokc)) {
    ISynopsis("delete an alias for a command");
    IHelp
      (IHelpArgs,
       "Discard  the alias  with name  <name>.   All aliases  are deleted by",
       "\"unalias *\".",
       "",
       "SEE ALSO",
       "alias",
       NULL);
    return 1;
  }
  commandName = *tokv ;
  ++tokv, --tokc ;

  while (tokc) {
    if (strcmp(*tokv, "*") == 0) {
      char	**list = listAliases() ;
      char	name[BUFSIZ] ;
      while (*list != NULL && sscanf(*list, "%[^\t]", name)) {
	deleteAlias(name) ;
	++list ;
      }
    } else {
      deleteAlias(*tokv) ;
    }
    ++tokv, --tokc ;
  }
  return 1 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createAliasTable
 *	Description:	create and initialize the hash tables to store
 *			the aliases
 *	Parameters:
 *	  int		maxItems - the maximum number of items
 *	Return Value:
 *	  static void	createAliasTable - NONE
 *********************************************************************/
static void	createAliasTable(maxItems) 
  int		maxItems ;
{
  if (aliasTable == NULL) {
    aliasTable   = INewTable((HASH_TABLE *)0, maxItems, maxItems, ID_INDEX);
    aliasContext = INewTableContext(aliasTable);
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		insertAlias
 *	Description:	inserts or updates an alias in the internal 
 *			hash table
 *	Parameters:
 *	  const char	*name  - the name of the alias
 *	  const char	*value - what it expands to
 *	Return Value:
 *	  static char	*insertAlias - the name of the alias, NULL on error.
 *********************************************************************/
static int	checkAliasSyntax(name)
  const String	name ;
{
  static Lex	lex = NULL ;
  char		**tokv ;
  int		tokc ;
  char		*tmpName = strdup(name) ;

  if (lex == NULL) {
    lex = duplicateLexObj(defaultLex) ;
    LexObjSetSyntax(lex, command_syntax) ;
  }

  tokv = LexObjAnalyse(lex, tmpName, &tokc);
  free(tmpName) ;

  if (itf_errno == 0 && tokc != 1) {
    itf_value_buffer[0]=0;
    itf_errno = ILLEGAL_ALIASNAME ;
  }

  return itf_errno ;
}
/********************************************************************/
static char	*insertAlias(name, value)
  const char	*name ;
  const char	*value ;
{
  TBL_ITEM	*item ;

  if (aliasTable == NULL) 
    createAliasTable(MAX_ALIASES) ;

  if (checkAliasSyntax(name)) {
    IError("%s", name);
    return NULL ;
  }

  if (value == NULL)
    value = "" ;

  item = ITableLookup(aliasTable, name, aliasContext);
  if (item == NULL)
    item = ITableInsert(aliasTable, name, aliasContext, NULL);
    
  if (item == NULL && hash_errno == HT_OVERFLOW) {
    panic("insertAlias", "table overflow on: %s", name) ;
    return NULL ;
  }

  if (item->data.ptr != NULL)
    free(item->data.ptr) ;
  item->data.ptr = strdup(value) ;
  
  return item->name ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		deleteAlias
 *	Description:	deletes an alias from the internal hash table
 *	Parameters:
 *	  const char	*name - the name of the alias to delete
 *	Return Value:
 *	  void	deleteAlias - NONE
 *********************************************************************/
static void	deleteAlias(name)
  const char	*name ;
{
  TBL_ITEM	*item ;

  if (aliasTable == NULL)
    return ;

  if (name == NULL)
    return ;

  item = ITableLookup(aliasTable, name, aliasContext);
  if (item == NULL)
    return ;

  free (item->data.ptr) ;
  ITableDelete(aliasTable, name, aliasContext) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		getAlias
 *	Description:	returns the string an alias expands to
 *	Parameters:
 *	  const char	*name - the name of the alias
 *	Return Value:
 *	  char	*getAlias - the string for the alias, NULL if no alias
 *********************************************************************/
char	*getAlias(name)
  const char	*name ;
{
  TBL_ITEM 	*item ;

  if (aliasTable == NULL) 
    return NULL ;

  item = ITableLookup(aliasTable, name, aliasContext);
  if (item != NULL)
    return item->data.ptr ;
  else
    return NULL ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		listAliases
 *	Description:	returns a STATIC, NULL terminated list of
 *			all aliases and what they expand to
 *	Parameters:	NONE
 *	Return Value:
 *	  char	**listAliases - a STATIC, NULL terminated list of aliases
 *********************************************************************/
static char	**listAliases()
{
  static char	*list[MAX_ALIASES] ;

  TBL_ITEM 	*item ;
  char		string[BUFSIZ] ;
  int		id, idx ;

  if (aliasTable == NULL) 
    createAliasTable(MAX_ALIASES) ;

  for (idx = 0, id = 0 ; item = ITableNext(aliasTable, &id) ; ++idx) {
    sprintf(string, "%s\t%s", item->name, item->data.ptr) ;
    if (list[idx])
      free(list[idx]) ;
    list[idx] = strdup(string) ;
  }
  if (list[idx])
    free(list[idx]) ;
  list[idx] = NULL ;

  return list ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		expandAlias
 *	Description:	expands a command line with aliases, calling
 *			itself recursively. Aborts if any alias is
 *			cyclic.
 *	Parameters:
 *	  const char	*alias   - the alias to expand
 *	  char		*command - an array to write the expanded alias to
 *	  int		maxLen   - the max length for command
 *	Return Value:
 *	  static char	*expandAlias - a pointer to command (space at end).
 *********************************************************************/
char		*aliasSubstitute(alias, command, maxLen)
  const char	*alias ;
  char		*command ;
  int		maxLen ;
{
  static const char	*matched[MAX_ALIASES] ;
  static int		numMatched = 0 ;
  static int		level = 0 ;
  static Lex		lex ;

  int		tokn, tokc, idx ;
  char		**tokv ;
  int		commandLen ;
  char		expandedAlias[BUFSIZ] ;
  const char	*ptr ;

  if (lex == NULL) {
    lex = duplicateLexObj(defaultLex) ;
    LexObjSetSyntax(lex, command_syntax) ;
  }

  /* if we're called from outside, don't 
   * expand the alias (it's already a command line) */
  if (level == 0) {
    ptr = (char *)alias ;
  } else {
    /* if we're called recursively, find the expansion for the
     * alias, then check for loops */
    ptr = getAlias(alias) ;
    if (ptr == NULL)
      return NULL ;

    /* If this is the first level, clear the matches */
    if (level == 1)
      numMatched = 0 ;
    
    /* check for loops */
    for (idx = 0 ; idx < numMatched ; ++idx) {
      if (strcmp(matched[idx], ptr) == 0) {
	int	printLevel = level - 1 ;
	level = 0 ;
	itf_value_buffer[0]=0;
	itf_errno = RECURSIVE_ALIAS ;
	IErrorAbort("%s at level %d", alias, printLevel);
      }
    }
    matched[numMatched++] = ptr ;
  }

  /* parse the expansded alias */
  /* Make sure that the escapes don't get removed */
  idx = 0 ;
  while (idx < maxLen && ptr && *ptr) {
    if (*ptr == '\\' && !IS_QUOTE(*(ptr+1)) && idx < maxLen - 1)
      command[idx++] = '\\' ;
    command[idx++] = *ptr++ ;
  }
  if (idx < maxLen)
    command[idx] = '\0' ;
  tokn = 0 ;
  tokv = LexObjAnalysePush(lex, command, &tokc);
  if (tokv == NULL)
    tokc = 0 ;

  /* iteratively find candidates for alias expansion
   * (first word or words after ";" ) */
  while (tokn < tokc) {
    ++level ;
    ptr = aliasSubstitute(tokv[tokn], expandedAlias, BUFSIZ) ;
    --level ;

    /* rebuild the command line, then reparse it */
    if (ptr != NULL) {
      LexObjReconstruct(lex, command, maxLen, tokn, tokv) ;
      commandLen = strlen(command) ;

      /* Make sure that the escapes don't get removed */
      while (commandLen < maxLen && ptr && *ptr) {
	if (*ptr == '\\' && !IS_QUOTE(*(ptr+1)) && commandLen < maxLen - 1)
	  command[commandLen++] = '\\' ;
	command[commandLen++] = *ptr++ ;
      }
      if (commandLen < maxLen)
	command[commandLen] = '\0' ;
      LexObjReconstruct(lex, command + commandLen, maxLen - commandLen,
			tokc - tokn - 1, tokv + tokn + 1) ;
      commandLen = strlen(command) ;
      LexObjAnalysePop(lex);
      tokv = LexObjAnalysePush(lex, command, &tokc);
      if (tokv == NULL)
	tokc = 0 ;
    }

    /* find the next candidate for expansion (everything up to tokn has
     * already been expanded by this point) */
    for (++tokn ; tokn < tokc && !IS_SEPARATOR(tokv[tokn]) ; ++tokn)
      ;
    ++tokn ;
  }
  /* finally rebuild the command line and stick white-space at the end */
  LexObjReconstruct(defaultLex, command, maxLen, tokc, tokv) ;
  strncat(command, level == 0 ? "\n" : " ", maxLen - strlen(command)) ;
  LexObjAnalysePop(lex);
  return command ;
}
/********************************************************************/
