
/**********************************************************************
 * $Id: outputCommand.c,v 1.4 92/11/30 11:29:01 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <string.h>

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include <xerion/commands.h>

#include "outputCommand.h"

/**********************************************************************/
static int	output		ARGS((OutputCommand)) ;
static String	setFile		ARGS((OutputCommand, String)) ;
static String	setTrace	ARGS((OutputCommand, String)) ;
static String	setSuffixProc 	ARGS((OutputCommand, SuffixProc, void *)) ;
static String	setCommand	ARGS((OutputCommand, String)) ;
static int	setInterval	ARGS((OutputCommand, int)) ;
static String	fileName	ARGS((OutputCommand)) ;
static String	command		ARGS((OutputCommand)) ;
static void	destroy		ARGS((OutputCommand)) ;
/**********************************************************************/
static void	deleteTrace	ARGS((String trace, String commandName)) ;
static void	addTrace	ARGS((String trace, String commandName, 
				      int freq)) ;
/**********************************************************************/


/*********************************************************************
 *	Name:		createOutputCommand
 *	Description:	creates an object for sending a command's output
 *			to a file
 *	Parameters:
 *	  String	traceName - the name of the trace to trigger the
 *			command
 *	  String	commandString - the command string, may contain
 *			*one* "%s" string to be replaced witht the file 
 *			name
 *	  String	file - the file name, may be NULL if you want
 *			to write to stdout
 *	Return Value:
 *	  OutputCommand	createOutputCommand - the object
 *********************************************************************/
OutputCommand	createOutputCommand(traceName, commandString, file)
  String	traceName ;
  String	commandString ;
  String	file ;
{
  OutputCommand	this = (OutputCommand)malloc(sizeof(OutputCommandRec)) ;
  
  if (this == NULL)
    return NULL ;
  
  this->commandFormat	= NULL ;
  this->fileBaseName	= NULL ;
  this->trace		= NULL ;
  this->commandString	= NULL ;
  this->fileNameString	= NULL ;
  this->interval	= 0 ;
  this->suffix		= NULL ;
  this->suffixData	= NULL ;

  this->output		= output ;
  this->setFile		= setFile ;
  this->setTrace	= setTrace ;
  this->setSuffixProc	= setSuffixProc ;
  this->setCommand	= setCommand ;
  this->setInterval	= setInterval ;
  this->fileName	= fileName ;
  this->command		= command ;
  this->destroy		= destroy ;

  MOCsetTrace(this, traceName) ;
  MOCsetCommand(this, commandString) ;
  MOCsetFile(this, file) ;

  return this ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroy
 *	Description:	destroys the object, frees all memory etc.
 *	Parameters:
 *	  OutputCommand	this - the object
 *	Return Value:
 *	  static void	destroy - NONE
 *********************************************************************/
static void	destroy(this)
  OutputCommand	this ;
{
  if (this == NULL)
    return ;

  if (this->commandFormat)
    free((void *)this->commandFormat) ;

  if (this->fileBaseName)
    free((void *)this->fileBaseName) ;

  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setFile
 *	Description:	sets the (base) name of the file
 *	Parameters:
 *	  OutputCommand	this - the object
 *	  String	fileBaseName - the (base) name of the file.
 *	Return Value:
 *	  static String	setFile - the (complete) name of the file
 *********************************************************************/
static String	setFile(this, fileBaseName)
  OutputCommand	this ;
  String	fileBaseName ;
{
  String	command, trace ;

  if (this == NULL)
    return NULL ;

  command = MOCcommand(this) ;
  trace   = MOCtrace(this) ;
  if (trace && command)
    deleteTrace(trace, command) ;

  if (this->fileBaseName)
    free((void *)this->fileBaseName) ;

  if (fileBaseName)
    this->fileBaseName = strdup(fileBaseName) ;
  else
    this->fileBaseName = NULL ;

  command = MOCcommand(this) ;
  if (trace && command && MOCinterval(this))
    addTrace(trace, command, MOCinterval(this)) ;

  return MOCfileName(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setTrace
 *	Description:	sets the trace that triggers the command
 *	Parameters:
 *	  OutputCommand	this - the object
 *	  String	trace - the name of the trace.
 *	Return Value:
 *	  static String	setTrace - the name of the trace
 *********************************************************************/
static String	setTrace(this, trace)
  OutputCommand	this ;
  String	trace ;
{
  String	command, oldTrace ;

  if (this == NULL)
    return NULL ;

  command  = MOCcommand(this) ;
  oldTrace = MOCtrace(this) ;
  if (oldTrace, command)
    deleteTrace(oldTrace, command) ;

  if (this->trace)
    free((void *)this->trace) ;

  if (trace)
    this->trace = strdup(trace) ;
  else
    this->trace = NULL ;

  trace   = MOCtrace(this) ;
  if (trace && command && MOCinterval(this))
    addTrace(trace, command, MOCinterval(this)) ;

  return MOCfileName(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setSuffixProc
 *	Description:	sets the procedure used to get the suffix
 *			for the output file
 *	Parameters:
 *	  OutputCommand	this - the object
 *	  SuffixProc	proc - the procedure of the form
 *				String	(*proc)(OutputCommand, void *data) ;
 *	  void		*data - the data to pass to the procedure
 *	Return Value:
 *	  static String	setSuffixProc - the suffix
 *********************************************************************/
static String	setSuffixProc(this, proc, data)
  OutputCommand	this ;
  SuffixProc	proc ;
  void		*data ;
{
  String	command, trace ;

  if (this == NULL)
    return FALSE ;

  command = MOCcommand(this) ;
  trace   = MOCtrace(this) ;
  if (trace && command)
    deleteTrace(trace, command) ;

  this->suffix     = proc ;
  this->suffixData = data ;

  command = MOCcommand(this) ;
  if (trace && command && MOCinterval(this))
    addTrace(trace, command, MOCinterval(this)) ;

  return MOCsuffix(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setCommand
 *	Description:	sets the command to execute
 *	Parameters:
 *	  OutputCommand	this - the object
 *	  String	commandFormat - the command string. May
 *			contain *one* "%s" to be replaced with the file
 *			name, otherwise output is redirected to the file
 *	Return Value:
 *	  static String	setCommand - the *full* command (with redirection
 *			etc.
 *********************************************************************/
static String	setCommand(this, commandFormat)
  OutputCommand	this ;
  String	commandFormat ;
{
  String	command, trace ;

  if (this == NULL)
    return NULL ;

  command = MOCcommand(this) ;
  trace   = MOCtrace(this) ;
  if (trace && command)
    deleteTrace(trace, command) ;

  if (this->commandFormat)
    free((void *)this->commandFormat) ;

  if (commandFormat)
    this->commandFormat = strdup(commandFormat) ;
  else
    this->commandFormat = NULL ;

  command = MOCcommand(this) ;
  if (trace && command && MOCinterval(this))
    addTrace(trace, command, MOCinterval(this)) ;

  return MOCcommand(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setInterval
 *	Description:	sets the interval between when the command is
 *			done
 *	Parameters:
 *	  OutputCommand	this - the object
 *	  int		interval - the interval
 *	Return Value:
 *	  static int	setInterval - the interval
 *********************************************************************/
static int	setInterval(this, interval)
  OutputCommand	this ;
  int		interval ;
{
  String	trace, command ;

  if (this == NULL)
    return 0 ;

  this->interval = interval ;

  command = MOCcommand(this) ;
  trace   = MOCtrace(this) ;
  if (trace == NULL || command == NULL)
    return 0 ;

  if (MOCinterval(this))
    addTrace(trace, command, MOCinterval(this)) ;
  else
    deleteTrace(trace, command) ;

  return MOCinterval(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		fileName
 *	Description:	returns the *full* file name
 *	Parameters:
 *	  OutputCommand	this - the object
 *	Return Value:
 *	  static String	fileName - a read only copy of the *full* file
 *			name. May change without notice.
 *********************************************************************/
static String	fileName(this)
  OutputCommand	this ;
{
  char	buffer[128] ;
  char	*suffix ;

  if (this->fileBaseName == NULL)
    return NULL ;

  suffix = MOCsuffix(this) ;

  if (this->fileBaseName[0] && suffix)
    sprintf(buffer, "%s%s", this->fileBaseName, suffix) ;
  else
    sprintf(buffer, "%s", this->fileBaseName) ;

  if (this->fileNameString)
    free((void *)this->fileNameString) ;
  this->fileNameString = strdup(buffer) ;

  return this->fileNameString ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		command
 *	Description:	the full command (with file and redirection)
 *	Parameters:
 *	  OutputCommand	this - the object
 *	Return Value:
 *	  static String	command - a read only copy of the *full* command.
 *			May change without notice.
 *********************************************************************/
static String	command(this)
  OutputCommand	this ;
{
  char		commandBuffer[512], buffer[512] ;
  String	fileName ;

  if (this == NULL || this->commandFormat == NULL)
    return NULL ;

  fileName = MOCfileName(this) ;

  if (this->commandFormat[0] == '\0')
    strcpy(commandBuffer, "") ;
  else if (strstr(this->commandFormat, "%s"))
    sprintf(commandBuffer, this->commandFormat, fileName ? fileName : "") ;
  else if (fileName && fileName[0])
    sprintf(commandBuffer, "%s > %s", this->commandFormat, fileName) ;
  else
    sprintf(commandBuffer, "%s", this->commandFormat) ;

  if (fileName && fileName[0] && commandBuffer[0])
    sprintf(buffer, "sh \"rm -f %s\" ; %s", fileName, commandBuffer) ;
  else
    strcpy(buffer, commandBuffer) ;

  if (this->commandString)
    free((void *)this->commandString) ;
  this->commandString = strdup(buffer) ;

  return this->commandString ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		output
 *	Description:	causes the object to execute the command
 *	Parameters:
 *	  OutputCommand	this - the object
 *	Return Value:
 *	  static int	output - 0 if successful, non-zero otherwise
 *********************************************************************/
static int	output(this)
  OutputCommand	this ;
{
  String	command ;

  if (this == NULL)
    return 0 ;
  
  command = MOCcommand(this) ;
  if (command == NULL)
    return 0 ;

  return IDoCommandLine(command) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		addTrace
 *	Description:	adds a command to a trace
 *	Parameters:	
 *		String	traceName - name of the trace.
 *		Strin	commandName - the name of the command to add
 *			to the trace.
 *		int	freq - the interval between triggering the trace.
 *	Return Value:	NONE
 ***********************************************************************/
static void	addTrace(traceName, commandName, freq)
  String	traceName ;
  String	commandName ;
  int		freq ;
{
  TRACE		*trace;
  int		idx ;

  for (trace = first_trace ; 
       trace != NULL && strcmp(traceName, trace->name) != 0 ;
       trace = trace->next)
    ;

  if (trace == NULL || commandName == NULL)
    return ;

  /* If it's already there, return */
  for (idx = 0; idx < trace->n_traces ; ++idx) {
    if (strcmp(trace->trace[idx].command, commandName) == 0) {
      trace->trace[idx].freq = freq;
      return ;
    }
  }

  /* The following is copied from commandTrace (with a few rephrases)
   * in itf/trace.c */
  /* add the new trace */
  idx = trace->n_traces;
  if (trace->n_traces > 0)
    trace->trace = (struct TRACE_RECORD *)
      realloc(trace->trace,
	      (unsigned)(sizeof(struct TRACE_RECORD)*(trace->n_traces+1))) ;
  else
    trace->trace = (struct TRACE_RECORD *)
      calloc(sizeof(struct TRACE_RECORD), 1);

  if (trace->trace == NULL) {
    IOutOfMemory("addTrace", "allocating trace record");
    return;
  }

  trace->n_traces++;
  trace->trace[idx].command = strdup(commandName) ;
  if (trace->trace[idx].command == NULL) {
    IOutOfMemory("addTrace", "allocating command string");
    return;
  }
  trace->trace[idx].active = 1;
  trace->trace[idx].freq = freq;
  trace->trace[idx].count = 0;
  trace->trace[idx].stream_not_open = 0;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		deleteTrace
 *	Description:	deletes a command from a trace
 *		String	traceName - name of the trace.
 *		String	commandName - the name of the command to add
 *			to the trace.
 *	Return Value:	NONE
 ***********************************************************************/
static void	deleteTrace(traceName, commandName)
  String	traceName ;
  String	commandName ;
{
  TRACE		*trace ;
  int		idx ;

  for (trace = first_trace ; 
       trace != NULL && strcmp(traceName, trace->name) != 0 ;
       trace = trace->next)
    ;

  if (trace == NULL || commandName == NULL)
    return ;

  /* find the command and free its memory */
  for (idx = 0 ; idx < trace->n_traces ; ++idx) {
    if (strcmp(trace->trace[idx].command, commandName) == 0) {
      free(trace->trace[idx].command);
      break ;
    }
  }
  /* if we found the command, move all the others forward */
  if (idx < trace->n_traces) {
    for ( ; idx < trace->n_traces ; ++idx)
      trace->trace[idx] = trace->trace[idx + 1] ;
    --trace->n_traces ;
    trace->trace = (struct TRACE_RECORD *)
      realloc(trace->trace,
	      (unsigned)(sizeof(struct TRACE_RECORD)*(trace->n_traces+1))) ;
  }
}
/**********************************************************************/
