/**********************************************************************
 * $Id: mixtureDisplay.c,v 1.10 92/12/02 10:44:43 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

/**********************************************************************
 * A whole mess o' include files.
 **********************************************************************/
#include <math.h>
#include <values.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

#include <xerion/MenuPane.h>
#include <xerion/Selection.h>
#include <xerion/Plot.h>

#include <xerion/display.h>
#undef getValues
#undef setValues

#include <xerion/displayUtils.h>
#include <xerion/display2Itf.h>
#include <xerion/Trace.h>

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include <xerion/commands.h>
#include <xerion/gaussian.h>
#include <xerion/costModel.h>

#include "mixtureDisplay.h"

/*******************************************************************/
typedef struct {
  Widget	shell ;
  Widget	fixedMin ;
  Widget	fixedMax ;
  Widget	fixedHeight ;
  Widget	maxValue ;
  Widget	minValue ;
  Widget	height ;
  Widget	format ;
  Widget	numPoints ;
} *DensityDialog ;
/*******************************************************************/
typedef struct {
  Widget	shell ;
  String	titleFormat ;
  Widget	selection ;
  String	*list ;
  Widget	density ;
  Widget	label ;
  String	labelFormat ;
  CanvasSegment	*segment ;
  int		numSegments ;
  float		min, max ;
  float		height ;
  Boolean	fixedMin ;
  Boolean	fixedMax ;
  Boolean	fixedHeight ;
  int		gaussianIdx ;
  DensityDialog	dialog ;
  Widget	mixtureList ;
  String	mixtureName ;
} DensityDisplayRec, *DensityDisplay ;
/*******************************************************************/
static XtResource	resources[] = {
  {"fixedMin", "Fixed", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(DensityDisplayRec, fixedMin), XtRImmediate, (XtPointer)FALSE },
  {"fixedMax", "Fixed", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(DensityDisplayRec, fixedMax), XtRImmediate, (XtPointer)FALSE },
  {"fixedHeight", "Fixed", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(DensityDisplayRec,fixedHeight),XtRImmediate,(XtPointer)FALSE },
  {"min", "Min", XtRFloat, sizeof(float),
     XtOffsetOf(DensityDisplayRec, min), XtRString, (XtPointer)"-1.0" },
  {"max", "Max", XtRFloat, sizeof(float),
     XtOffsetOf(DensityDisplayRec, max), XtRString, (XtPointer)"1.0" },
  {"densityHeight", "densityHeight", XtRFloat, sizeof(float),
     XtOffsetOf(DensityDisplayRec, height), XtRString, (XtPointer)"1.0" },
  {"numPoints", "NumPoints", XtRInt, sizeof(int),
     XtOffsetOf(DensityDisplayRec, numSegments), XtRImmediate, (XtPointer)500 }
} ;
/*******************************************************************/


/*******************************************************************/
static void	selectMixture ARGS((DensityDisplay)) ;
static void	redrawDensity ARGS((DensityDisplay)) ;
static Boolean	gaussiansChanged ARGS((Mixture, String *)) ;
static String	*listGaussians ARGS((Mixture, String *)) ;
static void	setRange ARGS((DensityDisplay, Mixture)) ;
/*******************************************************************/
static void  destroyCB	  ARGS((Widget, XtPointer, XtPointer)) ;
static void  redrawCB	  ARGS((Widget, XtPointer, XtPointer)) ;
static void  dialogCB	  ARGS((Widget, XtPointer, XtPointer)) ;
static void  saveDialogCB ARGS((Widget, XtPointer, XtPointer)) ;
static void  setGaussianIdxCB ARGS((Widget, XtPointer, XtPointer)) ;
static void  selectMixtureCB ARGS((Widget, XtPointer, XtPointer)) ;
static void  changeMixtureCB ARGS((Widget, XtPointer, XtPointer)) ;
static void  createMixtureDisplayCB ARGS((Widget, XtPointer, XtPointer)) ;
/*******************************************************************/
static CostModel	*listModels	 ARGS(()) ;
static String		*listModelNames ARGS(()) ;
static CostModel	modelFromName ARGS((String)) ;
static Mixture		mixtureFromName ARGS((String)) ;
static int		compare ARGS((const void *, const void *)) ;
/*******************************************************************/
static Cardinal		numDisplays = 0 ;
/*******************************************************************/


/*********************************************************************
 *	Name:		createMixtureQuery
 *	Description:	creates a query box that then allows you
 *			to create a mixture of gaussians display
 *	Parameters:
 *	  Widget	shell - the shell to hold the query box
 *	Return Value:
 *	  static void	createMixtureQuery - NONE
 *********************************************************************/
static void	listModelNamesCB ARGS((Widget, XtPointer, XtPointer)) ;
/*******************************************************************/
static Widget	 buildMixtureQuery(shell, okCB, okData)
  Widget	 shell ;
  XtCallbackProc okCB ;
  XtPointer	 okData ;
{
  Widget	selection ;
  Widget	button ;

  selection = XtVaCreateManagedWidget("selection", selectionWidgetClass, shell,
				      XtNlist,	listModelNames(), 
				      NULL) ;
  
  button = XtSelectionAddButton(selection, "ok",
				popdownWidgetCB, (XtPointer)shell) ;
  XtAddCallback(button, XtNcallback, okCB, okData) ;
  
  button = XtSelectionAddButton(selection, "cancel", 
				popdownWidgetCB, (XtPointer)shell) ;
				
  XtAddCallback(shell, XtNpopupCallback,
		listModelNamesCB, (XtPointer)selection) ;

  XtInstallAllAccelerators(selection, selection) ;

  return selection ;
}
/*******************************************************************/
export void	createMixtureQuery(shell)
  Widget	shell ;
{
  buildMixtureQuery(shell, createMixtureDisplayCB, (XtPointer)shell) ;

  XtVaSetValues(shell, XtNcreatePopupChildProc, NULL, NULL) ;
}
/********************************************************************/
static void	listModelNamesCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	selection = (Widget)clientData ;

  XtVaSetValues(selection, XtNlist, listModelNames(), NULL) ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		createMixtureDisplay
 *	Description:	procedure to create the mixture display
 *	Parameters:
 *	  Widget	popup - the popup shell that will contain the
 *				display
 *	Return Value:
 *	  export void	createMixtureDisplay - NONE
 *********************************************************************/
static void	createMixtureDisplayCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	shell = (Widget)clientData ;
  Widget	selection = XtNameToWidget(shell, "selection") ;

  if (selection) {
    XawListReturnStruct	*rs = XtSelectionGetSelected(selection) ;
    if (rs && rs->list_index != XAW_LIST_NONE)
      createMixtureDisplay(rs->string, NULL) ;
  }
}
/*********************************************************************/
export Widget	createMixtureDisplay(name, geometry)
  String	name ;
  String	geometry ;
{
  DensityDisplay this ;
  Widget	shell, form, menu, menuPane ;
  Widget	widget ;
  float		width ;
  char		buffer[BUFSIZ] ;

  if (topLevel == NULL)
    return NULL ;

  this	       = (DensityDisplay)XtMalloc(sizeof(*this)) ;
  this->dialog = NULL ;
  this->shell  = XtVaCreatePopupShell("densityDisplay", 
				      topLevelShellWidgetClass, topLevel, 
				      NULL) ;
  this->titleFormat = getTitle(this->shell) ;
  
  form = XtVaCreateManagedWidget("form", formWidgetClass, this->shell, NULL) ;

  /********************************************
   * Pulldown menu pane 
   */
  menuPane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form,
				     NULL) ;

  /********************************************
   * Make the list of gaussians
   */
  this->list      = listGaussians(NULL, NULL) ;
  this->selection = XtVaCreateManagedWidget("list", selectionWidgetClass, form,
					    XtNlist,	this->list,
					    NULL) ;
  XtAddCallback(this->selection, 
		XtNcallback, setGaussianIdxCB, (XtPointer)this) ;
  this->gaussianIdx = -1 ;

  /********************************************
   * Make the label for the plot widget:
   */
  this->label = XtVaCreateManagedWidget("statLabel", labelWidgetClass, form, 
					XtNmappedWhenManaged,	FALSE,
					NULL) ;
  this->labelFormat = getLabel(this->label) ;

  /********************************************
   * Make the plot widget
   */
  this->density = XtVaCreateManagedWidget("plot", plotWidgetClass, form, 
					  NULL) ;
  
  XtVaSetValues(XtNameToWidget(this->density, "canvas"),
		XtNautoScale,	FALSE, 
		NULL) ;

  XtGetApplicationResources(this->density, (XtPointer)this,
			    resources, XtNumber(resources), NULL, 0) ;
  width = this->max - this->min ;
  XtVaSetValues(this->density,
		XtNcanvasXOffset,	&this->min,
		XtNcanvasWidth,		&width,
		XtNcanvasHeight,	&this->height,
		NULL) ;

  /********************************************
   * Display Updates popup shell
   */
  sprintf(buffer, "mixtureDisplay %d", numDisplays++) ;
  shell = XtVaCreatePopupShell(buffer, traceWidgetClass, this->shell,
			       NULL) ;
  XtAddCallback(shell, XtNcallback, redrawCB, (XtPointer)this) ;

  /**********************
   * Options pulldown menu 
   */
  widget = XtVaCreateManagedWidget("options", menuButtonWidgetClass, menuPane,
				   XtNmenuName, "optionsMenu",
				   NULL) ;
  
  menu   = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, widget,
				NULL) ;
  
  widget = XtVaCreateManagedWidget("updates", smeBSBObjectClass,  menu, NULL) ;
  XtAddCallback(widget, XtNcallback, popupCenteredCB, (XtPointer)shell) ;

  widget = XtVaCreateManagedWidget("reset",   smeBSBObjectClass,  menu, NULL) ;
  XtAddCallback(widget, XtNcallback, redrawCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("line",    smeLineObjectClass, menu, NULL) ;

  widget = XtVaCreateManagedWidget("close",   smeBSBObjectClass,  menu, NULL) ;
  XtAddCallback(widget, XtNcallback, popdownWidgetCB, (XtPointer)this->shell) ;

  /**********************
   * Density menu
   */
  widget = XtVaCreateManagedWidget("density", menuButtonWidgetClass, menuPane,
				   XtNmenuName, "densityMenu",
				   NULL) ;

  menu   = XtVaCreatePopupShell("densityMenu", simpleMenuWidgetClass, widget, 
				NULL) ;
  
  widget = XtVaCreateManagedWidget("select", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, selectMixtureCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("misc", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, dialogCB, 	      (XtPointer)this) ;

  if (this->numSegments < 0)
    this->numSegments = 100 ;
  this->segment = (CanvasSegment *)calloc(this->numSegments, 
					  sizeof(*this->segment)) ;

  XtAddCallback(this->shell, XtNpopupCallback,   redrawCB,  (XtPointer)this) ;
  XtAddCallback(this->shell, XtNpopdownCallback, destroyCB, (XtPointer)this) ;

  this->mixtureList = NULL ;
  this->mixtureName = strdup(name) ;

  if (geometry)
    XtVaSetValues(this->shell, XtNgeometry, geometry, NULL) ;

  XtPopup(this->shell, XtGrabNone) ;
  XSetWMProtocols(XtDisplay(this->shell), 
		  XtWindow(this->shell), &wm_delete_window, 1);

  XtAddCallback(controller, XtNcallback, redrawCB, (XtPointer)this) ;

  return this->shell ;
}
/*********************************************************************/


/*******************************************************************
 *	Name:		redrawDensity
 *	Description:	Draws into the plot widget.
 *			Called by the "redraw" button.
 *	Parameters:
 *	  DensityDisplay	this - the display ;
 *	Return Value:
 *	  static void	redrawDensity - NONE
 *******************************************************************/
static int	compare(ptr1, ptr2)
  const void	*ptr1 ;
  const void	*ptr2 ;
{
  float	diff = *(float *)ptr1 - *(float *)ptr2 ;

  if (diff < 0)
    return -1 ;
  else if (diff > 0)
    return 1 ;
  else
    return 0 ;
}
/*********************************************************************/
static void	redrawDensity(this)
  DensityDisplay	this ;
{
  Widget	plot = this->density ;
  CostModel	costModel ;
  Mixture	mixture ;
  Gaussian	oneGaussian ;
  float		width, height, interval ;
  float		lastY, lastX, y, x ;
  float		zero = 0.0 ;
  int		idx ;
  float		*mean, *meanPtr ;
  
  if (this == NULL)
    return ;

  costModel = modelFromName(this->mixtureName) ;
  mixture   = mixtureFromName(this->mixtureName) ;

  if (mixture == NULL) {
    widgetPrintf(this->shell, this->titleFormat, "No Mixture") ;
    this->list = listGaussians(NULL, this->list) ;
    XtVaSetValues(this->selection, XtNlist, this->list, NULL) ;
    if (XtIsRealized(this->label))
      XtUnmapWidget(this->label) ;
    XtPlotClear(plot, TRUE) ;
    return ;
  }

  MCMsync(costModel) ;

  if (this->gaussianIdx >= 0)
    oneGaussian = MMgaussian(mixture, this->gaussianIdx) ;
  else
    oneGaussian = NULL ;

  setRange(this, mixture) ;

  width    = this->max - this->min ;
  interval = width/this->numSegments ;

  if (oneGaussian) {
    mean = (float *)XtCalloc(2, sizeof(float)) ;
    mean[0] = MGmean(oneGaussian) ;
    mean[1] = this->max + 1.0 ;
  } else {
    int	numGaussians = MMnumGaussians(mixture) ;
    mean = (float *)XtCalloc(numGaussians + 1, sizeof(float)) ;
    for (idx = 0 ; idx < numGaussians ; ++idx)
      mean[idx] = MGmean(MMgaussian(mixture, idx)) ;
    qsort((void *)mean, numGaussians, sizeof(float), compare) ;
    mean[numGaussians] = this->max + 1.0 ;
  }

  height = 0.0 ;
  lastX  = this->min ;
  if (oneGaussian)
    lastY = MGproportion(oneGaussian) * MGdensity(oneGaussian, this->min) ;
  else
    lastY = MMdensity(mixture, this->min) ;

  for (meanPtr = mean ; lastX > *meanPtr ; ++meanPtr)
    ;

  for (idx = 0 ; idx < this->numSegments ; ++idx) {
    Real	nextX = this->min + interval*(idx + 1) ;

    if (nextX != this->max && lastX < *meanPtr && nextX > *meanPtr) {
      while (nextX > *(meanPtr + 1))
	meanPtr++ ;
      x = *meanPtr++ ;
    } else {
      x = nextX ;
    }
    
    if (oneGaussian)
      y = MGproportion(oneGaussian) * MGdensity(oneGaussian, x) ;
    else
      y = MMdensity(mixture, x) ;

    this->segment[idx].x1 = lastX ;
    this->segment[idx].y1 = lastY ;
    this->segment[idx].x2 = x ;
    this->segment[idx].y2 = y ;

    height = MAX(height, y) ;
    lastX = x ;
    lastY = y ;
  }
  if (this->fixedHeight)
    height = this->height ;
  else
    height *= 1.1 ;

  XtFree((void *)mean) ;

  widgetPrintf(this->shell, this->titleFormat, this->mixtureName) ;
  if (gaussiansChanged(mixture, this->list)) {
    this->list = listGaussians(mixture, this->list) ;
    XtVaSetValues(this->selection, XtNlist, this->list, NULL) ;
  }

  if (oneGaussian) {
    widgetPrintf(this->label, this->labelFormat, 
		 MGmean(oneGaussian), MGvariance(oneGaussian), 
		 MGproportion(oneGaussian)) ;
    XtMapWidget(this->label) ;
  } else {
    if (XtIsRealized(this->label))
      XtUnmapWidget(this->label) ;
  }
  XtPlotClear(plot, FALSE) ;

  XtVaSetValues(plot,
		XtNcanvasXOffset, &this->min,
		XtNcanvasWidth,   &width,
		XtNcanvasYOffset, &zero,
		XtNcanvasHeight,  &height,
		NULL) ;

  XtPlotAddSegments(plot, this->segment, this->numSegments, TRUE) ;
}
/*********************************************************************/
static void	setRange(this, mixture)
  DensityDisplay	this ;
  Mixture		mixture ;
{
  Gaussian	gaussian, oneGaussian ;
  int		numGaussians = MMnumGaussians(mixture) ; 
  float		interval, yMax, tmpReal ;
  Boolean	foundMin ;
  int		idx ;

  if (this->fixedMin && this->fixedMax)
    return ;

  if (this->gaussianIdx >= 0)
    oneGaussian = MMgaussian(mixture, this->gaussianIdx) ;
  else
    oneGaussian = NULL ;

  if (!this->fixedMin) {
    if (oneGaussian) {
      this->min = MGmean(oneGaussian) - 3*MGstdDeviation(oneGaussian) ;
    } else {
      this->min = MAXFLOAT ;
      for (idx = 0 ; idx < numGaussians ; ++idx) {
	gaussian  = MMgaussian(mixture, idx) ;
	tmpReal	  = MGmean(gaussian) - 3*MGstdDeviation(gaussian) ;
	this->min = MIN(this->min, tmpReal) ;
      }
    }
  }

  if (!this->fixedMax) {
    if (oneGaussian){
      this->max = MGmean(oneGaussian) + 3*MGstdDeviation(oneGaussian) ;
    } else {
      this->max = -MAXFLOAT ;
      for (idx = 0 ; idx < numGaussians ; ++idx) {
	gaussian  = MMgaussian(mixture, idx) ;
	tmpReal   = MGmean(gaussian) + 3*MGstdDeviation(gaussian) ;
	this->max = MAX(this->max, tmpReal) ;
      }
    }
  }

  interval = (this->max - this->min)/this->numSegments ;

  yMax = 0.0 ;
  for (idx = 0 ; idx < this->numSegments ; ++idx) {
    this->segment[idx].x1 = this->min + interval*idx ;
    if (oneGaussian)
      this->segment[idx].y1
	= MGproportion(oneGaussian)
	  * MGdensity(oneGaussian, this->min + interval*idx) ;
    else
      this->segment[idx].y1 = MMdensity(mixture, this->min + interval*idx);

    yMax = MAX(yMax, this->segment[idx].y1) ;
  }

  if (this->fixedMin)
    foundMin = TRUE ;
  else
    foundMin = FALSE ;
  yMax /= 50 ;
  for (idx = 0 ; idx < this->numSegments ; ++idx) {
    if (foundMin == FALSE) {
      if (this->segment[idx].y1 > yMax) {
	this->min = this->segment[idx].x1 - interval ;
	foundMin  = TRUE ;
      }
    } else {
      if (this->fixedMax)
	break ;
      if (this->segment[idx].y1 > yMax)
	this->max = this->segment[idx].x1 + interval ;
    }
  }
}
/*********************************************************************/


/********************************************************************/
static void	redrawCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  redrawDensity((DensityDisplay)clientData) ;
}
/********************************************************************/
static void	setGaussianIdxCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay	this = (DensityDisplay)clientData ;
  XawListReturnStruct	*rs = (XawListReturnStruct *)callData ;

  if (rs && rs->list_index != XAW_LIST_NONE)
    this->gaussianIdx = rs->list_index - 1 ;
  else
    this->gaussianIdx = -1 ;
  redrawDensity(this) ;
}
/********************************************************************/
static void	destroyCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay	this = (DensityDisplay)clientData ;

  if (this->titleFormat)
    XtFree((void *)this->titleFormat) ;

  if (this->labelFormat)
    XtFree((void *)this->labelFormat) ;
  
  if (this->segment)
    XtFree((void *)this->segment) ;
  
  if (this->mixtureName)
    XtFree((void *)this->mixtureName) ;
  
  XtDestroyWidget(this->shell) ;

  XtRemoveCallback(controller, XtNcallback, redrawCB, (XtPointer)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		selectMixtureCB
 *	Description:	pops up a dialog allowing you to select a
 *			mixture of gaussians.
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	selectMixtureCB - 
 *********************************************************************/
static void	selectMixtureCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay this = ((DensityDisplay)clientData) ;
  Widget	 selection = this->mixtureList ;

  if (selection == NULL) {
    Widget	shell = XtVaCreatePopupShell("selectShell", 
					     transientShellWidgetClass, 
					     this->shell,
					     NULL) ;
    selection = buildMixtureQuery(shell, changeMixtureCB, (XtPointer)this) ;
    this->mixtureList = selection ;
  }

  XtVaSetValues(selection, XtNlist, listModelNames(), NULL) ;
  popupCentered(XtParent(selection), XtGrabExclusive) ;
}
/********************************************************************/
static void	changeMixtureCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay	 this = (DensityDisplay)clientData ;
  XawListReturnStruct	*rs   = XtSelectionGetSelected(this->mixtureList) ;

  if (this->mixtureName)
    XtFree(this->mixtureName) ;

  if (rs && rs->list_index != XAW_LIST_NONE)
    this->mixtureName = strdup(rs->string) ;
  else
    this->mixtureName = NULL ;

  redrawDensity(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		miscellaneousCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	miscellaneousCB - 
 *********************************************************************/
static void	dialogCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay this = (DensityDisplay)clientData ;
  DensityDialog	dialog = this->dialog ;
  float		*min, *width, *height ;
  String	format ;

  if (dialog == NULL) {
    Widget	shell, form, button, table ;
    
    dialog = (DensityDialog)XtMalloc(sizeof(*this->dialog)) ;
    this->dialog = dialog ;

    shell = XtVaCreatePopupShell("miscShell", 
				 transientShellWidgetClass, this->shell, NULL);
    dialog->shell = shell ;
    form = XtVaCreateManagedWidget("form", formWidgetClass, shell, NULL) ;

    table = startTable(form) ;
    dialog->minValue  = addTableItem("minValue",  &table, TRUE) ;
    dialog->maxValue  = addTableItem("maxValue",  &table, TRUE) ;
    dialog->height    = addTableItem("height",    &table, TRUE) ;
    dialog->format    = addTableItem("format",    &table, FALSE) ;
    dialog->numPoints = addTableItem("numPoints", &table, FALSE) ;

    dialog->fixedMax	= XtNameToWidget(form, "maxValueToggle") ;
    dialog->fixedMin	= XtNameToWidget(form, "minValueToggle") ;
    dialog->fixedHeight	= XtNameToWidget(form, "heightToggle") ;

    button = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				     XtNfromVert,	dialog->numPoints,
				     NULL) ;
    XtAddCallback(button, XtNcallback, saveDialogCB,	(XtPointer)this) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB,	(XtPointer)shell) ;

    button = XtVaCreateManagedWidget("cancel", commandWidgetClass, form,
				     XtNfromVert,	dialog->numPoints,
				     XtNfromHoriz,	button,
				     NULL) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)shell) ;

    XtInstallAllAccelerators(form, form) ;
  }
  dialog = this->dialog ;

  XtVaGetValues(this->density,
		XtNcanvasXOffset,	&min,
		XtNcanvasWidth,		&width,
		XtNcanvasHeight,	&height,
		XtNformat,		&format,
		NULL) ;

  XtVaSetValues(dialog->fixedMin,    XtNstate, this->fixedMin,	  NULL) ;
  XtVaSetValues(dialog->fixedMax,    XtNstate, this->fixedMax,	  NULL) ;
  XtVaSetValues(dialog->fixedHeight, XtNstate, this->fixedHeight, NULL) ;

  widgetPrintf(dialog->minValue,  "%g", *min) ;
  widgetPrintf(dialog->maxValue,  "%g", *min + *width) ;
  widgetPrintf(dialog->height,    "%g", *height) ;
  widgetPrintf(dialog->format,    "%s", format) ;
  widgetPrintf(dialog->numPoints, "%d", this->numSegments) ;

  popupCentered(dialog->shell, XtGrabExclusive) ;
}
/********************************************************************/
static void	saveDialogCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  DensityDisplay this   = (DensityDisplay)clientData ;
  DensityDialog	 dialog = this->dialog ;
  String	 string ;

  string = getText(dialog->minValue) ;
  if (string != NULL)
    this->min = atof(string) ;

  string = getText(dialog->maxValue) ;
  if (string != NULL)
    this->max = atof(string) ;

  string = getText(dialog->height) ;
  if (string != NULL)
    this->height = atof(string) ;

  string = getText(dialog->format) ;
  if (string != NULL)
    XtVaSetValues(this->density, XtNformat, string, NULL) ;

  string = getText(dialog->numPoints) ;
  if (string != NULL 
      && !(atol(string) == this->numSegments || atol(string) <= 0)) {
    this->numSegments = atol(string) ;
    if (this->segment)
      XtFree((void *)this->segment) ;
    this->segment
      = (CanvasSegment *)XtCalloc(this->numSegments, sizeof(*this->segment)) ;
  }

  XtVaGetValues(dialog->fixedMin,    XtNstate, &this->fixedMin,    NULL) ;
  XtVaGetValues(dialog->fixedMax,    XtNstate, &this->fixedMax,    NULL) ;
  XtVaGetValues(dialog->fixedHeight, XtNstate, &this->fixedHeight, NULL) ;

  redrawDensity(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		listGaussians
 *	Description:	create a list of the gaussians in the cost model
 *	Parameters:	NONE
 *	Return Value:
 *	  static String	*listGaussians - a STATIC list (NULL terminated), 
 *					overwritten at the next call.
 *********************************************************************/
static Boolean	gaussiansChanged(mixture, list)
  Mixture	mixture ;
  String	*list ;
{
  int	listSize ;

  for (listSize = 0 ; list && list[listSize] ; ++listSize)
    ;

  if (mixture == NULL || listSize != MMnumGaussians(mixture) + 1)
    return TRUE ;
  else
    return FALSE ;
}
/********************************************************************/
static String	*listGaussians(mixture, list)
  Mixture	mixture ;
  String	*list ;
{
  int		idx, listSize ;

  if (!gaussiansChanged(mixture, list))
    return list ;
      
  for (listSize = 0 ; list && list[listSize] ; ++listSize)
    ;

  if (mixture == NULL || listSize != MMnumGaussians(mixture) + 1) {
    if (list) {
      for (idx = 0 ; idx < listSize ; ++idx)
	XtFree((void *)list[idx]) ;
      XtFree((void *)list) ;
    }
  }

  if (mixture == NULL) {
    listSize = 0 ;
    list = (String *)XtCalloc(listSize + 1, sizeof(String)) ;

  } else if (listSize != MMnumGaussians(mixture) + 1) {
    listSize = MMnumGaussians(mixture) + 1 ;
    list = (String *)XtCalloc(listSize + 1, sizeof(String)) ;
    list[0] = strdup("All") ;
    for (idx = 0 ; idx < MMnumGaussians(mixture) ; ++idx) {
      char	buf[32] ;
      sprintf(buf, "Gaussian %d", idx) ;
      list[idx + 1] = strdup(buf) ;
    }
  }
  list[listSize] = NULL ;
  return list ;
}
/********************************************************************/
static void	getModelName(group, data)
  Group		group ;
  void		*data ;
{
  String	**ptr = (String **)data ;
  
  if (McostModel(group) && MCMtype(McostModel(group)) == CM_MIXTURE) {
    **ptr = group->name ;
    ++(*ptr) ;
  }
}
/********************************************************************/
static String	*listModelNames()
{
  static String	*modelNames ;

  String	*ptr ;
  int		maxNum ;

  if (modelNames)
    XtFree((void *)modelNames) ;

  maxNum = currentNet ? currentNet->numGroups + 2 : 1 ;

  modelNames = (String *)XtCalloc(maxNum, sizeof(String)) ;

  ptr = modelNames ;
  if (currentNet) {
    if (McostModel(currentNet) 
	&& MCMtype(McostModel(currentNet)) == CM_MIXTURE)
      *(ptr++) = currentNet->name ;

    netForAllGroups(currentNet, ALL, getModelName, &ptr) ;
  }
  *ptr = NULL ;

  return modelNames ;
}
/********************************************************************/


/********************************************************************/
static void	getModel(group, data)
  Group		group ;
  void		*data ;
{
  CostModel	**ptr = (CostModel **)data ;
  
  if (McostModel(group) && MCMtype(McostModel(group)) == CM_MIXTURE) {
    **ptr = McostModel(group) ;
    ++(*ptr) ;
  }
}
/********************************************************************/
static CostModel	*listModels()
{
  static CostModel	*costModels ;

  CostModel	*ptr ;
  int		maxNum ;

  if (costModels)
    XtFree((void *)costModels) ;

  maxNum = currentNet ? currentNet->numGroups + 2 : 1 ;

  costModels = (CostModel *)XtCalloc(maxNum, sizeof(CostModel)) ;

  ptr = costModels ;
  if (currentNet) {
    if (McostModel(currentNet) 
	&& MCMtype(McostModel(currentNet)) == CM_MIXTURE)
      *(ptr++) = McostModel(currentNet) ;

    netForAllGroups(currentNet, ALL, getModel, &ptr) ;
  }
  *ptr = NULL ;

  return costModels ;
}
/********************************************************************/
static CostModel	modelFromName(name)
  String	name ;
{
  String	*names	   = listModelNames() ;
  CostModel	*costModels = listModels() ;
  int		idx ;

  if (name == NULL)
    return NULL ;

  for (idx = 0 ; names[idx] && strcmp(name, names[idx]) ; ++idx)
    ;

  return costModels[idx] ;
}
/*******************************************************************/
static Mixture	mixtureFromName(name)
  String	name ;
{
  CostModel	costModel = modelFromName(name) ;

  if (costModel)
    return costModel->costModelData.mixtureData->mixture ;
  else
    return NULL ;
}
/*******************************************************************/
