
/**********************************************************************
 * $Id: miniDisplay.c,v 1.14 92/11/30 11:28:51 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <signal.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>

#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

#include <xerion/Selection.h>
#include <xerion/MenuPane.h>

#include <xerion/display.h>
#undef getValues
#undef setValues

#include <xerion/useful.h>
#include <xerion/minimize.h>

#include "display2Itf.h"
#include "display2Sim.h"
#include "displayUtils.h"
#include "outputCommand.h"
#include "miniLists.h"

extern Minimize	currentMinimizeRecord     ARGS((void)) ;

#define NO_SELECTION(selectionPtr) \
  ((selectionPtr) == NULL || (selectionPtr)->list_index == XAW_LIST_NONE)

#define doResetCB(this, idx) 	\
  (this)->callback[idx].proc(this, (this)->callback[idx].data)

typedef struct MinimizeDisplayRec	*MinimizeDisplay ;
typedef void	(*ResetProc) ARGS((MinimizeDisplay, XtPointer)) ;
typedef struct Reset {
  ResetProc	proc ;
  XtPointer	data ;
} ResetCallback ;

typedef struct	SaveWeightsDialogRec {
  Widget	shell ;
  Widget	command ;
  Widget	fileName ;
  Widget	addSuffix ;
  Widget	interval ;
} SaveWeightsDialogRec, *SaveWeightsDialog ;

typedef struct MinimizeDisplayRec {
  Minimize	mz ;
  Widget	shell ;		/* the popup shell for the display 	*/
  String	titleFormat ;	/* the format for the title string	*/
  Widget	iterations ;
  Widget	batchSize ;
  Widget	reportInterval ;
  int		iterationsNum ;
  ResetCallback	*callback ;
  int		numCallbacks ;
  int		maxCallbacks ;
  Widget	saveInterval ;
  OutputCommand	saveCommand ;
  SaveWeightsDialog	saveWeightsDialog ;
} MinimizeDisplayRec ;

typedef struct {
  Boolean	showParams ;
  String	weightFile ;
  String	saveWeightsCommand ;
  Boolean	appendIteration ;
  int		saveInterval ;
} OptionsRec ;

static OptionsRec	options ;

#define Offset(x)	XtOffsetOf(OptionsRec, x)

static XtResource	resources[] = {
  {"showParameters", "ShowParameters", XtRBoolean, sizeof(Boolean),
     Offset(showParams), XtRImmediate, (XtPointer)FALSE },
  {"weightFile", "WeightFile", XtRString, sizeof(String),
     Offset(weightFile), XtRImmediate, (XtPointer)"xerion.wt" },
  {"saveWeightsCommand", "saveWeightsCommand", XtRString, sizeof(String),
     Offset(saveWeightsCommand), XtRImmediate, (XtPointer)"saveWeights" },
  {"appendIteration", "AppendIteration", XtRBoolean, sizeof(Boolean),
     Offset(appendIteration), XtRImmediate, (XtPointer)FALSE },
  {"saveInterval", "SaveInterval", XtRInt, sizeof(int),
     Offset(saveInterval), XtRString, (XtPointer)"0" }
} ;
#undef Offset

static void 	minimizeAction ARGS((Widget, XEvent *, String *, Cardinal *)) ;
static void 	setTextOptionsAction ARGS((Widget, XEvent *, 
					   String *, Cardinal *)) ;

static XtActionsRec	actions[] = {
  {"Minimize", 		minimizeAction},
  {"SetTextOptions",	setTextOptionsAction}
} ;

/* Routines for redisplaying the whole display */
static void 	resetDisplayCB	ARGS((Widget, XtPointer, XtPointer)) ;

static void 	setCheckCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setRestartCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setDirectionCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setStepCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setInitStepCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	saveMiniParamCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	saveNetParamCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	showMiniParamCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	showNetParamCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	showHelpCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	toggleManagedCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	minimizeCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	doCommandLineCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	saveWeightsCB	ARGS((Widget, XtPointer, XtPointer)) ;

static void	addResetCallback ARGS((MinimizeDisplay, ResetProc, XtPointer));
static void	syncCheckRCB     ARGS((MinimizeDisplay, XtPointer)) ;
static void	sensitizeRCB     ARGS((MinimizeDisplay, XtPointer)) ;
static void	sensitizeNetRCB  ARGS((MinimizeDisplay, XtPointer)) ;
static void 	showMiniParamRCB ARGS((MinimizeDisplay, XtPointer)) ;
static void 	showNetParamRCB  ARGS((MinimizeDisplay, XtPointer)) ;
static void 	syncDirectionRCB ARGS((MinimizeDisplay, XtPointer)) ;
static void 	syncStepRCB	 ARGS((MinimizeDisplay, XtPointer)) ;
static void 	syncInitStepRCB	 ARGS((MinimizeDisplay, XtPointer)) ;

static int	defaultStepMethod ARGS((int directionMethod)) ;
static String	stringFromFlag ARGS((int,    int flags[], String strings[])) ;
static int	flagFromString ARGS((String, int flags[], String strings[])) ;
static void	resetDisplay   ARGS((MinimizeDisplay this)) ;
static void 	resetNetParam  ARGS((Widget this)) ;
static void 	resetMiniParam ARGS((Widget this)) ;

static Widget	createRadioGroup ARGS((String list[], Widget parent,
				       XtCallbackProc, XtPointer)) ;
static Widget	createMiniSelection ARGS((String name, 	
					  String list[], Widget	parent)) ;
static Widget	createNetParamSelection ARGS((String name, Widget parent)) ;

static void 	setTextOptions ARGS((MinimizeDisplay)) ;
static void 	minimizeThis   ARGS((MinimizeDisplay)) ;

static String	weightFileSuffix	ARGS((OutputCommand, void *)) ;

static MinimizeDisplay	minimizeDisplay ;


/***********************************************************************
 *	Name:		createMinimizeDisplay
 *	Description:	creates the window for the network display
 *			and all of the sub windows and necessary methods
 *	Parameters:	
 *		Widget	popup - the popup shell we are going in
 *	Return Value:	
 *		NONE
 ***********************************************************************/
export void	createMinimizeDisplay(popup)
  Widget	popup ;
{
  MinimizeDisplay	this ;
  Widget	pane, menuButton, menu, line ;
  Widget	selection, form, box, table ;
  Widget	label, button, enlarge ;
  String	name ;

  minimizeDisplay = (MinimizeDisplay)XtMalloc(sizeof(MinimizeDisplayRec)) ;
  this = minimizeDisplay ;

  this->maxCallbacks = 0 ;
  this->numCallbacks = 0 ;
  this->callback     = NULL ;

  this->shell = popup ;
  XtGetApplicationResources(this->shell, (XtPointer)&options,
			    resources, XtNumber(resources), NULL, 0) ;
  XtAppAddActions(XtWidgetToApplicationContext(this->shell),
		  actions, XtNumber(actions)) ;

  XtVaSetValues(this->shell, XtNallowShellResize, TRUE, NULL) ;

  /********************************************
   * Network Name label 
   */
  this->titleFormat = getTitle(this->shell) ;

  /********************************************
   * Form to hold everything
   */
  form = XtVaCreateManagedWidget("form", formWidgetClass, popup, NULL) ;
  redirectKeyboardInput(form) ;

  /********************************************
   * Pulldown menu pane 
   */
  pane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form, NULL) ;

  /**********************
   * Options menu 
   */
  menuButton = XtVaCreateManagedWidget("options", menuButtonWidgetClass, pane,
				       XtNmenuName, "optionsMenu",
				       NULL);

  menu = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, menuButton,
			      NULL) ;
  
  button = XtVaCreateManagedWidget("reset", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, resetDisplayCB, (XtPointer)this) ;

  button = XtVaCreateManagedWidget("enlarge", smeBSBObjectClass, menu, 
				   XtNlabel, "Show Parameters", NULL) ;
  enlarge = button ;

  line = XtVaCreateManagedWidget("line", smeLineObjectClass, menu, NULL) ;

  button = XtVaCreateManagedWidget("close", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)popup) ;

  /**********************
   * Net menu
   */
  menuButton = XtVaCreateManagedWidget("net", menuButtonWidgetClass, pane,
				       XtNmenuName, "netMenu",
				       NULL);
  addResetCallback(this, sensitizeRCB, (XtPointer)menuButton) ;

  menu = XtVaCreatePopupShell("netMenu", simpleMenuWidgetClass, menuButton,
			      NULL) ;
  
  button = XtVaCreateManagedWidget("test", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, doCommandLineCB, (XtPointer)"test") ;

  button = XtVaCreateManagedWidget("validate", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, doCommandLineCB, (XtPointer)"validate") ;

  button = XtVaCreateManagedWidget("randomize", smeBSBObjectClass, menu, NULL);
  XtAddCallback(button, XtNcallback, doCommandLineCB, (XtPointer)"randomize") ;
  
  button = XtVaCreateManagedWidget("saveWeights", smeBSBObjectClass,menu,NULL);
  XtAddCallback(button, XtNcallback, saveWeightsCB, (XtPointer)this) ;
  this->saveWeightsDialog = NULL ;
  
  /**********************
   * Buttons to do common things
   */
  button = XtVaCreateManagedWidget("run", commandWidgetClass, pane, NULL) ;
  XtAddCallback(button, XtNcallback, minimizeCB, (XtPointer)this) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)button) ;

  button = XtVaCreateManagedWidget("restart", commandWidgetClass, pane, NULL) ;
  XtAddCallback(button, XtNcallback, setRestartCB, (XtPointer)this) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)button) ;

  button = XtVaCreateManagedWidget("check", toggleWidgetClass, pane, NULL) ;
  XtAddCallback(button, XtNcallback, setCheckCB, (XtPointer)this) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)button) ;
  addResetCallback(this, syncCheckRCB, (XtPointer)button) ;

  /**********************
   * Frequently used parameters
   */
  table = startTable(form) ;
  this->iterations  = addTableItem("maxIterations",   &table, FALSE) ;
  this->batchSize   = addTableItem("batchSize",       &table, FALSE) ;

  table = startTable(form) ;
  this->reportInterval	= addTableItem("reportInterval",  &table, FALSE) ;
  this->saveInterval	= addTableItem("saveInterval",    &table, FALSE) ;

  addResetCallback(this, sensitizeRCB, (XtPointer)this->iterations) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)this->batchSize) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)this->reportInterval) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)this->saveInterval) ;

  this->saveCommand = createOutputCommand("minimizeEndIteration",
					  options.saveWeightsCommand,
					  options.weightFile) ;
  MOCsetInterval(this->saveCommand, options.saveInterval) ;
  if (options.appendIteration)
    MOCsetSuffixProc(this->saveCommand, weightFileSuffix, NULL) ;


  /**********************
   * Direction methods
   */
  box   = XtVaCreateManagedWidget("directionBox", boxWidgetClass, form, NULL);
  addResetCallback(this, sensitizeRCB, (XtPointer)box) ;

  label  = XtVaCreateManagedWidget("label", labelWidgetClass, box, NULL) ;
  button = createRadioGroup(directionMethods, box,
			    setDirectionCB, (XtPointer)this) ;
  addResetCallback(this, syncDirectionRCB, (XtPointer)button) ;

  /**********************
   * Step methods
   */
  box   = XtVaCreateManagedWidget("stepBox", boxWidgetClass, form, NULL) ;
  addResetCallback(this, sensitizeRCB, (XtPointer)box) ;

  label  = XtVaCreateManagedWidget("label", labelWidgetClass, box, NULL) ;
  button = createRadioGroup(stepMethods, box, setStepCB, (XtPointer)this) ;
  addResetCallback(this, syncStepRCB, (XtPointer)button) ;

  label  = XtVaCreateManagedWidget("initLabel", labelWidgetClass, box, NULL) ;
  button = createRadioGroup(initStepMethods, box,
			    setInitStepCB, (XtPointer)this) ;
  addResetCallback(this, syncInitStepRCB, (XtPointer)button) ;

  /**********************
   * Parameters
   */
  box = XtVaCreateWidget("paramBox", boxWidgetClass, form, 
			 XtNresizable, TRUE, NULL) ;
  XtAddCallback(enlarge, XtNcallback, toggleManagedCB,	(XtPointer)box) ;
  if ( options.showParams)
    toggleManagedCB(enlarge, (XtPointer)box, NULL) ;
    
  addResetCallback(this, sensitizeRCB, (XtPointer)box) ;

  selection = createMiniSelection("directionParams", directionParams, box) ;
  addResetCallback(this, showMiniParamRCB, (XtPointer)selection) ;

  selection = createMiniSelection("stepParams", stepParams, box) ;
  addResetCallback(this, showMiniParamRCB, (XtPointer)selection) ;

  selection = createNetParamSelection("netParams", box) ;
  addResetCallback(this, sensitizeNetRCB, (XtPointer)selection) ;
  addResetCallback(this, showNetParamRCB, (XtPointer)selection) ;

  XtAddCallback(popup, XtNpopupCallback, resetDisplayCB, (XtPointer)this) ;
  XtAddCallback(controller, XtNcallback, resetDisplayCB, (XtPointer)this) ;

  resetDisplay(this) ;

  XtVaSetValues(popup, XtNcreatePopupChildProc, NULL, NULL) ;
}
/**********************************************************************/

/*********************************************************************
 *	Name:		(un)blockMiniDisplayDeviceEvents
 *	Description:	blocks input to the minimize display (useful
 *			to call this if you are running very long commands)
 *	Parameters:	NONE
 *	Return Value:
 *	  export void	blockMiniDisplayDeviceEvents - NONE
 *********************************************************************/
export void	blockMiniDisplayDeviceEvents()
{
  if (minimizeDisplay == NULL || minimizeDisplay->shell == NULL)
    return ;

  blockShellDeviceEvents(minimizeDisplay->shell) ;
}
/********************************************************************/
export void	unblockMiniDisplayDeviceEvents()
{
  if (minimizeDisplay == NULL || minimizeDisplay->shell == NULL)
    return ;

  blockShellDeviceEvents(minimizeDisplay->shell) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addResetCallback
 *	Description:	adds a "ResetCallback" to the MinimizeDisplay.
 *			It will be called whenever the display has to
 *			be reset.
 *	Parameters:
 *	  MinimizeDisplay	this - the MinimizeDisplay to add to
 *	  ResetProc		callback - the callback
 *	  XtPointer		data - data for the callback
 *	Return Value:
 *	  static void	addResetCallback - NONE
 *********************************************************************/
static void	addResetCallback(this, callback, data)
  MinimizeDisplay	this ;
  ResetProc		callback ;
  XtPointer		data ;
{
  if (this->numCallbacks >= this->maxCallbacks) {
    if (this->numCallbacks == 0)
      this->maxCallbacks = 8 ;
    else
      this->maxCallbacks = 2*this->numCallbacks ;
    this->callback
      = (ResetCallback *)XtRealloc((char *)this->callback,
				   this->maxCallbacks*sizeof(ResetCallback)) ;
  }

  this->callback[this->numCallbacks].proc = callback ;
  this->callback[this->numCallbacks].data = data ;
  ++(this->numCallbacks) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createRadioGroup
 *	Description:	
 *	Parameters:
 *	  String	name - 
 *	  String	list[] -
 *	Return Value:
 *	  static Widget	createRadioGroup - 
 *********************************************************************/
static Widget	createRadioGroup(list, parent, callback, data)
  String	list[] ;
  Widget	parent ;
  XtCallbackProc	callback ;
  XtPointer		data ;
{
  Widget	button, radioGroup ;
  int		idx ;

  if (list == NULL || list[0] == NULL)
    return NULL ;

  radioGroup = XtVaCreateManagedWidget(list[0], toggleWidgetClass, parent, 
				       NULL) ;
  XtVaSetValues(radioGroup, 
		XtNradioGroup,	radioGroup, 
		XtNradioData,	list[0],
		NULL) ;
  XtAddCallback(radioGroup, XtNcallback, callback, data) ;

  for (idx = 1 ; list[idx] != NULL ; ++idx) {
    button = XtVaCreateManagedWidget(list[idx], toggleWidgetClass, parent, 
				     XtNradioData, 	list[idx],
				     XtNradioGroup, 	radioGroup, 
				     NULL) ;
    XtAddCallback(button, XtNcallback, callback, data) ;
  }

  return radioGroup ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createMiniSelection
 *	Description:	
 *	Parameters:
 *	  String	name - 
 *	  String	list[] -
 *	Return Value:
 *	  static Widget	createMiniSelection - 
 *********************************************************************/
static Widget	createMiniSelection(name, list, parent)
  String	name ;
  String	list[] ;
  Widget	parent ;
{
  Widget	selection, button ;

  selection = XtVaCreateManagedWidget(name, selectionWidgetClass, parent,
				      XtNlist, list,
				      XtNvalue, "",
				      NULL) ;
  XtAddCallback(selection, XtNcallback, showMiniParamCB, (XtPointer)selection);
  XtSelectionHighlight(selection, 0) ;

  XtSelectionAddButton(selection, "save", 
		       saveMiniParamCB,	(XtPointer)selection);
  XtSelectionAddButton(selection, "help",	
		       showHelpCB,	(XtPointer)selection) ;

  XtInstallAllAccelerators(XtNameToWidget(selection, "value"), selection) ;

  return selection ;
}
/********************************************************************/


/**********************************************************************
 *	Name:		createNetParamSelection
 *	Description:	creates a list widget with a list of network
 *			parameters and some buttons for updating the 
 *			values and closing the widget
 *	Parameters:	
 *		Widget	popup - the popup shell we are going in
 *	Return Value:
 *		NONE
 **********************************************************************/
static Widget	createNetParamSelection(name, parent)
  String	name ;
  Widget	parent ;
{
  static String	*objectList ;
  Widget	selection, button ;

  objectList = listNetParamsShort() ;

  selection = XtVaCreateManagedWidget(name, selectionWidgetClass, parent,
				      XtNlist,	objectList,
				      XtNvalue,	"",
				      NULL) ;
  XtAddCallback(selection, XtNcallback, showNetParamCB, (XtPointer)selection) ;
  XtSelectionHighlight(selection, 0) ;

  XtSelectionAddButton(selection, "save", 
		       saveNetParamCB,	(XtPointer)selection) ;
  XtSelectionAddButton(selection, "help", 
		       showHelpCB,	(XtPointer)selection) ;

  XtInstallAllAccelerators(XtNameToWidget(selection, "value"), selection) ;

  return selection ;
}
/**********************************************************************/


/********************************************************************/
static String	stringFromFlag(flag, flags, strings)
  int		flag ;
  int		flags[] ;
  String	strings[] ;
{
  int		idx ;

  for (idx = 0 ; flags[idx] && flags[idx] != flag ; ++idx)
    ;
  return strings[idx] ;
}
/********************************************************************/
static int	flagFromString(string, flags, strings)
  String	string ;
  int		flags[] ;
  String	strings[] ;
{
  int		idx ;

  for (idx = 0 ; strings[idx] && strcmp(strings[idx], string) ; ++idx)
    ;
  return flags[idx] ;
}
/********************************************************************/
static int	defaultStepMethod(directionMethod)
  int		directionMethod ;
{
  int		stepMethod ;
  switch(directionMethod) {
  case MZCONJGRAD:
  case MZCONJGRADRST:
  case MZRUDICG:
    stepMethod = MZRAYSLS ;
    break ;
  case MZSTEEPEST:
  case MZMOMENTUM:
  case MZDELTABARDELTA:
  case MZQUICKPROP:
  default:
    stepMethod =  MZFIXEDSTEP;
    break ;
  }
  return stepMethod ;
}
/********************************************************************/



/*********************************************************************
 *	Name:		resetDisplayCB
 *	Description:	resets the display (i.e. sets the sensitivity
 *			of the widgets and rebuilds nets and example
 *			lists
 *	Parameters:
 *	  Widget	widget     - the widget the cb is attached to
 *	  XtPointer	clientData - UNUSED
 *	  XtPointer	callData   - UNUSED
 *	Return Value:
 *	  static void resetDisplayCB - NONE
 *********************************************************************/
static void resetDisplayCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  resetDisplay((MinimizeDisplay)clientData) ;
}
/********************************************************************/
static void	resetDisplay(this)
  MinimizeDisplay	this ;
{
  String		name, value ;
  int			idx, batchSize ;

  getNetValue(NAME, &name) ;
  widgetPrintf(this->shell, this->titleFormat, name) ;
  this->mz = currentMinimizeRecord() ;

  if (netExists() && getNetValue(BATCH_SIZE, &batchSize)) {
    XtSetSensitive(this->batchSize, TRUE) ;
    if (batchSize <= 0)
      widgetPrintf(this->batchSize, "%s", "all") ;
    else
      widgetPrintf(this->batchSize, "%d", batchSize) ;
  } else {
    XtSetSensitive(this->batchSize, FALSE) ;
  }

  if (this->mz) 
    widgetPrintf(this->reportInterval, 
		 "%d", MAX(1, this->mz->repetitionCount)) ;

  if (this->saveCommand) {
    int	interval = MOCinterval(this->saveCommand) ;
    if (interval)
      widgetPrintf(this->saveInterval, "%d", interval) ;
    else
      widgetPrintf(this->saveInterval, "%s", "no save") ;
  }

  for (idx = 0 ; idx < this->numCallbacks ; ++idx)
    doResetCB(this, idx) ;
}
/********************************************************************/
static void	syncCheckRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  if (this && this->mz)
    XtVaSetValues((Widget)data, 
		  XtNstate, !this->mz->doNotCheckFunctions, NULL) ;
}
/********************************************************************/
static void	sensitizeRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  XtSetSensitive((Widget)data, this->mz ? TRUE : FALSE) ;
}
/********************************************************************/
static void	sensitizeNetRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  XtSetSensitive((Widget)data, netExists() ? TRUE : FALSE) ;
}
/********************************************************************/
static void	syncDirectionRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  Widget	radioGroup = (Widget)data ;
  String	method ;

  if (this->mz == NULL)
    return ;

  method = stringFromFlag(this->mz->directionMethod, 
			  directionMethodFlags, directionMethods) ;
  if (method)
    XawToggleSetCurrent(radioGroup, method) ;
}
/********************************************************************/
static void	syncStepRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  Widget	radioGroup = (Widget)data ;
  String	method ;

  if (this->mz == NULL)
    return ;

  if (this->mz->stepMethod == 0)
    this->mz->stepMethod = defaultStepMethod(this->mz->directionMethod) ;

  method = stringFromFlag(this->mz->stepMethod, 
			  stepMethodFlags, stepMethods) ;
  if (method)
    XawToggleSetCurrent(radioGroup, method) ;
}
/********************************************************************/
static void	syncInitStepRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  Widget	radioGroup = (Widget)data ;
  String	method ;

  if (this->mz == NULL)
    return ;

  method = stringFromFlag(this->mz->initialStepMethod, 
			  initStepMethodFlags, initStepMethods) ;
  if (method)
    XawToggleSetCurrent(radioGroup, method) ;
}
/********************************************************************/


/**********************************************************************/
static void 	setTextOptionsAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  if (minimizeDisplay == NULL)
    return ;

  minimizeDisplay->mz = currentMinimizeRecord() ;
  if (minimizeDisplay->mz == NULL)
    return ;

  setTextOptions(minimizeDisplay) ;
}
/**********************************************************************/
static void	setTextOptions(this)
  MinimizeDisplay	this ;
{
  String		string ;
  int			batchSize, interval ;

  string = getText(this->iterations) ;
  if (string)
    this->iterationsNum = atol(string) ;
  else {
    this->iterationsNum = 0 ;
    if (this->mz->maxIterations)
      widgetPrintf(this->iterations, "%d", 0) ;
  }

  string = getText(this->batchSize) ;
  batchSize = string ? MAX(atol(string), 0) : 0 ;
  setNetValue(BATCH_SIZE, &batchSize) ;
  getNetValue(BATCH_SIZE, &batchSize) ;
  if (batchSize <= 0)
    widgetPrintf(this->batchSize, "%s", "all") ;
  else
    widgetPrintf(this->batchSize, "%d", batchSize) ;

  string = getText(this->reportInterval) ;
  if (string)
    this->mz->repetitionCount = atol(string) ;
  widgetPrintf(this->reportInterval, "%d", MAX(1, this->mz->repetitionCount)) ;

  if (this->saveCommand) {
    string   = getText(this->saveInterval) ;
    interval = string ? MAX(0, atol(string)) : 0 ;

    MOCsetInterval(this->saveCommand, interval) ;
    interval = MOCinterval(this->saveCommand) ;
    if (interval)
      widgetPrintf(this->saveInterval, "%d", interval) ;
    else
      widgetPrintf(this->saveInterval, "%s", "no save") ;
  }
}
/**********************************************************************/
static  void	minimizeThis(this)
  MinimizeDisplay	this ;
{
  jmp_buf		env ;
  void			*old_env ;
  int			id ;

  if (setjmp(env)) {
    unblockShellDeviceEvents(this->shell) ;
    popdownInterruptDialog(id) ;
    IReportError(stderr);
    return ;
  } else
    old_env = ISetAbortEnv(env) ;

  blockShellDeviceEvents(this->shell) ;
  id = popupInterruptDialog(this->shell) ;
  minimize(this->mz, 0, this->iterationsNum) ;
  popdownInterruptDialog(id) ;
  unblockShellDeviceEvents(this->shell) ;

  ISetAbortEnv(old_env) ;

  if (this->mz->mzVerbosity) {
    fprintf(dout, "minimize: code= %d nFE= %d f= %.8g\n",
	    this->mz->mzResultCode, this->mz->nFuncEvals, this->mz->E);
    fprintf(dout, "minimize: %s\n", this->mz->mzResult);
  }
}
/**********************************************************************/
static void 	minimizeAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  minimizeDisplay->mz = currentMinimizeRecord() ;
  if (minimizeDisplay->mz == NULL)
    return ;

  setTextOptions(minimizeDisplay) ;
  minimizeThis(minimizeDisplay) ;
}
/**********************************************************************/
static void	minimizeCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;

  this->mz = currentMinimizeRecord() ;
  if (this->mz == NULL)
    return ;

  setTextOptions(this) ;
  minimizeThis(this) ;
}
/**********************************************************************/
static void	setRestartCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;

  if (this->mz)
    this->mz->xLen = 0.0 ;

  fprintf(stdout, 
	  "Discarding old direction and line search information.\n") ;
}
/**********************************************************************/
static void	setCheckCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;
  char			**helpStrings ;
  Boolean		state ;

  XtVaGetValues(widget, XtNstate, &state, NULL) ;

  if (this->mz)
    this->mz->doNotCheckFunctions = !state ;

  helpStrings = getHelpStrings("doNotCheckFunctions") ;

  if (helpStrings != NULL) {
    while (*helpStrings != NULL)
      fprintf(stdout, "%s\n", *helpStrings++);
  } else {
    fprintf(stdout, "Sorry, no help available\n") ;
  }
  fprintf(stdout, 
	  "These checks will be %s.\n", state ? "done" : "skipped") ;
}
/**********************************************************************/
static void	setDirectionCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;
  Widget		radioGroup ;
  String		string ;
  int			flag ;

  XtVaGetValues(widget, XtNradioGroup, &radioGroup, NULL) ;
  if (radioGroup == NULL) 
    return ;
  
  string = (String)XawToggleGetCurrent(radioGroup) ;

  if (this->mz == NULL || string == NULL)
    return ;

  flag = flagFromString(string, directionMethodFlags, directionMethods) ;

  if (flag)
    this->mz->directionMethod = flag ;
  
  this->mz->stepMethod = defaultStepMethod(this->mz->directionMethod) ;
  resetDisplay(this) ;
}
/**********************************************************************/
static void	setStepCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;
  Widget		radioGroup ;
  String		string ;
  int			flag ;

  XtVaGetValues(widget, XtNradioGroup, &radioGroup, NULL) ;
  if (radioGroup == NULL) 
    return ;
  
  string = (String)XawToggleGetCurrent(radioGroup) ;

  if (this->mz == NULL || string == NULL)
    return ;
  
  flag = flagFromString(string, stepMethodFlags, stepMethods) ;

  if (flag)
    this->mz->stepMethod = flag ;
}
/**********************************************************************/
static void	setInitStepCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this = (MinimizeDisplay)clientData ;
  Widget		radioGroup ;
  String		string ;
  int			flag ;

  XtVaGetValues(widget, XtNradioGroup, &radioGroup, NULL) ;
  if (radioGroup == NULL) 
    return ;
  
  string = (String)XawToggleGetCurrent(radioGroup) ;

  if (this->mz == NULL || string == NULL)
    return ;
  
  flag = flagFromString(string, initStepMethodFlags, initStepMethods) ;

  if (flag)
    this->mz->initialStepMethod = flag ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		showMiniParmCB
 *	Description:	shows the value of the selected item in the
 *			selection widget. Returns if no network exists.
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XtPointer	clientData - the selection box
 *	  XtPointer	callData   - the call data UNUSED
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	showMiniParamCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  resetMiniParam((Widget)clientData) ;
}
/**********************************************************************/
static void	showMiniParamRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  resetMiniParam((Widget)data) ;
}
/********************************************************************/
static void	resetMiniParam(this)
  Widget	this ;
{
  XawListReturnStruct	*selected ;
  String		fullName, value ;

  if (currentMinimizeRecord() == NULL) {
    XtSelectionSetValueString(this, "") ;
    return ;
  }

  selected = XtSelectionGetSelected(this) ;
  if (NO_SELECTION(selected))
    return ;

  fullName = minimizeParamLongName(selected->string) ;
  value    = getSimpleObjectValue(fullName, NULL) ;

  XtSelectionSetValueString(this, value) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		saveMiniParamCB
 *	Description:	saves the value in the value widget of a 
 *			selection box to the selected network parameter
 *			field. Returns if no network exists.
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XPointer	clientData - the selection box widget
 *	  XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	saveMiniParamCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget		selectionWidget = (Widget)clientData ;
  XawListReturnStruct	*selected ;
  String		name, value ;

  if (currentMinimizeRecord() == NULL)
    return ;

  selected = XtSelectionGetSelected(selectionWidget) ;
  if (NO_SELECTION(selected))
    return ;

  name  = minimizeParamLongName(selected->string) ;
  value = XtSelectionGetValueString(selectionWidget) ;
  if (value != NULL) {
    saveSimpleObjectValue(name, value) ;
  }
  resetMiniParam(selectionWidget) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		showNetParamCB
 *	Description:	shows the value of the selected item in the
 *			selection widget. Returns if no network exists.
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XtPointer	clientData - the selection box
 *	  XtPointer	callData   - the call data UNUSED
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	showNetParamCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  resetNetParam((Widget)clientData) ;
}
/**********************************************************************/
static void	showNetParamRCB(this, data)
  MinimizeDisplay	this ;
  XtPointer		data ;
{
  resetNetParam((Widget)data) ;
}
/**********************************************************************/
static void 	resetNetParam(this)
  Widget	this ;
{
  XawListReturnStruct	*selected ;
  String		fullName, value ;

  if (!netExists()) {
    XtSelectionSetValueString(this, "") ;
    return ;
  }

  selected = XtSelectionGetSelected(this) ;
  if (NO_SELECTION(selected))
    return ;

  fullName = netParamLongName(selected->string) ;
  value    = getSimpleObjectValue(fullName, NULL) ;

  XtSelectionSetValueString(this, value) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		saveNetParamCB
 *	Description:	saves the value in the value widget of a 
 *			selection box to the selected network parameter
 *			field. Returns if no network exists.
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XPointer	clientData - the selection box widget
 *	  XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	saveNetParamCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget		selectionWidget = (Widget)clientData ;
  XawListReturnStruct	*selected ;
  String		name, value ;
  
  if (!netExists())
    return ;

  selected = XtSelectionGetSelected(selectionWidget) ;
  if (NO_SELECTION(selected))
    return ;

  name  = netParamLongName(selected->string) ;
  value = XtSelectionGetValueString(selectionWidget) ;
  if (value != NULL) {
    saveSimpleObjectValue(name, value) ;
  }
  resetNetParam(selectionWidget) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		showHelpCB
 *	Description:	prints the "extraHelp" string for a variable.
 *			the extraHelp string is registered by the itf
 *			module 
 *	Parameters:
 *	  Widget	widget - the widget the cb is attached to
 *	  XtPointer	clientData - the Selection widget to get the
 *				variable name from.
 *	  XtPointer	callData - UNUSED
 *	Return Value:
 *	  static void	showHelpCB - NONE
 *********************************************************************/
static void	showHelpCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget		selectionWidget = (Widget)clientData ;
  XawListReturnStruct	*selected ;
  char			**helpStrings ;

  selected = XtSelectionGetSelected(selectionWidget) ;
  if (NO_SELECTION(selected)) {
    fprintf(stdout, "\nSelect an item from the list, then press \"help\"\n") ;
    return ;
  }

  helpStrings = getHelpStrings(selected->string) ;

  fprintf(stdout, "\n%s:\n", selected->string) ;
  if (helpStrings != NULL) {
    while (*helpStrings != NULL)
      fprintf(stdout, "%s\n", *helpStrings++);
  } else {
    fprintf(stdout, "Sorry, no help available\n") ;
  }
}
/********************************************************************/
static void	toggleManagedCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	box = (Widget)clientData ;
  String	label = getLabel(widget) ;

  if (strcmp(label, "Show Parameters") == 0) { 
    XtManageChild(box) ;
    widgetPrintf(widget, "Hide Parameters") ;
  } else {
    XtUnmanageChild(box) ;
    widgetPrintf(widget, "Show Parameters") ;
  }
  XtFree(label) ;
}
/********************************************************************/
static void	doCommandLineCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  String	command = (String)clientData ;
  jmp_buf	env ;
  void		*old_env ;

  if (setjmp(env)) {
    unblockDeviceEvents() ;
    IReportError(stderr);
    return ;
  } else
    old_env = ISetAbortEnv(env) ;

  blockDeviceEvents() ;
  IDoCommandLine(command) ;
  unblockDeviceEvents() ;

  ISetAbortEnv(old_env) ;
}
/********************************************************************/
static void	changeSaveWeightsCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this   = (MinimizeDisplay)clientData ;
  SaveWeightsDialog	dialog = this->saveWeightsDialog ;
  String		string ;
  Boolean		set ;

  XtPopdown(dialog->shell) ;

  if (this->saveCommand == NULL)
    return ;

  string = getText(dialog->command) ;
  if (string)
    MOCsetCommand(this->saveCommand, string) ;

  string = getText(dialog->fileName) ;
  if (string)
    MOCsetFile(this->saveCommand, string) ;

  XtVaGetValues(dialog->addSuffix, XtNstate, &set, NULL) ;
  if (set)
    MOCsetSuffixProc(this->saveCommand, weightFileSuffix, NULL) ;
  else
    MOCsetSuffixProc(this->saveCommand, NULL, NULL) ;

  string = getText(dialog->interval) ;
  if (string)
    MOCsetInterval(this->saveCommand, MAX(0, atol(string))) ;

  resetDisplay(this) ;
}
/********************************************************************/
static void	saveWeightsCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MinimizeDisplay	this   = (MinimizeDisplay)clientData ;
  SaveWeightsDialog	dialog = this->saveWeightsDialog ;

  if (dialog == NULL) {
    Widget	form, table, button ;

    dialog = (SaveWeightsDialog)XtMalloc(sizeof(SaveWeightsDialogRec)) ;

    dialog->shell = XtVaCreatePopupShell("miscShell", 
					 transientShellWidgetClass, 
					 this->shell, NULL) ;
    form = XtVaCreateManagedWidget("form", formWidgetClass, dialog->shell, 
				   NULL) ;
    table = startTable(form) ;
    dialog->command  = addTableItem("command", &table, FALSE) ;
    dialog->fileName = addTableItem("fileName", &table, TRUE) ;
    dialog->interval = addTableItem("interval", &table, FALSE) ;
    
    dialog->addSuffix = XtNameToWidget(form, "fileNameToggle") ;

    button = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				     XtNfromVert,	dialog->interval,
				     NULL) ;
    XtAddCallback(button, XtNcallback, changeSaveWeightsCB, (XtPointer)this) ;

    button = XtVaCreateManagedWidget("cancel", commandWidgetClass, form,
				     XtNfromVert,	dialog->interval,
				     XtNfromHoriz,	button,
				     NULL) ;
    XtAddCallback(button, XtNcallback, 
		  popdownWidgetCB, (XtPointer)dialog->shell) ;

    XtInstallAllAccelerators(form, form) ;
    this->saveWeightsDialog = dialog ;
  }
  if (this->saveCommand) {
    int	interval = MOCinterval(this->saveCommand) ;
    if (interval)
      widgetPrintf(dialog->interval, "%d", interval) ;
    else
      widgetPrintf(dialog->interval, "%s", "no save") ;
    widgetPrintf(dialog->fileName, MOCfileBaseName(this->saveCommand)) ;
    widgetPrintf(dialog->command,  MOCcommandString(this->saveCommand)) ;
    XtVaSetValues(dialog->addSuffix, 
		  XtNstate, MOCsuffix(this->saveCommand) ? TRUE : FALSE, NULL);
    popupCentered(dialog->shell, XtGrabExclusive) ;
  }
}
/********************************************************************/

/********************************************************************/
static String	weightFileSuffix(this, data)
  OutputCommand	this ;
  void		*data ;
{
  return ".${currentNet.currentEpoch}" ;
}
/********************************************************************/
